/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.util;

import com.hazelcast.instance.Node;
import com.hazelcast.logging.ILogger;
import com.hazelcast.util.HealthMonitorLevel;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.lang.reflect.Method;
import java.util.logging.Level;

public class HealthMonitor
extends Thread {
    private final ILogger logger;
    private final Node node;
    private final Runtime runtime;
    private final OperatingSystemMXBean osMxBean;
    private final HealthMonitorLevel logLevel;
    private double treshold = 70.0;
    private static final String[] UNITS = new String[]{"", "K", "M", "G", "T", "P", "E"};

    public HealthMonitor(Node node, HealthMonitorLevel logLevel) {
        super(node.threadGroup, node.getThreadNamePrefix("HealthMonitor"));
        this.setDaemon(true);
        this.node = node;
        this.logger = node.getLogger(HealthMonitor.class.getName());
        this.runtime = Runtime.getRuntime();
        this.osMxBean = ManagementFactory.getOperatingSystemMXBean();
        this.logLevel = logLevel;
    }

    @Override
    public void run() {
        while (this.node.isActive()) {
            switch (this.logLevel) {
                case OFF: {
                    break;
                }
                case NOISY: {
                    HealthMetrics metrics = new HealthMetrics();
                    this.logger.log(Level.INFO, metrics.toString());
                    break;
                }
                case SILENT: {
                    HealthMetrics metrics = new HealthMetrics();
                    if (!metrics.exceedsTreshold()) break;
                    this.logger.log(Level.INFO, metrics.toString());
                    break;
                }
                default: {
                    throw new IllegalStateException("unrecognized logLevel:" + (Object)((Object)this.logLevel));
                }
            }
            try {
                Thread.sleep(10000L);
            }
            catch (InterruptedException e) {
                return;
            }
        }
    }

    private static Long get(OperatingSystemMXBean mbean, String methodName, Long defaultValue) {
        try {
            Method method = mbean.getClass().getMethod(methodName, new Class[0]);
            method.setAccessible(true);
            Object value = method.invoke((Object)mbean, new Object[0]);
            if (value == null) {
                return defaultValue;
            }
            if (value instanceof Integer) {
                return (long)((Integer)value);
            }
            if (value instanceof Double) {
                double v = (Double)value;
                return Math.round(v * 100.0);
            }
            if (value instanceof Long) {
                return (Long)value;
            }
            return defaultValue;
        }
        catch (Exception e) {
            return defaultValue;
        }
    }

    public static String percentageString(double p) {
        return String.format("%.2f", p) + "%";
    }

    public static String bytesToString(long bytes) {
        for (int i = 6; i > 0; --i) {
            double step = Math.pow(1024.0, i);
            if (!((double)bytes > step)) continue;
            return String.format("%3.1f%s", (double)bytes / step, UNITS[i]);
        }
        return Long.toString(bytes);
    }

    public class HealthMetrics {
        long memoryFree;
        long memoryTotal;
        long memoryUsed;
        long memoryMax;
        double memoryUsedOfTotalPercentage;
        double memoryUsedOfMaxPercentage;
        double processCpuLoad;
        double systemLoadAverage;
        double systemCpuLoad;

        public HealthMetrics() {
            this.memoryFree = HealthMonitor.this.runtime.freeMemory();
            this.memoryTotal = HealthMonitor.this.runtime.totalMemory();
            this.memoryUsed = this.memoryTotal - this.memoryFree;
            this.memoryMax = HealthMonitor.this.runtime.maxMemory();
            this.memoryUsedOfTotalPercentage = 100.0 * (double)this.memoryUsed / (double)this.memoryTotal;
            this.memoryUsedOfMaxPercentage = 100.0 * (double)this.memoryUsed / (double)this.memoryMax;
            this.processCpuLoad = HealthMonitor.get(HealthMonitor.this.osMxBean, "getProcessCpuLoad", -1L).longValue();
            this.systemLoadAverage = HealthMonitor.get(HealthMonitor.this.osMxBean, "getSystemLoadAverage", -1L).longValue();
            this.systemCpuLoad = HealthMonitor.get(HealthMonitor.this.osMxBean, "getSystemCpuLoad", -1L).longValue();
        }

        public boolean exceedsTreshold() {
            if (this.memoryUsedOfMaxPercentage > HealthMonitor.this.treshold) {
                return true;
            }
            if (this.processCpuLoad > HealthMonitor.this.treshold) {
                return true;
            }
            if (this.systemCpuLoad > HealthMonitor.this.treshold) {
                return true;
            }
            return this.systemCpuLoad > HealthMonitor.this.treshold;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("\nmemory ");
            sb.append("used=").append(HealthMonitor.bytesToString(this.memoryUsed)).append(", ");
            sb.append("free=").append(HealthMonitor.bytesToString(this.memoryFree)).append(", ");
            sb.append("total=").append(HealthMonitor.bytesToString(this.memoryTotal)).append(", ");
            sb.append("max=").append(HealthMonitor.bytesToString(this.memoryMax)).append(", ");
            sb.append("used/total=").append(HealthMonitor.percentageString(this.memoryUsedOfTotalPercentage)).append(" ");
            sb.append("used/max=").append(HealthMonitor.percentageString(this.memoryUsedOfMaxPercentage));
            sb.append("\n");
            sb.append("cpu ");
            sb.append("process-load=").append(String.format("%.2f", this.processCpuLoad)).append("%, ");
            sb.append("system-load=").append(String.format("%.2f", this.systemCpuLoad)).append("%, ");
            sb.append("system-loadaverage=").append(String.format("%.2f", this.systemLoadAverage)).append("%");
            return sb.toString();
        }
    }
}

