/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.geode.internal.offheap;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.geode.logging.internal.executors.LoggingExecutors;

public class NonRealTimeStatsUpdater {
  private final Runnable updateTask;
  private final ScheduledExecutorService updateNonRealTimeStatsExecutor;
  private final AtomicReference<ScheduledFuture<?>> updateNonRealTimeStatsFuture =
      new AtomicReference<>();

  NonRealTimeStatsUpdater(Runnable updateTask) {
    this.updateTask = updateTask;
    updateNonRealTimeStatsExecutor =
        LoggingExecutors.newSingleThreadScheduledExecutor("Update Freelist Stats thread");

  }

  void start(int updateOffHeapStatsFrequencyMs) {
    updateNonRealTimeStatsFuture
        .set(updateNonRealTimeStatsExecutor.scheduleAtFixedRate(updateTask, 0,
            updateOffHeapStatsFrequencyMs, TimeUnit.MILLISECONDS));
  }

  void stop() {
    updateNonRealTimeStatsFuture.get().cancel(true);
    updateNonRealTimeStatsExecutor.shutdown();
  }
}
