/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.s3guard;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.impl.DirectoryPolicy;
import org.apache.hadoop.fs.s3a.s3guard.DirListingMetadata;
import org.apache.hadoop.fs.s3a.s3guard.DynamoDBMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.LocalMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.MetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadata;
import org.apache.hadoop.fs.shell.CommandFormat;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"management tools"})
@InterfaceStability.Evolving
public abstract class S3GuardTool
extends Configured
implements Tool {
    private static final Logger LOG = LoggerFactory.getLogger(S3GuardTool.class);
    private static final String NAME = "s3guard";
    private static final String COMMON_USAGE = "When possible and not overridden by more specific options, metadata\nrepository information will be inferred from the S3A URL (if provided)\n\nGeneric options supported are:\n  -conf <config file> - specify an application configuration file\n  -D <property=value> - define a value for a given property\n";
    private static final String USAGE = "s3guard [command] [OPTIONS] [s3a://BUCKET]\n\nCommands: \n\tinit - initialize metadata repository\n\tdestroy - destroy Metadata Store data (all data in S3 is preserved)\n\timport - import metadata from existing S3 data\n\tbucket-info - provide/check S3Guard information about a specific bucket\n\tdiff - report on delta between S3 and repository\n\tprune - truncate older metadata from repository (all data in S3 is preserved)\n\tset-capacity - Alter metadata store IO capacity\n";
    private static final String DATA_IN_S3_IS_PRESERVED = "(all data in S3 is preserved)";
    static final int SUCCESS = 0;
    static final int INVALID_ARGUMENT = 40;
    static final int E_USAGE = 42;
    static final int ERROR = -1;
    static final int E_BAD_STATE = 46;
    static final int E_NOT_FOUND = 44;
    private S3AFileSystem filesystem;
    private MetadataStore store;
    private final CommandFormat commandFormat;
    public static final String META_FLAG = "meta";
    public static final String DAYS_FLAG = "days";
    public static final String HOURS_FLAG = "hours";
    public static final String MINUTES_FLAG = "minutes";
    public static final String SECONDS_FLAG = "seconds";
    public static final String REGION_FLAG = "region";
    public static final String READ_FLAG = "read";
    public static final String WRITE_FLAG = "write";
    private static S3GuardTool command;

    public abstract String getUsage();

    protected S3GuardTool(Configuration conf, String ... opts) {
        super(conf);
        this.commandFormat = new CommandFormat(0, Integer.MAX_VALUE, opts);
        this.commandFormat.addOptionWithValue(META_FLAG);
        this.commandFormat.addOptionWithValue(REGION_FLAG);
    }

    abstract String getName();

    void parseDynamoDBRegion(List<String> paths) throws IOException {
        boolean hasS3Path;
        Configuration conf = this.getConf();
        String fromCli = this.getCommandFormat().getOptValue(REGION_FLAG);
        String fromConf = conf.get("fs.s3a.s3guard.ddb.region");
        boolean bl = hasS3Path = !paths.isEmpty();
        if (fromCli != null) {
            if (fromCli.isEmpty()) {
                throw S3GuardTool.invalidArgs("No region provided with -region flag", new Object[0]);
            }
            if (hasS3Path) {
                throw S3GuardTool.invalidArgs("Providing both an S3 path and the -region flag is not supported. If you need to specify a different region than the S3 bucket, configure fs.s3a.s3guard.ddb.region", new Object[0]);
            }
            conf.set("fs.s3a.s3guard.ddb.region", fromCli);
            return;
        }
        if (fromConf != null) {
            if (fromConf.isEmpty()) {
                throw S3GuardTool.invalidArgs("No region provided with config %s", "fs.s3a.s3guard.ddb.region");
            }
            return;
        }
        if (hasS3Path) {
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            return;
        }
        throw S3GuardTool.invalidArgs("No region found from -region flag, config, or S3 bucket", new Object[0]);
    }

    MetadataStore initMetadataStore(boolean forceCreate) throws IOException {
        Configuration conf;
        block14: {
            block13: {
                if (this.getStore() != null) {
                    return this.getStore();
                }
                conf = this.filesystem == null ? this.getConf() : this.filesystem.getConf();
                String metaURI = this.getCommandFormat().getOptValue(META_FLAG);
                if (metaURI == null || metaURI.isEmpty()) break block13;
                URI uri = URI.create(metaURI);
                LOG.info("Create metadata store: {}", (Object)(uri + " scheme: " + uri.getScheme()));
                switch (uri.getScheme().toLowerCase(Locale.ENGLISH)) {
                    case "local": {
                        this.setStore(new LocalMetadataStore());
                        break;
                    }
                    case "dynamodb": {
                        this.setStore(new DynamoDBMetadataStore());
                        conf.set("fs.s3a.s3guard.ddb.table", uri.getAuthority());
                        if (forceCreate) {
                            conf.setBoolean("fs.s3a.s3guard.ddb.table.create", true);
                            break;
                        }
                        break block14;
                    }
                    default: {
                        throw new IOException(String.format("Metadata store %s is not supported", uri));
                    }
                }
                break block14;
            }
            this.setStore(new DynamoDBMetadataStore());
            if (forceCreate) {
                conf.setBoolean("fs.s3a.s3guard.ddb.table.create", true);
            }
        }
        if (this.filesystem == null) {
            this.getStore().initialize(conf);
        } else {
            this.getStore().initialize(this.filesystem);
        }
        LOG.info("Metadata store {} is initialized.", (Object)this.getStore());
        return this.getStore();
    }

    void initS3AFileSystem(String path) throws IOException {
        URI uri = S3GuardTool.toUri(path);
        Configuration conf = new Configuration(this.getConf());
        String nullStore = NullMetadataStore.class.getName();
        conf.set("fs.s3a.metadatastore.impl", nullStore);
        String bucket = uri.getHost();
        S3AUtils.setBucketOption(conf, bucket, "fs.s3a.metadatastore.impl", "org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore");
        String updatedBucketOption = S3AUtils.getBucketOption(conf, bucket, "fs.s3a.metadatastore.impl");
        LOG.debug("updated bucket store option {}", (Object)updatedBucketOption);
        Preconditions.checkState((boolean)"org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore".equals(updatedBucketOption), (String)"Expected bucket option to be %s but was %s", (Object[])new Object[]{"org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore", updatedBucketOption});
        FileSystem fs = FileSystem.newInstance((URI)uri, (Configuration)conf);
        if (!(fs instanceof S3AFileSystem)) {
            throw S3GuardTool.invalidArgs("URI %s is not a S3A file system: %s", uri, fs.getClass().getName());
        }
        this.filesystem = (S3AFileSystem)fs;
    }

    List<String> parseArgs(String[] args) {
        return this.getCommandFormat().parse(args, 1);
    }

    protected S3AFileSystem getFilesystem() {
        return this.filesystem;
    }

    protected void setFilesystem(S3AFileSystem filesystem) {
        this.filesystem = filesystem;
    }

    @VisibleForTesting
    public MetadataStore getStore() {
        return this.store;
    }

    @VisibleForTesting
    protected void setStore(MetadataStore store) {
        Preconditions.checkNotNull((Object)store);
        this.store = store;
    }

    protected CommandFormat getCommandFormat() {
        return this.commandFormat;
    }

    public final int run(String[] args) throws Exception {
        return this.run(args, System.out);
    }

    public abstract int run(String[] var1, PrintStream var2) throws Exception;

    protected static URI toUri(String s3Path) {
        URI uri;
        try {
            uri = new URI(s3Path);
        }
        catch (URISyntaxException e) {
            throw S3GuardTool.invalidArgs("Not a valid fileystem path: %s", s3Path);
        }
        return uri;
    }

    private static void printHelp() {
        if (command == null) {
            S3GuardTool.errorln("Usage: hadoop s3guard [command] [OPTIONS] [s3a://BUCKET]\n\nCommands: \n\tinit - initialize metadata repository\n\tdestroy - destroy Metadata Store data (all data in S3 is preserved)\n\timport - import metadata from existing S3 data\n\tbucket-info - provide/check S3Guard information about a specific bucket\n\tdiff - report on delta between S3 and repository\n\tprune - truncate older metadata from repository (all data in S3 is preserved)\n\tset-capacity - Alter metadata store IO capacity\n");
            S3GuardTool.errorln("\tperform S3Guard metadata store administrative commands.");
        } else {
            S3GuardTool.errorln("Usage: hadoop " + command.getUsage());
        }
        S3GuardTool.errorln();
        S3GuardTool.errorln(COMMON_USAGE);
    }

    private static void errorln() {
        System.err.println();
    }

    private static void errorln(String x) {
        System.err.println(x);
    }

    private static void println(PrintStream out, String format, Object ... args) {
        out.println(String.format(format, args));
    }

    protected static void printStoreDiagnostics(PrintStream out, MetadataStore store) throws IOException {
        Map<String, String> diagnostics = store.getDiagnostics();
        out.println("Metadata Store Diagnostics:");
        for (Map.Entry<String, String> entry : diagnostics.entrySet()) {
            S3GuardTool.println(out, "\t%s=%s", entry.getKey(), entry.getValue());
        }
    }

    protected static ExitUtil.ExitException storeNotFound(FileNotFoundException e) {
        return new ExitUtil.ExitException(44, e.toString(), (Throwable)e);
    }

    protected static ExitUtil.ExitException invalidArgs(String format, Object ... args) {
        return new ExitUtil.ExitException(40, String.format(format, args));
    }

    protected static ExitUtil.ExitException badState(String format, Object ... args) {
        return new ExitUtil.ExitException(46, String.format(format, args));
    }

    public static int run(Configuration conf, String ... args) throws Exception {
        String[] otherArgs = new GenericOptionsParser(conf, args).getRemainingArgs();
        if (otherArgs.length == 0) {
            S3GuardTool.printHelp();
            throw new ExitUtil.ExitException(42, "No arguments provided");
        }
        String subCommand = otherArgs[0];
        LOG.debug("Executing command {}", (Object)subCommand);
        switch (subCommand) {
            case "init": {
                command = new Init(conf);
                break;
            }
            case "destroy": {
                command = new Destroy(conf);
                break;
            }
            case "import": {
                command = new Import(conf);
                break;
            }
            case "bucket-info": {
                command = new BucketInfo(conf);
                break;
            }
            case "diff": {
                command = new Diff(conf);
                break;
            }
            case "prune": {
                command = new Prune(conf);
                break;
            }
            case "set-capacity": {
                command = new SetCapacity(conf);
                break;
            }
            default: {
                S3GuardTool.printHelp();
                throw new ExitUtil.ExitException(42, "Unknown command " + subCommand);
            }
        }
        return ToolRunner.run((Configuration)conf, (Tool)command, (String[])otherArgs);
    }

    public static void main(String[] args) {
        try {
            int ret = S3GuardTool.run(new Configuration(), args);
            S3GuardTool.exit(ret, "");
        }
        catch (CommandFormat.UnknownOptionException e) {
            S3GuardTool.errorln(e.getMessage());
            S3GuardTool.printHelp();
            S3GuardTool.exit(42, e.getMessage());
        }
        catch (ExitUtil.ExitException e) {
            S3GuardTool.exit(e.getExitCode(), e.toString());
        }
        catch (Throwable e) {
            e.printStackTrace(System.err);
            S3GuardTool.exit(-1, e.toString());
        }
    }

    protected static void exit(int status, String text) {
        ExitUtil.terminate((int)status, (String)text);
    }

    static class BucketInfo
    extends S3GuardTool {
        public static final String NAME = "bucket-info";
        public static final String GUARDED_FLAG = "guarded";
        public static final String UNGUARDED_FLAG = "unguarded";
        public static final String AUTH_FLAG = "auth";
        public static final String NONAUTH_FLAG = "nonauth";
        public static final String ENCRYPTION_FLAG = "encryption";
        public static final String MAGIC_FLAG = "magic";
        public static final String MARKERS_FLAG = "markers";
        public static final String MARKERS_AWARE = "aware";
        public static final String PURPOSE = "provide/check S3Guard information about a specific bucket";
        private static final String USAGE = "bucket-info [OPTIONS] s3a://BUCKET\n\tprovide/check S3Guard information about a specific bucket\n\nCommon options:\n  -guarded - Require S3Guard\n  -encryption (none, sse-s3, sse-kms) - Require encryption policy\n  -markers (aware, keep, delete, authoritative) - directory markers policy\n";
        @VisibleForTesting
        public static final String IS_MARKER_AWARE = "The S3A connector can read data in S3 buckets where directory markers%nare not deleted (optional with later hadoop releases),%nand with buckets where they are.%n";

        public BucketInfo(Configuration conf) {
            super(conf, GUARDED_FLAG, UNGUARDED_FLAG, AUTH_FLAG, NONAUTH_FLAG);
            CommandFormat format = this.getCommandFormat();
            format.addOptionWithValue(ENCRYPTION_FLAG);
            format.addOptionWithValue(MARKERS_FLAG);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            String desiredEncryption;
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                S3GuardTool.errorln(this.getUsage());
                throw BucketInfo.invalidArgs("No bucket specified", new Object[0]);
            }
            String s3Path = paths.get(0);
            S3AFileSystem fs = (S3AFileSystem)FileSystem.newInstance((URI)BucketInfo.toUri(s3Path), (Configuration)this.getConf());
            this.setFilesystem(fs);
            Configuration conf = fs.getConf();
            URI fsUri = fs.getUri();
            MetadataStore store = fs.getMetadataStore();
            S3GuardTool.println(out, "Filesystem %s", new Object[]{fsUri});
            S3GuardTool.println(out, "Location: %s", new Object[]{fs.getBucketLocation()});
            boolean usingS3Guard = !(store instanceof NullMetadataStore);
            boolean authMode = false;
            if (usingS3Guard) {
                out.printf("Filesystem %s is using S3Guard with store %s%n", fsUri, store.toString());
                this.printOption(out, "Authoritative S3Guard", "fs.s3a.metadatastore.authoritative", "false");
                authMode = conf.getBoolean("fs.s3a.metadatastore.authoritative", false);
                BucketInfo.printStoreDiagnostics(out, store);
            } else {
                S3GuardTool.println(out, "Filesystem %s is not using S3Guard", new Object[]{fsUri});
            }
            S3GuardTool.println(out, "%nS3A Client", new Object[0]);
            String endpoint = conf.getTrimmed("fs.s3a.endpoint", "");
            S3GuardTool.println(out, "\tEndpoint: %s=%s", new Object[]{"fs.s3a.endpoint", StringUtils.isNotEmpty((String)endpoint) ? endpoint : "(unset)"});
            String encryption = this.printOption(out, "\tEncryption", "fs.s3a.server-side-encryption-algorithm", "none");
            this.printOption(out, "\tInput seek policy", "fs.s3a.experimental.input.fadvise", "normal");
            CommandFormat commands = this.getCommandFormat();
            if (usingS3Guard) {
                if (commands.getOpt(UNGUARDED_FLAG)) {
                    throw BucketInfo.badState("S3Guard is enabled for %s", fsUri);
                }
                if (commands.getOpt(AUTH_FLAG) && !authMode) {
                    throw BucketInfo.badState("S3Guard is enabled for %s, but not in authoritative mode", fsUri);
                }
                if (commands.getOpt(NONAUTH_FLAG) && authMode) {
                    throw BucketInfo.badState("S3Guard is enabled in authoritative mode for %s", fsUri);
                }
            } else if (commands.getOpt(GUARDED_FLAG)) {
                throw BucketInfo.badState("S3Guard is not enabled for %s", fsUri);
            }
            if (StringUtils.isNotEmpty((String)(desiredEncryption = this.getCommandFormat().getOptValue(ENCRYPTION_FLAG))) && !desiredEncryption.equalsIgnoreCase(encryption)) {
                throw BucketInfo.badState("Bucket %s: required encryption is %s but actual encryption is %s", fsUri, desiredEncryption, encryption);
            }
            this.processMarkerOption(out, fs, this.getCommandFormat().getOptValue(MARKERS_FLAG));
            out.flush();
            return 0;
        }

        private void processMarkerOption(PrintStream out, S3AFileSystem fs, String marker) {
            DirectoryPolicy markerPolicy = fs.getDirectoryMarkerPolicy();
            String desc = markerPolicy.describe();
            S3GuardTool.println(out, "%nThe directory marker policy is \"%s\"%n", new Object[]{desc});
            DirectoryPolicy.MarkerPolicy mp = markerPolicy.getMarkerPolicy();
            String desiredMarker = marker == null ? "" : marker.trim();
            String optionName = mp.getOptionName();
            if (!desiredMarker.isEmpty()) {
                if (MARKERS_AWARE.equalsIgnoreCase(desiredMarker)) {
                    S3GuardTool.println(out, IS_MARKER_AWARE, new Object[0]);
                    S3GuardTool.println(out, "Available Policies: delete", new Object[0]);
                } else if (!optionName.equalsIgnoreCase(desiredMarker)) {
                    throw BucketInfo.badState("Bucket %s: required marker policy is \"%s\" but actual policy is \"%s\"", fs.getUri(), desiredMarker, optionName);
                }
            }
        }

        private String printOption(PrintStream out, String description, String key, String defVal) {
            String t = this.getFilesystem().getConf().getTrimmed(key, defVal);
            S3GuardTool.println(out, "%s: %s=%s", new Object[]{description, key, t});
            return t;
        }
    }

    static class Prune
    extends S3GuardTool {
        public static final String NAME = "prune";
        public static final String PURPOSE = "truncate older metadata from repository (all data in S3 is preserved)";
        private static final String USAGE = "prune [OPTIONS] [s3a://BUCKET]\n\ttruncate older metadata from repository (all data in S3 is preserved)\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Prune(Configuration conf) {
            super(conf, new String[0]);
            CommandFormat format = this.getCommandFormat();
            format.addOptionWithValue(S3GuardTool.DAYS_FLAG);
            format.addOptionWithValue(S3GuardTool.HOURS_FLAG);
            format.addOptionWithValue(S3GuardTool.MINUTES_FLAG);
            format.addOptionWithValue(S3GuardTool.SECONDS_FLAG);
        }

        @VisibleForTesting
        void setMetadataStore(MetadataStore ms) {
            Preconditions.checkNotNull((Object)ms);
            this.setStore(ms);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        private long getDeltaComponent(TimeUnit unit, String arg) {
            String raw = this.getCommandFormat().getOptValue(arg);
            if (raw == null || raw.isEmpty()) {
                return 0L;
            }
            Long parsed = Long.parseLong(raw);
            return unit.toMillis(parsed);
        }

        @Override
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            List<String> paths = this.parseArgs(args);
            try {
                this.parseDynamoDBRegion(paths);
            }
            catch (ExitUtil.ExitException e) {
                S3GuardTool.errorln(USAGE);
                throw e;
            }
            this.initMetadataStore(false);
            Configuration conf = this.getConf();
            long confDelta = conf.getLong("fs.s3a.s3guard.cli.prune.age", 0L);
            long cliDelta = 0L;
            cliDelta += this.getDeltaComponent(TimeUnit.DAYS, S3GuardTool.DAYS_FLAG);
            cliDelta += this.getDeltaComponent(TimeUnit.HOURS, S3GuardTool.HOURS_FLAG);
            cliDelta += this.getDeltaComponent(TimeUnit.MINUTES, S3GuardTool.MINUTES_FLAG);
            if (confDelta <= 0L && (cliDelta += this.getDeltaComponent(TimeUnit.SECONDS, S3GuardTool.SECONDS_FLAG)) <= 0L) {
                S3GuardTool.errorln("You must specify a positive age for metadata to prune.");
            }
            long delta = confDelta;
            if (cliDelta > 0L) {
                delta = cliDelta;
            }
            long now = System.currentTimeMillis();
            long divide = now - delta;
            this.getStore().prune(divide);
            out.flush();
            return 0;
        }
    }

    static class Diff
    extends S3GuardTool {
        public static final String NAME = "diff";
        public static final String PURPOSE = "report on delta between S3 and repository";
        private static final String USAGE = "diff [OPTIONS] s3a://BUCKET\n\treport on delta between S3 and repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";
        private static final String SEP = "\t";
        static final String S3_PREFIX = "S3";
        static final String MS_PREFIX = "MS";

        Diff(Configuration conf) {
            super(conf, new String[0]);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        private static String formatFileStatus(FileStatus status) {
            return String.format("%s%s%d%s%s", status.isDirectory() ? "D" : "F", SEP, status.getLen(), SEP, status.getPath().toString());
        }

        private static boolean differ(FileStatus thisOne, FileStatus thatOne) {
            Preconditions.checkArgument((thisOne != null || thatOne != null ? 1 : 0) != 0);
            return thisOne == null || thatOne == null || thisOne.getLen() != thatOne.getLen() || thisOne.isDirectory() != thatOne.isDirectory() || !thisOne.isDirectory() && thisOne.getModificationTime() != thatOne.getModificationTime();
        }

        private static void printDiff(FileStatus msStatus, FileStatus s3Status, PrintStream out) {
            Preconditions.checkArgument((msStatus != null || s3Status != null ? 1 : 0) != 0);
            if (msStatus != null && s3Status != null) {
                Preconditions.checkArgument((boolean)msStatus.getPath().equals((Object)s3Status.getPath()), (Object)String.format("The path from metadata store and s3 are different: ms=%s s3=%s", msStatus.getPath(), s3Status.getPath()));
            }
            if (Diff.differ(msStatus, s3Status)) {
                if (s3Status != null) {
                    S3GuardTool.println(out, "%s%s%s", new Object[]{S3_PREFIX, SEP, Diff.formatFileStatus(s3Status)});
                }
                if (msStatus != null) {
                    S3GuardTool.println(out, "%s%s%s", new Object[]{MS_PREFIX, SEP, Diff.formatFileStatus(msStatus)});
                }
            }
        }

        private void compareDir(FileStatus msDir, FileStatus s3Dir, PrintStream out) throws IOException {
            DirListingMetadata dirMeta;
            Preconditions.checkArgument((msDir != null || s3Dir != null ? 1 : 0) != 0);
            if (msDir != null && s3Dir != null) {
                Preconditions.checkArgument((boolean)msDir.getPath().equals((Object)s3Dir.getPath()), (Object)String.format("The path from metadata store and s3 are different: ms=%s s3=%s", msDir.getPath(), s3Dir.getPath()));
            }
            HashMap<Path, FileStatus> s3Children = new HashMap<Path, FileStatus>();
            if (s3Dir != null && s3Dir.isDirectory()) {
                for (FileStatus status : this.getFilesystem().listStatus(s3Dir.getPath())) {
                    s3Children.put(status.getPath(), status);
                }
            }
            HashMap<Path, FileStatus> msChildren = new HashMap<Path, FileStatus>();
            if (msDir != null && msDir.isDirectory() && (dirMeta = this.getStore().listChildren(msDir.getPath())) != null) {
                for (PathMetadata meta : dirMeta.getListing()) {
                    FileStatus status = meta.getFileStatus();
                    msChildren.put(status.getPath(), status);
                }
            }
            HashSet allPaths = new HashSet(s3Children.keySet());
            allPaths.addAll(msChildren.keySet());
            for (Path path : allPaths) {
                FileStatus s3Status = (FileStatus)s3Children.get(path);
                FileStatus msStatus = (FileStatus)msChildren.get(path);
                Diff.printDiff(msStatus, s3Status, out);
                if ((s3Status == null || !s3Status.isDirectory()) && (msStatus == null || !msStatus.isDirectory())) continue;
                this.compareDir(msStatus, s3Status, out);
            }
            out.flush();
        }

        private void compareRoot(Path path, PrintStream out) throws IOException {
            Path qualified = this.getFilesystem().qualify(path);
            FileStatus s3Status = null;
            try {
                s3Status = this.getFilesystem().getFileStatus(qualified);
            }
            catch (FileNotFoundException fileNotFoundException) {
                // empty catch block
            }
            PathMetadata meta = this.getStore().get(qualified);
            FileStatus msStatus = meta != null && !meta.isDeleted() ? meta.getFileStatus() : null;
            this.compareDir(msStatus, s3Status, out);
        }

        @Override
        @VisibleForTesting
        public int run(String[] args, PrintStream out) throws IOException {
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                out.println(USAGE);
                throw Diff.invalidArgs("no arguments", new Object[0]);
            }
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            this.initMetadataStore(false);
            URI uri = Diff.toUri(s3Path);
            Path root = uri.getPath().isEmpty() ? new Path("/") : new Path(uri.getPath());
            root = this.getFilesystem().qualify(root);
            this.compareRoot(root, out);
            out.flush();
            return 0;
        }
    }

    static class Import
    extends S3GuardTool {
        public static final String NAME = "import";
        public static final String PURPOSE = "import metadata from existing S3 data";
        private static final String USAGE = "import [OPTIONS] [s3a://BUCKET]\n\timport metadata from existing S3 data\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";
        private final Set<Path> dirCache = new HashSet<Path>();

        Import(Configuration conf) {
            super(conf, new String[0]);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        private void putParentsIfNotPresent(FileStatus f) throws IOException {
            Preconditions.checkNotNull((Object)f);
            for (Path parent = f.getPath().getParent(); parent != null; parent = parent.getParent()) {
                if (this.dirCache.contains(parent)) {
                    return;
                }
                FileStatus dir = DynamoDBMetadataStore.makeDirStatus(parent, f.getOwner());
                this.getStore().put(new PathMetadata(dir));
                this.dirCache.add(parent);
            }
        }

        private long importDir(FileStatus status) throws IOException {
            Preconditions.checkArgument((boolean)status.isDirectory());
            RemoteIterator<LocatedFileStatus> it = this.getFilesystem().listFilesAndEmptyDirectories(status.getPath(), true);
            long items = 0L;
            while (it.hasNext()) {
                FileStatus child;
                LocatedFileStatus located = (LocatedFileStatus)it.next();
                if (located.isDirectory()) {
                    child = DynamoDBMetadataStore.makeDirStatus(located.getPath(), located.getOwner());
                    this.dirCache.add(child.getPath());
                } else {
                    child = new S3AFileStatus(located.getLen(), located.getModificationTime(), located.getPath(), located.getBlockSize(), located.getOwner());
                }
                this.putParentsIfNotPresent(child);
                this.getStore().put(new PathMetadata(child));
                ++items;
            }
            return items;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                S3GuardTool.errorln(this.getUsage());
                throw Import.invalidArgs("no arguments", new Object[0]);
            }
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            URI uri = Import.toUri(s3Path);
            String filePath = uri.getPath();
            if (filePath.isEmpty()) {
                filePath = "/";
            }
            Path path = new Path(filePath);
            FileStatus status = this.getFilesystem().getFileStatus(path);
            try {
                this.initMetadataStore(false);
            }
            catch (FileNotFoundException e) {
                throw Import.storeNotFound(e);
            }
            long items = 1L;
            if (status.isFile()) {
                PathMetadata meta = new PathMetadata(status);
                this.getStore().put(meta);
            } else {
                items = this.importDir(status);
            }
            S3GuardTool.println(out, "Inserted %d items into Metadata Store", new Object[]{items});
            return 0;
        }
    }

    static class Destroy
    extends S3GuardTool {
        public static final String NAME = "destroy";
        public static final String PURPOSE = "destroy Metadata Store data (all data in S3 is preserved)";
        private static final String USAGE = "destroy [OPTIONS] [s3a://BUCKET]\n\tdestroy Metadata Store data (all data in S3 is preserved)\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Destroy(Configuration conf) {
            super(conf, new String[0]);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            List<String> paths = this.parseArgs(args);
            try {
                this.parseDynamoDBRegion(paths);
            }
            catch (ExitUtil.ExitException e) {
                S3GuardTool.errorln(USAGE);
                throw e;
            }
            try {
                this.initMetadataStore(false);
            }
            catch (FileNotFoundException e) {
                S3GuardTool.println(out, "Metadata Store does not exist.", new Object[0]);
                LOG.debug("Failed to bind to store to be destroyed", (Throwable)e);
                return 0;
            }
            Preconditions.checkState((this.getStore() != null ? 1 : 0) != 0, (Object)"Metadata Store is not initialized");
            this.getStore().destroy();
            S3GuardTool.println(out, "Metadata store is deleted.", new Object[0]);
            return 0;
        }
    }

    static class SetCapacity
    extends S3GuardTool {
        public static final String NAME = "set-capacity";
        public static final String PURPOSE = "Alter metadata store IO capacity";
        private static final String USAGE = "set-capacity [OPTIONS] [s3a://BUCKET]\n\tAlter metadata store IO capacity\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -read UNIT - Provisioned read throughput units\n  -write UNIT - Provisioned write through put units\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        SetCapacity(Configuration conf) {
            super(conf, new String[0]);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.READ_FLAG);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.WRITE_FLAG);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            String writeCap;
            List<String> paths = this.parseArgs(args);
            HashMap<String, String> options = new HashMap<String, String>();
            String readCap = this.getCommandFormat().getOptValue(S3GuardTool.READ_FLAG);
            if (StringUtils.isNotEmpty((String)readCap)) {
                S3GuardTool.println(out, "Read capacity set to %s", new Object[]{readCap});
                options.put("fs.s3a.s3guard.ddb.table.capacity.read", readCap);
            }
            if (StringUtils.isNotEmpty((String)(writeCap = this.getCommandFormat().getOptValue(S3GuardTool.WRITE_FLAG)))) {
                S3GuardTool.println(out, "Write capacity set to %s", new Object[]{writeCap});
                options.put("fs.s3a.s3guard.ddb.table.capacity.write", writeCap);
            }
            try {
                this.parseDynamoDBRegion(paths);
            }
            catch (ExitUtil.ExitException e) {
                S3GuardTool.errorln(USAGE);
                throw e;
            }
            MetadataStore store = this.initMetadataStore(false);
            store.updateParameters(options);
            SetCapacity.printStoreDiagnostics(out, store);
            return 0;
        }
    }

    static class Init
    extends S3GuardTool {
        public static final String NAME = "init";
        public static final String PURPOSE = "initialize metadata repository";
        private static final String USAGE = "init [OPTIONS] [s3a://BUCKET]\n\tinitialize metadata repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n  -read UNIT - Provisioned read throughput units\n  -write UNIT - Provisioned write through put units\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Init(Configuration conf) {
            super(conf, new String[0]);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.READ_FLAG);
            this.getCommandFormat().addOptionWithValue(S3GuardTool.WRITE_FLAG);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        @Override
        public int run(String[] args, PrintStream out) throws Exception {
            String writeCap;
            List<String> paths = this.parseArgs(args);
            String readCap = this.getCommandFormat().getOptValue(S3GuardTool.READ_FLAG);
            if (readCap != null && !readCap.isEmpty()) {
                int readCapacity = Integer.parseInt(readCap);
                this.getConf().setInt("fs.s3a.s3guard.ddb.table.capacity.read", readCapacity);
            }
            if ((writeCap = this.getCommandFormat().getOptValue(S3GuardTool.WRITE_FLAG)) != null && !writeCap.isEmpty()) {
                int writeCapacity = Integer.parseInt(writeCap);
                this.getConf().setInt("fs.s3a.s3guard.ddb.table.capacity.write", writeCapacity);
            }
            try {
                this.parseDynamoDBRegion(paths);
            }
            catch (ExitUtil.ExitException e) {
                S3GuardTool.errorln(USAGE);
                throw e;
            }
            MetadataStore store = this.initMetadataStore(true);
            Init.printStoreDiagnostics(out, store);
            return 0;
        }
    }
}

