/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hop.www;

import static junit.framework.Assert.assertFalse;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import org.apache.hop.core.gui.Point;
import org.apache.hop.core.logging.HopLogStore;
import org.apache.hop.core.logging.ILogChannel;
import org.apache.hop.pipeline.Pipeline;
import org.apache.hop.pipeline.PipelineConfiguration;
import org.apache.hop.pipeline.PipelineExecutionConfiguration;
import org.apache.hop.pipeline.PipelineMeta;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.owasp.encoder.Encode;

public class PrepareExecutionPipelineServletTest {
  private PipelineMap mockPipelineMap;

  private PrepareExecutionPipelineServlet prepareExecutionPipelineServlet;

  @Before
  public void setup() {
    mockPipelineMap = mock(PipelineMap.class);
    prepareExecutionPipelineServlet = new PrepareExecutionPipelineServlet(mockPipelineMap);
  }

  @Test
  public void testPausePipelineServletEscapesHtmlWhenPipelineNotFound()
      throws ServletException, IOException {
    HttpServletRequest mockHttpServletRequest = mock(HttpServletRequest.class);
    HttpServletResponse mockHttpServletResponse = mock(HttpServletResponse.class);

    StringWriter out = new StringWriter();
    PrintWriter printWriter = new PrintWriter(out);

    Mockito.spy(Encode.class);
    when(mockHttpServletRequest.getContextPath())
        .thenReturn(PrepareExecutionPipelineServlet.CONTEXT_PATH);
    when(mockHttpServletRequest.getParameter(anyString()))
        .thenReturn(ServletTestUtils.BAD_STRING_TO_TEST);
    when(mockHttpServletResponse.getWriter()).thenReturn(printWriter);

    prepareExecutionPipelineServlet.doGet(mockHttpServletRequest, mockHttpServletResponse);
    assertFalse(ServletTestUtils.hasBadText(ServletTestUtils.getInsideOfTag("H1", out.toString())));
  }

  @Test
  public void testPausePipelineServletEscapesHtmlWhenPipelineFound()
      throws ServletException, IOException {
    HopLogStore.init();
    HttpServletRequest mockHttpServletRequest = mock(HttpServletRequest.class);
    HttpServletResponse mockHttpServletResponse = mock(HttpServletResponse.class);
    Pipeline mockPipeline = mock(Pipeline.class);
    PipelineConfiguration mockPipelineConf = mock(PipelineConfiguration.class);
    PipelineMeta mockPipelineMeta = mock(PipelineMeta.class);
    PipelineExecutionConfiguration mockPipelineExecutionConf =
        mock(PipelineExecutionConfiguration.class);

    ILogChannel mockChannelInterface = mock(ILogChannel.class);
    StringWriter out = new StringWriter();
    PrintWriter printWriter = new PrintWriter(out);

    Mockito.spy(Encode.class);
    when(mockHttpServletRequest.getContextPath())
        .thenReturn(PrepareExecutionPipelineServlet.CONTEXT_PATH);
    when(mockHttpServletRequest.getParameter(anyString()))
        .thenReturn(ServletTestUtils.BAD_STRING_TO_TEST);
    when(mockHttpServletResponse.getWriter()).thenReturn(printWriter);
    when(mockPipelineMap.getPipeline(any(HopServerObjectEntry.class))).thenReturn(mockPipeline);
    when(mockPipelineMap.getConfiguration(any(HopServerObjectEntry.class)))
        .thenReturn(mockPipelineConf);
    when(mockPipelineConf.getPipelineExecutionConfiguration())
        .thenReturn(mockPipelineExecutionConf);
    when(mockPipeline.getLogChannel()).thenReturn(mockChannelInterface);
    when(mockPipeline.getPipelineMeta()).thenReturn(mockPipelineMeta);
    when(mockPipelineMeta.getMaximum()).thenReturn(new Point(10, 10));

    prepareExecutionPipelineServlet.doGet(mockHttpServletRequest, mockHttpServletResponse);
    assertFalse(ServletTestUtils.hasBadText(ServletTestUtils.getInsideOfTag("H1", out.toString())));
  }
}
