/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.Futures;
import java.lang.management.ManagementFactory;
import java.net.UnknownHostException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.WeakHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.LongSupplier;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.exceptions.RequestFailureReason;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.gms.VersionedValue;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.NoPayload;
import org.apache.cassandra.net.RequestCallback;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationCoordinator {
    private static final Logger logger = LoggerFactory.getLogger(MigrationCoordinator.class);
    private static final Future<Void> FINISHED_FUTURE = Futures.immediateFuture(null);
    private static LongSupplier getUptimeFn = () -> ManagementFactory.getRuntimeMXBean().getUptime();
    private static final int MIGRATION_DELAY_IN_MS = CassandraRelevantProperties.MIGRATION_DELAY.getInt();
    private static final int MAX_OUTSTANDING_VERSION_REQUESTS = 3;
    public static final MigrationCoordinator instance = new MigrationCoordinator();
    private static final long BACKOFF_DELAY_MS = CassandraRelevantProperties.SCHEMA_PULL_BACKOFF_DELAY_MS.getInt();
    private final WeakHashMap<InetAddressAndPort, Long> lastPullAttemptTimestamps = new WeakHashMap();
    private static final Set<UUID> IGNORED_VERSIONS = MigrationCoordinator.getIgnoredVersions();
    private final Map<UUID, VersionInfo> versionInfo = new HashMap<UUID, VersionInfo>();
    private final Map<InetAddressAndPort, UUID> endpointVersions = new HashMap<InetAddressAndPort, UUID>();
    private final AtomicInteger inflightTasks = new AtomicInteger();
    private final Set<InetAddressAndPort> ignoredEndpoints = MigrationCoordinator.getIgnoredEndpoints();

    @VisibleForTesting
    public static void setUptimeFn(LongSupplier supplier) {
        getUptimeFn = supplier;
    }

    private static ImmutableSet<UUID> getIgnoredVersions() {
        String s = CassandraRelevantProperties.IGNORED_SCHEMA_CHECK_VERSIONS.getString();
        if (s == null || s.isEmpty()) {
            return ImmutableSet.of();
        }
        ImmutableSet.Builder versions = ImmutableSet.builder();
        for (String version : s.split(",")) {
            versions.add((Object)UUID.fromString(version));
        }
        return versions.build();
    }

    private static Set<InetAddressAndPort> getIgnoredEndpoints() {
        HashSet<InetAddressAndPort> endpoints = new HashSet<InetAddressAndPort>();
        String s = CassandraRelevantProperties.IGNORED_SCHEMA_CHECK_ENDPOINTS.getString();
        if (s == null || s.isEmpty()) {
            return endpoints;
        }
        for (String endpoint : s.split(",")) {
            try {
                endpoints.add(InetAddressAndPort.getByName(endpoint));
            }
            catch (UnknownHostException e) {
                throw new RuntimeException(e);
            }
        }
        return endpoints;
    }

    public void start() {
        int interval = CassandraRelevantProperties.SCHEMA_PULL_INTERVAL_MS.getInt();
        ScheduledExecutors.scheduledTasks.scheduleWithFixedDelay(this::pullUnreceivedSchemaVersions, interval, interval, TimeUnit.MILLISECONDS);
    }

    public synchronized void reset() {
        logger.info("Resetting migration coordinator...");
        this.endpointVersions.clear();
        this.clearVersionsInfo();
    }

    synchronized List<Future<Void>> pullUnreceivedSchemaVersions() {
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (VersionInfo info : this.versionInfo.values()) {
            if (info.wasReceived() || info.outstandingRequests.size() > 0) {
                logger.trace("Skipping pull of schema {} because it has been already recevied, or it is being received ({})", (Object)info.version, (Object)info);
                continue;
            }
            Future<Void> future = this.maybePullSchema(info);
            if (future == null || future == FINISHED_FUTURE) continue;
            futures.add(future);
        }
        return futures;
    }

    synchronized Future<Void> maybePullSchema(VersionInfo info) {
        if (info.endpoints.isEmpty() || info.wasReceived() || !this.shouldPullSchema(info.version)) {
            logger.trace("Not pulling schema {} because it was received, there is no endpoint to provide it, or we should not pull it ({})", (Object)info.version, (Object)info);
            return FINISHED_FUTURE;
        }
        if (info.outstandingRequests.size() >= this.getMaxOutstandingVersionRequests()) {
            logger.trace("Not pulling schema {} because the number of outstanding requests has been exceeded ({} >= {})", new Object[]{info.version, info.outstandingRequests.size(), this.getMaxOutstandingVersionRequests()});
            return FINISHED_FUTURE;
        }
        int isize = info.requestQueue.size();
        for (int i = 0; i < isize; ++i) {
            InetAddressAndPort endpoint = info.requestQueue.remove();
            if (!info.endpoints.contains(endpoint)) {
                logger.trace("Skipping request of schema {} from {} because the endpoint does not have that schema any longer", (Object)info.version, (Object)endpoint);
                continue;
            }
            if (this.shouldPullFromEndpoint(endpoint) && info.outstandingRequests.add(endpoint)) {
                return this.scheduleSchemaPull(endpoint, info);
            }
            logger.trace("Could not pull schema {} from {} - the request will be added back to the queue", (Object)info.version, (Object)endpoint);
            info.requestQueue.offer(endpoint);
        }
        return null;
    }

    public synchronized Map<UUID, Set<InetAddressAndPort>> outstandingVersions() {
        HashMap<UUID, Set<InetAddressAndPort>> map = new HashMap<UUID, Set<InetAddressAndPort>>();
        for (VersionInfo info : this.versionInfo.values()) {
            if (info.wasReceived()) continue;
            map.put(info.version, (Set<InetAddressAndPort>)ImmutableSet.copyOf(info.endpoints));
        }
        return map;
    }

    @VisibleForTesting
    protected VersionInfo getVersionInfoUnsafe(UUID version) {
        return this.versionInfo.get(version);
    }

    @VisibleForTesting
    protected int getMaxOutstandingVersionRequests() {
        return 3;
    }

    @VisibleForTesting
    protected boolean isAlive(InetAddressAndPort endpoint) {
        return FailureDetector.instance.isAlive(endpoint);
    }

    @VisibleForTesting
    protected boolean shouldPullSchema(UUID version) {
        if (Schema.instance.getVersion() == null) {
            logger.debug("Not pulling schema {} because the local schama version is not known yet", (Object)version);
            return false;
        }
        if (Schema.instance.isSameVersion(version)) {
            logger.debug("Not pulling schema {} because it is the same as the local schema", (Object)version);
            return false;
        }
        return true;
    }

    private static boolean is30Compatible(int version) {
        return version == 12 || version == 11;
    }

    @VisibleForTesting
    protected boolean shouldPullFromEndpoint(InetAddressAndPort endpoint) {
        if (endpoint.equals(FBUtilities.getBroadcastAddressAndPort())) {
            logger.trace("Not pulling schema from local endpoint");
            return false;
        }
        EndpointState state = Gossiper.instance.getEndpointStateForEndpoint(endpoint);
        if (state == null) {
            logger.trace("Not pulling schema from endpoint {} because its state is unknown", (Object)endpoint);
            return false;
        }
        VersionedValue releaseVersionValue = state.getApplicationState(ApplicationState.RELEASE_VERSION);
        if (releaseVersionValue == null) {
            return false;
        }
        String releaseVersion = releaseVersionValue.value;
        String ourMajorVersion = FBUtilities.getReleaseVersionMajor();
        if (!releaseVersion.startsWith(ourMajorVersion)) {
            logger.debug("Not pulling schema from {} because release version in Gossip is not major version {}, it is {}", new Object[]{endpoint, ourMajorVersion, releaseVersion});
            return false;
        }
        if (!MessagingService.instance().versions.knows(endpoint)) {
            logger.debug("Not pulling schema from {} because their messaging version is unknown", (Object)endpoint);
            return false;
        }
        if (MessagingService.instance().versions.getRaw(endpoint) != 12) {
            logger.debug("Not pulling schema from {} because their schema format is incompatible", (Object)endpoint);
            return false;
        }
        if (Gossiper.instance.isGossipOnlyMember(endpoint)) {
            logger.debug("Not pulling schema from {} because it's a gossip only member", (Object)endpoint);
            return false;
        }
        return true;
    }

    @VisibleForTesting
    protected boolean shouldPullImmediately(InetAddressAndPort endpoint, UUID version) {
        if (Schema.instance.isEmpty() || getUptimeFn.getAsLong() < (long)MIGRATION_DELAY_IN_MS) {
            logger.debug("Immediately submitting migration task for {}, schema versions: local={}, remote={}", new Object[]{endpoint, Schema.schemaVersionToString(Schema.instance.getVersion()), Schema.schemaVersionToString(version)});
            return true;
        }
        return false;
    }

    @VisibleForTesting
    protected boolean isLocalVersion(UUID version) {
        return Schema.instance.isSameVersion(version);
    }

    synchronized boolean shouldApplySchemaFor(VersionInfo info) {
        if (info.wasReceived()) {
            return false;
        }
        return !this.isLocalVersion(info.version);
    }

    public synchronized Future<Void> reportEndpointVersion(InetAddressAndPort endpoint, UUID version) {
        logger.debug("Reported schema {} at endpoint {}", (Object)version, (Object)endpoint);
        if (this.ignoredEndpoints.contains(endpoint) || IGNORED_VERSIONS.contains(version)) {
            this.endpointVersions.remove(endpoint);
            this.removeEndpointFromVersion(endpoint, null);
            logger.debug("Discarding endpoint {} or schema {} because either endpoint or schema version were marked as ignored", (Object)endpoint, (Object)version);
            return FINISHED_FUTURE;
        }
        UUID current = this.endpointVersions.put(endpoint, version);
        if (current != null && current.equals(version)) {
            logger.trace("Skipping report of schema {} from {} because we already know that", (Object)version, (Object)endpoint);
            return FINISHED_FUTURE;
        }
        VersionInfo info = this.versionInfo.computeIfAbsent(version, VersionInfo::new);
        if (this.isLocalVersion(version)) {
            info.markReceived();
            logger.trace("Schema {} from {} has been marked as recevied because it is equal the local schema", (Object)version, (Object)endpoint);
        } else {
            info.requestQueue.addFirst(endpoint);
        }
        info.endpoints.add(endpoint);
        logger.trace("Added endpoint {} to schema {}: {}", new Object[]{endpoint, info.version, info});
        this.removeEndpointFromVersion(endpoint, current);
        return this.maybePullSchema(info);
    }

    public Future<Void> reportEndpointVersion(InetAddressAndPort endpoint, EndpointState state) {
        if (state == null) {
            return FINISHED_FUTURE;
        }
        UUID version = state.getSchemaVersion();
        if (version == null) {
            return FINISHED_FUTURE;
        }
        return this.reportEndpointVersion(endpoint, version);
    }

    private synchronized void removeEndpointFromVersion(InetAddressAndPort endpoint, UUID version) {
        if (version == null) {
            return;
        }
        VersionInfo info = this.versionInfo.get(version);
        if (info == null) {
            return;
        }
        info.endpoints.remove(endpoint);
        logger.trace("Removed endpoint {} from schema {}: {}", new Object[]{endpoint, version, info});
        if (info.endpoints.isEmpty()) {
            info.waitQueue.signalAll();
            this.versionInfo.remove(version);
            logger.trace("Removed schema info: {}", (Object)info);
        }
    }

    private synchronized void clearVersionsInfo() {
        Iterator<Map.Entry<UUID, VersionInfo>> it = this.versionInfo.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<UUID, VersionInfo> entry = it.next();
            it.remove();
            entry.getValue().waitQueue.signalAll();
        }
    }

    public synchronized void removeAndIgnoreEndpoint(InetAddressAndPort endpoint) {
        logger.debug("Removing and ignoring endpoint {}", (Object)endpoint);
        Preconditions.checkArgument((endpoint != null ? 1 : 0) != 0);
        this.ignoredEndpoints.add(endpoint);
        ImmutableSet versions = ImmutableSet.copyOf(this.versionInfo.keySet());
        for (UUID version : versions) {
            this.removeEndpointFromVersion(endpoint, version);
        }
    }

    Future<Void> scheduleSchemaPull(InetAddressAndPort endpoint, VersionInfo info) {
        FutureTask<Object> task = new FutureTask<Object>(() -> this.pullSchema(new Callback(endpoint, info)), null);
        if (this.shouldPullImmediately(endpoint, info.version)) {
            long now;
            long nextAttempt = this.lastPullAttemptTimestamps.getOrDefault(endpoint, 0L) + BACKOFF_DELAY_MS;
            if (nextAttempt <= (now = System.currentTimeMillis())) {
                logger.debug("Pulling {} immediately from {}", (Object)info, (Object)endpoint);
                MigrationCoordinator.submitToMigrationIfNotShutdown(task);
            } else {
                long delay = nextAttempt - now;
                logger.debug("Previous pull of {} from {} failed. Postponing next attempt for {}ms", new Object[]{info, endpoint, delay});
                ScheduledExecutors.nonPeriodicTasks.schedule(() -> MigrationCoordinator.submitToMigrationIfNotShutdown(task), delay, TimeUnit.MILLISECONDS);
            }
        } else {
            logger.debug("Postponing pull of {} from {} for {}ms", new Object[]{info, endpoint, MIGRATION_DELAY_IN_MS});
            ScheduledExecutors.nonPeriodicTasks.schedule(() -> MigrationCoordinator.submitToMigrationIfNotShutdown(task), (long)MIGRATION_DELAY_IN_MS, TimeUnit.MILLISECONDS);
        }
        return task;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Future<?> submitToMigrationIfNotShutdown(Runnable task) {
        boolean skipped = false;
        try {
            if (Stage.MIGRATION.executor().isShutdown() || Stage.MIGRATION.executor().isTerminated()) {
                skipped = true;
                Future<?> future = null;
                return future;
            }
            Future<?> future = Stage.MIGRATION.submit(task);
            return future;
        }
        catch (RejectedExecutionException ex) {
            skipped = true;
            Future<?> future = null;
            return future;
        }
        finally {
            if (skipped) {
                logger.info("Skipped scheduled pulling schema from other nodes: the MIGRATION executor service has been shutdown.");
            }
        }
    }

    @VisibleForTesting
    protected void mergeSchemaFrom(InetAddressAndPort endpoint, Collection<Mutation> mutations) {
        Schema.instance.mergeAndAnnounceVersion(mutations);
    }

    private void pullSchema(Callback callback) {
        this.lastPullAttemptTimestamps.put(callback.endpoint, System.currentTimeMillis());
        if (!this.isAlive(callback.endpoint)) {
            logger.warn("Can't send schema pull request: node {} is down.", (Object)callback.endpoint);
            callback.fail();
            return;
        }
        if (!this.shouldPullFromEndpoint(callback.endpoint)) {
            logger.info("Skipped sending a migration request: node {} has a higher major version now.", (Object)callback.endpoint);
            callback.fail();
            return;
        }
        logger.debug("Requesting schema from {}", (Object)callback.endpoint);
        this.sendMigrationMessage(callback);
    }

    protected void sendMigrationMessage(Callback callback) {
        this.inflightTasks.getAndIncrement();
        Message<NoPayload> message = Message.out(Verb.SCHEMA_PULL_REQ, NoPayload.noPayload);
        logger.info("Sending schema pull request to {}", (Object)callback.endpoint);
        MessagingService.instance().sendWithCallback(message, callback.endpoint, callback);
    }

    private synchronized Future<Void> pullComplete(InetAddressAndPort endpoint, VersionInfo info, boolean wasSuccessful) {
        this.inflightTasks.decrementAndGet();
        if (wasSuccessful) {
            info.markReceived();
            this.lastPullAttemptTimestamps.remove(endpoint);
        }
        info.outstandingRequests.remove(endpoint);
        info.requestQueue.add(endpoint);
        return this.maybePullSchema(info);
    }

    public int getInflightTasks() {
        return this.inflightTasks.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean awaitSchemaRequests(long waitMillis) {
        if (!FBUtilities.getBroadcastAddressAndPort().equals(InetAddressAndPort.getLoopbackAddress())) {
            Gossiper.waitToSettle();
        }
        if (this.versionInfo.isEmpty()) {
            logger.debug("Nothing in versionInfo - so no schemas to wait for");
        }
        WaitQueue.Signal signal = null;
        try {
            MigrationCoordinator migrationCoordinator = this;
            synchronized (migrationCoordinator) {
                ArrayList<WaitQueue.Signal> signalList;
                block14: {
                    signalList = new ArrayList<WaitQueue.Signal>(this.versionInfo.size());
                    for (VersionInfo version : this.versionInfo.values()) {
                        if (version.wasReceived()) continue;
                        signalList.add(version.register());
                    }
                    if (!signalList.isEmpty()) break block14;
                    boolean bl = true;
                    return bl;
                }
                WaitQueue.Signal[] signals = new WaitQueue.Signal[signalList.size()];
                signalList.toArray(signals);
                signal = WaitQueue.all(signals);
            }
            boolean bl = signal.awaitUntil(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(waitMillis));
            return bl;
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (signal != null) {
                signal.cancel();
            }
        }
    }

    class Callback
    implements RequestCallback<Collection<Mutation>> {
        final InetAddressAndPort endpoint;
        final VersionInfo info;

        public Callback(InetAddressAndPort endpoint, VersionInfo info) {
            this.endpoint = endpoint;
            this.info = info;
        }

        @Override
        public void onFailure(InetAddressAndPort from, RequestFailureReason failureReason) {
            this.fail();
        }

        Future<Void> fail() {
            return MigrationCoordinator.this.pullComplete(this.endpoint, this.info, false);
        }

        @Override
        public void onResponse(Message<Collection<Mutation>> message) {
            this.response((Collection)message.payload);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        Future<Void> response(Collection<Mutation> mutations) {
            VersionInfo versionInfo = this.info;
            synchronized (versionInfo) {
                if (MigrationCoordinator.this.shouldApplySchemaFor(this.info)) {
                    try {
                        MigrationCoordinator.this.mergeSchemaFrom(this.endpoint, mutations);
                    }
                    catch (Exception e) {
                        logger.error(String.format("Unable to merge schema from %s", this.endpoint), (Throwable)e);
                        return this.fail();
                    }
                }
                return MigrationCoordinator.this.pullComplete(this.endpoint, this.info, true);
            }
        }

        public boolean isLatencyForSnitch() {
            return false;
        }
    }

    static class VersionInfo {
        final UUID version;
        final Set<InetAddressAndPort> endpoints = Sets.newConcurrentHashSet();
        final Set<InetAddressAndPort> outstandingRequests = Sets.newConcurrentHashSet();
        final Deque<InetAddressAndPort> requestQueue = new ArrayDeque<InetAddressAndPort>();
        private final WaitQueue waitQueue = new WaitQueue();
        volatile boolean receivedSchema;

        VersionInfo(UUID version) {
            this.version = version;
        }

        WaitQueue.Signal register() {
            return this.waitQueue.register();
        }

        void markReceived() {
            if (this.receivedSchema) {
                return;
            }
            this.receivedSchema = true;
            this.waitQueue.signalAll();
        }

        boolean wasReceived() {
            return this.receivedSchema;
        }

        public String toString() {
            return "VersionInfo{version=" + this.version + ", outstandingRequests=" + this.outstandingRequests + ", requestQueue=" + this.requestQueue + ", waitQueue.waiting=" + this.waitQueue.getWaiting() + ", receivedSchema=" + this.receivedSchema + '}';
        }
    }
}

