/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import React, { useEffect, useState } from "react";
import {
  Input,
  Modal,
  Tag,
  Table,
  Button,
  Space,
  message,
  Popconfirm,
  Descriptions,
} from "antd";
import {
  deletePlugin,
  getPlugins,
  getRemoteFiles,
} from "../services/plugin-manage";
import { AddTemplateModal } from "./uploadModal";
import $i18n from "@/components/i18n";

const { Search } = Input;

export const GeaflowPlugManage: React.FC<{}> = ({}) => {
  const [state, setState] = useState({
    pluginManage: [],
    search: "",
    isAddMd: false,
    files: [],
    visible: false,
    item: {},
  });
  const isAdminLogin = localStorage.getItem("IS_GEAFLOW_ADMIN");
  const currentInstance = localStorage.getItem("GEAFLOW_CURRENT_INSTANCE")
    ? JSON.parse(localStorage.getItem("GEAFLOW_CURRENT_INSTANCE"))
    : {};
  const instanceName = currentInstance.value;
  const handelResources = async () => {
    const respData = await getPlugins({
      searchType: isAdminLogin ? "ALL" : "USER_ONLY",
      keyword: state.search,
    });
    const files = await getRemoteFiles();
    setState({
      ...state,
      pluginManage: respData,
      files,
      isAddMd: false,
    });
  };

  useEffect(() => {
    handelResources();
  }, [state.search, isAdminLogin]);

  const handleDelete = async (record) => {
    const resp = await deletePlugin(record.id);
    if (resp) {
      message.success(
        $i18n.get({
          id: "openpiece-geaflow.geaflow.function-manage.DeletedSuccessfully",
          dm: "删除成功",
        })
      );
      handelResources();
    }
  };
  const showCurrentClusterConfigInfo = (record) => {
    setState({
      ...state,
      visible: true,
      item: record.jarPackage,
    });
  };

  const columns = [
    {
      title: $i18n.get({
        id: "openpiece-geaflow.geaflow.plugin-manage.pluginName",
        dm: "插件名称",
      }),
      dataIndex: "name",
      key: "name",
      render: (_, record: any) => (
        <span>
          {record.system ? (
            <span> {record.name}</span>
          ) : (
            <a onClick={() => showCurrentClusterConfigInfo(record)}>
              {record.name}
            </a>
          )}

          <br />
          {record?.comment && (
            <span style={{ fontSize: 12, color: "#ccc" }}>
              {record.comment}
            </span>
          )}
        </span>
      ),
    },

    {
      title: $i18n.get({
        id: "openpiece-geaflow.geaflow.plugin-manage.pluginType",
        dm: "插件类型",
      }),
      dataIndex: "type",
      key: "type",
    },
    {
      title: $i18n.get({
        id: "openpiece-geaflow.geaflow.plugin-manage.pluginCategory",
        dm: "插件种类",
      }),
      dataIndex: "category",
      key: "category",
    },
    {
      title: $i18n.get({
        id: "openpiece-geaflow.geaflow.plugin-manage.pluginSystem",
        dm: "系统插件",
      }),
      dataIndex: "system",
      key: "system",
      render: (_, record: any) => (
        <span>
          {record.system
            ? $i18n.get({
                id: "openpiece-geaflow.geaflow.plugin-true.system",
                dm: "是",
              })
            : $i18n.get({
                id: "openpiece-geaflow.geaflow.plugin-false.system",
                dm: "否",
              })}
        </span>
      ),
    },
    {
      title: $i18n.get({
        id: "openpiece-geaflow.geaflow.function-manage.Operator",
        dm: "操作人",
      }),
      key: "creatorName",
      render: (_, record: any) => (
        <span>
          {$i18n.get({
            id: "openpiece-geaflow.geaflow.function-manage.Creator",
            dm: "创建人：",
          })}
          {record.creatorName} <br />
          {record?.modifierName && (
            <span>
              {$i18n.get(
                {
                  id: "openpiece-geaflow.geaflow.function-manage.ModifiedByRecordmodifiername",
                  dm: "修改人：{recordModifierName}",
                },
                { recordModifierName: record.modifierName }
              )}
            </span>
          )}
        </span>
      ),
    },
    {
      title: $i18n.get({
        id: "openpiece-geaflow.geaflow.function-manage.OperationTime",
        dm: "操作时间",
      }),
      key: "createTime",
      render: (_, record: any) => (
        <span>
          {$i18n.get({
            id: "openpiece-geaflow.geaflow.function-manage.CreationTime",
            dm: "创建时间：",
          })}
          {record.createTime} <br />
          {record?.modifyTime && (
            <span>
              {$i18n.get(
                {
                  id: "openpiece-geaflow.geaflow.function-manage.ModificationTimeRecordmodifytime",
                  dm: "修改时间：{recordModifyTime}",
                },
                { recordModifyTime: record.modifyTime }
              )}
            </span>
          )}
        </span>
      ),
    },
    {
      title: $i18n.get({
        id: "openpiece-geaflow.geaflow.function-manage.Operation",
        dm: "操作",
      }),
      dataIndex: "container",
      key: "container",
      hideInSearch: true,
      render: (_, record) => {
        return (
          <Space>
            <Popconfirm
              disabled={record.system}
              title={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.AreYouSureYouWant",
                dm: "确定删除该记录吗？",
              })}
              onConfirm={() => handleDelete(record)}
              okText={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.Ok",
                dm: "确定",
              })}
              cancelText={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.Cancel",
                dm: "取消",
              })}
            >
              <a style={{ color: record.system && "rgba(0, 0, 0, 0.25)" }}>
                {$i18n.get({
                  id: "openpiece-geaflow.geaflow.function-manage.Delete",
                  dm: "删除",
                })}
              </a>
            </Popconfirm>
          </Space>
        );
      },
    },
  ];

  const { item } = state;
  return (
    <>
      <div style={{ padding: 16, background: "#fff" }}>
        <div
          style={{
            display: "flex",
            justifyContent: "space-between",
            marginBottom: 16,
          }}
        >
          <div style={{ fontWeight: 500, fontSize: 16 }}>
            {$i18n.get({
              id: "openpiece-geaflow.geaflow.plugin-definition.FunctionDefinitionList",
              dm: "插件列表",
            })}
          </div>
          <div>
            <Search
              style={{ width: 286, marginRight: 16 }}
              placeholder={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.EnterASearchKeyword",
                dm: "请输入搜索关键词",
              })}
              onSearch={(value) => {
                setState({ ...state, search: value });
              }}
            />
            {!isAdminLogin && (
              <Button
                onClick={() => {
                  setState({
                    ...state,
                    isAddMd: true,
                  });
                }}
                type="primary"
              >
                {$i18n.get({
                  id: "openpiece-geaflow.geaflow.function-manage.Add",
                  dm: "新增",
                })}
              </Button>
            )}
          </div>
        </div>
        <Table
          dataSource={state.pluginManage}
          columns={columns}
          pagination={{
            hideOnSinglePage: true,
            showQuickJumper: true,
            size: "small",
          }}
        />

        <AddTemplateModal
          isAddMd={state.isAddMd}
          onclose={() => {
            setState({
              ...state,
              isAddMd: false,
            });
          }}
          onLoad={() => {
            handelResources();
          }}
          instanceName={instanceName}
          files={state.files}
        />

        <Modal
          title={$i18n.get({
            id: "openpiece-geaflow.geaflow.plugin-manage.FunctionConfigurationParameters",
            dm: "插件详情",
          })}
          width={850}
          visible={state.visible}
          onCancel={() => {
            setState({ ...state, visible: false });
          }}
          onOk={() => {
            setState({ ...state, visible: false });
          }}
          okText={$i18n.get({
            id: "openpiece-geaflow.geaflow.function-manage.Ok",
            dm: "确定",
          })}
        >
          <Descriptions column={2}>
            <Descriptions.Item
              label={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.JarPackageName",
                dm: "JAR包名称",
              })}
            >
              {item?.name}
            </Descriptions.Item>
            <Descriptions.Item
              label={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.Type",
                dm: "类型",
              })}
            >
              {item?.type}
            </Descriptions.Item>
            <Descriptions.Item label="MD5">{item?.md5}</Descriptions.Item>
            <Descriptions.Item
              label={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.Path",
                dm: "路径",
              })}
            >
              {item?.path}
            </Descriptions.Item>
            <Descriptions.Item label="URL">{item?.url}</Descriptions.Item>
            <Descriptions.Item
              label={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.CreatorId",
                dm: "创建人ID",
              })}
            >
              {item?.creatorId}
            </Descriptions.Item>
            <Descriptions.Item
              label={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.CreationTime.1",
                dm: "创建时间",
              })}
            >
              {item?.createTime}
            </Descriptions.Item>
            <Descriptions.Item
              label={$i18n.get({
                id: "openpiece-geaflow.geaflow.function-manage.UpdateTime",
                dm: "更新时间",
              })}
            >
              {item?.modifyTime}
            </Descriptions.Item>
          </Descriptions>
        </Modal>
      </div>
    </>
  );
};
