/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iggy.client.blocking.http;

import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.iggy.client.blocking.ConsumerGroupsClient;
import org.apache.iggy.consumergroup.ConsumerGroup;
import org.apache.iggy.consumergroup.ConsumerGroupDetails;
import org.apache.iggy.identifier.ConsumerId;
import org.apache.iggy.identifier.StreamId;
import org.apache.iggy.identifier.TopicId;
import java.util.List;
import java.util.Optional;

class ConsumerGroupsHttpClient implements ConsumerGroupsClient {

    private final InternalHttpClient httpClient;

    public ConsumerGroupsHttpClient(InternalHttpClient httpClient) {
        this.httpClient = httpClient;
    }

    @Override
    public Optional<ConsumerGroupDetails> getConsumerGroup(StreamId streamId, TopicId topicId, ConsumerId groupId) {
        var request = httpClient.prepareGetRequest(path(streamId, topicId) + "/" + groupId);
        return httpClient.executeWithOptionalResponse(request, ConsumerGroupDetails.class);
    }

    @Override
    public List<ConsumerGroup> getConsumerGroups(StreamId streamId, TopicId topicId) {
        var request = httpClient.prepareGetRequest(path(streamId, topicId));
        return httpClient.execute(request, new TypeReference<>() {
        });
    }

    @Override
    public ConsumerGroupDetails createConsumerGroup(StreamId streamId, TopicId topicId, Optional<Long> groupId, String name) {
        var request = httpClient.preparePostRequest(path(streamId, topicId),
                new CreateConsumerGroup(groupId, name));
        return httpClient.execute(request, new TypeReference<>() {
        });
    }

    @Override
    public void deleteConsumerGroup(StreamId streamId, TopicId topicId, ConsumerId groupId) {
        var request = httpClient.prepareDeleteRequest(path(streamId, topicId) + "/" + groupId);
        httpClient.execute(request);
    }

    @Override
    public void joinConsumerGroup(StreamId streamId, TopicId topicId, ConsumerId groupId) {
        throw new UnsupportedOperationException("Method not available in HTTP client");
    }

    @Override
    public void leaveConsumerGroup(StreamId streamId, TopicId topicId, ConsumerId groupId) {
        throw new UnsupportedOperationException("Method not available in HTTP client");
    }

    private static String path(StreamId streamId, TopicId topicId) {
        return "/streams/" + streamId + "/topics/" + topicId + "/consumer-groups";
    }

    private record CreateConsumerGroup(Optional<Long> groupId, String name) {
    }

}
