/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.shaded.org.apache.commons.io.FileUtils;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class DiskChecker {
    public static final Logger LOG = LoggerFactory.getLogger(DiskChecker.class);
    private static AtomicReference<FileIoProvider> fileIoProvider = new AtomicReference<DefaultFileIoProvider>(new DefaultFileIoProvider());
    private static final String DISK_IO_FILE_PREFIX = "DiskChecker.OK_TO_DELETE_.";
    @VisibleForTesting
    static final int DISK_IO_MAX_ITERATIONS = 3;

    public static void checkDir(File dir) throws DiskErrorException {
        DiskChecker.checkDirInternal(dir);
    }

    public static void checkDirWithDiskIo(File dir) throws DiskErrorException {
        DiskChecker.checkDirInternal(dir);
        DiskChecker.doDiskIo(dir);
    }

    private static void checkDirInternal(File dir) throws DiskErrorException {
        if (!DiskChecker.mkdirsWithExistsCheck(dir)) {
            throw new DiskErrorException("Cannot create directory: " + dir.toString());
        }
        DiskChecker.checkAccessByFileMethods(dir);
    }

    public static void checkDir(LocalFileSystem localFS, Path dir, FsPermission expected) throws DiskErrorException, IOException {
        DiskChecker.checkDirInternal(localFS, dir, expected);
    }

    public static void checkDirWithDiskIo(LocalFileSystem localFS, Path dir, FsPermission expected) throws DiskErrorException, IOException {
        DiskChecker.checkDirInternal(localFS, dir, expected);
        DiskChecker.doDiskIo(localFS.pathToFile(dir));
    }

    private static void checkDirInternal(LocalFileSystem localFS, Path dir, FsPermission expected) throws DiskErrorException, IOException {
        DiskChecker.mkdirsWithExistsAndPermissionCheck(localFS, dir, expected);
        DiskChecker.checkAccessByFileMethods(localFS.pathToFile(dir));
    }

    private static void checkAccessByFileMethods(File dir) throws DiskErrorException {
        if (!dir.isDirectory()) {
            throw new DiskErrorException("Not a directory: " + dir.toString());
        }
        if (!FileUtil.canRead(dir)) {
            throw new DiskErrorException("Directory is not readable: " + dir.toString());
        }
        if (!FileUtil.canWrite(dir)) {
            throw new DiskErrorException("Directory is not writable: " + dir.toString());
        }
        if (!FileUtil.canExecute(dir)) {
            throw new DiskErrorException("Directory is not executable: " + dir.toString());
        }
    }

    private static boolean mkdirsWithExistsCheck(File dir) {
        File canonDir;
        if (dir.mkdir() || dir.exists()) {
            return true;
        }
        try {
            canonDir = dir.getCanonicalFile();
        }
        catch (IOException e) {
            return false;
        }
        String parent = canonDir.getParent();
        return parent != null && DiskChecker.mkdirsWithExistsCheck(new File(parent)) && (canonDir.mkdir() || canonDir.exists());
    }

    static void mkdirsWithExistsAndPermissionCheck(LocalFileSystem localFS, Path dir, FsPermission expected) throws IOException {
        File directory = localFS.pathToFile(dir);
        boolean created = false;
        if (!directory.exists()) {
            created = DiskChecker.mkdirsWithExistsCheck(directory);
        }
        if (created || !localFS.getFileStatus(dir).getPermission().equals(expected)) {
            localFS.setPermission(dir, expected);
        }
    }

    private static void doDiskIo(File dir) throws DiskErrorException {
        try {
            IOException ioe = null;
            for (int i = 0; i < 3; ++i) {
                File file = DiskChecker.getFileNameForDiskIoCheck(dir, i + 1);
                try {
                    DiskChecker.diskIoCheckWithoutNativeIo(file);
                    return;
                }
                catch (IOException e) {
                    ioe = e;
                    continue;
                }
            }
            throw ioe;
        }
        catch (IOException e) {
            throw new DiskErrorException("Error checking directory " + dir, e);
        }
    }

    private static void diskIoCheckWithoutNativeIo(File file) throws IOException {
        FileOutputStream fos = null;
        try {
            FileIoProvider provider = fileIoProvider.get();
            fos = provider.get(file);
            provider.write(fos, new byte[1]);
            fos.getFD().sync();
            fos.close();
            fos = null;
            if (!file.delete() && file.exists()) {
                throw new IOException("Failed to delete " + file);
            }
            file = null;
        }
        catch (Throwable throwable) {
            IOUtils.cleanupWithLogger(LOG, fos);
            FileUtils.deleteQuietly((File)file);
            throw throwable;
        }
        IOUtils.cleanupWithLogger(LOG, fos);
        FileUtils.deleteQuietly((File)file);
    }

    @VisibleForTesting
    static File getFileNameForDiskIoCheck(File dir, int iterationCount) {
        if (iterationCount < 3) {
            return new File(dir, DISK_IO_FILE_PREFIX + String.format("%03d", iterationCount));
        }
        return new File(dir, DISK_IO_FILE_PREFIX + UUID.randomUUID());
    }

    @VisibleForTesting
    static FileIoProvider replaceFileOutputStreamProvider(FileIoProvider newFosProvider) {
        return fileIoProvider.getAndSet(newFosProvider);
    }

    @VisibleForTesting
    static FileIoProvider getFileOutputStreamProvider() {
        return fileIoProvider.get();
    }

    private static class DefaultFileIoProvider
    implements FileIoProvider {
        private DefaultFileIoProvider() {
        }

        @Override
        public FileOutputStream get(File f) throws FileNotFoundException {
            return new FileOutputStream(f);
        }

        @Override
        public void write(FileOutputStream fos, byte[] data) throws IOException {
            fos.write(data);
        }
    }

    static interface FileIoProvider {
        public FileOutputStream get(File var1) throws FileNotFoundException;

        public void write(FileOutputStream var1, byte[] var2) throws IOException;
    }

    public static class DiskOutOfSpaceException
    extends IOException {
        public DiskOutOfSpaceException(String msg) {
            super(msg);
        }
    }

    public static class DiskErrorException
    extends IOException {
        public DiskErrorException(String msg) {
            super(msg);
        }

        public DiskErrorException(String msg, Throwable cause) {
            super(msg, cause);
        }
    }
}

