/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.viewController;

/**
 * Each view controller configured with the constructor will be asked to map the given viewId.
 * The first one not returning NULL wins. 
 * <p>
 * As an example, this can be configured to first try a mapper that converts the viewId to
 * a beanname, then if that fails to find a match try a mapper that uses class annotations
 * to find an appropriate bean.
 */
public class CompositeViewControllerNameMapper implements ViewControllerNameMapper
{
    private final ViewControllerNameMapper[] viewControllerNameMappers;

    public CompositeViewControllerNameMapper(ViewControllerNameMapper[] viewControllerNameMappers)
    {
        this.viewControllerNameMappers = new ViewControllerNameMapper[viewControllerNameMappers.length];
        System.arraycopy(
                viewControllerNameMappers, 0, 
                this.viewControllerNameMappers, 0,
                viewControllerNameMappers.length);
    }

    public String mapViewId(String viewId)
    {
        for (int i = 0; i<viewControllerNameMappers.length; i++)
        {
            String beanName = viewControllerNameMappers[i].mapViewId(viewId);
            if (beanName != null)
            {
                return beanName;
            }
        }

        return null;
    }
}
