/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.ratis.ContainerCommandRequestMessage;
import org.apache.hadoop.hdds.ratis.RatisHelper;
import org.apache.hadoop.hdds.scm.OzoneClientConfig;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.XceiverClientManager;
import org.apache.hadoop.hdds.scm.XceiverClientMetrics;
import org.apache.hadoop.hdds.scm.XceiverClientRatis;
import org.apache.hadoop.hdds.scm.XceiverClientReply;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.storage.ByteBufferStreamOutput;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.hdds.scm.storage.StreamBuffer;
import org.apache.hadoop.hdds.scm.storage.StreamCommitWatcher;
import org.apache.hadoop.ozone.common.Checksum;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.ratis.client.api.DataStreamApi;
import org.apache.ratis.client.api.DataStreamOutput;
import org.apache.ratis.io.StandardWriteOption;
import org.apache.ratis.io.WriteOption;
import org.apache.ratis.protocol.DataStreamReply;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockDataStreamOutput
implements ByteBufferStreamOutput {
    public static final Logger LOG = LoggerFactory.getLogger(BlockDataStreamOutput.class);
    public static final int PUT_BLOCK_REQUEST_LENGTH_MAX = 0x100000;
    public static final String EXCEPTION_MSG = "Unexpected Storage Container Exception: ";
    private static final CompletableFuture[] EMPTY_FUTURE_ARRAY = new CompletableFuture[0];
    private AtomicReference<BlockID> blockID;
    private final ContainerProtos.BlockData.Builder containerBlockData;
    private XceiverClientFactory xceiverClientFactory;
    private XceiverClientRatis xceiverClient;
    private OzoneClientConfig config;
    private int chunkIndex;
    private final AtomicLong chunkOffset = new AtomicLong();
    private List<StreamBuffer> bufferList;
    private final AtomicReference<IOException> ioException;
    private final ExecutorService responseExecutor;
    private long totalDataFlushedLength;
    private long writtenDataLength;
    private final StreamCommitWatcher commitWatcher;
    private Queue<CompletableFuture<ContainerProtos.ContainerCommandResponseProto>> putBlockFutures = new LinkedList<CompletableFuture<ContainerProtos.ContainerCommandResponseProto>>();
    private final List<DatanodeDetails> failedServers;
    private final Checksum checksum;
    private int flushPeriod;
    private final Token<? extends TokenIdentifier> token;
    private final DataStreamOutput out;
    private CompletableFuture<DataStreamReply> dataStreamCloseReply;
    private List<CompletableFuture<DataStreamReply>> futures = new ArrayList<CompletableFuture<DataStreamReply>>();
    private final long syncSize = 0L;
    private long syncPosition = 0L;
    private StreamBuffer currentBuffer;
    private XceiverClientMetrics metrics;
    private List<StreamBuffer> buffersForPutBlock;
    private boolean isDatastreamPipelineMode;

    public BlockDataStreamOutput(BlockID blockID, XceiverClientFactory xceiverClientManager, Pipeline pipeline, OzoneClientConfig config, Token<? extends TokenIdentifier> token, List<StreamBuffer> bufferList) throws IOException {
        this.xceiverClientFactory = xceiverClientManager;
        this.config = config;
        this.isDatastreamPipelineMode = config.isDatastreamPipelineMode();
        this.blockID = new AtomicReference<BlockID>(blockID);
        ContainerProtos.KeyValue keyValue = ContainerProtos.KeyValue.newBuilder().setKey("TYPE").setValue("KEY").build();
        this.containerBlockData = ContainerProtos.BlockData.newBuilder().setBlockID(blockID.getDatanodeBlockIDProtobuf()).addMetadata(keyValue);
        this.xceiverClient = (XceiverClientRatis)xceiverClientManager.acquireClient(pipeline, true);
        this.token = token;
        this.out = this.setupStream(pipeline);
        this.bufferList = bufferList;
        this.flushPeriod = (int)(config.getStreamBufferFlushSize() / (long)config.getStreamBufferSize());
        Preconditions.checkArgument(((long)this.flushPeriod * (long)config.getStreamBufferSize() == config.getStreamBufferFlushSize() ? 1 : 0) != 0);
        this.responseExecutor = Executors.newSingleThreadExecutor();
        this.commitWatcher = new StreamCommitWatcher(this.xceiverClient, bufferList);
        this.totalDataFlushedLength = 0L;
        this.writtenDataLength = 0L;
        this.failedServers = new ArrayList<DatanodeDetails>(0);
        this.ioException = new AtomicReference<Object>(null);
        this.checksum = new Checksum(config.getChecksumType(), config.getBytesPerChecksum());
        this.metrics = XceiverClientManager.getXceiverClientMetrics();
    }

    private DataStreamOutput setupStream(Pipeline pipeline) throws IOException {
        ContainerProtos.WriteChunkRequestProto.Builder writeChunkRequest = ContainerProtos.WriteChunkRequestProto.newBuilder().setBlockID(this.blockID.get().getDatanodeBlockIDProtobuf());
        String id = pipeline.getFirstNode().getUuidString();
        ContainerProtos.ContainerCommandRequestProto.Builder builder = ContainerProtos.ContainerCommandRequestProto.newBuilder().setCmdType(ContainerProtos.Type.StreamInit).setContainerID(this.blockID.get().getContainerID()).setDatanodeUuid(id).setWriteChunk(writeChunkRequest);
        if (this.token != null) {
            builder.setEncodedToken(this.token.encodeToUrlString());
        }
        ContainerCommandRequestMessage message = ContainerCommandRequestMessage.toMessage((ContainerProtos.ContainerCommandRequestProto)builder.build(), null);
        if (this.isDatastreamPipelineMode) {
            return ((DataStreamApi)Preconditions.checkNotNull((Object)this.xceiverClient.getDataStreamApi())).stream(message.getContent().asReadOnlyByteBuffer(), RatisHelper.getRoutingTable((Pipeline)pipeline));
        }
        return ((DataStreamApi)Preconditions.checkNotNull((Object)this.xceiverClient.getDataStreamApi())).stream(message.getContent().asReadOnlyByteBuffer());
    }

    public BlockID getBlockID() {
        return this.blockID.get();
    }

    public long getWrittenDataLength() {
        return this.writtenDataLength;
    }

    public List<DatanodeDetails> getFailedServers() {
        return this.failedServers;
    }

    @VisibleForTesting
    public XceiverClientRatis getXceiverClient() {
        return this.xceiverClient;
    }

    public IOException getIoException() {
        return this.ioException.get();
    }

    @Override
    public void write(ByteBuffer b, int off, int len) throws IOException {
        this.checkOpen();
        if (b == null) {
            throw new NullPointerException();
        }
        if (len == 0) {
            return;
        }
        while (len > 0) {
            this.allocateNewBufferIfNeeded();
            int writeLen = Math.min(len, this.currentBuffer.length());
            StreamBuffer buf = new StreamBuffer(b, off, writeLen);
            this.currentBuffer.put(buf);
            this.writeChunkIfNeeded();
            off += writeLen;
            this.writtenDataLength += (long)writeLen;
            len -= writeLen;
            this.doFlushIfNeeded();
        }
    }

    private void writeChunkIfNeeded() throws IOException {
        if (this.currentBuffer.length() == 0) {
            this.writeChunk(this.currentBuffer);
            this.currentBuffer = null;
        }
    }

    private void writeChunk(StreamBuffer sb) throws IOException {
        this.bufferList.add(sb);
        if (this.buffersForPutBlock == null) {
            this.buffersForPutBlock = new ArrayList<StreamBuffer>();
        }
        this.buffersForPutBlock.add(sb);
        ByteBuffer dup = sb.duplicate();
        dup.position(0);
        dup.limit(sb.position());
        this.writeChunkToContainer(dup);
    }

    private void allocateNewBufferIfNeeded() {
        if (this.currentBuffer == null) {
            this.currentBuffer = StreamBuffer.allocate(this.config.getDataStreamMinPacketSize());
        }
    }

    private void doFlushIfNeeded() throws IOException {
        long boundary = this.config.getDataStreamBufferFlushSize() / (long)this.config.getDataStreamMinPacketSize();
        long streamWindow = this.config.getStreamWindowSize() / (long)this.config.getDataStreamMinPacketSize();
        if (!this.bufferList.isEmpty() && (long)this.bufferList.size() % boundary == 0L && this.buffersForPutBlock != null && !this.buffersForPutBlock.isEmpty()) {
            this.updateFlushLength();
            this.executePutBlock(false, false);
        }
        if ((long)this.bufferList.size() == streamWindow) {
            try {
                this.checkOpen();
                if (!this.putBlockFutures.isEmpty()) {
                    this.putBlockFutures.remove().get();
                }
            }
            catch (ExecutionException e) {
                this.handleExecutionException(e);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                this.handleInterruptedException(ex, true);
            }
            this.watchForCommit(true);
        }
    }

    private void updateFlushLength() {
        this.totalDataFlushedLength = this.writtenDataLength;
    }

    @VisibleForTesting
    public long getTotalDataFlushedLength() {
        return this.totalDataFlushedLength;
    }

    public void writeOnRetry(long len) throws IOException {
        if (len == 0L) {
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrying write length {} for blockID {}", (Object)len, this.blockID);
        }
        int count = 0;
        while (len > 0L) {
            StreamBuffer buf = this.bufferList.get(count);
            long writeLen = Math.min((long)buf.position(), len);
            if (this.buffersForPutBlock == null) {
                this.buffersForPutBlock = new ArrayList<StreamBuffer>();
            }
            this.buffersForPutBlock.add(buf);
            ByteBuffer duplicated = buf.duplicate();
            duplicated.position(0);
            duplicated.limit(buf.position());
            this.writeChunkToContainer(duplicated);
            len -= writeLen;
            ++count;
            this.writtenDataLength += writeLen;
        }
    }

    private void watchForCommit(boolean bufferFull) throws IOException {
        this.checkOpen();
        try {
            List dnList;
            XceiverClientReply reply;
            XceiverClientReply xceiverClientReply = reply = bufferFull ? this.commitWatcher.watchOnFirstIndex() : this.commitWatcher.watchOnLastIndex();
            if (reply != null && !(dnList = reply.getDatanodes()).isEmpty()) {
                Pipeline pipe = this.xceiverClient.getPipeline();
                LOG.warn("Failed to commit BlockId {} on {}. Failed nodes: {}", new Object[]{this.blockID, pipe, dnList});
                this.failedServers.addAll(dnList);
            }
        }
        catch (IOException ioe) {
            this.setIoException(ioe);
            throw this.getIoException();
        }
    }

    private void executePutBlock(boolean close, boolean force) throws IOException {
        List<StreamBuffer> byteBufferList;
        this.checkOpen();
        long flushPos = this.totalDataFlushedLength;
        if (!force) {
            Preconditions.checkNotNull(this.bufferList);
            byteBufferList = this.buffersForPutBlock;
            this.buffersForPutBlock = null;
            Preconditions.checkNotNull(byteBufferList);
        } else {
            byteBufferList = null;
        }
        this.waitFuturesComplete();
        ContainerProtos.BlockData blockData = this.containerBlockData.build();
        if (close) {
            ContainerProtos.ContainerCommandRequestProto putBlockRequest = ContainerProtocolCalls.getPutBlockRequest((Pipeline)this.xceiverClient.getPipeline(), (ContainerProtos.BlockData)blockData, (boolean)true, this.token);
            this.dataStreamCloseReply = BlockDataStreamOutput.executePutBlockClose(putBlockRequest, 0x100000, this.out);
            this.dataStreamCloseReply.whenComplete((reply, e) -> {
                if (e != null || reply == null || !reply.isSuccess()) {
                    LOG.warn("Failed executePutBlockClose, reply=" + reply, e);
                    try {
                        this.executePutBlock(true, false);
                    }
                    catch (IOException ex) {
                        throw new CompletionException(ex);
                    }
                }
            });
        }
        try {
            XceiverClientReply asyncReply = ContainerProtocolCalls.putBlockAsync((XceiverClientSpi)this.xceiverClient, (ContainerProtos.BlockData)blockData, (boolean)close, this.token);
            CompletionStage flushFuture = ((CompletableFuture)asyncReply.getResponse().thenApplyAsync(e -> {
                try {
                    this.validateResponse((ContainerProtos.ContainerCommandResponseProto)e);
                }
                catch (IOException sce) {
                    throw new CompletionException(sce);
                }
                if (this.getIoException() == null && !force) {
                    BlockID responseBlockID = BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)e.getPutBlock().getCommittedBlockLength().getBlockID());
                    Preconditions.checkState((boolean)this.blockID.get().getContainerBlockID().equals((Object)responseBlockID.getContainerBlockID()));
                    this.blockID.set(responseBlockID);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Adding index " + asyncReply.getLogIndex() + " commitMap size " + this.commitWatcher.getCommitIndexMap().size() + " flushLength " + flushPos + " blockID " + this.blockID);
                    }
                    this.commitWatcher.updateCommitInfoMap(asyncReply.getLogIndex(), byteBufferList);
                }
                return e;
            }, (Executor)this.responseExecutor)).exceptionally(e -> {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("putBlock failed for blockID {} with exception {}", this.blockID, (Object)e.getLocalizedMessage());
                }
                CompletionException ce = new CompletionException((Throwable)e);
                this.setIoException(ce);
                throw ce;
            });
            this.putBlockFutures.add((CompletableFuture<ContainerProtos.ContainerCommandResponseProto>)flushFuture);
        }
        catch (IOException | ExecutionException e2) {
            throw new IOException(EXCEPTION_MSG + e2.toString(), e2);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, false);
        }
    }

    public static CompletableFuture<DataStreamReply> executePutBlockClose(ContainerProtos.ContainerCommandRequestProto putBlockRequest, int max, DataStreamOutput out) {
        ByteBuffer putBlock = ContainerCommandRequestMessage.toMessage((ContainerProtos.ContainerCommandRequestProto)putBlockRequest, null).getContent().asReadOnlyByteBuffer();
        ByteBuffer protoLength = BlockDataStreamOutput.getProtoLength(putBlock, max);
        RatisHelper.debug((ByteBuffer)putBlock, (String)"putBlock", (Logger)LOG);
        out.writeAsync(putBlock, new WriteOption[0]);
        RatisHelper.debug((ByteBuffer)protoLength, (String)"protoLength", (Logger)LOG);
        return out.writeAsync(protoLength, new WriteOption[]{StandardWriteOption.CLOSE});
    }

    public static ByteBuffer getProtoLength(ByteBuffer putBlock, int max) {
        int protoLength = putBlock.remaining();
        Preconditions.checkState((protoLength <= max ? 1 : 0) != 0, (String)"protoLength== %s > max = %s", (int)protoLength, (int)max);
        ByteBuffer buffer = ByteBuffer.allocate(4);
        buffer.putInt(protoLength);
        buffer.flip();
        LOG.debug("protoLength = {}", (Object)protoLength);
        Preconditions.checkState((buffer.remaining() == 4 ? 1 : 0) != 0);
        return buffer.asReadOnlyBuffer();
    }

    @Override
    public void flush() throws IOException {
        if (this.xceiverClientFactory != null && this.xceiverClient != null && !this.config.isStreamBufferFlushDelay()) {
            this.waitFuturesComplete();
        }
    }

    public void waitFuturesComplete() throws IOException {
        try {
            CompletableFuture.allOf(this.futures.toArray(EMPTY_FUTURE_ARRAY)).get();
            this.futures.clear();
        }
        catch (Exception e) {
            LOG.warn("Failed to write all chunks through stream: " + e);
            throw new IOException(e);
        }
    }

    private void handleFlush(boolean close) throws IOException, InterruptedException, ExecutionException {
        this.checkOpen();
        if (this.totalDataFlushedLength < this.writtenDataLength) {
            if (this.currentBuffer != null) {
                this.writeChunk(this.currentBuffer);
                this.currentBuffer = null;
            }
            this.updateFlushLength();
            this.executePutBlock(close, false);
        } else if (close) {
            this.executePutBlock(true, true);
        }
        CompletableFuture.allOf(this.putBlockFutures.toArray(EMPTY_FUTURE_ARRAY)).get();
        this.watchForCommit(false);
        this.checkOpen();
    }

    @Override
    public void close() throws IOException {
        if (this.xceiverClientFactory != null && this.xceiverClient != null) {
            try {
                this.handleFlush(true);
                this.dataStreamCloseReply.get();
            }
            catch (ExecutionException e) {
                this.handleExecutionException(e);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                this.handleInterruptedException(ex, true);
            }
            finally {
                this.cleanup(false);
            }
        }
    }

    private void validateResponse(ContainerProtos.ContainerCommandResponseProto responseProto) throws IOException {
        try {
            IOException exception = this.getIoException();
            if (exception != null) {
                throw exception;
            }
            ContainerProtocolCalls.validateContainerResponse((ContainerProtos.ContainerCommandResponseProto)responseProto);
        }
        catch (StorageContainerException sce) {
            this.setIoException(sce);
            throw sce;
        }
    }

    private void setIoException(Throwable e) {
        IOException ioe = this.getIoException();
        if (ioe == null) {
            IOException exception = new IOException(EXCEPTION_MSG + e.toString(), e);
            this.ioException.compareAndSet(null, exception);
        } else {
            LOG.debug("Previous request had already failed with " + ioe.toString() + " so subsequent request also encounters Storage Container Exception ", e);
        }
    }

    public void cleanup(boolean invalidateClient) {
        if (this.xceiverClientFactory != null) {
            this.xceiverClientFactory.releaseClient(this.xceiverClient, invalidateClient, true);
        }
        this.xceiverClientFactory = null;
        this.xceiverClient = null;
        this.commitWatcher.cleanup();
        this.responseExecutor.shutdown();
    }

    private void checkOpen() throws IOException {
        if (this.isClosed()) {
            throw new IOException("BlockDataStreamOutput has been closed.");
        }
        if (this.getIoException() != null) {
            throw this.getIoException();
        }
    }

    public boolean isClosed() {
        return this.xceiverClient == null;
    }

    private boolean needSync(long position) {
        return false;
    }

    private void writeChunkToContainer(ByteBuffer buf) throws IOException {
        int effectiveChunkSize = buf.remaining();
        long offset = this.chunkOffset.getAndAdd(effectiveChunkSize);
        ChecksumData checksumData = this.checksum.computeChecksum(buf.asReadOnlyBuffer());
        ContainerProtos.ChunkInfo chunkInfo = ContainerProtos.ChunkInfo.newBuilder().setChunkName(this.blockID.get().getLocalID() + "_chunk_" + ++this.chunkIndex).setOffset(offset).setLen((long)effectiveChunkSize).setChecksumData(checksumData.getProtoBufMessage()).build();
        this.metrics.incrPendingContainerOpsMetrics(ContainerProtos.Type.WriteChunk);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Writing chunk {} length {} at offset {}", new Object[]{chunkInfo.getChunkName(), effectiveChunkSize, offset});
        }
        CompletionStage future = (this.needSync(offset + (long)effectiveChunkSize) ? this.out.writeAsync(buf, new WriteOption[]{StandardWriteOption.SYNC}) : this.out.writeAsync(buf, new WriteOption[0])).whenCompleteAsync((r, e) -> {
            if (e != null || !r.isSuccess()) {
                if (e == null) {
                    e = new IOException("result is not success");
                }
                String msg = "Failed to write chunk " + chunkInfo.getChunkName() + " into block " + this.blockID;
                LOG.debug("{}, exception: {}", (Object)msg, (Object)e.getLocalizedMessage());
                CompletionException ce = new CompletionException(msg, (Throwable)e);
                this.setIoException(ce);
                throw ce;
            }
            if (r.isSuccess()) {
                this.xceiverClient.updateCommitInfosMap(r.getCommitInfos());
            }
        }, (Executor)this.responseExecutor);
        this.futures.add((CompletableFuture<DataStreamReply>)future);
        this.containerBlockData.addChunks(chunkInfo);
    }

    @VisibleForTesting
    public void setXceiverClient(XceiverClientRatis xceiverClient) {
        this.xceiverClient = xceiverClient;
    }

    private void handleInterruptedException(Exception ex, boolean processExecutionException) throws IOException {
        LOG.error("Command execution was interrupted.");
        if (!processExecutionException) {
            throw new IOException(EXCEPTION_MSG + ex.toString(), ex);
        }
        this.handleExecutionException(ex);
    }

    private void handleExecutionException(Exception ex) throws IOException {
        this.setIoException(ex);
        throw this.getIoException();
    }

    public long getTotalAckDataLength() {
        return this.commitWatcher.getTotalAckDataLength();
    }
}

