/*
 * Decompiled with CFR 0.152.
 */
package org.apache.streampark.console.core.task;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import java.util.Date;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.streampark.console.core.entity.Application;
import org.apache.streampark.console.core.entity.SavePoint;
import org.apache.streampark.console.core.enums.CheckPointStatus;
import org.apache.streampark.console.core.enums.FailoverStrategy;
import org.apache.streampark.console.core.metrics.flink.CheckPoints;
import org.apache.streampark.console.core.service.ApplicationService;
import org.apache.streampark.console.core.service.SavePointService;
import org.apache.streampark.console.core.service.alert.AlertService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class CheckpointProcessor {
    private final Cache<String, Long> checkPointCache = Caffeine.newBuilder().expireAfterAccess(1L, TimeUnit.DAYS).build();
    private final Map<Long, Counter> checkPointFailedCache = new ConcurrentHashMap<Long, Counter>(0);
    @Autowired
    private ApplicationService applicationService;
    @Autowired
    private AlertService alertService;
    @Autowired
    private SavePointService savePointService;

    public void process(Long appId, CheckPoints checkPoints) {
        CheckPoints.Latest latest = checkPoints.getLatest();
        if (latest == null || latest.getCompleted() == null) {
            return;
        }
        CheckPoints.CheckPoint checkPoint = latest.getCompleted();
        Application application = (Application)this.applicationService.getById(appId);
        CheckPointStatus status = checkPoint.getCheckPointStatus();
        if (CheckPointStatus.COMPLETED.equals(status)) {
            String cacheId = appId + "_" + application.getJobId();
            Long latestId = (Long)this.checkPointCache.get((Object)cacheId, key -> {
                SavePoint savePoint = this.savePointService.getLatest(appId);
                return Optional.ofNullable(savePoint).map(SavePoint::getChkId).orElse(null);
            });
            if (latestId == null || latestId < checkPoint.getId()) {
                this.saveSavepoint(checkPoint, application);
                this.checkPointCache.put((Object)cacheId, (Object)checkPoint.getId());
            }
        } else if (CheckPointStatus.FAILED.equals(status) && application.cpFailedTrigger()) {
            Counter counter = this.checkPointFailedCache.get(appId);
            if (counter == null) {
                this.checkPointFailedCache.put(appId, new Counter(checkPoint.getTriggerTimestamp()));
            } else {
                long minute = counter.getDuration(checkPoint.getTriggerTimestamp());
                if (minute <= (long)application.getCpFailureRateInterval().intValue() && counter.getCount() >= application.getCpMaxFailureInterval()) {
                    this.checkPointFailedCache.remove(appId);
                    FailoverStrategy failoverStrategy = FailoverStrategy.of(application.getCpFailureAction());
                    if (failoverStrategy == null) {
                        throw new IllegalArgumentException("Unexpected cpFailureAction: " + application.getCpFailureAction());
                    }
                    switch (failoverStrategy) {
                        case ALERT: {
                            this.alertService.alert(application, CheckPointStatus.FAILED);
                            break;
                        }
                        case RESTART: {
                            try {
                                this.applicationService.restart(application);
                                break;
                            }
                            catch (Exception e) {
                                throw new RuntimeException(e);
                            }
                        }
                    }
                } else {
                    counter.increment();
                }
            }
        }
    }

    private void saveSavepoint(CheckPoints.CheckPoint checkPoint, Application application) {
        SavePoint savePoint = new SavePoint();
        savePoint.setAppId(application.getId());
        savePoint.setChkId(checkPoint.getId());
        savePoint.setLatest(true);
        savePoint.setType(checkPoint.getCheckPointType().get());
        savePoint.setPath(checkPoint.getExternalPath());
        savePoint.setTriggerTime(new Date(checkPoint.getTriggerTimestamp()));
        savePoint.setCreateTime(new Date());
        this.savePointService.save(savePoint);
    }

    public static class Counter {
        private final Long timestamp;
        private final AtomicInteger count;

        public Counter(Long timestamp) {
            this.timestamp = timestamp;
            this.count = new AtomicInteger(1);
        }

        public void increment() {
            this.count.incrementAndGet();
        }

        public Integer getCount() {
            return this.count.get();
        }

        public long getDuration(Long currentTimestamp) {
            return (currentTimestamp - this.timestamp) / 1000L / 60L;
        }
    }
}

