/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.protocols.raft.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.primitives.Longs;
import io.atomix.cluster.MemberId;
import io.atomix.primitive.PrimitiveId;
import io.atomix.primitive.PrimitiveType;
import io.atomix.primitive.service.ServiceConfig;
import io.atomix.primitive.session.SessionId;
import io.atomix.primitive.session.SessionMetadata;
import io.atomix.protocols.raft.RaftException;
import io.atomix.protocols.raft.RaftServer;
import io.atomix.protocols.raft.impl.MetadataResult;
import io.atomix.protocols.raft.impl.OperationResult;
import io.atomix.protocols.raft.impl.RaftContext;
import io.atomix.protocols.raft.service.RaftServiceContext;
import io.atomix.protocols.raft.session.RaftSession;
import io.atomix.protocols.raft.storage.log.RaftLog;
import io.atomix.protocols.raft.storage.log.RaftLogReader;
import io.atomix.protocols.raft.storage.log.entry.CloseSessionEntry;
import io.atomix.protocols.raft.storage.log.entry.CommandEntry;
import io.atomix.protocols.raft.storage.log.entry.ConfigurationEntry;
import io.atomix.protocols.raft.storage.log.entry.InitializeEntry;
import io.atomix.protocols.raft.storage.log.entry.KeepAliveEntry;
import io.atomix.protocols.raft.storage.log.entry.MetadataEntry;
import io.atomix.protocols.raft.storage.log.entry.OpenSessionEntry;
import io.atomix.protocols.raft.storage.log.entry.QueryEntry;
import io.atomix.protocols.raft.storage.log.entry.RaftLogEntry;
import io.atomix.protocols.raft.storage.snapshot.Snapshot;
import io.atomix.protocols.raft.storage.snapshot.SnapshotReader;
import io.atomix.protocols.raft.storage.snapshot.SnapshotWriter;
import io.atomix.storage.StorageLevel;
import io.atomix.storage.journal.Indexed;
import io.atomix.utils.concurrent.ComposableFuture;
import io.atomix.utils.concurrent.Futures;
import io.atomix.utils.concurrent.OrderedFuture;
import io.atomix.utils.concurrent.ThreadContext;
import io.atomix.utils.concurrent.ThreadContextFactory;
import io.atomix.utils.config.ConfigurationException;
import io.atomix.utils.logging.ContextualLoggerFactory;
import io.atomix.utils.logging.LoggerContext;
import io.atomix.utils.serializer.Namespace;
import io.atomix.utils.serializer.Serializer;
import io.atomix.utils.time.WallClockTimestamp;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import org.slf4j.Logger;

public class RaftServiceManager
implements AutoCloseable {
    private static final Duration SNAPSHOT_INTERVAL = Duration.ofSeconds(10L);
    private static final Duration SNAPSHOT_COMPLETION_DELAY = Duration.ofSeconds(10L);
    private static final Duration COMPACT_DELAY = Duration.ofSeconds(10L);
    private static final int SEGMENT_BUFFER_FACTOR = 5;
    private final Logger logger;
    private final RaftContext raft;
    private final ThreadContext stateContext;
    private final ThreadContext compactionContext;
    private final ThreadContextFactory threadContextFactory;
    private final RaftLog log;
    private final RaftLogReader reader;
    private final Map<Long, CompletableFuture> futures = Maps.newHashMap();
    private volatile CompletableFuture<Void> compactFuture;
    private long lastEnqueued;
    private long lastCompacted;

    public RaftServiceManager(RaftContext raft, ThreadContext stateContext, ThreadContext compactionContext, ThreadContextFactory threadContextFactory) {
        this.raft = (RaftContext)Preconditions.checkNotNull((Object)raft, (Object)"state cannot be null");
        this.log = raft.getLog();
        this.reader = this.log.openReader(1L, RaftLogReader.Mode.COMMITS);
        this.stateContext = stateContext;
        this.compactionContext = compactionContext;
        this.threadContextFactory = threadContextFactory;
        this.logger = ContextualLoggerFactory.getLogger(this.getClass(), (LoggerContext)LoggerContext.builder(RaftServer.class).addValue((Object)raft.getName()).build());
        this.lastEnqueued = this.reader.getFirstIndex() - 1L;
        this.scheduleSnapshots();
    }

    public ThreadContext executor() {
        return this.stateContext;
    }

    private boolean isRunningOutOfDiskSpace() {
        return this.raft.getStorage().statistics().getUsableSpace() < (long)(this.raft.getStorage().maxLogSegmentSize() * 5) || (double)this.raft.getStorage().statistics().getUsableSpace() / (double)this.raft.getStorage().statistics().getTotalSpace() < this.raft.getStorage().freeDiskBuffer();
    }

    private boolean isRunningOutOfMemory() {
        StorageLevel level = this.raft.getStorage().storageLevel();
        if (level == StorageLevel.MEMORY || level == StorageLevel.MAPPED) {
            long freeMemory = this.raft.getStorage().statistics().getFreeMemory();
            long totalMemory = this.raft.getStorage().statistics().getTotalMemory();
            if (freeMemory > 0L && totalMemory > 0L) {
                return (double)freeMemory / (double)totalMemory < this.raft.getStorage().freeMemoryBuffer();
            }
        }
        return false;
    }

    private void scheduleSnapshots() {
        this.raft.getThreadContext().schedule(SNAPSHOT_INTERVAL, () -> this.takeSnapshots(true, false));
    }

    public CompletableFuture<Void> compact() {
        return this.takeSnapshots(false, true);
    }

    private CompletableFuture<Void> takeSnapshots(boolean rescheduleAfterCompletion, boolean force) {
        if (this.compactFuture != null) {
            if (rescheduleAfterCompletion) {
                this.compactFuture.whenComplete((r, e) -> this.scheduleSnapshots());
            }
            return this.compactFuture;
        }
        long lastApplied = this.raft.getLastApplied();
        if (this.raft.getLog().isCompactable(lastApplied) && this.raft.getLog().getCompactableIndex(lastApplied) > this.lastCompacted) {
            boolean runningOutOfDiskSpace = this.isRunningOutOfDiskSpace();
            boolean runningOutOfMemory = this.isRunningOutOfMemory();
            if (!force && !runningOutOfMemory && this.raft.getStorage().dynamicCompaction() && !runningOutOfDiskSpace && this.raft.getLoadMonitor().isUnderHighLoad()) {
                this.logger.debug("Skipping compaction due to high load");
                if (rescheduleAfterCompletion) {
                    this.scheduleSnapshots();
                }
                return CompletableFuture.completedFuture(null);
            }
            this.logger.debug("Snapshotting services");
            this.lastCompacted = lastApplied;
            this.compactFuture = new OrderedFuture();
            this.takeSnapshots(lastApplied).whenCompleteAsync((snapshot, error) -> {
                if (error == null) {
                    this.scheduleCompletion(snapshot.persist());
                }
            }, (Executor)this.compactionContext);
            if (rescheduleAfterCompletion) {
                this.compactFuture.whenComplete((r, e) -> this.scheduleSnapshots());
            }
            return this.compactFuture;
        }
        if (rescheduleAfterCompletion) {
            this.scheduleSnapshots();
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Snapshot> takeSnapshots(long index) {
        ComposableFuture future = new ComposableFuture();
        this.stateContext.execute(() -> {
            try {
                future.complete((Object)this.snapshot(index));
            }
            catch (Exception e) {
                future.completeExceptionally((Throwable)e);
            }
        });
        return future;
    }

    private void scheduleCompletion(Snapshot snapshot) {
        this.stateContext.schedule(SNAPSHOT_COMPLETION_DELAY, () -> {
            if (this.completeSnapshot(snapshot.index())) {
                this.logger.debug("Completing snapshot {}", (Object)snapshot.index());
                snapshot.complete();
                if (!this.raft.getLoadMonitor().isUnderHighLoad() || this.isRunningOutOfDiskSpace() || this.isRunningOutOfMemory()) {
                    this.compactLogs(snapshot.index());
                } else {
                    this.scheduleCompaction(snapshot.index());
                }
            } else {
                this.scheduleCompletion(snapshot);
            }
        });
    }

    private void scheduleCompaction(long lastApplied) {
        this.logger.trace("Scheduling compaction in {}", (Object)COMPACT_DELAY);
        this.stateContext.schedule(COMPACT_DELAY, () -> this.compactLogs(lastApplied));
    }

    private void compactLogs(long compactIndex) {
        this.raft.getThreadContext().execute(() -> {
            this.logger.debug("Compacting logs up to index {}", (Object)compactIndex);
            try {
                this.raft.getLog().compact(compactIndex);
            }
            catch (Exception e) {
                this.logger.error("An exception occurred during log compaction: {}", (Throwable)e);
            }
            finally {
                this.compactFuture.complete(null);
                this.compactFuture = null;
                this.takeSnapshots(false, false);
            }
        });
    }

    public void applyAll(long index) {
        this.enqueueBatch(index);
    }

    public <T> CompletableFuture<T> apply(long index) {
        CompletableFuture future = this.futures.computeIfAbsent(index, i -> new CompletableFuture());
        this.enqueueBatch(index);
        return future;
    }

    private void enqueueBatch(long index) {
        while (this.lastEnqueued < index) {
            this.enqueueIndex(++this.lastEnqueued);
        }
    }

    private void enqueueIndex(long index) {
        this.raft.getThreadContext().execute(() -> this.applyIndex(index));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void applyIndex(long index) {
        if (this.reader.hasNext() && this.reader.getNextIndex() == index) {
            Indexed entry = this.reader.next();
            try {
                if (entry.index() != index) {
                    throw new IllegalStateException("inconsistent index applying entry " + index + ": " + entry);
                }
                CompletableFuture future = this.futures.remove(index);
                this.apply((Indexed<? extends RaftLogEntry>)entry).whenComplete((r, e) -> {
                    if (future != null) {
                        if (e == null) {
                            future.complete(r);
                        } else {
                            future.completeExceptionally((Throwable)e);
                        }
                    }
                });
            }
            catch (Exception e2) {
                this.logger.error("Failed to apply {}: {}", (Object)entry, (Object)e2);
            }
            finally {
                this.raft.setLastApplied(index);
            }
        } else {
            CompletableFuture future = this.futures.remove(index);
            if (future != null) {
                this.logger.error("Cannot apply index " + index);
                future.completeExceptionally(new IndexOutOfBoundsException("Cannot apply index " + index));
            }
        }
    }

    public <T> CompletableFuture<T> apply(Indexed<? extends RaftLogEntry> entry) {
        CompletableFuture future = new CompletableFuture();
        this.stateContext.execute(() -> {
            this.logger.trace("Applying {}", (Object)entry);
            try {
                if (entry.type() == QueryEntry.class) {
                    this.applyQuery((Indexed<QueryEntry>)entry.cast()).whenComplete((r, e) -> {
                        if (e != null) {
                            future.completeExceptionally((Throwable)e);
                        } else {
                            future.complete(r);
                        }
                    });
                } else {
                    this.install(entry.index());
                    if (entry.type() == CommandEntry.class) {
                        future.complete(this.applyCommand((Indexed<CommandEntry>)entry.cast()));
                    } else if (entry.type() == OpenSessionEntry.class) {
                        future.complete(this.applyOpenSession((Indexed<OpenSessionEntry>)entry.cast()));
                    } else if (entry.type() == KeepAliveEntry.class) {
                        future.complete(this.applyKeepAlive((Indexed<KeepAliveEntry>)entry.cast()));
                    } else if (entry.type() == CloseSessionEntry.class) {
                        this.applyCloseSession((Indexed<CloseSessionEntry>)entry.cast());
                        future.complete(null);
                    } else if (entry.type() == MetadataEntry.class) {
                        future.complete(this.applyMetadata((Indexed<MetadataEntry>)entry.cast()));
                    } else if (entry.type() == InitializeEntry.class) {
                        future.complete(this.applyInitialize((Indexed<InitializeEntry>)entry.cast()));
                    } else if (entry.type() == ConfigurationEntry.class) {
                        future.complete(this.applyConfiguration((Indexed<ConfigurationEntry>)entry.cast()));
                    } else {
                        future.completeExceptionally(new RaftException.ProtocolException("Unknown entry type", new Object[0]));
                    }
                }
            }
            catch (Exception e2) {
                future.completeExceptionally(e2);
            }
        });
        return future;
    }

    private Snapshot snapshot(long index) {
        Snapshot snapshot = this.raft.getSnapshotStore().newTemporarySnapshot(index, new WallClockTimestamp());
        try (SnapshotWriter writer = snapshot.openWriter();){
            for (RaftServiceContext service : this.raft.getServices()) {
                writer.buffer().mark();
                SnapshotWriter serviceWriter = new SnapshotWriter(writer.buffer().writeInt(0).slice(), writer.snapshot());
                this.snapshotService(serviceWriter, service);
                int length = serviceWriter.buffer().position();
                writer.buffer().reset().writeInt(length).skip(length);
            }
        }
        catch (Exception e) {
            snapshot.close();
            throw e;
        }
        return snapshot;
    }

    private void snapshotService(SnapshotWriter writer, RaftServiceContext service) {
        writer.writeLong((Long)service.serviceId().id());
        writer.writeString(service.serviceType().name());
        writer.writeString(service.serviceName());
        byte[] config = Serializer.using((Namespace)service.serviceType().namespace()).encode(service.serviceConfig());
        writer.writeInt(config.length).writeBytes(config);
        service.takeSnapshot(writer);
    }

    private void install(long index) {
        Snapshot snapshot = this.raft.getSnapshotStore().getSnapshot(index - 1L);
        if (snapshot != null) {
            this.logger.debug("Installing snapshot {}", (Object)snapshot);
            try (SnapshotReader reader = snapshot.openReader();){
                while (reader.hasRemaining()) {
                    int length = reader.readInt();
                    if (length <= 0) continue;
                    SnapshotReader serviceReader = new SnapshotReader(reader.buffer().slice(length), reader.snapshot());
                    this.installService(serviceReader);
                    reader.skip(length);
                }
            }
        }
    }

    private void installService(SnapshotReader reader) {
        PrimitiveId primitiveId = PrimitiveId.from((long)reader.readLong());
        try {
            PrimitiveType primitiveType = this.raft.getPrimitiveTypes().getPrimitiveType(reader.readString());
            String serviceName = reader.readString();
            byte[] serviceConfig = reader.readBytes(reader.readInt());
            this.logger.debug("Installing service {} {}", (Object)primitiveId, (Object)serviceName);
            RaftServiceContext service = this.initializeService(primitiveId, primitiveType, serviceName, serviceConfig);
            if (service != null) {
                service.installSnapshot(reader);
            }
        }
        catch (ConfigurationException e) {
            this.logger.error(e.getMessage(), (Throwable)e);
        }
    }

    private boolean completeSnapshot(long index) {
        long lastCompleted = index;
        for (RaftSession session : this.raft.getSessions().getSessions()) {
            lastCompleted = Math.min(lastCompleted, session.getLastCompleted());
        }
        return lastCompleted >= index;
    }

    private CompletableFuture<Void> applyInitialize(Indexed<InitializeEntry> entry) {
        for (RaftServiceContext service : this.raft.getServices()) {
            service.keepAliveSessions(entry.index(), ((InitializeEntry)entry.entry()).timestamp());
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> applyConfiguration(Indexed<ConfigurationEntry> entry) {
        for (RaftServiceContext service : this.raft.getServices()) {
            service.keepAliveSessions(entry.index(), ((ConfigurationEntry)entry.entry()).timestamp());
        }
        return CompletableFuture.completedFuture(null);
    }

    private long[] applyKeepAlive(Indexed<KeepAliveEntry> entry) {
        long[] sessionIds = ((KeepAliveEntry)entry.entry()).sessionIds();
        long[] commandSequences = ((KeepAliveEntry)entry.entry()).commandSequenceNumbers();
        long[] eventIndexes = ((KeepAliveEntry)entry.entry()).eventIndexes();
        ArrayList<Long> successfulSessionIds = new ArrayList<Long>(sessionIds.length);
        HashSet<RaftServiceContext> services = new HashSet<RaftServiceContext>();
        for (int i = 0; i < sessionIds.length; ++i) {
            long sessionId = sessionIds[i];
            long commandSequence = commandSequences[i];
            long eventIndex = eventIndexes[i];
            RaftSession session = this.raft.getSessions().getSession(sessionId);
            if (session == null || !session.getService().keepAlive(entry.index(), ((KeepAliveEntry)entry.entry()).timestamp(), session, commandSequence, eventIndex)) continue;
            successfulSessionIds.add(sessionId);
            services.add(session.getService());
        }
        for (RaftServiceContext service : services) {
            service.completeKeepAlive(entry.index(), ((KeepAliveEntry)entry.entry()).timestamp());
        }
        return Longs.toArray(successfulSessionIds);
    }

    private RaftServiceContext getOrInitializeService(PrimitiveId primitiveId, PrimitiveType primitiveType, String serviceName, byte[] config) {
        RaftServiceContext service = this.raft.getServices().getService(serviceName);
        if (service == null) {
            service = this.initializeService(primitiveId, primitiveType, serviceName, config);
        }
        return service;
    }

    private RaftServiceContext initializeService(PrimitiveId primitiveId, PrimitiveType primitiveType, String serviceName, byte[] config) {
        RaftServiceContext oldService = this.raft.getServices().getService(serviceName);
        ServiceConfig serviceConfig = (ServiceConfig)Serializer.using((Namespace)primitiveType.namespace()).decode(config);
        RaftServiceContext service = new RaftServiceContext(primitiveId, serviceName, primitiveType, serviceConfig, primitiveType.newService(serviceConfig), this.raft, this.threadContextFactory);
        this.raft.getServices().registerService(service);
        if (oldService != null) {
            this.raft.getSessions().removeSessions(oldService.serviceId());
        }
        return service;
    }

    private long applyOpenSession(Indexed<OpenSessionEntry> entry) {
        PrimitiveType primitiveType = this.raft.getPrimitiveTypes().getPrimitiveType(((OpenSessionEntry)entry.entry()).serviceType());
        RaftServiceContext service = this.getOrInitializeService(PrimitiveId.from((long)entry.index()), primitiveType, ((OpenSessionEntry)entry.entry()).serviceName(), ((OpenSessionEntry)entry.entry()).serviceConfig());
        if (service == null) {
            throw new RaftException.UnknownService("Unknown service type " + ((OpenSessionEntry)entry.entry()).serviceType(), new Object[0]);
        }
        SessionId sessionId = SessionId.from((long)entry.index());
        RaftSession session = this.raft.getSessions().addSession(new RaftSession(sessionId, MemberId.from((String)((OpenSessionEntry)entry.entry()).memberId()), ((OpenSessionEntry)entry.entry()).serviceName(), primitiveType, ((OpenSessionEntry)entry.entry()).readConsistency(), ((OpenSessionEntry)entry.entry()).minTimeout(), ((OpenSessionEntry)entry.entry()).maxTimeout(), ((OpenSessionEntry)entry.entry()).timestamp(), service.serializer(), service, this.raft, this.threadContextFactory));
        return service.openSession(entry.index(), ((OpenSessionEntry)entry.entry()).timestamp(), session);
    }

    private void applyCloseSession(Indexed<CloseSessionEntry> entry) {
        RaftSession session = this.raft.getSessions().getSession(((CloseSessionEntry)entry.entry()).session());
        if (session == null) {
            throw new RaftException.UnknownSession("Unknown session: " + ((CloseSessionEntry)entry.entry()).session(), new Object[0]);
        }
        RaftServiceContext service = session.getService();
        service.closeSession(entry.index(), ((CloseSessionEntry)entry.entry()).timestamp(), session, ((CloseSessionEntry)entry.entry()).expired());
    }

    private MetadataResult applyMetadata(Indexed<MetadataEntry> entry) {
        if (((MetadataEntry)entry.entry()).session() > 0L) {
            RaftSession session = this.raft.getSessions().getSession(((MetadataEntry)entry.entry()).session());
            if (session == null) {
                this.logger.warn("Unknown session: " + ((MetadataEntry)entry.entry()).session());
                throw new RaftException.UnknownSession("Unknown session: " + ((MetadataEntry)entry.entry()).session(), new Object[0]);
            }
            HashSet<SessionMetadata> sessions = new HashSet<SessionMetadata>();
            for (RaftSession s : this.raft.getSessions().getSessions()) {
                if (!s.primitiveName().equals(session.primitiveName())) continue;
                sessions.add(new SessionMetadata(((Long)s.sessionId().id()).longValue(), s.primitiveName(), s.primitiveType().name()));
            }
            return new MetadataResult(sessions);
        }
        HashSet<SessionMetadata> sessions = new HashSet<SessionMetadata>();
        for (RaftSession session : this.raft.getSessions().getSessions()) {
            sessions.add(new SessionMetadata(((Long)session.sessionId().id()).longValue(), session.primitiveName(), session.primitiveType().name()));
        }
        return new MetadataResult(sessions);
    }

    private OperationResult applyCommand(Indexed<CommandEntry> entry) {
        RaftSession session = this.raft.getSessions().getSession(((CommandEntry)entry.entry()).session());
        if (session == null) {
            this.logger.debug("Unknown session: " + ((CommandEntry)entry.entry()).session());
            throw new RaftException.UnknownSession("unknown session: " + ((CommandEntry)entry.entry()).session(), new Object[0]);
        }
        this.raft.getLoadMonitor().recordEvent();
        return session.getService().executeCommand(entry.index(), ((CommandEntry)entry.entry()).sequenceNumber(), ((CommandEntry)entry.entry()).timestamp(), session, ((CommandEntry)entry.entry()).operation());
    }

    private CompletableFuture<OperationResult> applyQuery(Indexed<QueryEntry> entry) {
        RaftSession session = this.raft.getSessions().getSession(((QueryEntry)entry.entry()).session());
        if (session == null) {
            this.logger.warn("Unknown session: " + ((QueryEntry)entry.entry()).session());
            return Futures.exceptionalFuture((Throwable)new RaftException.UnknownSession("unknown session " + ((QueryEntry)entry.entry()).session(), new Object[0]));
        }
        return session.getService().executeQuery(entry.index(), ((QueryEntry)entry.entry()).sequenceNumber(), ((QueryEntry)entry.entry()).timestamp(), session, ((QueryEntry)entry.entry()).operation());
    }

    @Override
    public void close() {
    }
}

