/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.submarine.hadoop;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.interpreter.thrift.ParagraphInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsClient {
    private static Logger LOGGER = LoggerFactory.getLogger(HdfsClient.class);
    private ZeppelinConfiguration zConf = ZeppelinConfiguration.create();
    private Configuration hadoopConf;
    private boolean isSecurityEnabled;
    private FileSystem fs;
    private static Pattern REPL_PATTERN = Pattern.compile("(\\s*)%([\\w\\.]+)(\\(.*?\\))?.*", 32);

    public HdfsClient(Properties properties) {
        String krb5conf = properties.getProperty("submarine.hadoop.krb5.conf", "");
        if (!StringUtils.isEmpty(krb5conf)) {
            System.setProperty("java.security.krb5.conf", krb5conf);
        }
        this.hadoopConf = new Configuration();
        this.hadoopConf.set("fs.file.impl", "org.apache.hadoop.hdfs.DistributedFileSystem");
        this.isSecurityEnabled = UserGroupInformation.isSecurityEnabled();
        if (this.isSecurityEnabled) {
            String keytab = properties.getProperty("SUBMARINE_HADOOP_KEYTAB", "");
            String principal = properties.getProperty("SUBMARINE_HADOOP_PRINCIPAL", "");
            ZeppelinConfiguration zConf = ZeppelinConfiguration.create();
            if (StringUtils.isEmpty(keytab)) {
                keytab = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_SERVER_KERBEROS_KEYTAB);
            }
            if (StringUtils.isEmpty(principal)) {
                principal = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_SERVER_KERBEROS_PRINCIPAL);
            }
            if (StringUtils.isBlank(keytab) || StringUtils.isBlank(principal)) {
                throw new RuntimeException("keytab and principal can not be empty, keytab: " + keytab + ", principal: " + principal);
            }
            try {
                UserGroupInformation.loginUserFromKeytab((String)principal, (String)keytab);
            }
            catch (IOException e) {
                throw new RuntimeException("Fail to login via keytab:" + keytab + ", principal:" + principal, e);
            }
            catch (Exception e) {
                LOGGER.error(e.getMessage(), e);
            }
        }
        try {
            this.fs = FileSystem.get((URI)new URI("/"), (Configuration)this.hadoopConf);
        }
        catch (IOException e) {
            LOGGER.error(e.getMessage(), e);
        }
        catch (URISyntaxException e) {
            LOGGER.error(e.getMessage(), e);
        }
    }

    public FileSystem getFs() {
        return this.fs;
    }

    public Path makeQualified(Path path) {
        return this.fs.makeQualified(path);
    }

    public boolean exists(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<Boolean>(){

            @Override
            public Boolean call() throws IOException {
                return HdfsClient.this.fs.exists(path);
            }
        });
    }

    public void tryMkDir(final Path dir) throws IOException {
        this.callHdfsOperation(new HdfsOperation<Void>(){

            @Override
            public Void call() throws IOException {
                if (!HdfsClient.this.fs.exists(dir)) {
                    HdfsClient.this.fs.mkdirs(dir);
                    LOGGER.info("Create dir {} in hdfs", (Object)dir.toString());
                }
                if (HdfsClient.this.fs.isFile(dir)) {
                    throw new IOException(dir.toString() + " is file instead of directory, please remove it or specify another directory");
                }
                HdfsClient.this.fs.mkdirs(dir);
                return null;
            }
        });
    }

    public List<Path> list(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<List<Path>>(){

            @Override
            public List<Path> call() throws IOException {
                ArrayList<Path> paths = new ArrayList<Path>();
                for (FileStatus status : HdfsClient.this.fs.globStatus(path)) {
                    paths.add(status.getPath());
                }
                return paths;
            }
        });
    }

    public List<Path> listAll(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<List<Path>>(){

            @Override
            public List<Path> call() throws IOException {
                ArrayList<Path> paths = new ArrayList<Path>();
                this.collectNoteFiles(path, paths);
                return paths;
            }

            private void collectNoteFiles(Path folder, List<Path> noteFiles) throws IOException {
                FileStatus[] paths;
                for (FileStatus path2 : paths = HdfsClient.this.fs.listStatus(folder)) {
                    if (path2.isDirectory()) {
                        this.collectNoteFiles(path2.getPath(), noteFiles);
                        continue;
                    }
                    if (path2.getPath().getName().endsWith(".zpln")) {
                        noteFiles.add(path2.getPath());
                        continue;
                    }
                    LOGGER.warn("Unknown file: " + path2.getPath());
                }
            }
        });
    }

    public boolean delete(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<Boolean>(){

            @Override
            public Boolean call() throws IOException {
                return HdfsClient.this.fs.delete(path, true);
            }
        });
    }

    public String readFile(final Path file) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<String>(){

            @Override
            public String call() throws IOException {
                LOGGER.debug("Read from file: " + file);
                ByteArrayOutputStream noteBytes = new ByteArrayOutputStream();
                IOUtils.copyBytes((InputStream)HdfsClient.this.fs.open(file), (OutputStream)noteBytes, (Configuration)HdfsClient.this.hadoopConf);
                return new String(noteBytes.toString(HdfsClient.this.zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_ENCODING)));
            }
        });
    }

    public void writeFile(final String content, final Path file) throws IOException {
        this.callHdfsOperation(new HdfsOperation<Void>(){

            @Override
            public Void call() throws IOException {
                ByteArrayInputStream in = new ByteArrayInputStream(content.getBytes(HdfsClient.this.zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_ENCODING)));
                Path tmpFile = new Path(file.toString() + ".tmp");
                IOUtils.copyBytes((InputStream)in, (OutputStream)HdfsClient.this.fs.create(tmpFile), (Configuration)HdfsClient.this.hadoopConf);
                HdfsClient.this.fs.delete(file, true);
                HdfsClient.this.fs.rename(tmpFile, file);
                return null;
            }
        });
    }

    public void move(Path src, Path dest) throws IOException {
        this.callHdfsOperation(() -> {
            this.fs.rename(src, dest);
            return null;
        });
    }

    private synchronized <T> T callHdfsOperation(final HdfsOperation<T> func) throws IOException {
        if (this.isSecurityEnabled) {
            try {
                return (T)UserGroupInformation.getCurrentUser().doAs(new PrivilegedExceptionAction<T>(){

                    @Override
                    public T run() throws Exception {
                        return func.call();
                    }
                });
            }
            catch (InterruptedException e) {
                throw new IOException(e);
            }
        }
        return func.call();
    }

    public String parseText(String text) {
        String script = "";
        String intpText = "";
        if (text != null) {
            Matcher matcher = REPL_PATTERN.matcher(text);
            if (matcher.matches()) {
                String headingSpace = matcher.group(1);
                intpText = matcher.group(2);
                if (matcher.groupCount() == 3 && matcher.group(3) != null) {
                    String[] splits;
                    String localPropertiesText = matcher.group(3);
                    for (String split : splits = localPropertiesText.substring(1, localPropertiesText.length() - 1).split(",")) {
                        String[] kv = split.split("=");
                        if (StringUtils.isBlank(split) || kv.length == 0 || kv.length <= 2) continue;
                        throw new RuntimeException("Invalid paragraph properties format: " + split);
                    }
                    script = text.substring(headingSpace.length() + intpText.length() + localPropertiesText.length() + 1).trim();
                } else {
                    script = text.substring(headingSpace.length() + intpText.length() + 1).trim();
                }
            } else {
                script = text.trim();
            }
        }
        return script;
    }

    public String saveParagraphToFiles(String noteId, List<ParagraphInfo> paragraphInfos, String dirName, Properties properties) throws Exception {
        StringBuffer outputMsg = new StringBuffer();
        String hdfsUploadPath = properties.getProperty("submarine.algorithm.hdfs.path", "");
        HashMap<String, StringBuffer> mapParagraph = new HashMap<String, StringBuffer>();
        for (int i = 0; i < paragraphInfos.size(); ++i) {
            StringBuffer mergeScript;
            ParagraphInfo paragraph = paragraphInfos.get(i);
            String paragraphTitle = paragraph.getParagraphTitle();
            if (StringUtils.isEmpty(paragraphTitle)) {
                String message = "WARN: The title of the [" + i + "] paragraph is empty and was not submitted to HDFS.\n";
                LOGGER.warn(message);
                outputMsg.append(message);
                continue;
            }
            if (!mapParagraph.containsKey(paragraphTitle)) {
                mergeScript = new StringBuffer();
                mapParagraph.put(paragraphTitle, mergeScript);
            }
            mergeScript = (StringBuffer)mapParagraph.get(paragraphTitle);
            String parapraphText = paragraph.getParagraphText();
            String text = this.parseText(parapraphText);
            mergeScript.append(text + "\n\n");
        }
        if (!StringUtils.isEmpty(dirName)) {
            String noteDir = dirName + "/" + noteId;
            File fileNoteDir = new File(noteDir);
            if (fileNoteDir.exists()) {
                fileNoteDir.delete();
            }
            fileNoteDir.mkdirs();
        }
        if (!StringUtils.isEmpty(hdfsUploadPath)) {
            Path hdfsPath = new Path(hdfsUploadPath + "/" + noteId);
            try {
                if (this.exists(hdfsPath)) {
                    this.delete(hdfsPath);
                    this.tryMkDir(hdfsPath);
                }
            }
            catch (IOException e) {
                LOGGER.error(e.getMessage(), e);
                throw new Exception(e);
            }
        }
        for (Map.Entry entry : mapParagraph.entrySet()) {
            try {
                String fileName = (String)entry.getKey();
                String fileContext = ((StringBuffer)entry.getValue()).toString();
                String paragraphFile = dirName + "/" + noteId + "/" + fileName;
                if (!StringUtils.isEmpty(dirName)) {
                    File fileParagraph = new File(paragraphFile);
                    if (!fileParagraph.exists()) {
                        fileParagraph.createNewFile();
                    }
                    FileWriter writer = new FileWriter(paragraphFile);
                    writer.write(fileContext);
                    writer.close();
                }
                if (StringUtils.isEmpty(hdfsUploadPath)) continue;
                String fileDir = hdfsUploadPath + "/" + noteId + "/" + fileName;
                LOGGER.info("Commit algorithm to HDFS: {}", (Object)fileDir);
                Path filePath = new Path(fileDir);
                this.writeFile(fileContext, filePath);
            }
            catch (IOException e) {
                LOGGER.error(e.getMessage(), e);
                throw new Exception(e);
            }
        }
        return outputMsg.toString();
    }

    private static interface HdfsOperation<T> {
        public T call() throws IOException;
    }
}

