/*
 *	BIRD Library -- SHA and HMAC-SHA functions tests
 *
 *	(c) 2015 CZ.NIC z.s.p.o.
 *
 *	Can be freely distributed and used under the terms of the GNU GPL.
 */

#include "test/birdtest.h"
#include "test/bt-utils.h"

#include "lib/mac.h"


#define define_test_hash_fn(name,id)					\
static int								\
test_##name(void *out_, const void *in_, const void *expected_out_)	\
{									\
  char *out = out_;							\
  const char *in = in_;							\
  const char *expected_out = expected_out_;				\
									\
  struct mac_context ctx;						\
  mac_init(&ctx, id, NULL, 0);						\
  mac_update(&ctx, in, strlen(in));					\
  byte *out_bin = mac_final(&ctx);					\
									\
  uint len = mac_type_length(id);					\
  bt_bytes_to_hex(out, out_bin, len);					\
									\
  return strncmp(out, expected_out, 2*len+1) == 0; \
}

define_test_hash_fn(md5,	ALG_MD5)
define_test_hash_fn(sha1,	ALG_SHA1)
define_test_hash_fn(sha224,	ALG_SHA224)
define_test_hash_fn(sha256,	ALG_SHA256)
define_test_hash_fn(sha384,	ALG_SHA384)
define_test_hash_fn(sha512,	ALG_SHA512)


static int
t_md5(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = "",
      .out = "d41d8cd98f00b204e9800998ecf8427e",
    },
    {
      .in  = "a",
      .out = "0cc175b9c0f1b6a831c399e269772661",
    },
    {
      .in  = "abc",
      .out = "900150983cd24fb0d6963f7d28e17f72",
    },
    {
      .in  = "message digest",
      .out = "f96b697d7cb7938d525a2f31aaf161d0",
    },
    {
      .in  = "abcdefghijklmnopqrstuvwxyz",
      .out = "c3fcd3d76192e4007dfb496cca67e13b",
    },
    {
      .in  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
      .out = "d174ab98d277d9f5a5611c2c9f419d9f",
    },
    {
      .in  = "12345678901234567890123456789012345678901234567890123456789012345678901234567890",
      .out = "57edf4a22be3c955ac49da2e2107b67a",
    },
  };

  return bt_assert_batch(test_vectors, test_md5, bt_fmt_str, bt_fmt_str);
}


/*
 * Testing SHAxxx functions
 */


static int
t_sha1(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = "",
      .out = "da39a3ee5e6b4b0d3255bfef95601890afd80709",
    },
    {
      .in  = "a",
      .out = "86f7e437faa5a7fce15d1ddcb9eaeaea377667b8",
    },
    {
      .in  = "abc",
      .out = "a9993e364706816aba3e25717850c26c9cd0d89d",
    },
    {
      .in  = "message digest",
      .out = "c12252ceda8be8994d5fa0290a47231c1d16aae3",
    },
    {
      .in  = "abcdefghijklmnopqrstuvwxyz",
      .out = "32d10c7b8cf96570ca04ce37f2a19d84240d3a89",
    },
    {
      .in  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
      .out = "761c457bf73b14d27e9e9265c46f4b4dda11f940",
    },
    {
      .in  = "12345678901234567890123456789012345678901234567890123456789012345678901234567890",
      .out = "50abf5706a150990a08b2c5ea40fa0e585554732",
    },
    {
      .in  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
      .out = "6a64fcc1fb970f7339ce886601775d2efea5cd4b",
    },
  };

  return bt_assert_batch(test_vectors, test_sha1, bt_fmt_str, bt_fmt_str);
}

static int
t_sha224(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = "",
      .out = "d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f",
    },
    {
      .in  = "a",
      .out = "abd37534c7d9a2efb9465de931cd7055ffdb8879563ae98078d6d6d5",
    },
    {
      .in  = "abc",
      .out = "23097d223405d8228642a477bda255b32aadbce4bda0b3f7e36c9da7",
    },
    {
      .in  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
      .out = "75388b16512776cc5dba5da1fd890150b0c6455cb4f58b1952522525",
    },
    {
      .in  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
      .out = "cca7dd1a332a17775d8b0429bdb45055c2d4368ebaab0c7cf385586e",
    },
  };

  return bt_assert_batch(test_vectors, test_sha224, bt_fmt_str, bt_fmt_str);
}

static int
t_sha256(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = "",
      .out = "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855",
    },
    {
      .in  = "a",
      .out = "ca978112ca1bbdcafac231b39a23dc4da786eff8147c4e72b9807785afee48bb",
    },
    {
      .in  = "abc",
      .out = "ba7816bf8f01cfea414140de5dae2223b00361a396177a9cb410ff61f20015ad",
    },
    {
      .in  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
      .out = "248d6a61d20638b8e5c026930c3e6039a33ce45964ff2167f6ecedd419db06c1",
    },
    {
      .in  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
      .out = "bf18b43b61652b5d73f41ebf3d72e5e43aebf5076f497dde31ea3de9de4998ef",
    },
  };

  return bt_assert_batch(test_vectors, test_sha256, bt_fmt_str, bt_fmt_str);
}

static int
t_sha384(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = "",
      .out = "38b060a751ac96384cd9327eb1b1e36a21fdb71114be07434c0cc7bf63f6e1da274edebfe76f65fbd51ad2f14898b95b",
    },
    {
      .in  = "a",
      .out = "54a59b9f22b0b80880d8427e548b7c23abd873486e1f035dce9cd697e85175033caa88e6d57bc35efae0b5afd3145f31",
    },
    {
      .in  = "abc",
      .out = "cb00753f45a35e8bb5a03d699ac65007272c32ab0eded1631a8b605a43ff5bed8086072ba1e7cc2358baeca134c825a7",
    },
    {
      .in  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
      .out = "3391fdddfc8dc7393707a65b1b4709397cf8b1d162af05abfe8f450de5f36bc6b0455a8520bc4e6f5fe95b1fe3c8452b",
    },
    {
      .in  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
      .out = "6452928a62ca915a60f2d16ea22cc832d8ecb35443d78a3ff6986e7def9174a1dc16ce2ff65d3ed1666db98357f3c05e",
    },
  };

  return bt_assert_batch(test_vectors, test_sha384, bt_fmt_str, bt_fmt_str);
}

static int
t_sha512(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = "",
      .out = "cf83e1357eefb8bdf1542850d66d8007d620e4050b5715dc83f4a921d36ce9ce47d0d13c5d85f2b0ff8318d2877eec2f63b931bd47417a81a538327af927da3e",
    },
    {
      .in  = "a",
      .out = "1f40fc92da241694750979ee6cf582f2d5d7d28e18335de05abc54d0560e0f5302860c652bf08d560252aa5e74210546f369fbbbce8c12cfc7957b2652fe9a75",
    },
    {
      .in  = "abc",
      .out = "ddaf35a193617abacc417349ae20413112e6fa4e89a97ea20a9eeee64b55d39a2192992a274fc1a836ba3c23a3feebbd454d4423643ce80e2a9ac94fa54ca49f",
    },
    {
      .in  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
      .out = "204a8fc6dda82f0a0ced7beb8e08a41657c16ef468b228a8279be331a703c33596fd15c13b1b07f9aa1d3bea57789ca031ad85c7a71dd70354ec631238ca3445",
    },
    {
      .in  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
      .out = "415509a1c345371acb3e27a88b3835e3b6dfebcbbab5134850596f4db64d7bb22ac42c3cd179446a80c92b8be955460eb536eac01389a7e1fdf09d1dca83922f",
    },
  };

  return bt_assert_batch(test_vectors, test_sha512, bt_fmt_str, bt_fmt_str);
}


/*
 * Testing SHAxxx HMAC functions
 */

#define HMAC_BUFFER_SIZE 160
struct hmac_data_in {
  byte key[HMAC_BUFFER_SIZE];
  uint key_len;
  byte data[HMAC_BUFFER_SIZE];
  uint data_len;
};

static void
hmac_in_fmt(char *buf, size_t size, const void *data_)
{
  uint i;
  const struct hmac_data_in *data = data_;

  snprintf(buf, size, "data: '");
  for (i = 0; i < data->data_len; i++)
    snprintf(buf+strlen(buf), size-strlen(buf), bt_is_char(data->data[i]) ? "%c" : " 0x%02x", data->data[i]);

  snprintf(buf+strlen(buf), size-strlen(buf), "', key: '");
  for (i = 0; i < data->key_len; i++)
    snprintf(buf+strlen(buf), size-strlen(buf), bt_is_char(data->key[i]) ? "%c" : " 0x%02x", data->key[i]);
  snprintf(buf+strlen(buf), size-strlen(buf), "'");
}

#define define_test_hmac_fn(name,id)					\
static int								\
test_##name##_hmac(void *out_, const void *in_, const void *expected_out_)	\
{									\
  char *out = out_;							\
  const struct hmac_data_in *in = in_;					\
  const char *expected_out = expected_out_;				\
  									\
  struct mac_context ctx;						\
  mac_init(&ctx, id, in->key, in->key_len);				\
  mac_update(&ctx, in->data, in->data_len);				\
  byte *out_bin = mac_final(&ctx);					\
									\
  uint len = mac_type_length(id);					\
  bt_bytes_to_hex(out, out_bin, len);					\
									\
  return strncmp(out, expected_out, 2*len+1) == 0; \
}

define_test_hmac_fn(md5,	ALG_HMAC_MD5)
define_test_hmac_fn(sha1,	ALG_HMAC_SHA1)
define_test_hmac_fn(sha224,	ALG_HMAC_SHA224)
define_test_hmac_fn(sha256,	ALG_HMAC_SHA256)
define_test_hmac_fn(sha384,	ALG_HMAC_SHA384)
define_test_hmac_fn(sha512,	ALG_HMAC_SHA512)


static int
t_md5_hmac(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	},
	.key_len = 16,
	.data = "Hi There",
	.data_len = 8,
      },
      .out = "9294727a3638bb1c13f48ef8158bfc9d",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = "Jefe",
	.key_len = 4,
	.data = "what do ya want for nothing?",
	.data_len = 28,
      },
      .out = "750c783e6ab0b503eaa86e310a5db738",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 16,
	.data = {
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	},
	.data_len = 50,
      },
      .out = "56be34521d144c88dbb8c733f0e8b3f6",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
	  0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
	  0x15, 0x16, 0x17, 0x18, 0x19,
	},
	.key_len = 25,
	.data = {
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	},
	.data_len = 50,
      },
      .out = "697eaf0aca3a3aea3a75164746ffaa79",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	},
	.key_len = 16,
	.data = "Test With Truncation",
	.data_len = 20,
      },
      .out = "56461ef2342edc00f9bab995690efd4c",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 80,
	.data = "Test Using Larger Than Block-Size Key - Hash Key First",
	.data_len = 54,
      },
      .out = "6b1ab7fe4bd7bf8f0b62e6ce61b9d0cd",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 80,
	.data = "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data",
	.data_len = 73,
      },
      .out = "6f630fad67cda0ee1fb1f562db3aa53e",
    },
  };

  return bt_assert_batch(test_vectors, test_md5_hmac, hmac_in_fmt, bt_fmt_str);
}

static int
t_sha1_hmac(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	},
	.key_len = 20,
	.data = "Hi There",
	.data_len = 8,
      },
      .out = "b617318655057264e28bc0b6fb378c8ef146be00",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = "Jefe",
	.key_len = 4,
	.data = "what do ya want for nothing?",
	.data_len = 28,
      },
      .out = "effcdf6ae5eb2fa2d27416d5f184df9c259a7c79",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 20,
	.data = {
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	},
	.data_len = 50,
      },
      .out = "125d7342b9ac11cd91a39af48aa17b4f63f175d3",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
	  0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
	  0x15, 0x16, 0x17, 0x18, 0x19,
	},
	.key_len = 25,
	.data = {
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	},
	.data_len = 50,
      },
      .out = "4c9007f4026250c6bc8414f9bf50c86c2d7235da",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	},
	.key_len = 20,
	.data = "Test With Truncation",
	.data_len = 20,
      },
      .out = "4c1a03424b55e07fe7f27be1d58bb9324a9a5a04",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 80,
	.data = "Test Using Larger Than Block-Size Key - Hash Key First",
	.data_len = 54,
      },
      .out = "aa4ae5e15272d00e95705637ce8a3b55ed402112",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 80,
	.data = "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data",
	.data_len = 73,
      },
      .out = "e8e99d0f45237d786d6bbaa7965c7808bbff1a91",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
	  0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
	  0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
	  0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
	  0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
	  0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
	  0x61, 0x61, 0x61, 0x61,
	},
	.key_len = 64,
	.data = "Test Using key 64 bytes sized",
	.data_len = 29,
      },
      .out = "a55d4fb80962a6b3d2e720705314bee417d68cf6",
    },
  };

  return bt_assert_batch(test_vectors, test_sha1_hmac, hmac_in_fmt, bt_fmt_str);
}

static int
t_sha224_hmac(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	},
	.key_len = 20,
	.data = "Hi There",
	.data_len = 8,
      },
      .out = "896fb1128abbdf196832107cd49df33f47b4b1169912ba4f53684b22",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = "Jefe",
	.key_len = 4,
	.data = "what do ya want for nothing?",
	.data_len = 28,
      },
      .out = "a30e01098bc6dbbf45690f3a7e9e6d0f8bbea2a39e6148008fd05e44",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 20,
	.data = {
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	},
	.data_len = 50,
      },
      .out = "7fb3cb3588c6c1f6ffa9694d7d6ad2649365b0c1f65d69d1ec8333ea",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
	  0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
	  0x15, 0x16, 0x17, 0x18, 0x19,
	},
	.key_len = 25,
	.data = {
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	},
	.data_len = 50,
      },
      .out = "6c11506874013cac6a2abc1bb382627cec6a90d86efc012de7afec5a",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	},
	.key_len = 20,
	.data = "Test With Truncation",
	.data_len = 20,
      },
      .out = "0e2aea68a90c8d37c988bcdb9fca6fa8099cd857c7ec4a1815cac54c",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "Test Using Larger Than Block-Size Key - Hash Key First",
	.data_len = 54,
      },
      .out = "95e9a0db962095adaebe9b2d6f0dbce2d499f112f2d2b7273fa6870e",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
	.data_len = 152,
      },
      .out = "3a854166ac5d9f023f54d517d0b39dbd946770db9c2b95c9f6f565d1",
    },
  };

  return bt_assert_batch(test_vectors, test_sha224_hmac, hmac_in_fmt, bt_fmt_str);
}

static int
t_sha256_hmac(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	},
	.key_len = 20,
	.data = "Hi There",
	.data_len = 8,
      },
      .out = "b0344c61d8db38535ca8afceaf0bf12b881dc200c9833da726e9376c2e32cff7",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = "Jefe",
	.key_len = 4,
	.data = "what do ya want for nothing?",
	.data_len = 28,
      },
      .out = "5bdcc146bf60754e6a042426089575c75a003f089d2739839dec58b964ec3843",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 20,
	.data = {
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	},
	.data_len = 50,
      },
      .out = "773ea91e36800e46854db8ebd09181a72959098b3ef8c122d9635514ced565fe",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
	  0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
	  0x15, 0x16, 0x17, 0x18, 0x19,
	},
	.key_len = 25,
	.data = {
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	},
	.data_len = 50,
      },
      .out = "82558a389a443c0ea4cc819899f2083a85f0faa3e578f8077a2e3ff46729665b",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	},
	.key_len = 20,
	.data = "Test With Truncation",
	.data_len = 20,
      },
      .out = "a3b6167473100ee06e0c796c2955552bfa6f7c0a6a8aef8b93f860aab0cd20c5",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "Test Using Larger Than Block-Size Key - Hash Key First",
	.data_len = 54,
      },
      .out = "60e431591ee0b67f0d8a26aacbf5b77f8e0bc6213728c5140546040f0ee37f54",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
	.data_len = 152,
      },
      .out = "9b09ffa71b942fcb27635fbcd5b0e944bfdc63644f0713938a7f51535c3a35e2",
    },
  };

  return bt_assert_batch(test_vectors, test_sha256_hmac, hmac_in_fmt, bt_fmt_str);
}

static int
t_sha384_hmac(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	},
	.key_len = 20,
	.data = "Hi There",
	.data_len = 8,
      },
      .out = "afd03944d84895626b0825f4ab46907f15f9dadbe4101ec682aa034c7cebc59cfaea9ea9076ede7f4af152e8b2fa9cb6",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = "Jefe",
	.key_len = 4,
	.data = "what do ya want for nothing?",
	.data_len = 28,
      },
      .out = "af45d2e376484031617f78d2b58a6b1b9c7ef464f5a01b47e42ec3736322445e8e2240ca5e69e2c78b3239ecfab21649",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 20,
	.data = {
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	},
	.data_len = 50,
      },
      .out = "88062608d3e6ad8a0aa2ace014c8a86f0aa635d947ac9febe83ef4e55966144b2a5ab39dc13814b94e3ab6e101a34f27",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
	  0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
	  0x15, 0x16, 0x17, 0x18, 0x19,
	},
	.key_len = 25,
	.data = {
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	},
	.data_len = 50,
      },
      .out = "3e8a69b7783c25851933ab6290af6ca77a9981480850009cc5577c6e1f573b4e6801dd23c4a7d679ccf8a386c674cffb",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	},
	.key_len = 20,
	.data = "Test With Truncation",
	.data_len = 20,
      },
      .out = "3abf34c3503b2a23a46efc619baef897f4c8e42c934ce55ccbae9740fcbc1af4ca62269e2a37cd88ba926341efe4aeea",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "Test Using Larger Than Block-Size Key - Hash Key First",
	.data_len = 54,
      },
      .out = "4ece084485813e9088d2c63a041bc5b44f9ef1012a2b588f3cd11f05033ac4c60c2ef6ab4030fe8296248df163f44952",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
	.data_len = 152,
      },
      .out = "6617178e941f020d351e2f254e8fd32c602420feb0b8fb9adccebb82461e99c5a678cc31e799176d3860e6110c46523e",
    },
  };

  return bt_assert_batch(test_vectors, test_sha384_hmac, hmac_in_fmt, bt_fmt_str);
}

static int
t_sha512_hmac(void)
{
  struct bt_pair test_vectors[] = {
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	  0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	},
	.key_len = 20,
	.data = "Hi There",
	.data_len = 8,
      },
      .out = "87aa7cdea5ef619d4ff0b4241a1d6cb02379f4e2ce4ec2787ad0b30545e17cdedaa833b7d6b8a702038b274eaea3f4e4be9d914eeb61f1702e696c203a126854",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = "Jefe",
	.key_len = 4,
	.data = "what do ya want for nothing?",
	.data_len = 28,
      },
      .out = "164b7a7bfcf819e2e395fbe73b56e0a387bd64222e831fd610270cd7ea2505549758bf75c05a994a6d034f65f8f0e6fdcaeab1a34d4a6b4b636e070a38bce737",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	},
	.key_len = 20,
	.data = {
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	  0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	},
	.data_len = 50,
      },
      .out = "fa73b0089d56a284efb0f0756c890be9b1b5dbdd8ee81a3655f83e33b2279d39bf3e848279a722c806b485a47e67c807b946a337bee8942674278859e13292fb",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
	  0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
	  0x15, 0x16, 0x17, 0x18, 0x19,
	},
	.key_len = 25,
	.data = {
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	  0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	},
	.data_len = 50,
      },
      .out = "b0ba465637458c6990e5a8c5f61d4af7e576d97ff94b872de76f8050361ee3dba91ca5c11aa25eb4d679275cc5788063a5f19741120c4f2de2adebeb10a298dd",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	  0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	},
	.key_len = 20,
	.data = "Test With Truncation",
	.data_len = 20,
      },
      .out = "415fad6271580a531d4179bc891d87a650188707922a4fbb36663a1eb16da008711c5b50ddd0fc235084eb9d3364a1454fb2ef67cd1d29fe6773068ea266e96b",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "Test Using Larger Than Block-Size Key - Hash Key First",
	.data_len = 54,
      },
      .out = "80b24263c7c1a3ebb71493c1dd7be8b49b46d1f41b4aeec1121b013783f8f3526b56d037e05f2598bd0fd2215d6a1e5295e64f73f63f0aec8b915a985d786598",
    },
    {
      .in  = & (struct hmac_data_in) {
	.key = {
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	  0xaa,
	},
	.key_len = 131,
	.data = "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
	.data_len = 152,
      },
      .out = "e37b6a775dc87dbaa4dfa9f96e5e3ffddebd71f8867289865df5a32d20cdc944b6022cac3c4982b10d5eeb55c3e4de15134676fb6de0446065c97440fa8c6a58",
    },
  };

  return bt_assert_batch(test_vectors, test_sha512_hmac, hmac_in_fmt, bt_fmt_str);
}


/*
 * Testing SHAxxx concating independence
 */

#include "lib/sha256.h"
#include "lib/sha512.h"

static int
t_sha256_concating(void)
{
  char hash_a[SHA256_HEX_SIZE];
  char hash_b[SHA256_HEX_SIZE];

  char *str_a  = "a" "bb" "ccc" "dddd" "eeeee" "ffffff";
  char *str_b1 = "a"                                   ;
  char *str_b2 =     "bb"                              ;
  char *str_b3 =          "ccc"                        ;
  char *str_b4 =                "dddd"                 ;
  char *str_b5 =                       "eeeee"         ;
  char *str_b6 =                               "ffffff";

  struct hash_context ctx_a;
  sha256_init(&ctx_a);
  sha256_update(&ctx_a, str_a, strlen(str_a));
  byte *hash_a_ = sha256_final(&ctx_a);
  bt_bytes_to_hex(hash_a, hash_a_, SHA256_SIZE);

  struct hash_context ctx_b;
  sha256_init(&ctx_b);
  sha256_update(&ctx_b, str_b1, strlen(str_b1));
  sha256_update(&ctx_b, str_b2, strlen(str_b2));
  sha256_update(&ctx_b, str_b3, strlen(str_b3));
  sha256_update(&ctx_b, str_b4, strlen(str_b4));
  sha256_update(&ctx_b, str_b5, strlen(str_b5));
  sha256_update(&ctx_b, str_b6, strlen(str_b6));
  byte *hash_b_ = sha256_final(&ctx_b);
  bt_bytes_to_hex(hash_b, hash_b_, SHA256_SIZE);

  int are_hash_a_b_equal = (strncmp(hash_a, hash_b, sizeof(hash_a)) == 0);
  bt_assert_msg(are_hash_a_b_equal, "Hashes A: %s, B: %s should be same", hash_a, hash_b);

  return 1;
}


static int
t_sha512_concating(void)
{
  char hash_a[SHA512_HEX_SIZE];
  char hash_b[SHA512_HEX_SIZE];

  char *str_a  = "a" "bb" "ccc" "dddd" "eeeee" "ffffff";
  char *str_b1 = "a"                                   ;
  char *str_b2 =     "bb"                              ;
  char *str_b3 =          "ccc"                        ;
  char *str_b4 =                "dddd"                 ;
  char *str_b5 =                       "eeeee"         ;
  char *str_b6 =                               "ffffff";

  struct hash_context ctx_a;
  sha512_init(&ctx_a);
  sha512_update(&ctx_a, str_a, strlen(str_a));
  byte *hash_a_ = sha512_final(&ctx_a);
  bt_bytes_to_hex(hash_a, hash_a_, SHA512_SIZE);

  struct hash_context ctx_b;
  sha512_init(&ctx_b);
  sha512_update(&ctx_b, str_b1, strlen(str_b1));
  sha512_update(&ctx_b, str_b2, strlen(str_b2));
  sha512_update(&ctx_b, str_b3, strlen(str_b3));
  sha512_update(&ctx_b, str_b4, strlen(str_b4));
  sha512_update(&ctx_b, str_b5, strlen(str_b5));
  sha512_update(&ctx_b, str_b6, strlen(str_b6));
  byte *hash_b_ = sha512_final(&ctx_b);
  bt_bytes_to_hex(hash_b, hash_b_, SHA512_SIZE);

  int are_hash_a_b_equal = (strncmp(hash_a, hash_b, sizeof(hash_a)) == 0);
  bt_assert_msg(are_hash_a_b_equal, "Hashes A: %s, B: %s should be same", hash_a, hash_b);

  return 1;
}

int
main(int argc, char *argv[])
{
  bt_init(argc, argv);

  bt_test_suite(t_md5,    "Testing MD5 by RFC 1321");
  bt_test_suite(t_sha1,   "Testing SHA-1");
  bt_test_suite(t_sha224, "Testing SHA-224");
  bt_test_suite(t_sha256, "Testing SHA-256");
  bt_test_suite(t_sha384, "Testing SHA-384");
  bt_test_suite(t_sha512, "Testing SHA-512");

  bt_test_suite(t_md5_hmac, "Testing HMAC-MD5 by RFC 2202");
  bt_test_suite(t_sha1_hmac,   "Testing HMAC-SHA-1 by RFC 2202");
  bt_test_suite(t_sha224_hmac, "Testing HMAC-SHA-224 by RFC 4231");
  bt_test_suite(t_sha256_hmac, "Testing HMAC-SHA-256 by RFC 4231");
  bt_test_suite(t_sha384_hmac, "Testing HMAC-SHA-384 by RFC 4231");
  bt_test_suite(t_sha512_hmac, "Testing HMAC-SHA-512 by RFC 4231");

  bt_test_suite(t_sha256_concating, "Testing concatenation input string to hash using sha256_update");
  bt_test_suite(t_sha512_concating, "Testing concatenation input string to hash using sha512_update");

  return bt_exit_value();
}
