/*
 * fhist - file history and comparison tools
 * Copyright (C) 2000, 2002, 2004, 2008, 2010, 2012 Peter Miller
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <common/ac/stdio.h>
#include <common/ac/sys/stat.h>
#include <libexplain/fread.h>
#include <libexplain/getchar.h>

#include <common/error_intl.h>
#include <common/input/private.h>
#include <common/input/stdin.h>
#include <common/str.h>


static string_ty *
standard_input(void)
{
    static string_ty *name;
    sub_context_ty  *scp;

    if (!name)
    {
        scp = sub_context_new();
        name = subst_intl(scp, i18n("standard input"));
        sub_context_delete(scp);
    }
    return name;
}


static void
destruct(input_ty *this)
{
    (void)this;
}


static long
iread(input_ty *this, void *data, long len)
{
    (void)this;
    if (len <= 0)
        return 0;
    return explain_fread_or_die(data, 1, len, stdin);
}


static int
get(input_ty *this)
{
    (void)this;
    return explain_getchar_or_die();
}


static long
itell(input_ty *this)
{
    (void)this;
    return ftell(stdin);
}


static const char *
name(input_ty *this)
{
    (void)this;
    return standard_input()->str_text;
}


static long
length(input_ty *this)
{
    struct stat     st;

    (void)this;
    if (fstat(fileno(stdin), &st) < 0)
        return -1;
    if (!S_ISREG(st.st_mode))
        return -1;
    return st.st_size;
}


static input_vtbl_ty vtbl =
{
    sizeof(input_ty),
    destruct,
    iread,
    get,
    itell,
    name,
    length,
};


input_ty *
input_stdin(void)
{
    return input_new(&vtbl);
}


/* vim: set ts=8 sw=4 et : */
