% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_LMCurve.R
\name{fit_LMCurve}
\alias{fit_LMCurve}
\title{Non-linear Least Squares Fit for LM-OSL curves}
\usage{
fit_LMCurve(
  values,
  values.bg,
  n.components = 3,
  start_values,
  input.dataType = "LM",
  fit.method = "port",
  sample_code = "",
  sample_ID = "",
  LED.power = 36,
  LED.wavelength = 470,
  fit.trace = FALSE,
  fit.advanced = FALSE,
  fit.calcError = FALSE,
  bg.subtraction = "polynomial",
  verbose = TRUE,
  plot = TRUE,
  plot.BG = FALSE,
  method_control = list(),
  ...
)
}
\arguments{
\item{values}{\linkS4class{RLum.Data.Curve} or \link{data.frame} (\strong{required}):
x,y data of measured values (time and counts). See examples.}

\item{values.bg}{\linkS4class{RLum.Data.Curve} or \link{data.frame} (\emph{optional}):
x,y data of measured values (time and counts) for background subtraction.}

\item{n.components}{\link{integer} (\emph{with default}):
fixed number of components that are to be recognised during fitting
(min = 1, max = 7).}

\item{start_values}{\link{data.frame} (\emph{optional}):
start parameters for \code{lm} and \code{xm} data for the fit. If no start values are given,
an automatic start value estimation is attempted (see details).}

\item{input.dataType}{\link{character} (\emph{with default}):
alter the plot output depending on the input data: \code{"LM"} or \code{"pLM"} (pseudo-LM).
See: \link{convert_CW2pLM}}

\item{fit.method}{\link{character} (\emph{with default}):
select fit method, allowed values: \code{'port'} and \code{'LM'}. \code{'port'} uses the 'port'
routine from the function \link{nls} \code{'LM'} utilises the function \code{nlsLM} from
the package \code{minpack.lm} and with that the Levenberg-Marquardt algorithm.}

\item{sample_code}{\link{character} (\emph{optional}):
sample code used for the plot and the optional output table (mtext).}

\item{sample_ID}{\link{character} (\emph{optional}):
additional identifier used as column header for the table output.}

\item{LED.power}{\link{numeric} (\emph{with default}):
LED power (max.) used for intensity ramping in mW/cm².
\strong{Note:} This value is used for the calculation of the absolute
photoionisation cross section.}

\item{LED.wavelength}{\link{numeric} (\emph{with default}):
LED wavelength in nm used for stimulation.
\strong{Note:} This value is used for the calculation of the absolute
photoionisation cross section.}

\item{fit.trace}{\link{logical} (\emph{with default}):
traces the fitting process on the terminal.}

\item{fit.advanced}{\link{logical} (\emph{with default}):
enables advanced fitting attempt for automatic start parameter recognition.
Works only if no start parameters are provided.
\strong{Note:} It may take a while and it is not compatible with \code{fit.method = "LM"}.}

\item{fit.calcError}{\link{logical} (\emph{with default}):
calculate 1-sigma error range of components using \link[stats:confint]{stats::confint}.}

\item{bg.subtraction}{\link{character} (\emph{with default}):
specifies method for background subtraction (\code{polynomial}, \code{linear}, \code{channel},
see Details). \strong{Note:} requires input for \code{values.bg}.}

\item{verbose}{\link{logical} (\emph{with default}):
enable/disable output to the terminal.}

\item{plot}{\link{logical} (\emph{with default}):
enable/disable the plot output.}

\item{plot.BG}{\link{logical} (\emph{with default}):
returns a plot of the background values with the fit used for the
background subtraction.}

\item{method_control}{\link{list} (\emph{optional}): options to control the output
produced. Currently only the 'export.comp.contrib.matrix' (logical) option
is supported, to enable/disable export of the component contribution
matrix.}

\item{...}{Further arguments that may be passed to the plot output, e.g.
\code{xlab}, \code{xlab}, \code{main}, \code{log}.}
}
\value{
Various types of plots are returned. For details see above. Furthermore an
\code{RLum.Results} object is returned with the following structure:

\strong{\verb{@data:}}

\code{.. $data} : \link{data.frame} with fitting results\cr
\code{.. $fit} : nls (\link{nls} object)\cr
\code{.. $component_matrix} : \link{matrix} with numerical xy-values of the single fitted components with the resolution of the input data
\code{.. $component.contribution.matrix} : \link{list} component distribution matrix
(produced only if \code{method_control$export.comp.contrib.matrix = TRUE})

\strong{\verb{info:}}

\code{.. $call} : \link{call} the original function call

Matrix structure for the distribution matrix:

Column 1 and 2: time and \code{rev(time)} values\cr
Additional columns are used for the components, two for each component,
containing I0 and n0. The last columns \code{cont.} provide information on
the relative component contribution for each time interval including the row
sum for this values.
}
\description{
The function determines weighted non-linear least-squares estimates of the
component parameters of an LM-OSL curve (Bulur 1996) for a given number of
components and returns various component parameters. The fitting procedure
uses the function \link{nls} with the \code{port} algorithm.
}
\details{
\strong{Fitting function}

The function for the fitting has the general
form:

\deqn{y = (exp(0.5)*Im_1*x/xm_1)*exp(-x^2/(2*xm_1^2)) + ,\ldots, + exp(0.5)*Im_i*x/xm_i)*exp(-x^2/(2*xm_i^2))}

where \eqn{1 < i < 8}

This function and the equations for the conversion to b (detrapping probability)
and n0 (proportional to initially trapped charge) have been taken from Kitis
et al. (2008):

\deqn{xm_i=\sqrt{max(t)/b_i}}
\deqn{Im_i=exp(-0.5)n0/xm_i}

\strong{Background subtraction}

Three methods for background subtraction
are provided for a given background signal (\code{values.bg}).
\itemize{
\item \code{polynomial}: default method. A polynomial function is fitted using \link{glm}
and the resulting function is used for background subtraction:
\deqn{y = a*x^4 + b*x^3 + c*x^2 + d*x + e}
\item \code{linear}: a linear function is fitted using \link{glm} and the resulting function
is used for background subtraction:
\deqn{y = a*x + b}
\item \code{channel}: the measured
background signal is subtracted channel wise from the measured signal.
}

\strong{Start values}

The choice of the initial parameters for the \code{nls}-fitting is a crucial
point and the fitting procedure may mainly fail due to ill chosen start
parameters. Here, three options are provided:

\strong{(a)}
If no start values (\code{start_values}) are provided by the user, a cheap guess is made
by using the detrapping values found by Jain et al. (2003) for quartz for a
maximum of 7 components. Based on these values, the pseudo start parameters
\code{xm} and \code{Im} are recalculated for the given data set. In all cases, the fitting
starts with the ultra-fast component and (depending on \code{n.components})
steps through the following values. If no fit could be achieved, an error
plot (for \code{plot = TRUE}) with the pseudo curve (based on the
pseudo start parameters) is provided. This may give the opportunity to
identify appropriate start parameters visually.

\strong{(b)}
If start values are provided, the function works like a simple \link{nls}
fitting approach.

\strong{(c)}
If no start parameters are provided and
the option \code{fit.advanced = TRUE} is chosen, an advanced start parameter
estimation is applied using a stochastic attempt. Therefore, the
recalculated start parameters \strong{(a)} are used to construct a normal
distribution. The start parameters are then sampled randomly from this
distribution. A maximum of 100 attempts will be made. \strong{Note:} This
process may be time consuming.

\strong{Goodness of fit}

The goodness of the fit is given by a pseudo-R² value (pseudo coefficient of
determination). According to Lave (1970), the value is calculated as:

\deqn{pseudoR^2 = 1 - RSS/TSS}

where \eqn{RSS = Residual~Sum~of~Squares}
and \eqn{TSS = Total~Sum~of~Squares}

\strong{Error of fitted component parameters}

The 1-sigma error for the components is calculated using
the function \link[stats:confint]{stats::confint}. Due to considerable calculation time, this
option is deactivated by default. In addition, the error for the components
can be estimated by using internal R functions like \link{summary}. See the
\link{nls} help page for more information.

\emph{For more details on the nonlinear regression in R, see Ritz & Streibig (2008).}
}
\note{
The pseudo-R² may not be the best parameter to describe the goodness
of the fit. The trade off between the \code{n.components} and the pseudo-R²
value currently remains unconsidered.

The function \strong{does not} ensure that the fitting procedure has reached a
global minimum rather than a local minimum! In any case of doubt, the use of
manual start values is highly recommended.
}
\section{Function version}{
 0.3.5
}

\examples{

##(1) fit LM data without background subtraction
data(ExampleData.FittingLM, envir = environment())
fit_LMCurve(values = values.curve, n.components = 3, log = "x")

##(2) fit LM data with background subtraction and export as JPEG
## -alter file path for your preferred system
##jpeg(file = "~/Desktop/Fit_Output\\%03d.jpg", quality = 100,
## height = 3000, width = 3000, res = 300)
data(ExampleData.FittingLM, envir = environment())
fit_LMCurve(values = values.curve, values.bg = values.curveBG,
            n.components = 2, log = "x", plot.BG = TRUE)
##dev.off()

##(3) fit LM data with manual start parameters
data(ExampleData.FittingLM, envir = environment())
fit_LMCurve(values = values.curve,
            values.bg = values.curveBG,
            n.components = 3,
            log = "x",
            start_values = data.frame(Im = c(170,25,400), xm = c(56,200,1500)))

} 

\section{How to cite}{
Kreutzer, S., 2025. fit_LMCurve(): Non-linear Least Squares Fit for LM-OSL curves. Function version 0.3.5. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., Steinbuch, L., Boer, A.d., 2025. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 1.0.1. https://r-lum.github.io/Luminescence/
}

\references{
Bulur, E., 1996. An Alternative Technique For Optically
Stimulated Luminescence (OSL) Experiment. Radiation Measurements, 26, 5,
701-709.

Jain, M., Murray, A.S., Boetter-Jensen, L., 2003. Characterisation of
blue-light stimulated luminescence components in different quartz samples:
implications for dose measurement. Radiation Measurements, 37 (4-5),
441-449.

Kitis, G. & Pagonis, V., 2008. Computerized curve deconvolution analysis for
LM-OSL. Radiation Measurements, 43, 737-741.

Lave, C.A.T., 1970. The Demand for Urban Mass Transportation. The Review of
Economics and Statistics, 52 (3), 320-323.

Ritz, C. & Streibig, J.C., 2008. Nonlinear Regression with R. R. Gentleman,
K. Hornik, & G. Parmigiani, eds., Springer, p. 150.
}
\seealso{
\link{fit_CWCurve}, \link{plot}, \link{nls}, \link[minpack.lm:nlsLM]{minpack.lm::nlsLM}, \link{get_RLum}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)
, RLum Developer Team}
\keyword{dplot}
\keyword{models}
