// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// star.h - The Star class.
//

#ifndef _STAR_H_
#define _STAR_H_

#include "../../lib/compat.h"
#include "strings.h"
#include "specclass.h"
#include "vector3.h"
#include "viewer.h"
#include <cstdlib>
#include <cstdio>
#include <cctype>
#include <cmath>

// Specify the characters used to separate fields in a star text record
const char FIELD_DELIMITER = ';';
const char SUBFIELD_DELIMITER = ',';

// How many pixels a secondary star must be from its primary before
//  being drawn (implemented in StarArray::Read())
const unsigned int MIN_SECONDARY_DISTANCE = 2;

// Default radius, in pixels, with which a star is drawn
//  (implemented in Star::Display())
const unsigned int STAR_PIXEL_RADIUS = 2;

// function to hash spectral class.  This takes the one-letter class
//  description as argument and outputs the position of the class in
//  the Rules.StarClasses array (see comments below).  
inline size_t SpecHash(char spectrum)
{
  switch (toupper(spectrum)) {
    case 'O': case 'W': return 0;             // O and Wolf-Rayet stars
    case 'B': return 1;   
    case 'A': return 2;   
    case 'F': return 3;   
    case 'G': return 4;
    case 'K': return 5;
    case 'M': return 6;
    case 'D': return 7;                       // White dwarfs
    case '*': return 9;                       // Nonstellar objects
    default : return 8;                       // Unclassified stars
  }
}


// an enum to specify what kind of labels should be attached to stars:
//  full name, name only for stars with Bayer designations, numerical index,
//  or none.
enum star_label_t { NO_LABEL, NUMBER_LABEL, STRING_LABEL, LANDMARK_LABEL };

// an enum to specify whether the diameter of star icons should depend upon
//  MK class or upon magnitude
enum star_diameter_t { MK_DIAMETERS, MAGNITUDE_DIAMETERS };

#define CELESTIAL 1
#define GALACTIC  0

// a struct to keep all these rules for which stars get displayed in
//  one place.  This will be a private member of StarArray

struct Rules {
  // where we are reading the data from
  StringList   ChartFileNames;

  // which stars get displayed: used in the Star::PassesFilter() check
  Vector3      ChartLocation;
  SolidAngle   ChartOrientation;
  double       ChartRadius;
  float        ChartDimmestMagnitude;
  float        ChartBrightestMagnitude;
  bool	       ChartHideCompanions;

  // the following is included in this struct for convenience, but is not
  // used in the core classes; implementation is up to the GUI interface.
  bool         ChartAutosetDimmestMagnitude;

  // The elements of StarClasses tell whether or not to display a specific
  //  spectral class of star.  See the SpecHash function comment above.
  bool         StarClasses[10];

  // some other display toggles: used in the displays
  bool         CelestialCoords;
  star_diameter_t StarDiameters;
  star_label_t StarLabels;
  bool         StarBars;
  bool         ChartGrid;
  bool         ChartLegend;
  distance_unit ChartUnits[4];

  // constructors for convenience
  void Copy(const StringList &files, const Vector3 &posn,
            const SolidAngle &orientation, double rad,
            float dimmag, float brightmag, bool hide_comps,
	    bool autoset, const bool classes[10],
            bool coords, star_diameter_t diam, star_label_t label,
            bool bars, bool grid, bool legend,
	    const distance_unit units[4])
  {
    ChartFileNames   = files;
    ChartLocation    = posn;
    ChartOrientation = orientation;
    ChartRadius      = rad;
    ChartDimmestMagnitude = dimmag;
    ChartBrightestMagnitude = brightmag;
    ChartHideCompanions = hide_comps;
    ChartAutosetDimmestMagnitude = autoset;
    for (unsigned int i = 0; i < 10; i++) StarClasses[i] = classes[i];
    CelestialCoords  = coords;
    StarDiameters    = diam;
    StarLabels       = label;
    StarBars         = bars;
    ChartGrid        = grid;
    ChartLegend      = legend;
    for (unsigned int i = 0; i < 4; i++) ChartUnits[i] = units[i];
  }

  Rules(const StringList &files, const Vector3 &posn,
	const SolidAngle &orientation, double rad, float dimmag,
	float brightmag, bool hide_comps, bool autoset, const bool classes[10],
        bool coords, star_diameter_t diam, star_label_t label,
        bool bars, bool grid, bool legend,
	const distance_unit units[4])
  { Copy(files, posn, orientation, rad, dimmag, brightmag, hide_comps, autoset,
	 classes, coords, diam, label, bars, grid, legend, units);
  }

  Rules()
  { bool classes[10];
    for (unsigned int i = 0; i < 10; i++) classes[i] = true;
    distance_unit units[4] = { DIST_LY, DIST_LY, DIST_AU, DIST_KM };
    Copy(StringList(), Vector3(0,0,0), SolidAngle(0,0), 10, 25, -25,
	 true, true, classes, CELESTIAL, MAGNITUDE_DIAMETERS, LANDMARK_LABEL,
	 true, true, true, units);
  }

  Rules(const Rules &r)
  { Copy(r.ChartFileNames, r.ChartLocation, r.ChartOrientation, r.ChartRadius,
	 r.ChartDimmestMagnitude, r.ChartBrightestMagnitude,
	 r.ChartHideCompanions, r.ChartAutosetDimmestMagnitude,
	 r.StarClasses, r.CelestialCoords,
	 r.StarDiameters, r.StarLabels, r.StarBars, r.ChartGrid,
	 r.ChartLegend, r.ChartUnits);
  }

  ~Rules() { }
  
  Rules & operator= (const Rules &r)
  { if (&r != this)
      Copy(r.ChartFileNames, r.ChartLocation, r.ChartOrientation, r.ChartRadius,
      r.ChartDimmestMagnitude, r.ChartBrightestMagnitude,
      r.ChartHideCompanions, r.ChartAutosetDimmestMagnitude,
      r.StarClasses, r.CelestialCoords,
      r.StarDiameters, r.StarLabels, r.StarBars, r.ChartGrid, 
      r.ChartLegend, r.ChartUnits); 
    return *this;
  }
};


class Star {
 private:
  StringList sNames;            // list of designations for the star
  StringList sMembership;       // in one or more clusters or multiple systems
  StringList sComments;         // miscellaneous text information

  SolidAngle sGridPosn;         // star location, relative to sun, and size
  double sDistance, sDiameter;  //  (in radians and L-Y)
  double sPrimaryDistance;      // distance (L-Y) from primary; 0 if n/a

  double sMagnitude;            // absolute visual magnitude
  SpecClass sSpectrum;          // spectral type and luminosity class (O9.5 Ia)

  mutable size_t sPlace;	// numerical order of the star, for labels
  mutable unsigned int xPixel;  // location of the star on the plot
  mutable unsigned int yPixel;
  mutable unsigned int rPixel;  // radius of the star in pixels
  mutable bool sLabelDraw;      // whether to draw label if in landmark mode

  // function to save some typing in the copy constructor and operator=
  void Obtain(StringList, StringList, StringList, SolidAngle,
	      double, double, double, double, SpecClass, size_t,
	      unsigned int, unsigned int, unsigned int, bool);

 public:
  Star();
  Star(const Star &s); // copy constructor
  Star & operator=(const Star &s);

  // constructs a Star by parsing a text record:
  Star(const std::string &record,
       bool fastconversion = false, bool nameconvert = true);

  inline unsigned int GetPlace()  const { return sPlace; }
  inline void SetPlace(size_t place) const { sPlace = place; }
  inline unsigned int GetXPixel() const { return xPixel; }
  inline unsigned int GetYPixel() const { return yPixel; }
  inline unsigned int GetRPixel() const { return rPixel; }
  inline void SetLabel(bool label) const { sLabelDraw = label; }

  // convert celestial to galactic coordinates and vice versa (note this
  //  does NOT test to see whether it's already in desired coord. system)
  inline void toGalactic()  
    { 
      if (sDistance == 0.0) sGridPosn = SolidAngle(0.0, 0.0);
      else sGridPosn = sGridPosn.toGalactic(); 
    }

  inline void toCelestial() 
    { 
      if (sDistance == 0.0) sGridPosn = SolidAngle(0.0, 0.0);
      else sGridPosn = sGridPosn.toCelestial(); 
    }

  // get various pieces of information about the star
  inline Vector3 GetStarXYZ() const { return sGridPosn.toCartesian(sDistance); }
  inline double GetStarDistance()  const { return sDistance; }
  inline double GetStarMagnitude() const { return sMagnitude; }
  inline double GetStarPrimaryDistance() const { return sPrimaryDistance; }
  inline SpecClass  GetStarClass() const { return sSpectrum; }
  inline StringList GetStarNames() const { return sNames; }
  inline StringList GetStarMembership() const { return sMembership; }

  // for use by starconvert:
  inline void SetStarMembership(std::string s) { sMembership.push_back(s); }
  inline void SetStarPrimaryDistance(double d) { sPrimaryDistance = d; }

  // tells whether star should be displayed given the Rules
  bool PassesFilter(const Rules &rules) const;

  // plot the star onto the painting device, as viewed from the specified
  //  point and orientation given in Rules.
  //
  //  Note: StarViewer is a generic display class defined in viewer.h -- I am
  //  trying to keep the graphics-library dependent functions contained in
  //  descendant classes of StarViewer (e.g. KDEViewer, GTKViewer, etc.,
  //  which will be wrapper classes around the given graphics libraries).
  //  This will make the code a bit more confusing, but make the program
  //  easier to port.
  void Display(const Rules &rules, StarViewer *sv) const;

  // a lighter-weight drawing function that just plots the star at a specific
  //  point (returns the radius of star drawn, in pixels)
  unsigned int Draw(const Rules &rules, StarViewer *sv, int xposn, int yposn,
		    int pixelradius = 0) const;

  // write star information to a StringList:
  StringList GetInfo(const Rules &rules, bool punctuation = true,
		     char sep = ' ') const;
};


#endif
