use core:lang;
use core:asm;

/**
 * Lambda expressions. There are two types of them: one requires specifying types in the parameter
 * list, but can be used anywhere. The other does not require types in the parameter list, but needs
 * to be used in a context where the types of the parameters can be inferred (not implemented yet).
 *
 * We should also support captures in some manner.
 */
class LambdaExpr extends Expr {
	// Parameters to the function. Parameters where the type is 'void' were not specified at the
	// declaration site and need to be inferred.
	private Array<ValParam> fnParams;

	// The type returned by this expression. Set to 'void' if we don't know due to unspecified parameters.
	private Value type;

	// Set of captured parameters if known beforehand.
	private Maybe<Set<Str>> captured;

	// Body of the lambda function.
	private SExpr body;

	// Parent block.
	private Block parent;

	// Create a lambda expression where the types of all parameters are known.
	init(SrcPos pos, Block parent, Array<NameParam> params, SExpr body) {
		var p = params.resolve(parent.scope);
		Scope scope = lambdaScope(parent);
		var info = lambdaInfo(scope, p, body, parent);
		Value result = info.result.type.asRef(false);

		SimpleName name;
		name.add("core");
		name.add("Fn", { var z = p.values(); z.insert(0, result); z; });

		init(pos) {
			fnParams = params.resolve(parent.scope);
			type = rootScope.value(name, pos);
			body = body;
			parent = parent;
		}
	}

	// Create a lambda expression where the type of the parameters are unknown and have to be deduced.
	init(SrcPos pos, Block parent, Array<SStr> paramNames, SExpr body) {
		ValParam[] params;
		for (i in paramNames)
			params << ValParam(Value(), i);

		init(pos) {
			fnParams = params;
			type = Value();
			body = body;
			parent = parent;
		}
	}

	ExprResult result() {
		ExprResult(type);
	}

	// Attempt automatic type deduction if we need to.
	Int castPenalty(Value to) {
		if (to == type) {
			0;
		} else if (acceptable(to)) {
			10;
		} else {
			-1;
		}
	}

	void code(CodeGen to, CodeResult r) {
		unless (fnType = r.type.type) {
			throw SyntaxError(pos, "Unable to deduce parameter types for the lambda function.");
			return;
		}

		if (!acceptable(r.type))
			throw SyntaxError(pos, "Unable to deduce parameter types for the lambda function.");

		if (!r.needed())
			return;

		ValParam[] params;
		for (i, p in fnParams)
			params << ValParam(fnType.params[i + 1], p.name);

		Str[] capture = capturedVars(params);
		if (capture.any)
			captureCode(to, r, capture, fnType.params[0], params);
		else
			plainCode(to, r, fnType.params[0], params);
	}

	private void plainCode(CodeGen to, CodeResult r, Value result, ValParam[] params) {
		BSTreeFn fn(result, SStr("<lambda>", pos), params, to.runOn.thread);
		fn.parentLookup = lambdaParent(parent);
		fn.body = {
			FnBody b(fn, lambdaScope(parent));
			b.add(body.transform(b));
			b;
		};

		var v = r.location(to);
		to.l << mov(v, objPtr(pointer(fn)));
		r.created(to);
	}

	private void captureCode(CodeGen to, CodeResult r, Str[] capture, Value result, ValParam[] params) {
		Type type("<lambda>", TypeFlags:typeClass);
		type.parentLookup = lambdaParent(parent);
		// TODO: Maybe we want to support 'known, but not statically' as well. I don't think it will
		// make much of a difference in this case though.
		if (t = to.runOn.thread)
			type.setThread(t);

		var create = setupType(type, capture);

		params.insert(0, thisParam(type));
		BSTreeFn fn(result, SStr("<run>", pos), params, null);
		type.add(fn);
		fn.body = {
			FnBody b(fn, parent.scope.child(type));
			b.add(body.transform(b));
			b;
		};

		FnPtr p(create, fn, pos);
		p.code(to, r);
	}

	// Set up the type 'type' for use as a container for the local variables 'capture'. Returns an
	// Expr that creates an instance of 'type' properly initialized.
	private Expr setupType(Type type, Str[] capture) {
		MemberVar[] vars;
		ValParam[] formals;
		formals << thisParam(type);

		Actuals actuals;
		for (c in capture) {
			if (v = parent.variable(SimplePart(c))) {
				MemberVar m(c, v.result, type);
				type.add(m);
				vars << m;

				formals << ValParam(v.result, c);

				actuals.add(LocalVarAccess(pos, v));
			}
		}

		// Create a constructor.
		BSTreeCtor ctor(formals, pos);
		ctor.body = {
			CtorBody b(ctor, rootScope); // We don't need a proper scope.
			Initializer[] init;
			for (v in vars) {
				if (formal = b.variable(SimplePart(v.name))) {
					init << Initializer(SStr(v.name), LocalVarAccess(pos, formal));
				}
			}
			InitBlock s(pos, b, Actuals(), init);
			b.add(s);
			b;
		};
		type.add(ctor);

		CtorCall(pos, rootScope, ctor, actuals);
	}

	protected void toS(StrBuf to) {
		to << "(";
		for (i, p in fnParams) {
			if (i > 0)
				to << ", ";
			if (p.type != Value())
				to << p.type << " ";
			to << p.name;
		}
		to << ") => <lambda body>";
		// TODO: It would be nice to output a nice formatted body here, but at the moment that is
		// not easily done since we need to transform the body in a suitable context for that to
		// work. As it is now, we don't necessarily know that context here.
	}

	// Is 'type' an acceptable return type given what we currently know?
	private Bool acceptable(Value type) {
		unless (fn = type.type as FnType)
			return false;

		// Proper number of parameters?
		if (fn.params.count != fnParams.count + 1)
			return false;

		// Make sure that all parameter types that are specified are properly set.
		for (i, p in fnParams) {
			if (p.type == Value()) {
				// Not specified. Anything goes!
			} else if (p.type != fn.params[i + 1]) {
				// Non-compatible type. Not possible!
				return false;
			}
		}

		// All is well!
		true;
	}

	// Get the captured parameters as an array.
	private Str[] capturedVars(ValParam[] params) {
		Set<Str> r = if (captured) {
			captured;
		} else {
			lambdaInfo(lambdaScope(parent), params, body, parent).capture;
		};

		Str[] result;
		for (i in r)
			result << i;
		result;
	}
}


/**
 * Wrap a lambda function body in a block that contains the parameters as variables, much like how a
 * BSRawFn would do.
 */
private ExprBlock wrapLambda(Scope scope, Array<ValParam> params, SExpr body) on Compiler {
	SrcPos pos = body.pos;
	ExprBlock r(pos, scope);

	for (p in params) {
		r.add(LocalVar(p.name, p.type, pos, true));
	}

	r.add(body.transform(r));

	r;
}

/**
 * Information about a lambda function.
 */
private class LambdaInfo extends NameLookup {
	private Block block;
	public Set<Str> capture;
	public ExprResult result;

	init(NameLookup? parent, Block block) {
		init() {
			block = block;
		}
		parentLookup = parent;
	}

	Named? find(SimplePart part, Scope source) {
		if (var = block.variable(part)) {
			capture.put(part.name);
			return var;
		} else {
			return null;
		}
	}
}

/**
 * Figure out the return type and any captured parameters of a lambda function.
 */
private LambdaInfo lambdaInfo(Scope scope, Array<ValParam> params, SExpr body, Block parent) on Compiler {
	LambdaInfo l(scope.top, parent);
	scope.top = l;

	l.result = wrapLambda(scope, params, body).result;

	l;
}

/**
 * Find the proper scope for a lambda function.
 */
private Scope lambdaScope(Block block) on Compiler {
	NameLookup at = block.lookup;
	while (true) {
		if (at as NameSet) {
			return block.scope.child(at);
		} else if (parent = at.parent) {
			at = parent;
		} else {
			return rootScope;
		}
	}

	// Crude fallback. Should not happen.
	return rootScope();
}

/**
 * Find the proper parent to a lambda function.
 */
private NameLookup lambdaParent(Block block) on Compiler {
	NameLookup at = block.lookup;
	while (true) {
		// print(at.toS);
		if (at as Package) {
			return at;
		} else if (at as Type) {
			return at;
		} else if (parent = at.parent) {
			at = parent;
		} else {
			return rootPkg;
		}
	}

	return rootPkg;
}
