static char rcsid[] = "$Id: kmer-search.c 223751 2020-12-14 04:44:47Z twu $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_MEMCPY
#define memcpy(d,s,n) bcopy((s),(d),(n))
#endif

#include "kmer-search.h"

#include "assert.h"
#include "mem.h"
#include "bool.h"
#include "types.h"
#include "chrnum.h"
#include "reader.h"
#include "oligo.h"

#ifndef LARGE_GENOMES
#include "merge-diagonals-simd-uint4.h"
#elif !defined(HAVE_AVX512) && !defined(HAVE_AVX2)
#include "merge-diagonals-heap.h" /* For Merge_diagonals_large */
#include "merge-diagonals-simd-uint4.h"
#else
#include "merge-diagonals-simd-uint8.h" /* For Merge_diagonals_large */
#include "merge-diagonals-simd-uint4.h"
#endif

#ifdef LARGE_GENOMES
#include "intersect-large.h"
#endif
#include "intersect.h"
#include "transcript.h"
#include "popcount.h"
#include "genome128_hr.h"
#include "substring.h"
#include "junction.h"

#include "univdiag.h"
#include "univdiagdef.h"
#include "genome128_consec.h"
#include "splice.h"
#include "indel.h"
#include "intron.h"
#include "maxent_hr.h"
#include "knownsplicing.h"
#include "sedgesort.h"


#if defined(HAVE_SSE2)
#include <emmintrin.h>
#endif
#ifdef HAVE_SSE4_1
#include <smmintrin.h>
#endif
#ifdef HAVE_AVX2
#include <immintrin.h>
#endif
#ifdef HAVE_AVX512
#include <immintrin.h>
#endif


#define MIN_SUPPORT_INDEL 6


/* General flow */
#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

/* Processing merged diagonals */
#ifdef DEBUG0
#define debug0(x) x
#else
#define debug0(x)
#endif

/* Exon and path overlap */
#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif

/* Transcriptome and one end */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif

/* Transcriptome and one end details*/
#ifdef DEBUG2A
#define debug2a(x) x
#else
#define debug2a(x)
#endif

/* One end */
#ifdef DEBUG3
#define debug3(x) x
#else
#define debug3(x)
#endif

/* Both_ends */
#ifdef DEBUG4
#define debug4(x) x
#else
#define debug4(x)
#endif

/* Both end details */
#ifdef DEBUG4A
#define debug4a(x) x
#else
#define debug4a(x)
#endif


/* Trimming */
#ifdef DEBUG8
#define debug8(x) x
#else
#define debug8(x)
#endif

/* Oligos and remapping */
#ifdef DEBUG9
#define debug9(x) x
#else
#define debug9(x)
#endif

/* Binary search */
#ifdef DEBUG10
#define debug10(x) x
#else
#define debug10(x)
#endif


/* Known splicing for genome */
#ifdef DEBUG4S
#define debug4s(x) x
#else
#define debug4s(x)
#endif

/* Find best path genome */
#ifdef DEBUG13
#define debug13(x) x
#else
#define debug13(x)
#endif


/* Merging faster than count table */
#define USE_MERGE 1

#define MAX_NEIGHBORS 3		/* Cannot be 0 */
#define SUBOPT 3

#define LONG_END 6
#define ALLOWED_END_MISMATCHES 2 /* For long ends */
#define ALLOWED_TRANSCRIPTOME_TRIM 3


static Mode_T mode;

static Univ_IIT_T chromosome_iit;
static Univcoord_T genomelength;
static Trcoord_T transcriptomelength;
static int circular_typeint;
static bool *circularp;

static Genome_T genomebits;
static Genome_T genomebits_alt;

static Indexdb_T indexdb;
static Indexdb_T indexdb2;
static Indexdb_T indexdb_tr;

static bool splicingp;
static Univcoord_T *splicesites;
static Splicetype_T *splicetypes;
static Chrpos_T *splicedists;
static int nsplicesites = 0;


static Chrpos_T min_intronlength;

static int index1part_tr;
static int index1part;
static int index1interval;
static int local1part;

static int leftreadshift;
static Oligospace_T oligobase_mask;


/* Indexdb: oligo + diagterm -> streams of diagonals */
/* Merge: an array of diagonals with duplicates */
/* Path: genomic endpoints + gaps + trnums */

/* All calls to Substring_new are for transcriptome.  May need to make call to Univ_IIT_update_chrnum */

/* Transcriptome ends */
/* Ultrafast: check ends only.  Require extension to ends and nmismatches <= nmismatches_allowed */

/* Transcriptome complete */
/* Most prevalent: merged diagonals -> most prevalent diagonals (called loci), unique */

/* Algorithm: Check for indels.  Require extension to ends - 3 bp.
   Create lefts,querystarts,queryends,adjustments.  Convert to genomic
   paths */

/* single_hits: paths -> hits */

/* Genome */
/* Ultrafast: check ends only */
/* find_local_sets: merged->diagonals -> middle_diagonal, left_diagonals, right_diagonals */
/* Algorithm 1a: find_best_path_genome -> complete_path (list of Univdiag_T) */
/* Algorithm 1b: Kmer_search_genome (solve_via_segments_genome): complete_path -> hits */
/* Algorithm 2: Stage3end_complete_path_run_gmap: complete_path -> hits */


#ifdef DEBUG8
#ifdef HAVE_SSE4_1
static void
print_vector_dec (__m128i x) {
  printf("%d %d %d %d\n",
	 _mm_extract_epi32(x,0),_mm_extract_epi32(x,1),_mm_extract_epi32(x,2),_mm_extract_epi32(x,3));
  return;
}
#endif
#endif


#if !defined(HAVE_SSE4_2)
#define count_leading_zeroes_32(diff) ((diff >> 16) ? clz_table[diff >> 16] : 16 + clz_table[diff])
#elif defined(HAVE_LZCNT)
#define count_leading_zeroes_32(diff) _lzcnt_u32(diff)
#elif defined(HAVE_BUILTIN_CLZ)
#define count_leading_zeroes_32(diff) __builtin_clz(diff)
#else
#define count_leading_zeroes_32(diff) ((diff >> 16) ? clz_table[diff >> 16] : 16 + clz_table[diff])
#endif

#if !defined(HAVE_SSE4_2)
#define count_trailing_zeroes_32(diff) mod_37_bit_position[(-diff & diff) % 37]
#elif defined(HAVE_TZCNT)
#define count_trailing_zeroes_32(diff) _tzcnt_u32(diff)
#elif defined(HAVE_BUILTIN_CTZ)
#define count_trailing_zeroes_32(diff) __builtin_ctz(diff)
#else
/* lowbit = -diff & diff */
#define count_trailing_zeroes_32(diff) mod_37_bit_position[(-diff & diff) % 37]
#endif



/* Overwrite values array */
/* Repetitive oligos can calse false indel diagonals */
static int
most_prevalent_uint (int *nloci, UINT4 *values, int nvalues) {
  int max_count, count;
  UINT4 *out, *ptr, *end, *first;

  assert(nvalues > 0);

  ptr = out = &(values[0]);	/* Reset */
  end = &(values[nvalues]);

  max_count = 1;
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %u",*first));
    if (ptr + max_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[max_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[max_count - 1] != ptr[max_count - 2]) {
	/* Can jump immediately */
	ptr += max_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      ptr += max_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
      debug0(printf(" => Count %d, index %d => printing\n",ptr - first,first - values));
      if ((count = ptr - first) > max_count) {
	out = &(values[0]);	/* Reset */
	max_count = count;
      }
      *out++ = *first;
    }
  }

  *nloci = out - &(values[0]);
  return max_count;
}


#if 0
static int
most_prevalent_count (UINT4 *values, int nvalues) {
  int max_count, count;
  UINT4 *ptr, *end, *first;

  assert(nvalues > 0);

  ptr = values;
  end = &(values[nvalues]);

  max_count = 1;
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %u",*first));
    if (ptr + max_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[max_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[max_count - 1] != ptr[max_count - 2]) {
	/* Can jump immediately */
	ptr += max_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      ptr += max_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
      debug0(printf(" => Count %d\n",ptr - first));
      if ((count = ptr - first) > max_count) {
	max_count = count;
      }
    }
  }

  return max_count;
}
#endif



#if 0
static int
most_prevalent_uint_simd (int *nloci, UINT4 *values, int nvalues) {
  int max_count, count;
  UINT4 *ptr, *end, last_value;
  int i;

  assert(nvalues > 0);

  ptr = &(values[0]);
  end = &(values[n]);

  max_count = 1;
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %u",*first));

    if (ptr + max_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[max_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      _first = _mm_set1_epi32(*first);
      _values = _mm_loadu_si128((__m128i *) ptr);

      match = _mm_cmpeq_epi32(_values,_first);
      matchbits = _mm_movemask_ps(_mm_castsi128_ps(matc));

#if !defined(HAVE_SSE4_2)
      nmatches = __builtin_popcount(matchbits);
#elif defined(HAVE_POPCNT)
      nmatches = _popcnt32(matchbits);
#elif defined HAVE_MM_POPCNT
      nmatches = _mm_popcnt_u32(matchbits);
#else
      nmatches = __builtin_popcount(matchbits);
#endif
      count = nmatches;

      while (nmatches == 4) {
	ptr += 4;
	_values = _mm_loadu_si128((__m128i *) ptr);

	match = _mm_cmpeq_epi32(_values,_first);
	matchbits = _mm_movemask_ps(_mm_castsi128_ps(matc));

#if !defined(HAVE_SSE4_2)
	nmatches = __builtin_popcount(matchbits);
#elif defined(HAVE_POPCNT)
	nmatches = _popcnt32(matchbits);
#elif defined HAVE_MM_POPCNT
	nmatches = _mm_popcnt_u32(matchbits);
#else
	nmatches = __builtin_popcount(matchbits);
#endif
	count += nmatches;
      }
    }
  }

  *out++ = count;
  ptr += nmatches;
}

#endif


struct Path_T {
  int total_nmatches;
  int n_shared_endpoints;		/* Set during disambiguate_paths to find best path */

  Transcript_T initial_transcript;

  List_T transcripts;		/* Set during disambiguate_paths */
  List_T transcripts_other;	/* Set during disambiguate_paths */

  Chrnum_T chrnum;
  Univcoord_T chroffset;
  Univcoord_T chrhigh;
  Chrpos_T chrlength;

  int nexons;
  Uintlist_T endpoints;		/* Chrpos_T */
  Intlist_T gaps;

  int trim_low;
  int trim_high;

  bool concordantp;
};

static void
Path_free (Path_T *old) {
  List_T p;
  Transcript_T transcript;

  /* Don't need to free initial_transcript, since it has been pushed onto transcripts or transcripts_other of a representative path */
  assert((*old)->initial_transcript == NULL);

  for (p = (*old)->transcripts; p != NULL; p = List_next(p)) {
    transcript = (Transcript_T) List_head(p);
    Transcript_free(&transcript);
  }
  List_free(&(*old)->transcripts);

  for (p = (*old)->transcripts_other; p != NULL; p = List_next(p)) {
    transcript = (Transcript_T) List_head(p);
    Transcript_free(&transcript);
  }
  List_free(&(*old)->transcripts_other);

  Uintlist_free(&(*old)->endpoints);
  Intlist_free(&(*old)->gaps);
  FREE(*old);

  return;
}


static Path_T
Path_new (int total_nmatches, Trnum_T trnum, int trstart, int trend, int trim_low, int trim_high,
	  Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh, Chrpos_T chrlength,
	  Uintlist_T endpoints, Intlist_T gaps, bool concordantp) {
  Path_T new = (Path_T) MALLOC(sizeof(*new));
  
  new->total_nmatches = total_nmatches;

  new->initial_transcript = Transcript_new(trnum,trstart,trend);

  new->transcripts = (List_T) NULL;
  new->transcripts_other = (List_T) NULL;

  new->chrnum = chrnum;
  new->chroffset = chroffset;
  new->chrhigh = chrhigh;
  new->chrlength = chrlength;

  new->nexons = Uintlist_length(endpoints)/2;
  new->endpoints = endpoints;
  new->gaps = gaps;

  new->trim_low = trim_low;
  new->trim_high = trim_high;

  new->concordantp = concordantp;

  return new;
}


static int
Path_cmp (const void *x, const void *y) {
  Path_T a = * (Path_T *) x;
  Path_T b = * (Path_T *) y;
  Chrpos_T a0, b0;
  
  if (a->total_nmatches < b->total_nmatches) {
    return -1;
  } else if (b->total_nmatches < a->total_nmatches) {
    return +1;

  } else if (a->n_shared_endpoints > b->n_shared_endpoints) {
    return -1;
  } else if (b->n_shared_endpoints > a->n_shared_endpoints) {
    return +1;

  } else if (a->chrnum < b->chrnum) {
    return -1;
  } else if (b->chrnum < a->chrnum) {
    return +1;

  } else {
    a0 = Uintlist_head(a->endpoints);
    b0 = Uintlist_head(b->endpoints);

    if (a0 < b0) {
      return -1;
    } else if (a0 > b0) {
      return +1;

    } else {
      a0 = Uintlist_last_value(a->endpoints);
      b0 = Uintlist_last_value(b->endpoints);
      if (a0 < b0) {
	return -1;
      } else if (a0 > b0) {
	return +1;
      } else {
	return 0;
      }
    }
  }
}


static bool
exon_overlap_fwd_p (Uintlist_T p, Uintlist_T q) {
  Chrpos_T low, high;
  int overlap, length1, length2;

  if (Uintlist_head(Uintlist_next(p)) < Uintlist_head(q)) {
    return false;
  } else if (Uintlist_head(Uintlist_next(q)) < Uintlist_head(p)) {
    return false;
  } else if (Uintlist_head(p) == Uintlist_head(q)) {
    return true;
  } else if (Uintlist_head(Uintlist_next(p)) == Uintlist_head(Uintlist_next(q))) {
    return true;
  } else {
    low = Uintlist_head(p);
    if (Uintlist_head(q) > low) {
      low = Uintlist_head(q);
    }

    high = Uintlist_head(Uintlist_next(p));
    if (Uintlist_head(Uintlist_next(q)) < high) {
      high = Uintlist_head(Uintlist_next(q));
    }
    debug1(printf("overlap: %u..%u\n",low,high));
    overlap = high - low - 1;

    low = Uintlist_head(p);
    high = Uintlist_head(Uintlist_next(p));
    length1 = high - low - 1;
    p = Uintlist_next(Uintlist_next(p));

    low = Uintlist_head(q);
    high = Uintlist_head(Uintlist_next(q));
    length2 = high - low - 1;
    q = Uintlist_next(Uintlist_next(q));

    if (length1 > length2) {
      if (overlap < 0.8 * (double) length1) {
	return false;
      }
    } else {
      if (overlap < 0.8 * (double) length2) {
	return false;
      }
    }

    return true;
  }
}

static bool
exon_overlap_rev_p (Uintlist_T p, Uintlist_T q) {
  Chrpos_T low, high;
  int overlap, length1, length2;

  if (Uintlist_head(p) < Uintlist_head(Uintlist_next(q))) {
    return false;
  } else if (Uintlist_head(q) < Uintlist_head(Uintlist_next(p))) {
    return false;
  } else if (Uintlist_head(Uintlist_next(p)) == Uintlist_head(Uintlist_next(q))) {
    return true;
  } else if (Uintlist_head(p) == Uintlist_head(q)) {
    return true;
  } else {
    low = Uintlist_head(Uintlist_next(p));
    if (Uintlist_head(Uintlist_next(q)) > low) {
      low = Uintlist_head(Uintlist_next(q));
    }

    high = Uintlist_head(p);
    if (Uintlist_head(q) < high) {
      high = Uintlist_head(q);
    }
    debug1(printf("overlap: %u..%u\n",low,high));
    overlap = high - low - 1;

    low = Uintlist_head(Uintlist_next(p));
    high = Uintlist_head(p);
    length1 = high - low - 1;
    p = Uintlist_next(Uintlist_next(p));

    low = Uintlist_head(Uintlist_next(q));
    high = Uintlist_head(q);
    length2 = high - low - 1;
    q = Uintlist_next(Uintlist_next(q));

    if (length1 > length2) {
      if (overlap < 0.8 * (double) length1) {
	return false;
      }
    } else {
      if (overlap < 0.8 * (double) length2) {
	return false;
      }
    }

    return true;
  }
}


static bool
Path_equivalent_p (Path_T a, Path_T b) {
  if (a->total_nmatches == b->total_nmatches) {
    return true;
  } else {
    return false;
  }
}


static bool
Path_overlap_p (Path_T a, Path_T b) {
  Uintlist_T p, q;
  Chrpos_T low, high;
  int overlap, length1, length2;

  if (a->chrnum != b->chrnum) {
    return false;

  } else if (Uintlist_last_value(a->endpoints) < Uintlist_head(b->endpoints)) {
    /* No overlap */
    return false;

  } else if (Uintlist_last_value(b->endpoints) < Uintlist_head(a->endpoints)) {
    /* No overlap */
    return false;

  } else {
    /* Compute amount of overlap */
    overlap = length1 = length2 = 0;
    p = a->endpoints;
    q = b->endpoints;

    while (p != NULL && q != NULL) {
      debug1(printf("%u..%u vs %u..%u\n",
		    Uintlist_head(p),Uintlist_head(Uintlist_next(p)),
		    Uintlist_head(q),Uintlist_head(Uintlist_next(q))));

      if (Uintlist_head(Uintlist_next(p)) < Uintlist_head(q)) {
	debug1(printf("p is entirely before q\n"));
	low = Uintlist_head(p);
	high = Uintlist_head(Uintlist_next(p));
	length1 += high - low - 1;
	p = Uintlist_next(Uintlist_next(p));

      } else if (Uintlist_head(Uintlist_next(q)) < Uintlist_head(p)) {
	debug1(printf("q is entirely before p\n"));
	low = Uintlist_head(q);
	high = Uintlist_head(Uintlist_next(q));
	length2 += high - low - 1;
	q = Uintlist_next(Uintlist_next(q));

      } else {
	low = Uintlist_head(p);
	if (Uintlist_head(q) > low) {
	  low = Uintlist_head(q);
	}

	high = Uintlist_head(Uintlist_next(p));
	if (Uintlist_head(Uintlist_next(q)) < high) {
	  high = Uintlist_head(Uintlist_next(q));
	}
	debug1(printf("overlap: %u..%u\n",low,high));
	overlap += high - low - 1;

	if (Uintlist_head(p) == Uintlist_head(q)) {
	  low = Uintlist_head(p);
	  high = Uintlist_head(Uintlist_next(p));
	  length1 += high - low - 1;
	  p = Uintlist_next(Uintlist_next(p));

	  low = Uintlist_head(q);
	  high = Uintlist_head(Uintlist_next(q));
	  length2 += high - low - 1;
	  q = Uintlist_next(Uintlist_next(q));

	} else if (Uintlist_head(p) < Uintlist_head(q)) {
	  low = Uintlist_head(p);
	  high = Uintlist_head(Uintlist_next(p));
	  length1 += high - low - 1;
	  p = Uintlist_next(Uintlist_next(p));

	} else {
	  low = Uintlist_head(q);
	  high = Uintlist_head(Uintlist_next(q));
	  length2 += high - low - 1;
	  q = Uintlist_next(Uintlist_next(q));
	}
      }
    }

    while (p != NULL) {
      low = Uintlist_head(p);
      high = Uintlist_head(Uintlist_next(p));
      length1 += high - low - 1;
      p = Uintlist_next(Uintlist_next(p));
    }
      
    while (q != NULL) {
      low = Uintlist_head(q);
      high = Uintlist_head(Uintlist_next(q));
      length2 += high - low - 1;
      q = Uintlist_next(Uintlist_next(q));
    }

    debug1(printf("Overlap is %d\n",overlap));
    if (length1 > length2) {
      if (overlap < 0.8 * (double) length1) {
	return false;
      }
    } else {
      if (overlap < 0.8 * (double) length2) {
	return false;
      }
    }

    /* Look for a common low point */
    p = a->endpoints;
    q = b->endpoints;
    while (p != NULL && q != NULL) {
      if (Uintlist_head(p) == Uintlist_head(q)) {
	return true;
      } else if (Uintlist_head(p) < Uintlist_head(q)) {
	p = Uintlist_next(Uintlist_next(p));
      } else {
	q = Uintlist_next(Uintlist_next(q));
      }
    }

    /* Look for a common high point */
    p = a->endpoints;
    q = b->endpoints;
    while (p != NULL && q != NULL) {
      if (Uintlist_head(Uintlist_next(p)) == Uintlist_head(Uintlist_next(q))) {
	return true;
      } else if (Uintlist_head(Uintlist_next(p)) < Uintlist_head(Uintlist_next(q))) {
	p = Uintlist_next(Uintlist_next(p));
      } else {
	q = Uintlist_next(Uintlist_next(q));
      }
    }
      
    return false;
  }
}


static int
Univcoord_cmp (const void *a, const void *b) {
  Univcoord_T x = * (Univcoord_T *) a;
  Univcoord_T y = * (Univcoord_T *) b;

  if (x < y) {
    return -1;
  } else if (y < x) {
    return +1;
  } else {
    return 0;
  }
}
  


/* IDEA: Instead of trying to predict overlap, try absorbing paths
   until they can no longer be combined.  Implement a combine_paths
   procedure */

/* Each path is in ascending order: low to high coordinates.  First entry is trnum, then trstart, then chrnum. */
static List_T
disambiguate_paths (Path_T *paths, int npaths) {
  List_T all_paths = NULL;
  Univcoordlist_T all_endpoints_list;
  int i, j, k, l, nendpoints, ndup;
  Uintlist_T final_endpoints, pstart, pend, p, *ptrs;
  Intlist_T final_gaps, gstart, g;
  Path_T path, representative_path, temp;
  Univcoord_T *all_endpoints, *dup_endpoints,
    overall_start, overall_end, initialexon_high, finalexon_low;
  int trim_low, trim_high;
  bool foundp;
  int nconcordant;


  debug2(printf("Entering disambiguate_paths\n"));

  /* Sort by goodness: nmatches, then number of endpoints in common with the group */
  all_endpoints_list = (Univcoordlist_T) NULL;
  for (i = 0; i < npaths; i++) {
    for (p = paths[i]->endpoints; p != NULL; p = Uintlist_next(p)) {
      all_endpoints_list = Univcoordlist_push(all_endpoints_list,paths[i]->chroffset + Uintlist_head(p));
    }
  }
  all_endpoints = (Univcoord_T *) Univcoordlist_to_array(&nendpoints,all_endpoints_list);
  Univcoordlist_free(&all_endpoints_list);
  qsort(all_endpoints,nendpoints,sizeof(Univcoord_T),Univcoord_cmp);

  ndup = 0;
  dup_endpoints = (Univcoord_T *) MALLOC(nendpoints*sizeof(Univcoord_T));
  i = 0;
  while (i < nendpoints) {
    j = i + 1;
    while (j < nendpoints && all_endpoints[j] == all_endpoints[i]) {
      j++;
    }
    if (j - i > 1) {
      dup_endpoints[ndup++] = all_endpoints[i];
    }
    i = j;
  }
  FREE(all_endpoints);

  for (i = 0; i < npaths; i++) {
    l = 0;
    k = 0;
    p = paths[i]->endpoints;
    while (k < ndup && p != NULL) {
      if (dup_endpoints[k] < paths[i]->chroffset + Uintlist_head(p)) {
	k++;
      } else if (paths[i]->chroffset + Uintlist_head(p) < dup_endpoints[k]) {
	p = Uintlist_next(p);
      } else {
	k++;
	p = Uintlist_next(p);
	l++;
      }
    }
    paths[i]->n_shared_endpoints = l;
  }
  FREE(dup_endpoints);
    
  qsort(paths,npaths,sizeof(Uintlist_T),Path_cmp);
  for (i = 0; i < npaths; i++) {
    debug2(printf("Path %d (%d matches, %d shared endpoints): #%d:%s\n",
		  i,paths[i]->total_nmatches,paths[i]->n_shared_endpoints,
		  paths[i]->chrnum,Uintlist_to_string(paths[i]->endpoints)));
  }
  /* Done with sort */


  ptrs = (Uintlist_T *) MALLOC(npaths*sizeof(Uintlist_T));

  i = 0;
  while (i < npaths) {
    debug2(printf("Path %d: %s\n",i,Uintlist_to_string(paths[i]->endpoints)));
    /* last_value = Uintlist_last_value(paths[i]); */
    j = i + 1;
    while (j < npaths && Path_equivalent_p(paths[j],paths[i]) == true &&
	   Path_overlap_p(paths[j],paths[i]) == true) {
      j++;
    }
    debug2(printf("Cluster ends at %d\n",j));
    
    if (j - i == 1) {
      /* Cluster size is 1 */
      representative_path = paths[i];
      if (representative_path->concordantp == true) {
	representative_path->transcripts = List_push(NULL,(void *) representative_path->initial_transcript);
	representative_path->initial_transcript = (Transcript_T) NULL;
	all_paths = List_push(all_paths,(void *) representative_path);
	debug2(printf("Putting initial transcript for %d onto %d\n",i,i));
      } else {
	/* Shortcut for freeing initial transcript */
	Transcript_free(&representative_path->initial_transcript);
	Path_free(&representative_path);
      }

    } else {
      nconcordant = 0;
      for (k = i; k < j; k++) {
	if (paths[k]->concordantp == true) {
	  nconcordant += 1;
	}
      }

      if (nconcordant == 0) {
	for (k = i; k < j; k++) {
	  /* Shortcut for freeing initial transcript */
	  Transcript_free(&paths[k]->initial_transcript);
	  Path_free(&(paths[k]));
	}

      } else {
	k = i;
	l = i + nconcordant;
	while (k < i + nconcordant && l < j) {
	  while (k < i + nconcordant && paths[k]->concordantp == true) {
	    k++;
	  }
	  while (l < j && paths[l]->concordantp == false) {
	    l++;
	  }
	  if (k < i + nconcordant && l < j) {
	    /* Swap */
	    temp = paths[k];
	    paths[k] = paths[l];
	    paths[l] = temp;
	    k++; l++;
	  }
	}
	
	representative_path = paths[i];
	representative_path->transcripts = List_push(NULL,(void *) representative_path->initial_transcript);
	representative_path->initial_transcript = (Transcript_T) NULL;
	debug2(printf("Putting initial transcript for %d onto %d\n",i,i));

	for (k = i + nconcordant; k < j; k++) {
	  path = paths[k];
	  representative_path->transcripts_other =
	    List_push(representative_path->transcripts_other,(void *) path->initial_transcript);
	  path->initial_transcript = (Transcript_T) NULL;
	  Path_free(&path);
	}

	if (nconcordant == 1) {
	  all_paths = List_push(all_paths,(void *) representative_path);

	} else {
	  /* Need to find first common exon across all clusters */
	  debug2(printf("Finding initialexon_high\n"));
	  foundp = false;
	  ptrs[i] = paths[i]->endpoints;
	  while (ptrs[i] != NULL && foundp == false) {
	    foundp = true;
	    for (k = i + 1; k < i + nconcordant; k++) {
	      ptrs[k] = paths[k]->endpoints;
	      while (ptrs[k] != NULL && exon_overlap_fwd_p(ptrs[i],ptrs[k]) == false) {
		ptrs[k] = Uintlist_next(Uintlist_next(ptrs[k]));
	      }
	      if (ptrs[k] == NULL) {
		foundp = false;
	      }
	    }
	    if (foundp == false) {
	      ptrs[i] = Uintlist_next(Uintlist_next(ptrs[i]));
	    }
	  }
      
	  debug2(printf("foundp is %d\n",foundp));
	  if (foundp == true) {
	    pstart = ptrs[i];
	    overall_start = Uintlist_head(ptrs[i]);
	    initialexon_high = Uintlist_head(Uintlist_next(ptrs[i]));
	    for (k = i + 1; k < i + nconcordant; k++) {
	      if (Uintlist_head(ptrs[k]) > overall_start) {
		overall_start = Uintlist_head(ptrs[k]);
	      }
	      if (Uintlist_head(Uintlist_next(ptrs[k])) < initialexon_high) {
		initialexon_high = Uintlist_head(Uintlist_next(ptrs[k]));
	      }
	    }
	    debug2(printf("overall_start is %u\n",overall_start));
	    debug2(printf("initialexon_high is %u\n",initialexon_high));
	  }

	  debug2(printf("Finding finalexon_low\n"));
	  /* Reverse all endpoints */
	  for (k = i; k < i + nconcordant; k++) {
	    paths[k]->endpoints = Uintlist_reverse(paths[k]->endpoints);
	  }

	  foundp = false;
	  ptrs[i] = paths[i]->endpoints;
	  while (ptrs[i] != NULL && foundp == false) {
	    foundp = true;
	    for (k = i + 1; k < i + nconcordant; k++) {
	      ptrs[k] = paths[k]->endpoints;
	      while (ptrs[k] != NULL && exon_overlap_rev_p(ptrs[i],ptrs[k]) == false) {
		ptrs[k] = Uintlist_next(Uintlist_next(ptrs[k]));
	      }
	      if (ptrs[k] == NULL) {
		foundp = false;
	      }
	    }
	    if (foundp == false) {
	      ptrs[i] = Uintlist_next(Uintlist_next(ptrs[i]));
	    }
	  }
      
	  debug2(printf("foundp is %d\n",foundp));
	  if (foundp == true) {
	    pend = Uintlist_next(ptrs[i]);
	    overall_end = Uintlist_head(ptrs[i]);
	    finalexon_low = Uintlist_head(Uintlist_next(ptrs[i]));
	    for (k = i + 1; k < i + nconcordant; k++) {
	      if (Uintlist_head(ptrs[k]) < overall_end) {
		overall_end = Uintlist_head(ptrs[k]);
	      }
	      if (Uintlist_head(Uintlist_next(ptrs[k])) > finalexon_low) {
		finalexon_low = Uintlist_head(Uintlist_next(ptrs[k]));
	      }
	    }
	    debug2(printf("finalexon_low is %u\n",finalexon_low));
	  }

#ifdef DEBUG2
	  if (foundp == true) {
	    printf("initialexon high %u, finalexon_low %u\n",initialexon_high,finalexon_low);
	  }
#endif
	  
	  /* Restore all endpoints */
	  for (k = i; k < i + nconcordant; k++) {
	    paths[k]->endpoints = Uintlist_reverse(paths[k]->endpoints);
	  }

	  /* Handle cases where one transcript may extend into intron and
	     another splices.  Want largest elt in each paths[k] that is
	     less than initialexon_high, and smallest elt that is greater
	     than finalexon_low. */
#if 0
	  overall_start = 0;
	  overall_end = (Univcoord_T) -1;
	  for (k = i; k < i + nconcordant; k++) {
	    p = paths[k]->endpoints = Uintlist_reverse(paths[k]->endpoints);
	    /* Skip entire exons */
	    while (Uintlist_head(Uintlist_next(p)) > initialexon_high) {
	      p = Uintlist_next(Uintlist_next(p));
	    }
	    p = Uintlist_next(p);
	    if (Uintlist_head(p) > overall_start) {
	      overall_start = Uintlist_head(p);
	    }

	    p = paths[k]->endpoints = Uintlist_reverse(paths[k]->endpoints);
	    /* Skip entire exons */
	    while (Uintlist_head(Uintlist_next(p)) < finalexon_low) {
	      p = Uintlist_next(Uintlist_next(p));
	    }
	    p = Uintlist_next(p);
	    if (Uintlist_head(p) < overall_end) {
	      overall_end = Uintlist_head(p);
	    }
	  }
#endif

#ifdef DEBUG2
	  if (foundp == true) {
	    printf("OVERALL START %u, END %u\n",overall_start,overall_end);
	  }
#endif

	  if (foundp == false) {
	    all_paths = List_push(all_paths,(void *) representative_path);
	    for (k = i + 1; k < i + nconcordant; k++) {
	      path = paths[k];
	      Transcript_free(&path->initial_transcript);
	      Path_free(&path);
	    }

	  } else {
	    /* Put overall_start and overall_end on the right exons */
	    trim_low = representative_path->trim_low;
	    trim_high = representative_path->trim_high;
	    debug2(printf("INITIAL TRIMS: low %d, high %d\n",trim_low,trim_high));

	    debug2(printf("\n"));

	    p = representative_path->endpoints;
	    g = representative_path->gaps;

	    /* Skip entire exons */
	    while (p != pstart) {
	      debug2(printf("Skipping exon %u..%u which is before initial exon\n",
			    Uintlist_head(p),Uintlist_head(Uintlist_next(p))));
	      trim_low += Uintlist_head(Uintlist_next(p)) - Uintlist_head(p);
	      debug2(printf("Adding to trim_low (whole exon): %d = %u - %u\n",
			    Uintlist_head(Uintlist_next(p)) - Uintlist_head(p),
			    Uintlist_head(Uintlist_next(p)),Uintlist_head(p)));
	      p = Uintlist_next(Uintlist_next(p));
	      g = Intlist_next(g);
	    }
	    gstart = g;

	    trim_low += overall_start - Uintlist_head(pstart);
	    debug2(printf("Adding to trim_low (partial exon): %d = %u - %u\n",
			  overall_start - Uintlist_head(pstart),
			  overall_start,Uintlist_head(pstart)));
	    Uintlist_head_set(pstart,overall_start);

	    while (p != pend) {
	      p = Uintlist_next(Uintlist_next(p));
	    }

	    trim_high += Uintlist_head(Uintlist_next(p)) - overall_end;
	    debug2(printf("Adding to trim_high (partial exon): %d = %u - %u\n",
			  Uintlist_head(Uintlist_next(p)) - overall_end,
			  Uintlist_head(Uintlist_next(p)),overall_end));
	    Uintlist_head_set(Uintlist_next(p),overall_end);

	    p = Uintlist_next(Uintlist_next(p));
	    while (p != NULL) {
	      trim_high += Uintlist_head(Uintlist_next(p)) - Uintlist_head(p);
	      debug2(printf("Adding to trim_high (whole exon): %d = %u - %u\n",
			    Uintlist_head(Uintlist_next(p)) - Uintlist_head(p),
			    Uintlist_head(Uintlist_next(p)),Uintlist_head(p)));
	      p = Uintlist_next(Uintlist_next(p));
	    }

	    representative_path->trim_low = trim_low;
	    representative_path->trim_high = trim_high;
	    debug2(printf("FINAL TRIMS: low %d, high %d\n",trim_low,trim_high));


	    for (k = i+1; k < i + nconcordant; k++) {
	      path = paths[k];
	      debug2(printf("Putting initial transcript for %d onto %d\n",k,i));
	      representative_path->transcripts =
		List_push(representative_path->transcripts,(void *) paths[k]->initial_transcript);
	      paths[k]->initial_transcript = (Transcript_T) NULL;
	      Path_free(&path);
	    }

	    final_endpoints = (Uintlist_T) NULL;
	    final_gaps = (Intlist_T) NULL;
	    for (p = pstart, g = gstart; p != pend; p = Uintlist_next(Uintlist_next(p))) {
	      final_endpoints = Uintlist_push(final_endpoints,Uintlist_head(p));
	      final_endpoints = Uintlist_push(final_endpoints,Uintlist_head(Uintlist_next(p)));
	      if (Uintlist_next(Uintlist_next(p)) != NULL) {
		final_gaps = Intlist_push(final_gaps,Intlist_head(g));
		g = Intlist_next(g);
	      }
	    }
	    /* Handle pend */
	    final_endpoints = Uintlist_push(final_endpoints,Uintlist_head(p));
	    final_endpoints = Uintlist_push(final_endpoints,Uintlist_head(Uintlist_next(p)));
#if 0
	    if (Uintlist_next(Uintlist_next(p)) != NULL) {
	      final_gaps = Intlist_push(final_gaps,Intlist_head(g));
	      g = Intlist_next(g);
	    }
#endif

	    assert(Intlist_length(final_gaps) == Uintlist_length(final_endpoints)/2 - 1);
	    
	    final_endpoints = Uintlist_reverse(final_endpoints);
	    final_gaps = Intlist_reverse(final_gaps);
	    debug2(printf("FINAL ENDPOINTS: %s.  FINAL GAPS: %s.  TRIM LOW: %d, TRIM HIGH: %d\n\n",
			  Uintlist_to_string(final_endpoints),Intlist_to_string(final_gaps),trim_low,trim_high));
	    Uintlist_free(&(representative_path->endpoints));
	    Intlist_free(&(representative_path->gaps));
	    representative_path->endpoints = final_endpoints;
	    representative_path->gaps = final_gaps;
	    
	    all_paths = List_push(all_paths,(void *) representative_path);
	  }
	}
      }
    }

    i = j;
  }

  FREE(ptrs);

#if 0
  printf("PERFORMING CHECK OF INITIAL TRANSCRIPTS\n");
  for (a = all_paths; a != NULL; a = List_next(a)) {
    path = (Path_T) List_head(a);
    if (path->initial_transcript != NULL) {
      printf("Still have an initial_transcript that was not put into transcripts or transcripts_other in a representative path\n");
      abort();
    }
    for (b = path->transcripts; b != NULL; b = List_next(b)) {
      printf("Transcript %p\n",List_head(b));
    }
    for (b = path->transcripts_other; b != NULL; b = List_next(b)) {
      printf("Transcript other %p\n",List_head(b));
    }
  }
#endif

  return all_paths;
}


static List_T
single_hits_gplus (int *found_score_overall, int *found_score_within_trims,
		   List_T hits, Path_T *gplus_paths, int ngplus,
		   int querylength, Compress_T query_compress_fwd,
		   int nmismatches_allowed, int sensedir,
		   Listpool_T listpool, Hitlistpool_T hitlistpool, int level) {
  Stage3end_T hit;
  List_T all_paths, substrings, junctions, q, r;
  Uintlist_T p;
  Intlist_T g;
  Path_T path, representative_path;
  int querystart, queryend, pos5, pos3;
  Univcoord_T alignstart, alignend, prev_alignend, left, univdiagonal;
  Chrpos_T splice_distance;
  int nmismatches_whole, nmismatches_refdiff, nmismatches, ref_nmismatches, nindels, ninserts;
  Substring_T substring;
  Junction_T junction;
  bool abortp;

  debug(printf("Entered single_hits_gplus with sensedir %d and %d gplus paths\n",sensedir,ngplus));

  if (ngplus == 0) {
    all_paths = (List_T) NULL;
  } else if (ngplus == 1) {
    representative_path = gplus_paths[0];
    debug2(printf("Putting initial transcript for 0 onto 0\n"));
    representative_path->transcripts = List_push(NULL,representative_path->initial_transcript);
    representative_path->initial_transcript = (Transcript_T) NULL;
    all_paths = List_push(NULL,(void *) representative_path);
  } else {
    all_paths = disambiguate_paths(gplus_paths,ngplus);
  }

  for (q = all_paths; q != NULL; q = List_next(q)) {
    abortp = false;
    nmismatches_whole = 0;
    nmismatches_refdiff = 0;
    substrings = junctions = (List_T) NULL;

    path = (Path_T) List_head(q);
    debug2(printf("path %p has %d exons.  trim_low %d, trim_high %d.  Endpoints %s\n",
		  path,path->nexons,path->trim_low,path->trim_high,Uintlist_to_string(path->endpoints)));
    /* gplus: Do not reverse endpoints or gaps */
    querystart = path->trim_low; /* gplus */
    prev_alignend = 0;
    for (p = path->endpoints, g = path->gaps; p != NULL; p = Uintlist_next(Uintlist_next(p))) {
      alignstart = Uintlist_head(p) + path->chroffset;
      alignend = Uintlist_head(Uintlist_next(p)) + path->chroffset;
      ninserts = 0;
      if (prev_alignend > 0) {
	if ((nindels = Intlist_head(g)) > 0) {
	  debug2(printf("Setting value of deletionpos to be %u\n",alignstart - nindels));
	  junction = Junction_new_deletion(nindels,/*deletionpos*/alignstart - nindels); /* gplus */
	} else if (nindels < 0) {
	  junction = Junction_new_insertion(-nindels);
	  ninserts = -nindels;
	  debug2(printf("Revising ninserts to be %d\n",ninserts));
	} else {
	  splice_distance = alignstart - prev_alignend; /* gplus */
	  debug2(printf("splice_distance %u = %u - %u\n",splice_distance,alignstart,prev_alignend));
	  junction = Junction_new_splice(splice_distance,sensedir,/*donor_prob*/2.0,/*acceptor_prob*/2.0);
	}
	junctions = Listpool_push(junctions,listpool,(void *) junction);
	g = Intlist_next(g);
      }
      querystart += ninserts;
      queryend = querystart + (alignend - alignstart); /* gplus */

      /* gplus */
      left = alignstart - querystart;
      univdiagonal = left + querylength;

      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5 = (univdiagonal + querystart >= path->chroffset + querylength) ? querystart : (int) (path->chroffset - left);
      pos3 = (univdiagonal + queryend <= path->chrhigh + querylength) ? queryend : (int) (path->chrhigh - left);

      debug2(printf("tr gplus query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
		    querystart,queryend,alignstart,alignend,alignstart - path->chroffset,alignend - path->chroffset,
		    left - path->chroffset,alignend - path->chroffset,querylength,queryend));
      nmismatches =
	Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress_fwd,left,
					  pos5,pos3,/*plusp*/true,/*genestrand*/0);
      nmismatches_whole += nmismatches;
      nmismatches_refdiff += ref_nmismatches;
      debug2(printf("nmismatches fwd from %d to %d: %d\n",pos5,pos3,nmismatches));

      if ((substring = Substring_new(nmismatches,ref_nmismatches,left,pos5,pos3,querylength,
				     /*plusp*/true,/*genestrand*/0,query_compress_fwd,
				     path->chrnum,path->chroffset,path->chrhigh,path->chrlength,
				     /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				     /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				     sensedir)) == NULL) {
	debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
	abortp = true;
      } else {
	debug2(printf(" => Created substring for %d..%d\n",querystart,queryend));
	substrings = Listpool_push(substrings,listpool,(void *) substring);
      }
	
      querystart = queryend;
      prev_alignend = alignend;
    }

    if (abortp == true || nmismatches_whole > nmismatches_allowed) {
      debug2(printf("ABORTING STAGE3END:  abortp %d.  nmismatches_whole %d vs nmismatches_allowed %d\n\n",
		    abortp,nmismatches_whole,nmismatches_allowed));
      for (r = substrings; r != NULL; r = List_next(r)) {
	substring = (Substring_T) List_head(r);
	Substring_free(&substring);
      }
      /* List_free(&substrings); -- allocated by Listpool_push */

      for (r = junctions; r != NULL; r = List_next(r)) {
	junction = (Junction_T) List_head(r);
	Junction_free(&junction);
      }
      /* List_free(&junctions); -- allocated by Listpool_push */

    } else {
      substrings = List_reverse(substrings);
      junctions = List_reverse(junctions);
      hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
				      /*nmismatches_bothdiff*/nmismatches_whole,nmismatches_refdiff,
				      substrings,junctions,path->transcripts,path->transcripts_other,
				      querylength,path->chrnum,path->chroffset,path->chrhigh,path->chrlength,
				      /*gplusp*/true,/*genestrand*/0,sensedir,listpool,/*method*/TR,level);
      debug(printf("Created new transcript hit %p with %d transcripts\n",hit,List_length(path->transcripts)));
      hits = Hitlist_push(hits,hitlistpool,(void *) hit);
      assert(path->initial_transcript == NULL);
      path->transcripts = (List_T) NULL; /* hit owns it now */
      path->transcripts_other = (List_T) NULL; /* hit owns it now */
    }
  }

  for (q = all_paths; q != NULL; q = List_next(q)) {
    path = (Path_T) List_head(q);
    Path_free(&path);
  }
  List_free(&all_paths);

  return hits;
}


static List_T
single_hits_gminus (int *found_score_overall, int *found_score_within_trims,
		    List_T hits, Path_T *gminus_paths, int ngminus,
		    int querylength, Compress_T query_compress_rev,
		    int nmismatches_allowed, int sensedir,
		    Listpool_T listpool, Hitlistpool_T hitlistpool, int level) {
  Stage3end_T hit;
  List_T all_paths, substrings, junctions, q, r;
  Uintlist_T p;
  Intlist_T g;
  Path_T path, representative_path;
  int querystart, queryend, pos5, pos3;
  Univcoord_T alignstart, alignend, prev_alignend, left, univdiagonal;
  Chrpos_T splice_distance;
  int nmismatches_whole, nmismatches_refdiff, nmismatches, ref_nmismatches, nindels, ninserts;
  Substring_T substring;
  Junction_T junction;
  bool abortp;

  debug(printf("Entered single_hits_gminus with sensedir %d and %d gminus paths\n",sensedir,ngminus));

  if (ngminus == 0) {
    all_paths = (List_T) NULL;
  } else if (ngminus == 1) {
    representative_path = gminus_paths[0];
    debug2(printf("Putting initial transcript for 0 onto 0\n"));
    representative_path->transcripts = List_push(NULL,representative_path->initial_transcript);
    representative_path->initial_transcript = (Transcript_T) NULL;
    all_paths = List_push(NULL,(void *) representative_path);
  } else {
    all_paths = disambiguate_paths(gminus_paths,ngminus);
  }

  for (q = all_paths; q != NULL; q = List_next(q)) {
    abortp = false;
    nmismatches_whole = 0;
    nmismatches_refdiff = 0;
    substrings = junctions = (List_T) NULL;

    path = (Path_T) List_head(q);
    debug2(printf("path %p has %d exons.  trim_low %d, trim_high %d.  Endpoints %s\n",
		  path,path->nexons,path->trim_low,path->trim_high,Uintlist_to_string(path->endpoints)));
    /* gminus: Reverse endpoints and gaps */
    path->endpoints = Uintlist_reverse(path->endpoints);
    path->gaps = Intlist_reverse(path->gaps);
    querystart = path->trim_high; /* gminus */
    prev_alignend = 0;
    for (p = path->endpoints, g = path->gaps; p != NULL; p = Uintlist_next(Uintlist_next(p))) {
      alignstart = Uintlist_head(p) + path->chroffset;
      alignend = Uintlist_head(Uintlist_next(p)) + path->chroffset;
      ninserts = 0;
      if (prev_alignend > 0) {
	if ((nindels = Intlist_head(g)) > 0) {
	  debug2(printf("Setting value of deletionpos to be %u\n",alignstart));
	  junction = Junction_new_deletion(nindels,/*deletionpos*/alignstart);  /* gminus */
	} else if (nindels < 0) {
	  junction = Junction_new_insertion(-nindels);
	  ninserts = -nindels;
	  debug2(printf("Revising ninserts to be %d\n",ninserts));
	} else {
	  splice_distance = prev_alignend - alignstart; /* gminus */
	  debug2(printf("splice_distance %u = %u - %u\n",splice_distance,prev_alignend,alignstart));
	  junction = Junction_new_splice(splice_distance,sensedir,/*donor_prob*/2.0,/*acceptor_prob*/2.0);
	}
	junctions = Listpool_push(junctions,listpool,(void *) junction);
	g = Intlist_next(g);
      }
      querystart += ninserts;
      queryend = querystart + (alignstart - alignend); /* gminus */

      /* gminus */
      left = alignend - (querylength - queryend);
      univdiagonal = left + querylength;

      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5 = (univdiagonal + querystart >= path->chroffset + querylength) ? querystart : (int) (path->chroffset - left);
      pos3 = (univdiagonal + queryend <= path->chrhigh + querylength) ? queryend : (int) (path->chrhigh - left);

      debug2(printf("tr gminus query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
		    querystart,queryend,alignstart,alignend,alignstart - path->chroffset,alignend - path->chroffset,
		    left - path->chroffset,alignend - path->chroffset,querylength,queryend));

      /* Verified that (querylength - pos3) and (querylength - pos5)
	 are correct here, possibly because of how querystart and
	 queryend are defined above*/
      nmismatches =
	Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress_rev,left,
					  querylength - pos3,querylength - pos5,/*plusp*/false,/*genestrand*/0);
      nmismatches_whole += nmismatches;
      nmismatches_refdiff += ref_nmismatches;
      debug2(printf("mismatches rev from %d to %d: %d\n",pos5,pos3,nmismatches));

      /* Verified that pos5 and pos3 are correct */
      if ((substring = Substring_new(nmismatches,ref_nmismatches,left,pos5,pos3,querylength,
				     /*plusp*/false,/*genestrand*/0,query_compress_rev,
				     path->chrnum,path->chroffset,path->chrhigh,path->chrlength,
				     /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				     /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				     sensedir)) == NULL) {
	debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
	abortp = true;
      } else {
	debug2(printf(" => Created substring for %d..%d\n",querystart,queryend));
	substrings = Listpool_push(substrings,listpool,(void *) substring);
      }

      querystart = queryend;
      prev_alignend = alignend;
    }

    if (abortp == true || nmismatches_whole > nmismatches_allowed) {
      debug2(printf("ABORTING STAGE3END:  abortp %d.  nmismatches_whole %d vs nmismatches_allowed %d\n\n",
		    abortp,nmismatches_whole,nmismatches_allowed));
      for (r = substrings; r != NULL; r = List_next(r)) {
	substring = (Substring_T) List_head(r);
	Substring_free(&substring);
      }
      /* List_free(&substrings); -- allocated by Listpool_push */

      for (r = junctions; r != NULL; r = List_next(r)) {
	junction = (Junction_T) List_head(r);
	Junction_free(&junction);
      }
      /* List_free(&junctions); -- allocated by Listpool_push */

    } else {
      substrings = List_reverse(substrings);
      junctions = List_reverse(junctions);
      hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
				      /*nmismatches_bothdiff*/nmismatches_whole,nmismatches_refdiff,
				      substrings,junctions,path->transcripts,path->transcripts_other,
				      querylength,path->chrnum,path->chroffset,path->chrhigh,path->chrlength,
				      /*gplusp*/false,/*genestrand*/0,sensedir,listpool,/*method*/TR,level);
      debug(printf("Created new transcript hit %p with %d transcripts\n",hit,List_length(path->transcripts)));
      hits = Hitlist_push(hits,hitlistpool,(void *) hit);
      assert(path->initial_transcript == NULL);
      path->transcripts = (List_T) NULL; /* hit owns it now */
      path->transcripts_other = (List_T) NULL; /* hit owns it now */
    }
  }

  for (q = all_paths; q != NULL; q = List_next(q)) {
    path = (Path_T) List_head(q);
    Path_free(&path);
  }
  List_free(&all_paths);

  return hits;
}



#if 0
static int
Path_trnum_cmp (const void *x, const void *y) {
  Path_T a = * (Path_T *) x;
  Path_T b = * (Path_T *) y;
  Transcript_T transcript_a, transcript_b;
  Trnum_T trnum_a, trnum_b;

  transcript_a = (Transcript_T) a->initial_transcript;
  transcript_b = (Transcript_T) b->initial_transcript;
  
  trnum_a = Transcript_num(transcript_a);
  trnum_b = Transcript_num(transcript_b);

  if (trnum_a < trnum_b) {
    return -1;
  } else if (trnum_b < trnum_a) {
    return +1;
  } else {
    return 0;
  }
}
#endif


#if 0
static void
filter_concordant_paths (Path_T *paths5, int npaths5, Path_T *paths3, int npaths3) {
  int i, j, k, l, a, b;
  Transcript_T transcript5, transcript3;
  Trnum_T trnum;
  
  debug2(printf("Entered Kmer_filter_concordant_paths with %d paths5 and %d paths3\n",
		npaths5,npaths3));

  qsort(paths5,npaths5,sizeof(Path_T),Path_trnum_cmp);
  qsort(paths3,npaths3,sizeof(Path_T),Path_trnum_cmp);
  i = j = 0;
  while (i < npaths5 && j < npaths3) {
    transcript5 = paths5[i]->initial_transcript;
    transcript3 = paths3[j]->initial_transcript;

    if ((trnum = Transcript_num(transcript5)) < Transcript_num(transcript3)) {
      i++;
    } else if (Transcript_num(transcript3) < Transcript_num(transcript5)) {
      j++;
    } else {
      k = i + 1;
      while (k < npaths5 && Transcript_num(paths5[k]->initial_transcript) == trnum) {
	k++;
      }

      l = j + 1;
      while (l < npaths3 && Transcript_num(paths3[l]->initial_transcript) == trnum) {
	l++;
      }

      for (a = i; a < k; a++) {
	transcript5 = paths5[a]->initial_transcript;
	for (b = j; b < l; b++) {
	  transcript3 = paths3[b]->initial_transcript;
	  if (Transcript_concordant_p(transcript5,transcript3) == true) {
	    debug2(printf(" => concordant\n"));
	    paths5[a]->concordantp = true;
	    paths3[b]->concordantp = true;
	  }
	}
      }
      i = k;
      j = l;
    }
  }

  return;
}
#endif


#ifdef HAVE_AVX2
#define all_zero_p(diff) _mm256_testz_si256(diff,diff)
#elif defined(HAVE_SSE4_1)
#define all_zero_p(diff) _mm_testz_si128(diff,diff)
#elif defined(HAVE_SSE2)
#define all_zero_p(diff) _mm_movemask_ps(_mm_castsi128_ps(diff)) == 0
#endif


#if !defined(HAVE_SSE4_2)
#define count_trailing_zeroes_32(diff) mod_37_bit_position[(-diff & diff) % 37]
#elif defined(HAVE_TZCNT)
#define count_trailing_zeroes_32(diff) _tzcnt_u32(diff)
#elif defined(HAVE_BUILTIN_CTZ)
#define count_trailing_zeroes_32(diff) __builtin_ctz(diff)
#else
/* lowbit = -diff & diff */
#define count_trailing_zeroes_32(diff) mod_37_bit_position[(-diff & diff) % 37]
#endif


/* Expecting transcriptome to not be a large genome */
static void
search_transcriptome_complete (Path_T **tplus_gplus_paths, int *n_tplus_gplus,
			       Path_T **tplus_gminus_paths, int *n_tplus_gminus,
			       Path_T **tminus_gminus_paths, int *n_tminus_gminus,
			       Path_T **tminus_gplus_paths, int *n_tminus_gplus,
			       
			       Trcoord_T *tplus_positions_5, int n_tplus_positions_5, int tplus_diagterm_5,
			       Trcoord_T *tminus_positions_5, int n_tminus_positions_5, int tminus_diagterm_5,
			       Trcoord_T *tplus_positions_3, int n_tplus_positions_3, int tplus_diagterm_3, 
			       Trcoord_T *tminus_positions_3, int n_tminus_positions_3, int tminus_diagterm_3,
			       
			       char *queryuc_ptr, int querylength,
			       Trcoord_T **tplus_stream_array, int *tplus_streamsize_array, int *tplus_diagterm_array,
			       Trcoord_T **tminus_stream_array, int *tminus_streamsize_array, int *tminus_diagterm_array,
			       Compress_T query_compress_fwd, Compress_T query_compress_rev,
			       Univ_IIT_T transcript_iit, Transcriptome_T transcriptome, Genome_T transcriptomebits, 
			       int nmismatches_allowed, bool concordantp) {
  int max_tplus_count, max_tminus_count;

  Trcoord_T tr_univdiagonal, tr_left, tr_left5, tr_left3, low, high;
  Trcoord_T *tplus_diagonals, *tminus_diagonals;
  int n_tplus_loci, n_tminus_loci, n_tplus_diagonals, n_tminus_diagonals,
    total_ndiagonals_tplus = 0, total_ndiagonals_tminus = 0, i;
  
  Trcoord_T *positions;
  int npositions;
  int tplus_streami = 0, tminus_streami = 0;

  Reader_T reader;
  int querypos;
  Oligostate_T last_state = INIT;
  Oligospace_T forward = 0, revcomp = 0, forward_oligo, revcomp_oligo;

#if 0 && defined(HAVE_AVX2)
  __m256i _diff, _exonbounds, _trstart;
  int matchbits;
  int *ptr;
#elif 0 && defined(HAVE_SSE2)
  __m128i _diff, _exonbounds, _trstart;
  int matchbits;
  int *ptr;
#endif

  Trnum_T trnum;
  int transcript_genestrand;
  Trcoord_T troffset, trhigh, trlength;
  Chrnum_T chrnum;
  int nexons, exoni;
  int overall_adj, adj, adj5, adj3, nindels;

  Uintlist_T tr_lefts, q;
  Intlist_T querystarts, queryends, adjustments, r, s, t;
  int trim5, trim3, pos5, pos3;
  int nmismatches5, nmismatches3, ref_nmismatches;

  bool want_lowest_coordinate_p;
  int indel_pos_5, indel_pos_3;
  int total_nmismatches, total_nmismatches_5, total_nmismatches_3;
  int total_nmatches, total_nmatches_5, total_nmatches_3;
  
  int *exonbounds, transcript_diag, overall_trstart, overall_trend, trstart;
#ifdef DEBUG2
  int trend;
  bool allocp;
  Univcoord_T left;
#endif
  Chrpos_T *exonstarts, genomicpos, chrlength;
  Univcoord_T alignstart, alignend, chroffset, chrhigh;
  int querystart, queryend, exonpos, align_residual, exon_residual, len;
  bool abortp;

  Uintlist_T endpoints;
  Intlist_T gaps;
  Path_T *gplus_paths, *gminus_paths;
  int ngplus, ngminus;


  debug2(printf("\n\n***Complete transcriptome\n"));


  if (n_tplus_positions_5 > 0) {
    tplus_stream_array[tplus_streami] = tplus_positions_5;
    tplus_streamsize_array[tplus_streami] = n_tplus_positions_5;
    tplus_diagterm_array[tplus_streami] = tplus_diagterm_5;
    tplus_streami++;
  }

  if (n_tplus_positions_3 > 0) {
    tplus_stream_array[tplus_streami] = tplus_positions_3;
    tplus_streamsize_array[tplus_streami] = n_tplus_positions_3;
    tplus_diagterm_array[tplus_streami] = tplus_diagterm_3;
    tplus_streami++;
  }

  if (n_tminus_positions_5 > 0) {
    tminus_stream_array[tminus_streami] = tminus_positions_5;
    tminus_streamsize_array[tminus_streami] = n_tminus_positions_5;
    tminus_diagterm_array[tminus_streami] = tminus_diagterm_5;
    tminus_streami++;
  }

  if (n_tminus_positions_3 > 0) {
    tminus_stream_array[tminus_streami] = tminus_positions_3;
    tminus_streamsize_array[tminus_streami] = n_tminus_positions_3;
    tminus_diagterm_array[tminus_streami] = tminus_diagterm_3;
    tminus_streami++;
  }


  reader = Reader_new(queryuc_ptr,/*querystart*/1,/*queryend*/querylength - 1);
  last_state = INIT;
  forward = revcomp = 0;

  while ((last_state = Oligo_next_5(last_state,&querypos,&forward,&revcomp,reader,/*genestrand*/0)) != DONE) {
    forward_oligo = forward & oligobase_mask;
    npositions = Indexdb_ptr(&positions,/*plus_indexdb*/indexdb_tr,forward_oligo);
    if (npositions > 0) {
      tplus_stream_array[tplus_streami] = positions;
      tplus_streamsize_array[tplus_streami] = npositions;
      tplus_diagterm_array[tplus_streami] = querylength - querypos;
      total_ndiagonals_tplus += npositions;
      tplus_streami++;
    }

    revcomp_oligo = (revcomp >> leftreadshift) & oligobase_mask;
    npositions = Indexdb_ptr(&positions,/*minus_indexdb*/indexdb_tr,revcomp_oligo);
    if (npositions > 0) {
      tminus_stream_array[tminus_streami] = positions;
      tminus_streamsize_array[tminus_streami] = npositions;
      tminus_diagterm_array[tminus_streami] = querypos + index1part_tr;
      total_ndiagonals_tminus += npositions;
      tminus_streami++;
    }
  }
  Reader_free(&reader);

  tplus_diagonals = Merge_diagonals(&n_tplus_diagonals,tplus_stream_array,
				    tplus_streamsize_array,tplus_diagterm_array,/*nstreams*/tplus_streami);

  if (n_tplus_diagonals == 0) {
    max_tplus_count = 0;
    n_tplus_loci = 0;
  } else {
    max_tplus_count = most_prevalent_uint(&n_tplus_loci,tplus_diagonals,n_tplus_diagonals);
  }

#ifdef DEBUG2
  printf("max_tplus_count: %d\n",max_tplus_count);
  for (i = 0; i < n_tplus_loci; i++) {
    printf("PLUS_LOCUS %u\n",tplus_diagonals[i]);
  }
#endif

  tminus_diagonals = Merge_diagonals(&n_tminus_diagonals,tminus_stream_array,
				     tminus_streamsize_array,tminus_diagterm_array,/*nstreams*/tminus_streami);

  if (n_tminus_diagonals == 0) {
    max_tminus_count = 0;
    n_tminus_loci = 0;
  } else {
    max_tminus_count = most_prevalent_uint(&n_tminus_loci,tminus_diagonals,n_tminus_diagonals);
  }


#ifdef DEBUG2
  printf("max_tminus_count: %d\n",max_tminus_count);
  for (i = 0; i < n_tminus_loci; i++) {
    printf("MINUS_LOCUS %u\n",tminus_diagonals[i]);
  }
#endif

  ngplus = ngminus = 0;	/* ngplus and ngminus partition n_tplus_loci */
  if (max_tplus_count < (querylength - index1part_tr)/5 &&
      max_tplus_count < max_tminus_count - SUBOPT) {
    /* Not enough kmers match or less than minus */
    debug2(printf("Not enough tplus kmers match (%d)\n",max_tplus_count));
    gplus_paths = gminus_paths = (Path_T *) NULL;

  } else if (n_tplus_loci == 0) {
    gplus_paths = gminus_paths = (Path_T *) NULL;

  } else {
    gplus_paths = (Path_T *) MALLOC(n_tplus_loci*sizeof(Path_T));
    gminus_paths = (Path_T *) MALLOC(n_tplus_loci*sizeof(Path_T));
    for (i = 0; i < n_tplus_loci; i++) {
      tr_univdiagonal = tplus_diagonals[i];
      tr_left = tr_univdiagonal - (Trcoord_T) querylength; /* NEW FORMULA for querystart of 0 */

      /* TRIM COORDINATES AT TRANSCRIPTOME BOUNDS */
      low = (tr_univdiagonal >= (Univcoord_T) querylength) ? tr_left : 0;
      high = (tr_univdiagonal <= transcriptomelength) ? tr_univdiagonal : transcriptomelength;
      trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high);
      debug2(printf("(1) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));

      /* TRIM QUERY AT TRANSCRIPT BOUNDS */
      pos5 = (tr_univdiagonal >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_left);
      pos3 = (tr_univdiagonal <= trhigh) ? querylength : (int) (trhigh - tr_left);
      debug2(printf("Trimming at transcript bounds: tr_univdiagonal %u relative to troffset %u and trhigh %u => pos5 %d, pos3 %d\n",
		    tr_univdiagonal,troffset,trhigh,pos5,pos3));

      trim5 = Genome_first_kmer_left(&nmismatches5,transcriptomebits,query_compress_fwd,tr_left,pos5,pos3,
				     /*plusp*/true,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
      debug2(printf("First kmer on left is at trim5 %d\n",trim5));
      queryend = Genome_first_kmer_right(&nmismatches3,transcriptomebits,query_compress_fwd,tr_left,pos5,pos3,
					 /*plusp*/true,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
      debug2(printf("First kmer on right is at queryend %d\n",queryend));
      trim3 = querylength - queryend;

      /* Look for conditions where we need not look for an indel */
      abortp = false;
      if (trim5 >= queryend) {
	abortp = true;
	total_nmismatches = pos3 - pos5;

      } else if ((total_nmismatches =
		  Genome_count_mismatches_substring(&ref_nmismatches,transcriptomebits,NULL,query_compress_fwd,
						    tr_left,/*pos5*/trim5,/*pos3*/queryend,/*plusp*/true,/*genestrand*/0)) < 3) {
	trim5 = pos5;
	trim3 = querylength - queryend;

	debug2(printf("tplus diagonal %u => trimmed %d..%d, nmismatches: %d total, %d at 5', and %d at 3'\n",
		      tr_univdiagonal,trim5,querylength - trim3,total_nmismatches,nmismatches5,nmismatches3));

      } else {
	total_nmismatches = pos3 - pos5;
	
	if (trim5 < LONG_END) {
	  if (nmismatches5 == 1) {
	    trim5 = pos5;
	  }
	} else {
	  if (nmismatches5 <= ALLOWED_END_MISMATCHES) {
	    trim5 = pos5;
	  }
	}

	if (trim3 < LONG_END) {
	  if (nmismatches3 == 1) {
	    trim3 = querylength - queryend;
	  }
	} else {
	  if (nmismatches3 <= ALLOWED_END_MISMATCHES) {
	    trim3 = querylength - queryend;
	  }
	}
      }

      total_nmatches = (querylength - trim5 - trim3) - total_nmismatches;

      if (abortp == true) {
	/* Skip */
      } else if ((chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum)) <= 0) {
	debug2(printf("No genomic alignment for trnum %d, so aborting\n",trnum));
	abortp = true;
      } else {
	if (transcript_genestrand > 0) {
	  want_lowest_coordinate_p = true;
	} else {
	  want_lowest_coordinate_p = false;
	}
	debug2(printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand));

	overall_adj = adj5 = adj3 = 0;
	total_nmismatches_5 = total_nmismatches_3 = querylength;
	debug2(printf("trim5 %d, trim3 %d\n",trim5,trim3));
	if (trim5 > pos5) {
	  indel_pos_5 = Indel_solve_end_low(&adj5,&total_nmismatches_5,tr_left,troffset,trhigh,
					    /*firstbound*/trim5,querylength,
					    query_compress_fwd,transcriptomebits,NULL,
					    /*max_mismatches*/total_nmismatches-1,/*plusp*/true,/*genestrand*/0,
					    want_lowest_coordinate_p);
	  total_nmatches_5 = querylength - total_nmismatches_5;
	  /* Compensate for nmismatches from indel to be fair to a straight alignment */
	  if (adj5 > 0) {
	    total_nmismatches_5 += adj5;
	  } else {
	    total_nmismatches_5 -= adj5;
	  }
	  debug2(printf("Indel_solve_end_low returns indel_pos %d, adj5 %d, with total_nmismatches %d\n",
			indel_pos_5,adj5,total_nmismatches_5));
	}
	if (trim3 > querylength - queryend) {
	  indel_pos_3 = Indel_solve_end_high(&adj3,&total_nmismatches_3,tr_left,troffset,trhigh,
					     /*lastbound*/querylength - trim3,querylength,
					     query_compress_fwd,transcriptomebits,NULL,
					     /*max_mismatches*/total_nmismatches-1,/*plusp*/true,/*genestrand*/0,
					     want_lowest_coordinate_p);
	  total_nmatches_3 = querylength - total_nmismatches_3;
	  /* Compensate for nmismatches from indel to be fair to a straight alignment */
	  if (adj3 > 0) {
	    total_nmismatches_3 += adj3;
	  } else {
	    total_nmismatches_3 -= adj3;
	  }
	  debug2(printf("Indel_solve_end_high returns indel_pos %d, adj3 %d, with total_nmismatches %d\n",
			indel_pos_3,adj3,total_nmismatches_3));
	}
	
	if (total_nmismatches_5 < total_nmismatches_3) {
	  if (total_nmismatches_5 >= total_nmismatches) {
	    overall_adj = 0;
	  } else {
	    total_nmatches = total_nmatches_5;
	    overall_adj = adj5;
	  }
	} else {
	  if (total_nmismatches_3 >= total_nmismatches) {
	    overall_adj = 0;
	  } else {
	    total_nmatches = total_nmatches_3;
	    overall_adj = adj3;
	  }
	}
	debug2(printf("overall_adj %d, adj5 %d, adj3 %d\n",overall_adj,adj5,adj3));
      }

      if (abortp == true) {
	/* Already aborted due to lack of genomic alignment for transcript */

      } else if (overall_adj == 0) {
	trim5 = pos5;
	trim3 = querylength - queryend;      /* Reset trims, since we cannot find an indel */
	tr_lefts = Uintlist_push(NULL,tr_left);
	querystarts = Intlist_push(NULL,trim5);
	queryends = Intlist_push(NULL,querylength - trim3);
	adjustments = (Intlist_T) NULL;
	debug2(printf("1 querystarts: %s\n",Intlist_to_string(querystarts)));
	debug2(printf("1 queryends: %s\n",Intlist_to_string(queryends)));

      } else if (overall_adj > 0) {
	/* Deletion */
	if (total_nmismatches_5 < total_nmismatches_3) {
	  if (total_nmismatches_5 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_5));
	    abortp = true;
	  } else {
	    overall_adj = adj5;
	    adj3 = 0;
	    tr_left5 = tr_left-adj5;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left),tr_left5);
	    querystarts = Intlist_push(Intlist_push(NULL,indel_pos_5),trim5);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - trim3),indel_pos_5);
	    adjustments = Intlist_push(NULL,adj5);
	    debug2(printf("2 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("2 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	} else {
	  if (total_nmismatches_3 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_3));
	    abortp = true;
	  } else {
	    overall_adj = adj3;
	    adj5 = 0;
	    tr_left3 = tr_left+adj3;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left3),tr_left);
	    querystarts = Intlist_push(Intlist_push(NULL,indel_pos_3),trim5);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - trim3),indel_pos_3);
	    adjustments = Intlist_push(NULL,adj3);
	    debug2(printf("3 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("3 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	}

      } else {
	/* Insertion */
	if (total_nmismatches_5 < total_nmismatches_3) {
	  if (total_nmismatches_5 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_5));
	    abortp = true;
	  } else {
	    overall_adj = adj5;
	    adj3 = 0;
	    tr_left5 = tr_left-adj5;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left),tr_left5);
	    querystarts = Intlist_push(Intlist_push(NULL,indel_pos_5 - adj5),trim5);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - trim3),indel_pos_5);
	    adjustments = Intlist_push(NULL,adj5);
	    debug2(printf("4 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("4 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	} else {
	  if (total_nmismatches_3 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_3));
	    abortp = true;
	  } else {
	    overall_adj = adj3;
	    adj5 = 0;
	    tr_left3 = tr_left+adj3;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left3),tr_left);
	    querystarts = Intlist_push(Intlist_push(NULL,indel_pos_3 - adj3),trim5);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - trim3),indel_pos_3);
	    adjustments = Intlist_push(NULL,adj3);
	    debug2(printf("5 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("5 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	}
      }

      if (abortp == false) {
	nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);

	if (transcript_genestrand > 0) {
	  /* Case 1: tplus, gplus.  transcript_plusp == true && transcript_genestrand > 0 */
	  debug2(printf("Case 1: tplus, gplus\n"));
	  /* sensedir = SENSE_FORWARD; */
	  /* plusp = true; */

	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;

	  for (q = tr_lefts, r = querystarts, s = queryends, t = adjustments; q != NULL;
	       q = Uintlist_next(q), r = Intlist_next(r), s = Intlist_next(s)) {
	    transcript_diag = Uintlist_head(q) - troffset;

	    /* Find initial exonpos */
	    querystart = Intlist_head(r); /* tplus: compute from querystart to queryend */
	    queryend = Intlist_head(s);
	    debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	    trstart = transcript_diag + querystart;	/* tplus */
#ifdef DEBUG2
	    trend = transcript_diag + queryend; /* tplus */
	    printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, overall_adj %d\n",
		   transcript_diag,trstart,trend,Uintlist_head(q),overall_adj);
#endif

	    if (q == tr_lefts) {
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif
	    } else {
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
	    }
	      
	    if (exoni >= nexons) {
	      /* Should not be possible because of the check above for aligning past end of transcript */
	      debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	      abortp = true;
	    } else {
	      debug2(printf("(1) exoni %d out of nexons %d\n",exoni,nexons));
	      if (exoni == 0) {
		exonpos = trstart;
	      } else {
		exonpos = trstart - exonbounds[exoni - 1];
	      }
	      exon_residual = exonbounds[exoni] - trstart;
	      debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));

	      /* Generate one substring per exon */
	      genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
	      debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
	      align_residual = queryend - querystart;
	      while (align_residual > 0) {
		debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		queryend = querystart + len; /* tplus */
		  
		alignstart = chroffset + genomicpos;
		alignend = alignstart + len; /* gplus */
		debug2(left = alignstart - querystart); /* tplus == gminus */
		debug2(printf("tr complete case 1 query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
			      querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			      left - chroffset,alignstart - chroffset,querystart));
	
		/* tplus: push alignstart first */
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));

		if (align_residual <= exon_residual) {
		  exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		  align_residual = 0;
		} else if (++exoni >= nexons) {
		  /* Aligning past the end of the transcript */
		  debug2(printf("2 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		  align_residual = 0;
		  abortp = true;
		} else {
		  /* nintrons += 1; */
		  align_residual -= len;
		  querystart += len; /* tplus */
		  genomicpos = exonstarts[exoni] - 1; /* gplus */
		  debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		  exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		  debug2(printf("INTRON\n"));
		  gaps = Intlist_push(gaps,0);
		}
	      }
	    
	      if (Uintlist_next(q) != NULL) {
		nindels = adj = Intlist_head(t);
		debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
		if (nindels < 0) {
		  nindels = -nindels;
		}
		debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
		if (abortp == true) {
		  /* Skip */
		} else if (/* len <= nindels ||*/exon_residual <= nindels) {
		  debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (1)\n"));
		  abortp = true;
		} else {
		  gaps = Intlist_push(gaps,(int) adj);
		}
		t = Intlist_next(t);
	      }
	    }
	  }
	  
	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    endpoints = Uintlist_reverse(endpoints); /* gplus */
	    gaps = Intlist_reverse(gaps);	     /* gplus */
	    debug2(printf("PLUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngplus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    overall_trstart = transcript_diag + Intlist_head(querystarts); /* tplus */
	    overall_trend = transcript_diag + queryend; /* tplus: queryend is the last one processed */
	    gplus_paths[ngplus++] = Path_new(total_nmatches,trnum,overall_trstart,overall_trend,
					     /*trim_low*/trim5,/*trim_high*/trim3,
					     chrnum,chroffset,chrhigh,chrlength,endpoints,gaps,concordantp);
	  }

	} else {
	  /* Case 2: tplus, gminus.  transcript_plusp == true && transcript_genestrand < 0 */
	  debug2(printf("Case 2: tplus, gminus\n"));
	  /* sensedir = SENSE_FORWARD; */
	  /* plusp = false; */
    
	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;
	  
	  for (q = tr_lefts, r = querystarts, s = queryends, t = adjustments; q != NULL;
	       q = Uintlist_next(q), r = Intlist_next(r), s = Intlist_next(s)) {
	    transcript_diag = Uintlist_head(q) - troffset;

	    /* Find initial exonpos */
	    querystart = Intlist_head(r); /* tplus: compute from querystart to queryend */
	    queryend = Intlist_head(s);
	    debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	    trstart = transcript_diag + querystart;	/* tplus */
#ifdef DEBUG2
	    trend = transcript_diag + queryend; /* tplus */
	    printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, overall_adj %d\n",
		   transcript_diag,trstart,trend,Uintlist_head(q),overall_adj);
#endif

	    if (q == tr_lefts) {
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif
	    } else {
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
	    }

	    if (exoni >= nexons) {
	      debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	      abortp = true;
	    } else {
	      debug2(printf("(2) exoni %d out of nexons %d\n",exoni,nexons));
	      if (exoni == 0) {
		exonpos = trstart;
	      } else {
		exonpos = trstart - exonbounds[exoni - 1];
	      }
	      exon_residual = exonbounds[exoni] - trstart;
	      debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));

	      /* Generate one substring per exon */
	      genomicpos = exonstarts[exoni] - exonpos; /* gminus */
	      debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
	      align_residual = queryend - querystart;
	      while (align_residual > 0) {
		debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		queryend = querystart + len; /* tplus */
		
		alignstart = chroffset + genomicpos;
		alignend = alignstart - len; /* gminus */
		debug2(left = alignend - (querylength - queryend)); /* tplus != gminus */
		debug2(printf("tr complete case 2 query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
			      querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			      left - chroffset,alignend - chroffset,querylength,queryend));
		
		/* tplus: push alignstart first */
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
		
		if (align_residual <= exon_residual) {
		  exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		  align_residual = 0;
		} else if (++exoni >= nexons) {
		  /* Aligning past the end of the transcript */
		  debug2(printf("4 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		  align_residual = 0;
		  abortp = true;
		} else {
		  /* nintrons += 1; */
		  align_residual -= len;
		  querystart += len; /* tplus */
		  genomicpos = exonstarts[exoni]; /* gminus */
		  debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		  exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		  debug2(printf("INTRON\n"));
		  gaps = Intlist_push(gaps,0);
		}
	      }
	      
	      if (Uintlist_next(q) != NULL) {
		nindels = adj = Intlist_head(t);
		debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
		if (nindels < 0) {
		  nindels = -nindels;
		}
		debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
		if (abortp == true) {
		  /* Skip */
		} else if (/*len <= nindels ||*/ exon_residual <= nindels) {
		  debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (2)\n"));
		  abortp = true;
		} else {
		  gaps = Intlist_push(gaps,(int) adj);
		}
		t = Intlist_next(t);
	      }
	    }
	  }
      
	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    /* gminus: Do not reverse endpoints or gaps */
	    debug2(printf("MINUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngminus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    overall_trstart = transcript_diag + Intlist_head(querystarts); /* tplus */
	    overall_trend = transcript_diag + queryend; /* tplus: queryend is the last one processed */
	    gminus_paths[ngminus++] = Path_new(total_nmatches,trnum,overall_trstart,overall_trend,
					       /*trim_low*/trim3,/*trim_high*/trim5,
					       chrnum,chroffset,chrhigh,chrlength,endpoints,gaps,concordantp);
	  }
	}

	Intlist_free(&adjustments);
	Intlist_free(&queryends);
	Intlist_free(&querystarts);
	Uintlist_free(&tr_lefts);
      }
    }

    debug2(printf("tplus loci %d = ngplus %d + ngminus %d\n",n_tplus_loci,ngplus,ngminus));
  }
#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(tplus_diagonals);
#else
  FREE(tplus_diagonals);
#endif
  *tplus_gplus_paths = gplus_paths;
  *n_tplus_gplus = ngplus;
  *tplus_gminus_paths = gminus_paths;
  *n_tplus_gminus = ngminus;


  ngplus = ngminus = 0;	/* ngplus and ngminus partition n_tminus_loci */
  if (max_tminus_count < (querylength - index1part_tr)/5 &&
      max_tminus_count < max_tplus_count - SUBOPT) {
    /* Not enough kmers match or less than plus */
    debug2(printf("Not enough tminus kmers match (%d)\n",max_tminus_count));
    gplus_paths = gminus_paths = (Path_T *) NULL;

  } else if (n_tminus_loci == 0) {
    gplus_paths = gminus_paths = (Path_T *) NULL;

  } else {
    gplus_paths = (Path_T *) MALLOC(n_tminus_loci*sizeof(Path_T));
    gminus_paths = (Path_T *) MALLOC(n_tminus_loci*sizeof(Path_T));
    for (i = 0; i < n_tminus_loci; i++) {
      tr_univdiagonal = tminus_diagonals[i];
      tr_left = tr_univdiagonal - (Trcoord_T) querylength; /* NEW FORMULA for queryend of querylength */

      /* TRIM COORDINATES AT TRANSCRIPTOME BOUNDS */
      low = (tr_univdiagonal >= (Univcoord_T) querylength) ? tr_left : 0;
      high = (tr_univdiagonal <= transcriptomelength) ? tr_univdiagonal : transcriptomelength;
      trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high);
      debug2(printf("(2) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));

      /* TRIM QUERY AT TRANSCRIPT BOUNDS */
      pos5 = (tr_univdiagonal >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_left);
      pos3 = (tr_univdiagonal <= trhigh) ? querylength : (int) (trhigh - tr_left);
      debug2(printf("Trimming at transcript bounds: tr_univdiagonal %u relative to troffset %u and trhigh %u => pos5 %d, pos3 %d\n",
		    tr_univdiagonal,troffset,trhigh,pos5,pos3));

      trim5 = Genome_first_kmer_left(&nmismatches5,transcriptomebits,query_compress_rev,tr_left,pos5,pos3,
				     /*plusp*/false,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
      debug2(printf("First kmer on left is at trim5 %d\n",trim5));
      queryend = Genome_first_kmer_right(&nmismatches3,transcriptomebits,query_compress_rev,tr_left,pos5,pos3,
					 /*plusp*/false,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
      debug2(printf("First kmer on right is at queryend %d\n",queryend));
      trim3 = querylength - queryend;

      /* Look for conditions where we need not look for an indel */
      abortp = false;
      if (trim5 >= queryend) {
	abortp = true;
	total_nmismatches = pos3 - pos5;

      } else if ((total_nmismatches =
		  Genome_count_mismatches_substring(&ref_nmismatches,transcriptomebits,NULL,query_compress_rev,
						    tr_left,/*pos5*/trim5,/*pos3*/queryend,/*plusp*/false,/*genestrand*/0)) < 3) {
	trim5 = pos5;
	trim3 = querylength - queryend;

	debug2(printf("tminus diagonal %u => trimmed %d..%d, nmismatches: %d total, %d at 5', and %d at 3'.\n",
		      tr_univdiagonal,trim5,querylength - trim3,total_nmismatches,nmismatches5,nmismatches3));

      } else {
	total_nmismatches = pos3 - pos5;

	if (trim5 < LONG_END) {
	  if (nmismatches5 == 1) {
	    trim5 = pos5;
	  }
	} else {
	  if (nmismatches5 <= ALLOWED_END_MISMATCHES) {
	    trim5 = pos5;
	  }
	}

	if (trim3 < LONG_END) {
	  if (nmismatches3 == 1) {
	    trim3 = querylength - queryend;
	  }
	} else {
	  if (nmismatches3 <= ALLOWED_END_MISMATCHES) {
	    trim3 = querylength - queryend;
	  }
	}
      }
      total_nmatches = (querylength - trim3 - trim5) - total_nmismatches;

      if (abortp == true) {
	/* Skip */
      } else if ((chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum)) <= 0) {
	debug2(printf("No genomic alignment for trnum %d, so aborting\n",trnum));
	abortp = true;
      } else {
	if (transcript_genestrand > 0) {
	  want_lowest_coordinate_p = true;
	} else {
	  want_lowest_coordinate_p = false;
	}
	debug2(printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand));

	overall_adj = adj5 = adj3 = 0;
	total_nmismatches_5 = total_nmismatches_3 = querylength;
	debug2(printf("trim5 %d, trim3 %d\n",trim5,trim3));
	if (trim5 > pos5) {
	  indel_pos_5 = Indel_solve_end_low(&adj5,&total_nmismatches_5,tr_left,troffset,trhigh,
					    /*firstbound*/trim5,querylength,
					    query_compress_rev,transcriptomebits,NULL,
					    /*max_mismatches*/total_nmismatches-1,/*plusp*/false,/*genestrand*/0,
					    want_lowest_coordinate_p);
	  total_nmatches_5 = querylength - total_nmismatches_5;
	  /* Compensate for nmismatches from indel to be fair to a straight alignment */
	  if (adj5 > 0) {
	    total_nmismatches_5 += adj5;
	  } else {
	    total_nmismatches_5 -= adj5;
	  }
	  debug2(printf("Indel_solve_end_low returns indel_pos %d, adj5 %d, with total_nmismatches %d\n",
			indel_pos_5,adj5,total_nmismatches_5));
	}
	if (trim3 > querylength - queryend) {
	  indel_pos_3 = Indel_solve_end_high(&adj3,&total_nmismatches_3,tr_left,troffset,trhigh,
					     /*lastbound*/querylength - trim3,querylength,
					     query_compress_rev,transcriptomebits,NULL,
					     /*max_mismatches*/total_nmismatches-1,/*plusp*/false,/*genestrand*/0,
					     want_lowest_coordinate_p);
	  total_nmatches_3 = querylength - total_nmismatches_3;
	  /* Compensate for nmismatches from indel to be fair to a straight alignment */
	  if (adj3 > 0) {
	    total_nmismatches_3 += adj3;
	  } else {
	    total_nmismatches_3 -= adj3;
	  }
	  debug2(printf("Indel_solve_end_high returns indel_pos %d, adj3 %d, with total_nmismatches %d\n",
			indel_pos_3,adj3,total_nmismatches_3));
	}
	
	if (total_nmismatches_5 < total_nmismatches_3) {
	  if (total_nmismatches_5 >= total_nmismatches) {
	    overall_adj = 0;
	  } else {
	    total_nmatches = total_nmatches_5;
	    overall_adj = adj5;
	  }
	} else {
	  if (total_nmismatches_3 >= total_nmismatches) {
	    overall_adj = 0;
	  } else {
	    total_nmatches = total_nmatches_3;
	    overall_adj = adj3;
	  }
	}
	debug2(printf("overall_adj %d, adj5 %d, adj3 %d\n",overall_adj,adj5,adj3));
      }


      if (abortp == true) {
	/* Already aborted due to lack of genomic alignment for transcript */

      } else if (overall_adj == 0) {
	trim5 = pos5;
	trim3 = querylength - queryend;      /* Reset trims, since we cannot find an indel */
	tr_lefts = Uintlist_push(NULL,tr_left);
	querystarts = Intlist_push(NULL,trim3);
	queryends = Intlist_push(NULL,querylength - trim5);
	adjustments = (Intlist_T) NULL;
	debug2(printf("6 querystarts: %s\n",Intlist_to_string(querystarts)));
	debug2(printf("6 queryends: %s\n",Intlist_to_string(queryends)));

      } else if (overall_adj > 0) {
	/* Deletion */
	if (total_nmismatches_5 < total_nmismatches_3) {
	  if (total_nmismatches_5 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_5));
	    abortp = true;
	  } else {
	    overall_adj = adj5;
	    adj3 = 0;
	    tr_left5 = tr_left-adj5;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left),tr_left5);
	    querystarts = Intlist_push(Intlist_push(NULL,trim3),querylength - indel_pos_5);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - indel_pos_5),querylength - trim5);
	    adjustments = Intlist_push(NULL,adj5);
	    debug2(printf("7 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("7 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	} else {
	  if (total_nmismatches_3 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_3));
	    abortp = true;
	  } else {
	    overall_adj = adj3;
	    adj5 = 0;
	    tr_left3 = tr_left+adj3;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left3),tr_left);
	    querystarts = Intlist_push(Intlist_push(NULL,trim3),querylength - indel_pos_3);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - indel_pos_3),querylength - trim5);
	    adjustments = Intlist_push(NULL,adj3);
	    debug2(printf("8 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("8 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	}
	
      } else {
	/* Insertion */
	if (total_nmismatches_5 < total_nmismatches_3) {
	  if (total_nmismatches_5 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_5));
	    abortp = true;
	  } else {
	    overall_adj = adj5;
	    adj3 = 0;
	    tr_left5 = tr_left-adj5;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left),tr_left5);
	    querystarts = Intlist_push(Intlist_push(NULL,trim3),querylength - indel_pos_5);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - indel_pos_5 + adj5),querylength - trim5);
	    adjustments = Intlist_push(NULL,adj5);
	    debug2(printf("9 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("9 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	} else {
	  if (total_nmismatches_3 > nmismatches_allowed) {
	    debug2(printf("ABORTING INDEL BECAUSE CANNOT EXTEND TO END OF TRANSCRIPT: %d mismatches\n",total_nmismatches_3));
	    abortp = true;
	  } else {
	    overall_adj = adj3;
	    adj5 = 0;
	    tr_left3 = tr_left+adj3;

	    trim5 = pos5;
	    trim3 = querylength - queryend;      /* Reset trims, since we have found the (single) indel */
	    tr_lefts = Uintlist_push(Uintlist_push(NULL,tr_left3),tr_left);
	    querystarts = Intlist_push(Intlist_push(NULL,trim3),querylength - indel_pos_3);
	    queryends = Intlist_push(Intlist_push(NULL,querylength - indel_pos_3 + adj3),querylength - trim5);
	    adjustments = Intlist_push(NULL,adj3);
	    debug2(printf("10 querystarts: %s\n",Intlist_to_string(querystarts)));
	    debug2(printf("10 queryends: %s\n",Intlist_to_string(queryends)));
	  }
	}
      }

      if (abortp == false) {
	nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);
	debug2(printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand));

	if (transcript_genestrand > 0) {
	  /* Case 3: tminus, gplus.  transcript_plusp == false && transcript_genestrand > 0 */
	  debug2(printf("Case 3: tminus, gplus\n"));
	  /* sensedir = SENSE_ANTI; */
	  /* plusp = false; */

	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;

	  for (q = tr_lefts, r = querystarts, s = queryends, t = adjustments; q != NULL;
	       q = Uintlist_next(q), r = Intlist_next(r), s = Intlist_next(s)) {
	    transcript_diag = Uintlist_head(q) - troffset;

	    /* Find initial exonpos */
	    queryend = Intlist_head(s); /* tminus: compute from queryend to querystart */
	    querystart = Intlist_head(r);
	    debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	    trstart = transcript_diag + (querylength - queryend); /* tminus */
#ifdef DEBUG2
	    trend = transcript_diag + (querylength - querystart); /* tminus */
	    printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, overall_adj %d\n",
		   transcript_diag,trstart,trend,Uintlist_head(q),overall_adj);
#endif	  

	    if (q == tr_lefts) {
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif
	    } else {
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
	    }

	    if (exoni >= nexons) {
	      debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	      abortp = true;
	    } else {
	      debug2(printf("(3) exoni %d out of nexons %d\n",exoni,nexons));
	      if (exoni == 0) {
		exonpos = trstart;
	      } else {
		exonpos = trstart - exonbounds[exoni - 1];
	      }
	      exon_residual = exonbounds[exoni] - trstart;
	      debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));

	      /* Generate one substring per exon */
	      genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
	      debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
	      align_residual = queryend - querystart;
	      while (align_residual > 0) {
		debug2(printf("abortp %d,align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		querystart = queryend - len; /* tminus */
      
		alignend = chroffset + genomicpos;
		alignstart = alignend + len; /* gplus */
		debug2(left = alignend - (querylength - queryend)); /* tminus != gplus */
		debug2(printf("tr complete case 3 query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
			      querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			      left - chroffset,alignend - chroffset,querylength,queryend));

		/* tminus: plus alignend first */
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));

		if (align_residual <= exon_residual) {
		  exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		  align_residual = 0;
		} else if (++exoni >= nexons) {
		  /* Aligning past the end of the transcript */
		  debug2(printf("6 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		  align_residual = 0;
		  abortp = true;
		} else {
		  /* nintrons += 1; */
		  align_residual -= len;
		  queryend -= len; /* tminus */
		  genomicpos = exonstarts[exoni] - 1; /* gplus */
		  debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		  exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		  debug2(printf("INTRON\n"));
		  gaps = Intlist_push(gaps,0);
		}
	      }

	      if (Uintlist_next(q) != NULL) {
		nindels = adj = Intlist_head(t);
		debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
		if (nindels < 0) {
		  nindels = -nindels;
		}
		debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
		if (abortp == true) {
		  /* Skip */
		} else if (/*len <= nindels ||*/exon_residual <= nindels) {
		  debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (3)\n"));
		  abortp = true;
		} else {
		  gaps = Intlist_push(gaps,(int) adj);
		}
		t = Intlist_next(t);
	      }
	    }
	  }

	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    endpoints = Uintlist_reverse(endpoints); /* gplus */
	    gaps = Intlist_reverse(gaps);	     /* gplus */
	    debug2(printf("MINUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngminus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    overall_trstart = transcript_diag + (querylength - Intlist_head(queryends)); /* tminus */
	    overall_trend = transcript_diag + (querylength - querystart); /* tminus: querystart is the last one processed */
	    gminus_paths[ngminus++] = Path_new(total_nmatches,trnum,overall_trend,overall_trstart,
					       /*trim_low*/trim3,/*trim_high*/trim5,chrnum,chroffset,chrhigh,chrlength,
					       endpoints,gaps,concordantp);
	  }

	} else {
	  /* Case 4: tminus, gminus.  transcript_plusp == false && transcript_genestrand < 0 */
	  debug2(printf("Case 4: tminus, gminus\n"));
	  /* sensedir = SENSE_ANTI; */
	  /* plusp = true; */

	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;

	  for (q = tr_lefts, r = querystarts, s = queryends, t = adjustments; q != NULL;
	       q = Uintlist_next(q), r = Intlist_next(r), s = Intlist_next(s)) {
	    transcript_diag = Uintlist_head(q) - troffset;

	    /* Find initial exonpos */
	    queryend = Intlist_head(s); /* tminus: compute from queryend to querystart */
	    querystart = Intlist_head(r);
	    debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	    trstart = transcript_diag + (querylength - queryend); /* tminus */
#ifdef DEBUG2
	    trend = transcript_diag + (querylength - querystart); /* tminus */
	    printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, overall_adj %d\n",
		   transcript_diag,trstart,trend,Uintlist_head(q),overall_adj);
#endif

	    if (q == tr_lefts) {
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif
	    } else {
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
	    }

	    if (exoni >= nexons) {
	      debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	      abortp = true;
	    } else {
	      debug2(printf("(4) exoni %d out of nexons %d\n",exoni,nexons));
	      if (exoni == 0) {
		exonpos = trstart;
	      } else {
		exonpos = trstart - exonbounds[exoni - 1];
	      }
	      exon_residual = exonbounds[exoni] - trstart;
	      debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));

	      /* Generate one substring per exon */
	      genomicpos = exonstarts[exoni] - exonpos; /* gminus */
	      debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
	      align_residual = queryend - querystart;
	      while (align_residual > 0) {
		debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		querystart = queryend - len; /* tminus */
	
		alignend = chroffset + genomicpos;
		alignstart = alignend - len; /* gminus */
		debug2(left = alignstart - querystart); /* tminus == gminus */
		debug2(printf("tr complete case 4 query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
			      querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			      left - chroffset,alignstart - chroffset,querystart));

		/* tminus: push alignend first */
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
		endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));

		if (align_residual <= exon_residual) {
		  exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		  align_residual = 0;
		} else if (++exoni >= nexons) {
		  /* Aligning past the end of the transcript */
		  debug2(printf("8 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		  align_residual = 0;
		  abortp = true;
		} else {
		  /* nintrons += 1; */
		  align_residual -= len;
		  queryend -= len; /* tminus */
		  genomicpos = exonstarts[exoni]; /* gminus */
		  debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		  exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		  debug2(printf("INTRON\n"));
		  gaps = Intlist_push(gaps,0);
		}
	      }

	      if (Uintlist_next(q) != NULL) {
		nindels = adj = Intlist_head(t);
		debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
		if (nindels < 0) {
		  nindels = -nindels;
		}
		debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
		if (abortp == true) {
		  /* Skip */
		} else if (/*len <= nindels ||*/exon_residual <= nindels) {
		  debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (4)\n"));
		  abortp = true;
		} else {
		  gaps = Intlist_push(gaps,(int) adj);
		}
		t = Intlist_next(t);
	      }
	    }
	  }

	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    /* gminus: Do not reverse endpoints or gaps */
	    debug2(printf("PLUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngplus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    overall_trstart = transcript_diag + (querylength - Intlist_head(queryends)); /* tminus */
	    overall_trend = transcript_diag + (querylength - querystart); /* tminus: querystart is the last one processed */
	    gplus_paths[ngplus++] = Path_new(total_nmatches,trnum,overall_trend,overall_trstart,
					     /*trim_low*/trim3,/*trim_high*/trim5,chrnum,chroffset,chrhigh,chrlength,
					     endpoints,gaps,concordantp);
	  }
	}

	Intlist_free(&adjustments);
	Intlist_free(&queryends);
	Intlist_free(&querystarts);
	Uintlist_free(&tr_lefts);
      }
    }

    debug2(printf("tminus loci %d = ngplus %d + ngminus %d\n",n_tminus_loci,ngplus,ngminus));
  }

#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(tminus_diagonals);
#else
  FREE(tminus_diagonals);
#endif
  *tminus_gminus_paths = gminus_paths;
  *n_tminus_gminus = ngminus;
  *tminus_gplus_paths = gplus_paths;
  *n_tminus_gplus = ngplus;

  return;
}



static int
search_transcriptome_ends (Path_T **tplus_gplus_paths, int *n_tplus_gplus,
			   Path_T **tplus_gminus_paths, int *n_tplus_gminus,
			   Path_T **tminus_gminus_paths, int *n_tminus_gminus,
			   Path_T **tminus_gplus_paths, int *n_tminus_gplus,
				
			   Trcoord_T **tplus_positions_5, int *n_tplus_positions_5, int *tplus_diagterm_5,
			   Trcoord_T **tminus_positions_5, int *n_tminus_positions_5, int *tminus_diagterm_5,
			   Trcoord_T **tplus_positions_3, int *n_tplus_positions_3, int *tplus_diagterm_3,
			   Trcoord_T **tminus_positions_3, int *n_tminus_positions_3, int *tminus_diagterm_3,
			   
			   char *queryuc_ptr, int querylength,
			   Compress_T query_compress_fwd, Compress_T query_compress_rev,
			   Univ_IIT_T transcript_iit, Transcriptome_T transcriptome, Genome_T transcriptomebits, 
			   int nmismatches_allowed, bool concordantp) {
  Path_T *gplus_paths, *gminus_paths;
  int ngplus, ngminus;
				
  Trcoord_T tr_left, tr_lefta, tr_leftb, tr_left0, tr_left1;
  Trcoord_T tr_univdiagonal, tr_univdiagonala, tr_univdiagonalb;
  Trcoord_T *tplus_diagpairs, *tminus_diagpairs;
  int n_tplus_diagpairs, n_tminus_diagpairs, i, k;
  bool tplus_exactp, tminus_exactp;

  int adj, nindels, pos5, pos3, pos5a, pos3a, pos5b, pos3b;
  int trim5 = 0, trim3 = 0;	/* Assuming that alignment goes to the ends (found that way) */
  int trim_a5, trim_a3, trim_b5, trim_b3;
  int nmismatches, ref_nmismatches, total_nmatches, nmismatches_a5, nmismatches_a3, nmismatches_b5, nmismatches_b3;
  int best_nmismatches_i, best_nmismatches_j;
  int best_ref_nmismatches_i, best_ref_nmismatches_j;
  bool want_lowest_coordinate_p;

  Reader_T reader;
  int querypos, query_lastpos, indel_pos, ninserts;
  Oligospace_T forward, revcomp, forward_oligo, revcomp_oligo;

#if 0 && defined(HAVE_AVX2)
  __m256i _diff, _exonbounds, _trstart;
  int matchbits;
  int *ptr;
#elif 0 && defined(HAVE_SSE2)
  __m128i _diff, _exonbounds, _trstart;
  int matchbits;
  int *ptr;
#endif

  Trnum_T trnum;
  int transcript_genestrand;
  Trcoord_T troffset, trhigh, trlength, low, high;
  Chrnum_T chrnum;
  int nexons, exoni;

  int *exonbounds, transcript_diag, transcript_diag0, transcript_diag1,
    overall_trstart, overall_trend, trstart;
#ifdef DEBUG2
  int trend;
  bool allocp;
  Univcoord_T left;
#endif
  Chrpos_T *exonstarts, genomicpos, chrlength;
  Univcoord_T alignstart, alignend, chroffset, chrhigh;
  int querystart, queryend, exonpos, align_residual, exon_residual, len;
  bool abortp;

  Uintlist_T endpoints;
  Intlist_T gaps;

  *tplus_positions_5 = *tminus_positions_5 = *tplus_positions_3 = *tminus_positions_3 = (Trcoord_T *) NULL;
  *n_tplus_positions_5 = *n_tminus_positions_5 = *n_tplus_positions_3 = *n_tminus_positions_3 = 0;
			

  debug3(printf("%s\n",queryuc_ptr));
  reader = Reader_new(queryuc_ptr,/*querystart*/0,/*queryend*/querylength);
  forward = revcomp = 0;
  if (Oligo_next_5(/*last_state*/INIT,&querypos,&forward,&revcomp,reader,/*genestrand*/0) != DONE) {
    forward_oligo = forward & oligobase_mask;
    *tplus_diagterm_5 = querylength - querypos;
    *n_tplus_positions_5 = Indexdb_ptr(&(*tplus_positions_5),/*plus_indexdb*/indexdb_tr,forward_oligo);

    revcomp_oligo = (revcomp >> leftreadshift) & oligobase_mask;
    *tminus_diagterm_5 = querypos + index1part_tr;
    *n_tminus_positions_5 = Indexdb_ptr(&(*tminus_positions_5),/*minus_indexdb*/indexdb_tr,revcomp_oligo);
    debug3(printf("5' end: %s %s: %d plus positions, %d minus positions\n",
		  Oligo_one_nt(forward_oligo,index1part_tr),Oligo_one_nt(revcomp_oligo,index1part_tr),
		  *n_tplus_positions_5,*n_tminus_positions_5));
  }
  Reader_free(&reader);

  query_lastpos = querylength - index1part_tr;
  reader = Reader_new(queryuc_ptr,/*querystart*/query_lastpos - /*index1interval*/1 + 1,/*queryend*/querylength);
  forward = revcomp = 0;
  if (Oligo_next_5(/*last_state*/INIT,&querypos,&forward,&revcomp,reader,/*genestrand*/0) != DONE) {
    forward_oligo = forward & oligobase_mask;
    *tplus_diagterm_3 = querylength - querypos;
    *n_tplus_positions_3 = Indexdb_ptr(&(*tplus_positions_3),/*plus_indexdb*/indexdb_tr,forward_oligo);

    revcomp_oligo = (revcomp >> leftreadshift) & oligobase_mask;
    *tminus_diagterm_3 = querypos + index1part_tr;
    *n_tminus_positions_3 = Indexdb_ptr(&(*tminus_positions_3),/*minus_indexdb*/indexdb_tr,revcomp_oligo);
    debug3(printf("3' end: %s %s: %d plus positions, %d minus positions\n",
		  Oligo_one_nt(forward_oligo,index1part_tr),Oligo_one_nt(revcomp_oligo,index1part_tr),
		  *n_tplus_positions_5,*n_tminus_positions_5));
  }
  Reader_free(&reader);
	 
  tplus_diagpairs = Intersect_approx(&tplus_exactp,&n_tplus_diagpairs,
				     *tplus_positions_5,*n_tplus_positions_5,*tplus_diagterm_5,
				     *tplus_positions_3,*n_tplus_positions_3,*tplus_diagterm_3,
				     /*maxdistance*/3);
  tminus_diagpairs = Intersect_approx(&tminus_exactp,&n_tminus_diagpairs,
				      *tminus_positions_5,*n_tminus_positions_5,*tminus_diagterm_5,
				      *tminus_positions_3,*n_tminus_positions_3,*tminus_diagterm_3,
				      /*maxdistance*/3);
  debug2(printf("***Ultrafast transcriptome: exactp %d and %d.  %d plus and %d minus diagpairs\n",
		tplus_exactp,tminus_exactp,n_tplus_diagpairs,n_tminus_diagpairs));


  if (n_tplus_diagpairs == 0) {
    FREE(tplus_diagpairs);	/* Occupies memory even if n_tplus_diagpairs == 0 */
    *tplus_gplus_paths = *tplus_gminus_paths = (Path_T *) NULL;
    *n_tplus_gplus = *n_tplus_gminus = 0;

  } else if (tplus_exactp == true) {
    ngplus = ngminus = 0;
    gplus_paths = (Path_T *) CALLOC(n_tplus_diagpairs,sizeof(Path_T));
    gminus_paths = (Path_T *) CALLOC(n_tplus_diagpairs,sizeof(Path_T));

    for (i = 0, k = 0; i < n_tplus_diagpairs; i++, k += 2) {
      debug2(printf("tplus diagpairs: %u and %u\n",tplus_diagpairs[k],tplus_diagpairs[k+1]));
      if ((tr_univdiagonal = tplus_diagpairs[k]) == tplus_diagpairs[k+1]) {
	tr_left = tr_univdiagonal - (Trcoord_T) querylength; /* NEW FORMULA for querystart of 0 */

	/* TRIM COORDINATES AT TRANSCRIPTOME BOUNDS */
	low = (tr_univdiagonal >= (Univcoord_T) querylength) ? tr_left : 0;
	high = (tr_univdiagonal <= transcriptomelength) ? tr_univdiagonal : transcriptomelength;
	trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high);
	debug2(printf("(3) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));

	/* TRIM QUERY AT TRANSCRIPT BOUDNS */
	pos5 = (tr_univdiagonal >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_left);
	pos3 = (tr_univdiagonal <= trhigh) ? querylength : (int) (trhigh - tr_left);
	debug2(printf("Trimming at transcript bounds: tr_univdiagonal %u relative to troffset %u and trhigh %u => pos5 %d, pos3 %d\n",
		      tr_univdiagonal,troffset,trhigh,pos5,pos3));

	debug2a(printf("ULTRAFAST PLUS DIAGONAL %u\n",tplus_diagpairs[k]));
	nmismatches = Genome_count_mismatches_substring(&ref_nmismatches,transcriptomebits,NULL,query_compress_fwd,tr_left,
							pos5,pos3,/*plusp*/true,/*genestrand*/0);
	if (nmismatches <= nmismatches_allowed) {
	  debug2a(printf(" => nmismatches %d\n",nmismatches));
	  total_nmatches = (pos3 - pos5) - nmismatches;

	  abortp = false;
	  if ((chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum)) <= 0) {
	    debug2(printf("No genomic alignment for trnum %d, so aborting\n",trnum));
	    abortp = true;
	  } else {
	    nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	    Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);
	    debug2a(printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand));

	    if (transcript_genestrand > 0) {
	      /* Case 1 ultrafast nogap: tplus, gplus.  transcript_plusp == true && transcript_genestrand > 0 */
	      debug2(printf("Case 1 ultrafast nogap: tplus, gplus\n"));
	      /* sensedir = SENSE_FORWARD; */
	      /* plusp = true; */
	      
	      endpoints = (Uintlist_T) NULL;
	      gaps = (Intlist_T) NULL;
	      abortp = false;
	      
	      /* Find initial exonpos */
	      querystart = 0;	/* tplus: compute from querystart to queryend */
	      queryend = querylength;
	      debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	      transcript_diag = tr_left - troffset;
	      trstart = transcript_diag /*+ querystart (0)*/; /* tplus */
#ifdef DEBUG2
	      trend = transcript_diag + queryend; /* tplus */
	      printf("transcript_diag %d = tr_left %u - troffset %u, trstart %d, trend %d\n",
		     transcript_diag,tr_left,troffset,trstart,trend);
#endif
	      
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif

	      if (exoni >= nexons) {
		debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
		abortp = true;
	      } else {
		debug2(printf("(5) exoni %d out of nexons %d\n",exoni,nexons));
		if (exoni == 0) {
		  exonpos = trstart;
		} else {
		  exonpos = trstart - exonbounds[exoni - 1];
		}
		exon_residual = exonbounds[exoni] - trstart;
		debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
		
		/* Generate one substring per exon */
		genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
		debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
		align_residual = queryend - querystart;
		while (align_residual > 0) {
		  debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		  len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		  queryend = querystart + len; /* tplus */
		  
		  alignstart = chroffset + genomicpos;
		  alignend = alignstart + len; /* gplus */
		  debug2(left = alignstart /*- querystart (0)*/); /* gplus == gminus */
		  debug2(printf("tr ultrafast nogap case 1 query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
				querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
				left - chroffset,alignstart - chroffset,querystart));
		  
		  /* tplus: push alignstart first */
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
		  
		  if (align_residual <= exon_residual) {
		    exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		    align_residual = 0;
		  } else if (++exoni >= nexons) {
		    /* Aligning past the end of the transcript */
		    debug2(printf("2 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		    align_residual = 0;
		    abortp = true;
		  } else {
		    /* nintrons += 1; */
		    align_residual -= len;
		    querystart += len; /* tplus */
		    genomicpos = exonstarts[exoni] - 1; /* gplus */
		    debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		    exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		    debug2(printf("INTRON\n"));
		    gaps = Intlist_push(gaps,0);
		  }
		}
	      }
	  
	      if (abortp == true) {
		debug2(printf("ABORTING PATH\n"));
		Uintlist_free(&endpoints);
		Intlist_free(&gaps);
	      } else {
		endpoints = Uintlist_reverse(endpoints); /* gplus */
		gaps = Intlist_reverse(gaps);	     /* gplus */
		debug2(printf("PLUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			      ngplus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
		overall_trstart = transcript_diag	 /*+ querystart (0)*/; /* tplus */
		overall_trend = transcript_diag + queryend; /* tplus */
		gplus_paths[ngplus++] = Path_new(total_nmatches,trnum,overall_trstart,overall_trend,
						 /*trim_low*/0,/*trim_high*/0,chrnum,chroffset,chrhigh,chrlength,
						 endpoints,gaps,concordantp);
	      }

	    } else {
	      /* Case 2 ultrafast nogap: tplus, gminus.  transcript_plusp == true && transcript_genestrand < 0 */
	      debug2(printf("Case 2 ultrafast nogap: tplus, gminus\n"));
	      /* sensedir = SENSE_FORWARD; */
	      /* plusp = false; */
	      
	      endpoints = (Uintlist_T) NULL;
	      gaps = (Intlist_T) NULL;
	      abortp = false;
	      
	      /* Find initial exonpos */
	      querystart = 0;	/* tplus: compute from querystart to queryend */
	      queryend = querylength;
	      debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	      transcript_diag = tr_left - troffset;
	      trstart = transcript_diag /*+ querystart (0) */; /* tplus */
#ifdef DEBUG2
	      trend = transcript_diag + queryend; /* tplus */
	      printf("transcript_diag %d = tr_left %u - troffset %u, trstart %d, trend %d\n",
		     transcript_diag,tr_left,troffset,trstart,trend);
#endif
	      
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif
	      
	      if (exoni >= nexons) {
		debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
		abortp = true;
	      } else {
		debug2(printf("(6) exoni %d out of nexons %d\n",exoni,nexons));
		if (exoni == 0) {
		  exonpos = trstart;
		} else {
		  exonpos = trstart - exonbounds[exoni - 1];
		}
		exon_residual = exonbounds[exoni] - trstart;
		debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
		
		/* Generate one substring per exon */
		genomicpos = exonstarts[exoni] - exonpos; /* gminus */
		debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
		align_residual = queryend - querystart;
		while (align_residual > 0) {
		  debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		  len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		  queryend = querystart + len; /* tplus */
		  
		  alignstart = chroffset + genomicpos;
		  alignend = alignstart - len; /* gminus */
		  debug2(left = alignend /*- (querylength - queryend) (0)*/); /* tplus != gminus */
		  debug2(printf("tr ultrafast nogap case 2 query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
				querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
				left - chroffset,alignend - chroffset,querylength,queryend));
		  
		  /* tplus: push alignstart first */
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
		  
		  if (align_residual <= exon_residual) {
		    exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		    align_residual = 0;
		  } else if (++exoni >= nexons) {
		    /* Aligning past the end of the transcript */
		    debug2(printf("4 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		    align_residual = 0;
		    abortp = true;
		  } else {
		    /* nintrons += 1; */
		    align_residual -= len;
		    querystart += len; /* tplus */
		    genomicpos = exonstarts[exoni]; /* gminus */
		    debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		    exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		    debug2(printf("INTRON\n"));
		    gaps = Intlist_push(gaps,0);
		  }
		}
	      }
      
	      if (abortp == true) {
		debug2(printf("ABORTING PATH\n"));
		Uintlist_free(&endpoints);
		Intlist_free(&gaps);
	      } else {
		/* gminus: Do not reverse endpoints or gaps */
		debug2(printf("MINUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			      ngminus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
		overall_trstart = transcript_diag	 /*+ querystart (0)*/; /* tplus */
		overall_trend = transcript_diag + queryend; /* tplus */
		gminus_paths[ngminus++] = Path_new(total_nmatches,trnum,overall_trstart,overall_trend,
						   /*trim_low*/0,/*trim_high*/0,chrnum,chroffset,chrhigh,chrlength,
						   endpoints,gaps,concordantp);
	      }
	    }
	  }
	}
      }
    }

    FREE(tplus_diagpairs);	/* Occupies memory even if n_tplus_diagpairs == 0 */
    *tplus_gplus_paths = gplus_paths;
    *n_tplus_gplus = ngplus;
    *tplus_gminus_paths = gminus_paths;
    *n_tplus_gminus = ngminus;

  } else if (tminus_exactp == true) {
    FREE(tplus_diagpairs);	/* Occupies memory even if n_tplus_diagpairs == 0 */
    *tplus_gplus_paths = *tplus_gminus_paths = (Path_T *) NULL;
    *n_tplus_gplus = *n_tplus_gminus = 0;

  } else {
    /* Handle approximate diagpairs below */
  }
    

  if (n_tminus_diagpairs == 0) {
    FREE(tminus_diagpairs);	/* Occupies memory even if n_tminus_diagpairs == 0 */
    *tminus_gplus_paths = *tminus_gminus_paths = (Path_T *) NULL;
    *n_tminus_gplus = *n_tminus_gminus = 0;

  } else if (tminus_exactp == true) {
    ngplus = ngminus = 0;
    gplus_paths = (Path_T *) CALLOC(n_tminus_diagpairs,sizeof(Path_T));
    gminus_paths = (Path_T *) CALLOC(n_tminus_diagpairs,sizeof(Path_T));

    for (i = 0, k = 0; i < n_tminus_diagpairs; i++, k += 2) {
      debug2(printf("tminus diagpairs: %u and %u\n",tminus_diagpairs[k],tminus_diagpairs[k+1]));
      if ((tr_univdiagonal = tminus_diagpairs[k]) == tminus_diagpairs[k+1]) {
	tr_left = tr_univdiagonal - (Trcoord_T) querylength; /* NEW FORMULA for querystart of 0 */

	/* TRIM COORDINATES AT TRANSCRIPTOME BOUNDS */
	low = (tr_univdiagonal >= (Univcoord_T) querylength) ? tr_left : 0;
	high = (tr_univdiagonal <= transcriptomelength) ? tr_univdiagonal : transcriptomelength;
	trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high);
	debug2(printf("(4) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));

	/* TRIM QUERY AT TRANSCRIPT BOUNDS */
	pos5 = (tr_univdiagonal >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_left);
	pos3 = (tr_univdiagonal <= trhigh) ? querylength : (int) (trhigh - tr_left);
	debug2(printf("Trimming at transcript bounds: tr_univdiagonal %u relative to troffset %u and trhigh %u => pos5 %d, pos3 %d\n",
		      tr_univdiagonal,troffset,trhigh,pos5,pos3));

	debug2a(printf("ULTRAFAST MINUS DIAGONAL %u\n",tminus_diagpairs[k]));
	nmismatches = Genome_count_mismatches_substring(&ref_nmismatches,transcriptomebits,NULL,query_compress_rev,tr_left,
							pos5,pos3,/*plusp*/false,/*genestrand*/0);
	
	if (nmismatches <= nmismatches_allowed) {
	  debug2a(printf(" => nmismatches %d\n",nmismatches));
	  total_nmatches = (pos3 - pos5) - nmismatches;

	  abortp = false;
	  if ((chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum)) <= 0) {
	    debug2(printf("No genomic alignment for trnum %d, so aborting\n",trnum));
	    abortp = true;
	  } else {
	    nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	    Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);
	    debug2a(printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand));

	    if (transcript_genestrand > 0) {
	      /* Case 3 ultrafast nogap: tminus, gplus.  transcript_plusp == false && transcript_genestrand > 0 */
	      debug2(printf("Case 3 ultrafast nogap: tminus, gplus\n"));
	      /* sensedir = SENSE_ANTI; */
	      /* plusp = false; */
	      
	      endpoints = (Uintlist_T) NULL;
	      gaps = (Intlist_T) NULL;
	      abortp = false;
	      
	      /* Find initial exonpos */
	      queryend = querylength; /* tminus: compute from queryend to querystart */
	      querystart = 0;
	      debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	      transcript_diag = tr_left - troffset;
	      trstart = transcript_diag /*+ (querylength - queryend) (0)*/; /* tminus */
#ifdef DEBUG2
	      trend = transcript_diag + (querylength /*- querystart (0)*/); /* tminus */
	      printf("transcript_diag %d = tr_left %u - troffset %u, trstart %d, trend %d\n",
		     transcript_diag,tr_left,troffset,trstart,trend);
#endif	  
	      
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif

	      if (exoni >= nexons) {
		debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
		abortp = true;
	      } else {
		debug2(printf("(7) exoni %d out of nexons %d\n",exoni,nexons));
		if (exoni == 0) {
		  exonpos = trstart;
		} else {
		  exonpos = trstart - exonbounds[exoni - 1];
		}
		exon_residual = exonbounds[exoni] - trstart;
		debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
		
		/* Generate one substring per exon */
		genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
		debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
		align_residual = queryend - querystart;
		while (align_residual > 0) {
		  debug2(printf("abortp %d,align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		  len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		  querystart = queryend - len; /* tminus */
		  
		  alignend = chroffset + genomicpos;
		  alignstart = alignend + len; /* gplus */
		  debug2(left = alignend /*- (querylength - queryend) (0)*/);  /* tminus != gplus */
		  debug2(printf("tr ultrafast nogap case 3 query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
				querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
				left - chroffset,alignend - chroffset,querylength,queryend));
		  
		  /* tminus: plus alignend first */
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
		  
		  if (align_residual <= exon_residual) {
		    exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		    align_residual = 0;
		  } else if (++exoni >= nexons) {
		    /* Aligning past the end of the transcript */
		    debug2(printf("6 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		    align_residual = 0;
		    abortp = true;
		  } else {
		    /* nintrons += 1; */
		    align_residual -= len;
		    queryend -= len; /* tminus */
		    genomicpos = exonstarts[exoni] - 1; /* gplus */
		    debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		    exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		    debug2(printf("INTRON\n"));
		    gaps = Intlist_push(gaps,0);
		  }
		}
	      }

	      if (abortp == true) {
		debug2(printf("ABORTING PATH\n"));
		Uintlist_free(&endpoints);
		Intlist_free(&gaps);
	      } else {
		endpoints = Uintlist_reverse(endpoints); /* gplus */
		gaps = Intlist_reverse(gaps);	     /* gplus */
		debug2(printf("MINUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			      ngminus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
		overall_trstart = transcript_diag /*+ (querylength - queryend) (0)*/; /* tminus */
		overall_trend = transcript_diag + (querylength /*- querystart (0)*/); /* tminus */
		gminus_paths[ngminus++] = Path_new(total_nmatches,trnum,overall_trend,overall_trstart,
						   /*trim_low*/0,/*trim_high*/0,chrnum,chroffset,chrhigh,chrlength,
						   endpoints,gaps,concordantp);
	      }

	    } else {
	      /* Case 4 ultrafast nogap: tminus, gminus.  transcript_plusp == false && transcript_genestrand < 0 */
	      debug2(printf("Case 4 ultrafast nogap: tminus, gminus\n"));
	      /* sensedir = SENSE_ANTI; */
	      /* plusp = true; */
	      
	      endpoints = (Uintlist_T) NULL;
	      gaps = (Intlist_T) NULL;
	      abortp = false;
	      
	      /* Find initial exonpos */
	      queryend = querylength; /* tminus: compute from queryend to querystart */
	      querystart = 0;
	      debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));

	      transcript_diag = tr_left - troffset;
	      trstart = transcript_diag /*+ (querylength - queryend) (0)*/; /* tminus */
#ifdef DEBUG2
	      trend = transcript_diag + (querylength /*- querystart (0)*/); /* tminus */
	      printf("transcript_diag %d = tr_left %u - troffset %u, trstart %d, trend %d\n",
		     transcript_diag,tr_left,troffset,trstart,trend);
#endif	  
	      
	      exoni = 0;
#if 0 && defined(HAVE_AVX2)
	      _trstart = _mm256_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	      _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 8;
		ptr += 8;
		_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
		_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	      _trstart = _mm_set1_epi32(trstart);
	      ptr = exonbounds;
	      _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	      _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      while (all_zero_p(_diff)) {
		exoni += 4;
		ptr += 4;
		_exonbounds = _mm_loadu_si128((__m128i *) ptr);
		_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	      }
	      matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	      exoni += count_trailing_zeroes_32(matchbits);
#else
	      while (exoni < nexons && exonbounds[exoni] <= trstart) {
		exoni++;
	      }
#endif

	      if (exoni >= nexons) {
		debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
		abortp = true;
	      } else {
		debug2(printf("(8) exoni %d out of nexons %d\n",exoni,nexons));
		if (exoni == 0) {
		  exonpos = trstart;
		} else {
		  exonpos = trstart - exonbounds[exoni - 1];
		}
		exon_residual = exonbounds[exoni] - trstart;
		debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
		
		/* Generate one substring per exon */
		genomicpos = exonstarts[exoni] - exonpos; /* gminus */
		debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
		align_residual = queryend - querystart;
		while (align_residual > 0) {
		  debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
		  len = (align_residual <= exon_residual) ? align_residual : exon_residual;
		  querystart = queryend - len; /* tminus */
		  
		  alignend = chroffset + genomicpos;
		  alignstart = alignend - len; /* gminus */
		  debug2(left = alignstart /*- querystart (0)*/); /* tminus == gminus */
		  debug2(printf("tr ultrafast nogap case 4 query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
				querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
				left - chroffset,alignstart - chroffset,querystart));
		  
		  /* tminus: push alignend first */
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
		  endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
		  
		  if (align_residual <= exon_residual) {
		    exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		    align_residual = 0;
		  } else if (++exoni >= nexons) {
		    /* Aligning past the end of the transcript */
		    debug2(printf("8 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		    align_residual = 0;
		    abortp = true;
		  } else {
		    /* nintrons += 1; */
		    align_residual -= len;
		    queryend -= len; /* tminus */
		    genomicpos = exonstarts[exoni]; /* gminus */
		    debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		    exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		    debug2(printf("INTRON\n"));
		    gaps = Intlist_push(gaps,0);
		  }
		}
	      }

	      if (abortp == true) {
		debug2(printf("ABORTING PATH\n"));
		Uintlist_free(&endpoints);
		Intlist_free(&gaps);
	      } else {
		/* gminus: Do not reverse endpoints or gaps */
		debug2(printf("PLUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			      ngplus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
		overall_trstart = transcript_diag /*+ (querylength - queryend) (0)*/; /* tminus */
		overall_trend = transcript_diag + (querylength /*- querystart (0)*/); /* tminus */
		gplus_paths[ngplus++] = Path_new(total_nmatches,trnum,overall_trend,overall_trstart,
						 /*trim_low*/0,/*trim_high*/0,chrnum,chroffset,chrhigh,chrlength,
						 endpoints,gaps,concordantp);
	      }
	    }
	  }
	}
      }
    }
    FREE(tminus_diagpairs);	/* Occupies memory even if n_tminus_diagpairs == 0 */
    *tminus_gplus_paths = gplus_paths;
    *n_tminus_gplus = ngplus;
    *tminus_gminus_paths = gminus_paths;
    *n_tminus_gminus = ngminus;

  } else if (tplus_exactp == true) {
    FREE(tminus_diagpairs);	/* Occupies memory even if n_tminus_diagpairs == 0 */
    *tminus_gplus_paths = *tminus_gminus_paths = (Path_T *) NULL;
    *n_tminus_gplus = *n_tminus_gminus = 0;

  } else {
    /* Handle approximate diagpairs below */
  }

  /* Handle approximate diagpairs */
  if (tplus_exactp == false && tminus_exactp == false) {

    /* Case for n_tplus_diagpairs == 0 already handled above */
    if (n_tplus_diagpairs > 0) {
      ngplus = ngminus = 0;
      gplus_paths = (Path_T *) CALLOC(n_tplus_diagpairs,sizeof(Path_T));
      gminus_paths = (Path_T *) CALLOC(n_tplus_diagpairs,sizeof(Path_T));

      for (i = 0, k = 0; i < n_tplus_diagpairs; i++, k += 2) {
	debug2(printf("tplus diagpairs: %u and %u\n",tplus_diagpairs[k],tplus_diagpairs[k+1]));
	tr_univdiagonala = tplus_diagpairs[k];
	tr_univdiagonalb = tplus_diagpairs[k+1];

	tr_lefta = tr_univdiagonala - (Trcoord_T) querylength; /* NEW FORMULA for querystart of 0 */
	tr_leftb = tr_univdiagonalb - (Trcoord_T) querylength; /* NEW FORMULA for querystart of 0 */

	/* TRIM COORDINATES AT TRANSCRIPTOME BOUNDS */
	low = (tr_univdiagonala >= (Trcoord_T) querylength) ? tr_lefta : 0;
	high = (tr_univdiagonala <- transcriptomelength) ? tr_univdiagonala : transcriptomelength;
	trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high);

	low = (tr_univdiagonalb >= (Trcoord_T) querylength) ? tr_lefta : 0;
	high = (tr_univdiagonalb <- transcriptomelength) ? tr_univdiagonalb : transcriptomelength;
	if (Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high) != trnum) {
	  /* Skip further computation */
	  adj = 0;
	} else {
	  debug2(printf("(5) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));

	  /* TRIM QUERY AT TRANSCRIPT BOUNDS */
	  pos5a = (tr_univdiagonala >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_lefta);
	  pos3a = (tr_univdiagonala <= trhigh) ? querylength : (int) (trhigh - tr_lefta);
	  trim_a5 = Genome_first_kmer_left(&nmismatches_a5,transcriptomebits,query_compress_fwd,tr_lefta,pos5a,pos3a,
					   /*plusp*/true,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  trim_a3 = querylength - Genome_first_kmer_right(&nmismatches_a3,transcriptomebits,query_compress_fwd,tr_lefta,pos5a,pos3a,
							  /*plusp*/true,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  
	  /* TRIM QUERY TO TRANSCRIPT BOUNDS */
	  pos5b = (tr_univdiagonalb >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_leftb);
	  pos3b = (tr_univdiagonalb <= trhigh) ? querylength : (int) (trhigh - tr_leftb);
	  trim_b5 = Genome_first_kmer_left(&nmismatches_b5,transcriptomebits,query_compress_fwd,tr_leftb,pos5b,pos3b,
					   /*plusp*/true,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  trim_b3 = querylength - Genome_first_kmer_right(&nmismatches_b3,transcriptomebits,query_compress_fwd,tr_leftb,pos5b,pos3b,
							  /*plusp*/true,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  
	  debug2a(printf("trimmed a %d..%d, trimmed b %d..%d\n",
			 trim_a5,querylength - trim_a3,trim_b5,querylength - trim_b3));
	  if (trim_a5 == pos5a || trim_b3 == pos3b) {
	    tr_left0 = tr_lefta; /* low0 = tr_left0 + pos5a; high0 = tr_left0 + pos3a; */
	    tr_left1 = tr_leftb; /* low1 = tr_left1 + pos5b; high1 = tr_left1 + pos3b; */
	    adj = (int) (tr_univdiagonalb - tr_univdiagonala);
	    debug2a(printf("Setting a first, b second, adj %d\n",adj));
	  } else if (trim_a3 == pos3a || trim_b5 == pos5b) {
	    tr_left0 = tr_leftb; /* low0 = tr_left0 + pos5b; high0 = tr_left0 + pos3b; */
	    tr_left1 = tr_lefta; /* low1 = tr_left1 + pos5a; high1 = tr_left1 + pos3a; */
	    adj = (int) (tr_univdiagonala - tr_univdiagonalb);
	    debug2a(printf("Setting b first, a second, adj %d\n",adj));
	  } else {
	    /* Read does not extend to the ends, so this method is not applicable */
	    adj = 0;
	  }
	}

	if (adj != 0) {
	  if ((chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum)) <= 0) {
	    adj = 0;
	  } else {
	    if (transcript_genestrand > 0) {
	      want_lowest_coordinate_p = true;
	    } else {
	      want_lowest_coordinate_p = false;
	    }
	    debug2(printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand));

	    if (adj > 0 &&
		(indel_pos = Indel_resolve_middle_deletion(&best_nmismatches_i,&best_nmismatches_j,
							   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
							   /*left*/tr_left0,/*indels*/-adj,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/transcriptomebits,/*ome_alt*/NULL,query_compress_fwd,
							   /*querystart*/0,/*queryend*/querylength,querylength,
							   nmismatches_allowed,/*plusp*/true,/*genestrand*/0,
							   want_lowest_coordinate_p)) > 0) {
	      ninserts = 0;
	      nindels = adj;
	      total_nmatches = querylength - best_nmismatches_i - best_nmismatches_j;
	    } else if (adj < 0 &&
		       (indel_pos = Indel_resolve_middle_insertion(&best_nmismatches_i,&best_nmismatches_j,
								   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
								   /*left*/tr_left0,/*indels*/-adj,
								   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
								   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
								   /*ome*/transcriptomebits,/*ome_alt*/NULL,query_compress_fwd,
								   /*querystart*/0,/*queryend*/querylength,querylength,
								   nmismatches_allowed,/*plusp*/true,/*genestrand*/0,
								   want_lowest_coordinate_p)) > 0) {
	      ninserts = nindels = -adj;
	      total_nmatches = querylength - best_nmismatches_i - best_nmismatches_j - ninserts;
	    } else {
	      adj = 0;
	    }
	  }
	}

	if (adj == 0) {
	  /* Skip finding an indel */

	} else if (transcript_genestrand > 0) {
	  /* Case 1 ultrafast gap: tplus, gplus.  transcript_plusp == true && transcript_genestrand > 0 */
	  debug2(printf("Case 1 ultrafast gap: tplus, gplus\n"));

	  /* sensedir = SENSE_FORWARD; */
	  /* plusp = true; */

	  nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	  Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);

	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;
	  
	  transcript_diag0 = tr_left0 - troffset;
	  transcript_diag1 = tr_left1 - troffset;
	  
	  /* first part */
	  tr_left = tr_left0;
	  querystart = 0;
	  queryend = indel_pos;
	  debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));
	  
	  overall_trstart = trstart = transcript_diag0 + querystart;	/* tplus */
#ifdef DEBUG2
	  trend = transcript_diag0 + queryend; /* tplus */
	  printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, adj %d\n",
		 transcript_diag0,trstart,trend, tr_left,adj);
#endif
	  
	  debug2(printf("Searching for exoni for trstart %d\n",trstart));
	  exoni = 0;
#if 0 && defined(HAVE_AVX2)
	  _trstart = _mm256_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	  _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 8;
	    ptr += 8;
	    _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	    _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	  _trstart = _mm_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	  _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 4;
	    ptr += 4;
	    _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	    _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#else
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
#endif
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(9) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
	    debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      queryend = querystart + len; /* tplus */
	      
	      alignstart = chroffset + genomicpos;
	      alignend = alignstart + len; /* gplus */
	      debug2(left = alignstart - querystart); /* tplus == gminus */
	      debug2(printf("tr ultrafast gap case 1 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignstart - chroffset,querystart));
	      
	      /* tplus: push alignstart first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("2 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		querystart += len; /* tplus */
		genomicpos = exonstarts[exoni] - 1; /* gplus */
		debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  /* gap */
	  debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
	  debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
	  if (abortp == true) {
	    /* Skip */
	  } else if (/*len <= nindels ||*/exon_residual <= nindels) {
	    debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (5)\n"));
	    abortp = true;
	  } else {
	    gaps = Intlist_push(gaps,(int) adj);
	  }
	  
	  /* second part */
	  tr_left = tr_left1;
	  querystart = indel_pos + ninserts;
	  queryend = querylength;
	  debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));
	  
	  trstart = transcript_diag1 + querystart;	/* tplus */
	  overall_trend = transcript_diag1 + queryend; /* tplus */
	  debug2(printf("transcript_diag %d, trstart %d, trend %d, tr_left %u\n",
			transcript_diag1,trstart,overall_trend,tr_left));
	  
	  debug2(printf("Searching for exoni for trstart %d\n",trstart));
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(10) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
	    debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      queryend = querystart + len; /* tplus */
	      
	      alignstart = chroffset + genomicpos;
	      alignend = alignstart + len; /* gplus */
	      debug2(left = alignstart - querystart); /* tplus == gminus */
	      debug2(printf("tr ultrafast gap case 1 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignstart - chroffset,querystart));
	      
	      /* tplus: push alignstart first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("2 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		querystart += len; /* tplus */
		genomicpos = exonstarts[exoni] - 1; /* gplus */
		debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    endpoints = Uintlist_reverse(endpoints); /* gplus */
	    gaps = Intlist_reverse(gaps);	     /* gplus */
	    debug2(printf("PLUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngplus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    gplus_paths[ngplus++] = Path_new(total_nmatches,trnum,overall_trstart,overall_trend,
					     /*trim_low*/trim5,/*trim_high*/trim3,
					     chrnum,chroffset,chrhigh,chrlength,endpoints,gaps,concordantp);
	  }
	  
	} else {
	  /* Case 2 ultrafast gap: tplus, gminus.  transcript_plusp == true && transcript_genestrand < 0 */
	  debug2(printf("Case 2 ultrafast gap: tplus, gminus\n"));
	  /* sensedir = SENSE_FORWARD; */
	  /* plusp = false; */
	    
	  nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	  Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);

	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;
	  
	  transcript_diag0 = tr_left0 - troffset;
	  transcript_diag1 = tr_left1 - troffset;
	  
	  /* first part */
	  tr_left = tr_left0;
	  querystart = 0; /* tplus: compute from querystart to queryend */
	  queryend = indel_pos;
	  debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));
	  
	  overall_trstart = trstart = transcript_diag0 + querystart;	/* tplus */
#ifdef DEBUG2
	  trend = transcript_diag0 + queryend; /* tplus */
	  printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, adj %d\n",
		 transcript_diag0,trstart,trend,tr_left,adj);
#endif
	  
	  exoni = 0;
#if 0 && defined(HAVE_AVX2)
	  _trstart = _mm256_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	  _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 8;
	    ptr += 8;
	    _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	    _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	  _trstart = _mm_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	  _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 4;
	    ptr += 4;
	    _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	    _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#else
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
#endif
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(11) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] - exonpos; /* gminus */
	    debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      queryend = querystart + len; /* tplus */
	      
	      alignstart = chroffset + genomicpos;
	      alignend = alignstart - len; /* gminus */
	      debug2(left = alignend - (querylength - queryend)); /* tplus != gminus */
	      debug2(printf("tr ultrafast gap case 2 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignend - chroffset,querylength,queryend));
	      
	      /* tplus: push alignstart first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("4 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		querystart += len; /* tplus */
		genomicpos = exonstarts[exoni]; /* gminus */
		debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  /* gap */
	  debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
	  debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
	  if (abortp == true) {
	    /* Skip */
	  } else if (/*len <= nindels ||*/exon_residual <= nindels) {
	    debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (6)\n"));
	    abortp = true;
	  } else {
	    gaps = Intlist_push(gaps,(int) adj);
	  }
	  
	  /* second part */
	  tr_left = tr_left1;
	  querystart = indel_pos + ninserts;
	  queryend = querylength;
	  debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));
	  
	  trstart = transcript_diag1 + querystart;	/* tplus */
	  overall_trend = transcript_diag1 + queryend; /* tplus */
	  debug2(printf("transcript_diag %d, trstart %d, trend %d, tr_left %u\n",
			transcript_diag1,trstart,overall_trend,tr_left));
	  
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(12) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] - exonpos; /* gminus */
	    debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      queryend = querystart + len; /* tplus */
	      
	      alignstart = chroffset + genomicpos;
	      alignend = alignstart - len; /* gminus */
	      debug2(left = alignend - (querylength - queryend)); /* tplus != gminus */
	      debug2(printf("tr ultrfast gap case 2 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignend - chroffset,querylength,queryend));
	      
	      /* tplus: push alignstart first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("4 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		querystart += len; /* tplus */
		genomicpos = exonstarts[exoni]; /* gminus */
		debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    /* gminus: Do not reverse endpoints or gaps */
	    debug2(printf("MINUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngminus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    gminus_paths[ngminus++] = Path_new(total_nmatches,trnum,overall_trstart,overall_trend,
					       /*trim_low*/trim3,/*trim_high*/trim5,
					       chrnum,chroffset,chrhigh,chrlength,endpoints,gaps,concordantp);
	  }
	}
      }

      FREE(tplus_diagpairs);
      *tplus_gplus_paths = gplus_paths;
      *n_tplus_gplus = ngplus;
      *tplus_gminus_paths = gminus_paths;
      *n_tplus_gminus = ngminus;
    }

    /* Case for n_tminus_diagpairs == 0 already handled above */
    if (n_tminus_diagpairs > 0) {
      ngplus = ngminus = 0;
      gplus_paths = (Path_T *) CALLOC(n_tminus_diagpairs,sizeof(Path_T));
      gminus_paths = (Path_T *) CALLOC(n_tminus_diagpairs,sizeof(Path_T));

      for (i = 0, k = 0; i < n_tminus_diagpairs; i++, k += 2) {
	debug2(printf("tminus diagpairs: %u and %u\n",tminus_diagpairs[k],tminus_diagpairs[k+1]));
	tr_univdiagonala = tminus_diagpairs[k];
	tr_univdiagonalb = tminus_diagpairs[k+1];

	tr_lefta = tr_univdiagonala - (Trcoord_T) querylength; /* NEW FORMULA for queryend of querylength */
	tr_leftb = tr_univdiagonalb - (Trcoord_T) querylength; /* NEW FORMULA for queryend of querylength */

	/* TRIM COORDINATES AT TRANSCRIPTOME BOUNDS */
	low = (tr_univdiagonala >= (Trcoord_T) querylength) ? tr_lefta : 0;
	high = (tr_univdiagonala <- transcriptomelength) ? tr_univdiagonala : transcriptomelength;
	trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high);

	low = (tr_univdiagonalb >= (Trcoord_T) querylength) ? tr_leftb : 0;
	high = (tr_univdiagonalb <- transcriptomelength) ? tr_univdiagonalb : transcriptomelength;
	if (Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,low,high) != trnum) {
	  /* Skip further computation */
	  adj = 0;
	} else {
	  debug2(printf("(6) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));

	  /* TRIM QUERY AT TRANSCRIPT BOUNDS */
	  pos5a = (tr_univdiagonala >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_lefta);
	  pos3a = (tr_univdiagonala <= trhigh) ? querylength : (int) (trhigh - tr_lefta);
	  trim_a5 = Genome_first_kmer_left(&nmismatches_a5,transcriptomebits,query_compress_rev,tr_lefta,pos5a,pos3a,
					   /*plusp*/false,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  trim_a3 = querylength - Genome_first_kmer_right(&nmismatches_a3,transcriptomebits,query_compress_rev,tr_lefta,pos5a,pos3a,
							  /*plusp*/false,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  
	  /* TRIM QUERY AT TRANSCRIPT BOUNDS */
	  pos5b = (tr_univdiagonalb >= troffset + (Trcoord_T) querylength) ? 0 : (int) (troffset - tr_leftb);
	  pos3b = (tr_univdiagonalb <= trhigh) ? querylength : (int) (trhigh - tr_leftb);
	  trim_b5 = Genome_first_kmer_left(&nmismatches_b5,transcriptomebits,query_compress_rev,tr_leftb,pos5b,pos3b,
					   /*plusp*/false,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  trim_b3 = querylength - Genome_first_kmer_right(&nmismatches_b3,transcriptomebits,query_compress_rev,tr_leftb,pos5b,pos3b,
							  /*plusp*/false,/*genestrand*/0,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  
	  debug2a(printf("trimmed a %d..%d, trimmed b %d..%d\n",
			 trim_a5,querylength - trim_a3,trim_b5,querylength - trim_b3));
	  if (trim_a5 == pos5a || trim_b3 == pos3b) {
	    tr_left0 = tr_lefta; /* low0 = tr_left0 + pos5a; high0 = tr_left0 + pos3a; */
	    tr_left1 = tr_leftb; /* low1 = tr_left1 + pos5b; high1 = tr_left1 + pos3b; */
	    adj = (int) (tr_univdiagonalb - tr_univdiagonala);
	    debug2a(printf("Setting a first, b second, adj %d\n",adj));
	  } else if (trim_a3 == pos3a || trim_b5 == pos5b) {
	    tr_left0 = tr_leftb; /* low0 = tr_left0 + pos5b; high0 = tr_left0 + pos3b; */
	    tr_left1 = tr_lefta; /* low1 = tr_left1 + pos5a; high1 = tr_left1 + pos3a; */
	    adj = (int) (tr_univdiagonala - tr_univdiagonalb);
	    debug2a(printf("Setting b first, a second, adj %d\n",adj));
	  } else {
	    /* Read does not extend to the ends, so this method is not applicable */
	    adj = 0;
	  }
	}

	if (adj != 0) {
	  if ((chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum)) <= 0) {
	    adj = 0;
	  } else {
	    if (transcript_genestrand > 0) {
	      want_lowest_coordinate_p = true;
	    } else {
	      want_lowest_coordinate_p = false;
	    }
	    debug2(printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand));
	    
	    if (adj > 0 &&
		(indel_pos = Indel_resolve_middle_deletion(&best_nmismatches_i,&best_nmismatches_j,
							   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
							   /*left*/tr_left0,/*indels*/-adj,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/transcriptomebits,/*ome_alt*/NULL,query_compress_rev,
							   /*querystart*/0,/*queryend*/querylength,querylength,
							   nmismatches_allowed,/*plusp*/false,/*genestrand*/0,
							   want_lowest_coordinate_p)) > 0) {
	      ninserts = 0;
	      nindels = adj;
	      total_nmatches = querylength - best_nmismatches_i - best_nmismatches_j;
	    } else if (adj < 0 &&
		       (indel_pos = Indel_resolve_middle_insertion(&best_nmismatches_i,&best_nmismatches_j,
								   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
								   /*left*/tr_left0,/*indels*/-adj,
								   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
								   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
								   /*ome*/transcriptomebits,/*ome_alt*/NULL,query_compress_rev,
								   /*querystart*/0,/*queryend*/querylength,querylength,
								   nmismatches_allowed,/*plusp*/false,/*genestrand*/0,
								   want_lowest_coordinate_p)) > 0) {
	      ninserts = nindels = -adj;
	      total_nmatches = querylength - best_nmismatches_i - best_nmismatches_j - ninserts;
	    } else {
	      adj = 0;
	    }
	  }
	}
	  
	if (adj == 0) {
	  /* Skip finding an indel */

	} else if (transcript_genestrand > 0) {
	  /* Case 3 ultrafast gap: tminus, gplus.  transcript_plusp == false && transcript_genestrand > 0 */
	  debug2(printf("Case 3 ultrafast gap: tminus, gplus\n"));
	  /* sensedir = SENSE_ANTI; */
	  /* plusp = false; */

	  nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	  Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);

	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;
	  
	  transcript_diag0 = tr_left0 - troffset;
	  transcript_diag1 = tr_left1 - troffset;
	  
	  /* first part */
	  tr_left = tr_left0;
	  queryend = querylength; /* tminus: compute from queryend to querystart */
	  querystart = querylength - indel_pos;
	  debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));
	  
	  overall_trstart = trstart = transcript_diag0 + (querylength - queryend); /* tminus */
#ifdef DEBUG2
	  trend = transcript_diag0 + (querylength - querystart); /* tminus */
	  printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, adj %d\n",
		 transcript_diag0,trstart,trend,tr_left,adj);
#endif
	  
	  exoni = 0;
#if 0 && defined(HAVE_AVX2)
	  _trstart = _mm256_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	  _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 8;
	    ptr += 8;
	    _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	    _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	  _trstart = _mm_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	  _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 4;
	    ptr += 4;
	    _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	    _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#else
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
#endif
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(13) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
	    debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d,align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      querystart = queryend - len; /* tminus */
	      
	      alignend = chroffset + genomicpos;
	      alignstart = alignend + len; /* gplus */
	      debug2(left = alignend - (querylength - queryend)); /* tminus != gplus */
	      debug2(printf("tr ultrafast gap case 3 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignend - chroffset,querylength,queryend));
	      
	      /* tminus: plus alignend first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("6 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		queryend -= len; /* tminus */
		genomicpos = exonstarts[exoni] - 1; /* gplus */
		debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  /* gap */
	  debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
	  debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
	  if (abortp == true) {
	    /* Skip */
	  } else if (/*len <= nindels ||*/exon_residual <= nindels) {
	    debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (7)\n"));
	    abortp = true;
	  } else {
	    gaps = Intlist_push(gaps,(int) adj);
	  }
	  
	  /* second part */
	  tr_left = tr_left1;
	  queryend = querylength - indel_pos - ninserts; /* tminus: compute from queryend to querystart */
	  querystart = 0;
	  
	  trstart = transcript_diag1 + (querylength - queryend); /* tminus */
	  overall_trend = transcript_diag1 + (querylength - querystart); /* tminus */
	  debug2(printf("transcript_diag %d, trstart %d, trend %d, tr_left %u\n",
			transcript_diag1,trstart,overall_trend,tr_left));
	  
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(14) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] + exonpos - 1; /* gplus */
	    debug2(printf("genomicpos %u = %u + %u - 1\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d,align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      querystart = queryend - len; /* tminus */
	      
	      alignend = chroffset + genomicpos;
	      alignstart = alignend + len; /* gplus */
	      debug2(left = alignend - (querylength - queryend)); /* tminus != gplus */
	      debug2(printf("tr ultrafast gap case 3 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignend - chroffset,querylength,queryend));
	      
	      /* tminus: plus alignend first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("6 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		queryend -= len; /* tminus */
		genomicpos = exonstarts[exoni] - 1; /* gplus */
		debug2(printf("genomicpos %u = %u - 1\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    endpoints = Uintlist_reverse(endpoints); /* gplus */
	    gaps = Intlist_reverse(gaps);	     /* gplus */
	    debug2(printf("MINUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngminus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    gminus_paths[ngminus++] = Path_new(total_nmatches,trnum,overall_trend,overall_trstart,
					       /*trim_low*/trim3,/*trim_high*/trim5,chrnum,chroffset,chrhigh,chrlength,
					       endpoints,gaps,concordantp);
	  }

	} else {
	  /* Case 4 ultrafast gap: tminus, gminus.  transcript_plusp == false && transcript_genestrand < 0 */
	  debug2(printf("Case 4 ultrafast gap: tminus, gminus\n"));
	  /* sensedir = SENSE_ANTI; */
	  /* plusp = true; */

	  nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
	  Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,chrnum,circular_typeint);

	  endpoints = (Uintlist_T) NULL;
	  gaps = (Intlist_T) NULL;
	  abortp = false;

	  transcript_diag0 = tr_left0 - troffset;
	  transcript_diag1 = tr_left1 - troffset;
	  
	  /* first part */
	  tr_left = tr_left0;
	  queryend = querylength; /* tminus: compute from queryend to querystart */
	  querystart = querylength - indel_pos;
	  debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));
	  
	  overall_trstart = trstart = transcript_diag0 + (querylength - queryend); /* tminus */
#ifdef DEBUG2
	  trend = transcript_diag0 + (querylength - querystart); /* tminus */
	  printf("transcript_diag %d, trstart %d, trend %d, tr_left %u, adj %d\n",
		 transcript_diag0,trstart,trend,tr_left,adj);
#endif
	  
	  exoni = 0;
#if 0 && defined(HAVE_AVX2)
	  _trstart = _mm256_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	  _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 8;
	    ptr += 8;
	    _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	    _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	  _trstart = _mm_set1_epi32(trstart);
	  ptr = exonbounds;
	  _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	  _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  while (all_zero_p(_diff)) {
	    exoni += 4;
	    ptr += 4;
	    _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	    _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	  }
	  matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	  exoni += count_trailing_zeroes_32(matchbits);
#else
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
#endif
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(15) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] - exonpos; /* gminus */
	    debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      querystart = queryend - len; /* tminus */
	      
	      alignend = chroffset + genomicpos;
	      alignstart = alignend - len; /* gminus */
	      debug2(left = alignstart - querystart); /* tminus == gminus */
	      debug2(printf("tr ultrafast gap case 4 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignstart - chroffset,querystart));
	      
	      /* tminus: push alignend first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("8 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		queryend -= len; /* tminus */
		genomicpos = exonstarts[exoni]; /* gminus */
		debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  /* gap */
	  debug2(printf("INDEL: last piece %d | nindels %d | next piece %d\n",len,nindels,exon_residual));
	  debug2(printf("exon residual %d vs nindels %d\n",exon_residual,nindels));
	  if (abortp == true) {
	    /* Skip */
	  } else if (/*len <= nindels ||*/exon_residual <= nindels) {
	    debug2(printf("ABORTING BECAUSE INDEL OVERLAPS SPLICE SITE (8)\n"));
	    abortp = true;
	  } else {
	    gaps = Intlist_push(gaps,(int) adj);
	  }
	  
	  /* second part */
	  tr_left = tr_left1;
	  queryend = querylength - indel_pos - ninserts; /* tminus: compute from queryend to querystart */
	  querystart = 0;
	  debug2(printf("\nAnalyzing query %d..%d\n",querystart,queryend));
	  
	  trstart = transcript_diag1 + (querylength - queryend); /* tminus */
	  overall_trend = transcript_diag1 + (querylength - querystart); /* tminus */
	  debug2(printf("transcript_diag %d, trstart %d, trend %d, tr_left %u\n",
			transcript_diag1,trstart,overall_trend,tr_left));
	  
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }
	  
	  if (exoni >= nexons) {
	    debug2(printf("ABORTING BECAUSE OF EXONI %d >= NEXONS %d\n",exoni,nexons));
	    abortp = true;
	  } else {
	    debug2(printf("(16) exoni %d out of nexons %d\n",exoni,nexons));
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    exon_residual = exonbounds[exoni] - trstart;
	    debug2(printf("exoni %d => exon_residual %d, exonpos %d\n",exoni,exon_residual,exonpos));
	    
	    /* Generate one substring per exon */
	    genomicpos = exonstarts[exoni] - exonpos; /* gminus */
	    debug2(printf("genomicpos %u = %u - %u\n",genomicpos,exonstarts[exoni],exonpos));
	    align_residual = queryend - querystart;
	    while (align_residual > 0) {
	      debug2(printf("abortp %d, align_residual %d, exon_residual %d\n",abortp,align_residual,exon_residual));
	      len = (align_residual <= exon_residual) ? align_residual : exon_residual;
	      querystart = queryend - len; /* tminus */
	      
	      alignend = chroffset + genomicpos;
	      alignstart = alignend - len; /* gminus */
	      debug2(left = alignstart - querystart); /* tminus == gminus */
	      debug2(printf("tr ultrafast gap case 4 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - %d\n",
			    querystart,queryend,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
			    left - chroffset,alignstart - chroffset,querystart));
	      
	      /* tminus: push alignend first */
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignend - chroffset));
	      endpoints = Uintlist_push(endpoints,(Chrpos_T) (alignstart - chroffset));
	      
	      if (align_residual <= exon_residual) {
		exon_residual -= align_residual; /* Need to know this if we have a following deletion */
		align_residual = 0;
	      } else if (++exoni >= nexons) {
		/* Aligning past the end of the transcript */
		debug2(printf("8 ABORTING BECAUSE ALIGNING PAST END OF TRANSCRIPT\n"));
		align_residual = 0;
		abortp = true;
	      } else {
		/* nintrons += 1; */
		align_residual -= len;
		queryend -= len; /* tminus */
		genomicpos = exonstarts[exoni]; /* gminus */
		debug2(printf("genomicpos %u = %u\n",genomicpos,exonstarts[exoni]));
		exon_residual = exonbounds[exoni] - exonbounds[exoni - 1];
		debug2(printf("INTRON\n"));
		gaps = Intlist_push(gaps,0);
	      }
	    }
	  }
	  
	  if (abortp == true) {
	    debug2(printf("ABORTING PATH\n"));
	    Uintlist_free(&endpoints);
	    Intlist_free(&gaps);
	  } else {
	    /* gminus: Do not reverse endpoints or gaps */
	    debug2(printf("PLUS PATH %d.  ENDPOINTS: %s.  GAPS: %s\n\n",
			  ngplus,Uintlist_to_string(endpoints),Intlist_to_string(gaps)));
	    gplus_paths[ngplus++] = Path_new(total_nmatches,trnum,overall_trend,overall_trstart,
					     /*trim_low*/trim3,/*trim_high*/trim5,chrnum,chroffset,chrhigh,chrlength,
					     endpoints,gaps,concordantp);
	  }
	}
      }
      
      FREE(tminus_diagpairs);
      *tminus_gplus_paths = gplus_paths;
      *n_tminus_gplus = ngplus;
      *tminus_gminus_paths = gminus_paths;
      *n_tminus_gminus = ngminus;
    }
  }

  debug2(printf("Counts: %d %d %d %d\n",
		*n_tplus_gplus,*n_tplus_gminus,*n_tminus_gplus,*n_tminus_gminus));

  return (*n_tplus_gplus) + (*n_tplus_gminus) + (*n_tminus_gplus) + (*n_tminus_gminus);
}


void
Kmer_search_transcriptome_single (int *found_score_overall, int *found_score_within_trims,
				  List_T *sense_hits_gplus, List_T *sense_hits_gminus,
				  List_T *antisense_hits_gplus, List_T *antisense_hits_gminus,

				  char *queryuc_ptr, int querylength,
				  Trcoord_T **tplus_stream_array, int *tplus_streamsize_array, int *tplus_diagterm_array,
				  Trcoord_T **tminus_stream_array, int *tminus_streamsize_array, int *tminus_diagterm_array,

				  Compress_T query_compress_fwd, Compress_T query_compress_rev,
				  Univ_IIT_T transcript_iit, Transcriptome_T transcriptome, Genome_T transcriptomebits, 
				  int nmismatches_allowed,
				  Listpool_T listpool, Hitlistpool_T hitlistpool, int level) {
  
  Path_T *tplus_gplus_paths, *tplus_gminus_paths, *tminus_gminus_paths, *tminus_gplus_paths;
  int n_tplus_gplus, n_tplus_gminus, n_tminus_gminus, n_tminus_gplus;

  Trcoord_T *tplus_positions_5, *tminus_positions_5, *tplus_positions_3, *tminus_positions_3;
  int n_tplus_positions_5, n_tminus_positions_5, n_tplus_positions_3, n_tminus_positions_3;
  int tplus_diagterm_5, tminus_diagterm_5, tplus_diagterm_3, tminus_diagterm_3;

  /* Set concordantp to be true, meaning that no further concordance is necessary */
  if (search_transcriptome_ends(&tplus_gplus_paths,&n_tplus_gplus,
				&tplus_gminus_paths,&n_tplus_gminus,
				&tminus_gminus_paths,&n_tminus_gminus,
				&tminus_gplus_paths,&n_tminus_gplus,
				
				&tplus_positions_5,&n_tplus_positions_5,&tplus_diagterm_5,
				&tminus_positions_5,&n_tminus_positions_5,&tminus_diagterm_5,
				&tplus_positions_3,&n_tplus_positions_3,&tplus_diagterm_3,
				&tminus_positions_3,&n_tminus_positions_3,&tminus_diagterm_3,

				queryuc_ptr,querylength,
				query_compress_fwd,query_compress_rev,
				transcript_iit,transcriptome,transcriptomebits, 
				nmismatches_allowed,/*concordantp*/true) > 0) {
    /* Found hits */
  } else {
    /* Will reassign paths from search_transcriptome_complete */
    FREE(tplus_gplus_paths);
    FREE(tplus_gminus_paths);
    FREE(tminus_gplus_paths);
    FREE(tminus_gminus_paths);

    search_transcriptome_complete(&tplus_gplus_paths,&n_tplus_gplus,
				  &tplus_gminus_paths,&n_tplus_gminus,
				  &tminus_gminus_paths,&n_tminus_gminus,
				  &tminus_gplus_paths,&n_tminus_gplus,
				
				  tplus_positions_5,n_tplus_positions_5,tplus_diagterm_5,
				  tminus_positions_5,n_tminus_positions_5,tminus_diagterm_5,
				  tplus_positions_3,n_tplus_positions_3,tplus_diagterm_3,
				  tminus_positions_3,n_tminus_positions_3,tminus_diagterm_3,
				 
				  queryuc_ptr,querylength,
				  tplus_stream_array,tplus_streamsize_array,tplus_diagterm_array,
				  tminus_stream_array,tminus_streamsize_array,tminus_diagterm_array,

				  query_compress_fwd,query_compress_rev,
				  transcript_iit,transcriptome,transcriptomebits, 
				  nmismatches_allowed,/*concordantp*/true);
  }


  debug(printf("Converting tplus paths to hits\n"));
  *sense_hits_gplus = single_hits_gplus(&(*found_score_overall),&(*found_score_within_trims),
					*sense_hits_gplus,tplus_gplus_paths,n_tplus_gplus,
					querylength,query_compress_fwd,nmismatches_allowed,
					/*sensedir*/SENSE_FORWARD,listpool,hitlistpool,level);
  *sense_hits_gminus = single_hits_gminus(&(*found_score_overall),&(*found_score_within_trims),
					  *sense_hits_gminus,tplus_gminus_paths,n_tplus_gminus,
					  querylength,query_compress_rev,nmismatches_allowed,
					  /*sensedir*/SENSE_FORWARD,listpool,hitlistpool,level);

  debug(printf("Converting tminus paths to hits\n"));
  *antisense_hits_gplus = single_hits_gplus(&(*found_score_overall),&(*found_score_within_trims),
					    *antisense_hits_gplus,tminus_gplus_paths,n_tminus_gplus,
					    querylength,query_compress_fwd,nmismatches_allowed,
					    /*sensedir*/SENSE_ANTI,listpool,hitlistpool,level);
  *antisense_hits_gminus = single_hits_gminus(&(*found_score_overall),&(*found_score_within_trims),
					      *antisense_hits_gminus,tminus_gminus_paths,n_tminus_gminus,
					      querylength,query_compress_rev,nmismatches_allowed,
					      /*sensedir*/SENSE_ANTI,listpool,hitlistpool,level);
  FREE(tplus_gplus_paths);
  FREE(tplus_gminus_paths);
  FREE(tminus_gminus_paths);
  FREE(tminus_gplus_paths);
  
  return;
}


#if 0
void
Kmer_search_transcriptome_paired (int *found_score_5, int *found_score_3,
				  List_T *hits5_gplus, List_T *hits5_gminus,
				  List_T *hits3_gplus, List_T *hits3_gminus,
				  
				  char *queryuc_ptr_5, int querylength5,
				  char *queryuc_ptr_3, int querylength3,
				  int *mismatch_positions_alloc_5, int *mismatch_positions_alloc_3,

				  Trcoord_T **tplus_stream_array_5, int *tplus_streamsize_array_5, int *tplus_diagterm_array_5,
				  Trcoord_T **tminus_stream_array_5, int *tminus_streamsize_array_5, int *tminus_diagterm_array_5,
				  Trcoord_T **tplus_stream_array_3, int *tplus_streamsize_array_3, int *tplus_diagterm_array_3,
				  Trcoord_T **tminus_stream_array_3, int *tminus_streamsize_array_3, int *tminus_diagterm_array_3,

				  Compress_T query5_compress_fwd, Compress_T query5_compress_rev,
				  Compress_T query3_compress_fwd, Compress_T query3_compress_rev,

				  Univ_IIT_T transcript_iit, Transcriptome_T transcriptome, Genome_T transcriptomebits, 
				  int nmismatches_allowed_5, int nmismatches_allowed_3, 
				  int kmer_search_sizelimit, int level) {
  
  Path_T *tplus_gplus_paths_5, *tplus_gminus_paths_5, *tminus_gminus_paths_5, *tminus_gplus_paths_5,
    *tplus_gplus_paths_3, *tplus_gminus_paths_3, *tminus_gminus_paths_3, *tminus_gplus_paths_3;
  int n_tplus_gplus_5, n_tplus_gminus_5, n_tminus_gminus_5, n_tminus_gplus_5,
    n_tplus_gplus_3, n_tplus_gminus_3, n_tminus_gminus_3, n_tminus_gplus_3;

  /* 5 and 3 here mean the 5' and 3' ends of each read, not the 5' and 3' reads */
  Trcoord_T *tplus_positions_5, *tminus_positions_5, *tplus_positions_3, *tminus_positions_3;
  int n_tplus_positions_5, n_tminus_positions_5, n_tplus_positions_3, n_tminus_positions_3;
  int tplus_diagterm_5, tminus_diagterm_5, tplus_diagterm_3, tminus_diagterm_3;


  /* Solve for 5' read */
  /* Set concordantp to be false, meaning that we want to find concordance later */
  if (search_transcriptome_ends(&tplus_gplus_paths_5,&n_tplus_gplus_5,
				&tplus_gminus_paths_5,&n_tplus_gminus_5,
				&tminus_gminus_paths_5,&n_tminus_gminus_5,
				&tminus_gplus_paths_5,&n_tminus_gplus_5,
				
				&tplus_positions_5,&n_tplus_positions_5,&tplus_diagterm_5,
				&tminus_positions_5,&n_tminus_positions_5,&tminus_diagterm_5,
				&tplus_positions_3,&n_tplus_positions_3,&tplus_diagterm_3,
				&tminus_positions_3,&n_tminus_positions_3,&tminus_diagterm_3,
				
				queryuc_ptr_5,querylength5,
				query5_compress_fwd,query5_compress_rev,
				transcript_iit,transcriptome,transcriptomebits, 
				nmismatches_allowed_5,/*concordantp*/false) > 0) {
    /* Found hits */
  } else {
    /* Will reassign paths from search_transcriptome_complete */
    FREE(tplus_gplus_paths_5);
    FREE(tplus_gminus_paths_5);
    FREE(tminus_gplus_paths_5);
    FREE(tminus_gminus_paths_5);

    search_transcriptome_complete(&tplus_gplus_paths_5,&n_tplus_gplus_5,
				  &tplus_gminus_paths_5,&n_tplus_gminus_5,
				  &tminus_gminus_paths_5,&n_tminus_gminus_5,
				  &tminus_gplus_paths_5,&n_tminus_gplus_5,
				
				  tplus_positions_5,n_tplus_positions_5,tplus_diagterm_5,
				  tminus_positions_5,n_tminus_positions_5,tminus_diagterm_5,
				  tplus_positions_3,n_tplus_positions_3,tplus_diagterm_3,
				  tminus_positions_3,n_tminus_positions_3,tminus_diagterm_3,
				 
				  queryuc_ptr_5,querylength5,
				  tplus_stream_array_5,tplus_streamsize_array_5,tplus_diagterm_array_5,
				  tminus_stream_array_5,tminus_streamsize_array_5,tminus_diagterm_array,
				  query5_compress_fwd,query5_compress_rev,
				  transcript_iit,transcriptome,transcriptomebits, 
				  nmismatches_allowed_5,/*concordantp*/false);
  }


  /* Solve for 3' read */
  /* Set concordantp to be false, meaning that we want to find concordance later */
  if (search_transcriptome_ends(&tplus_gplus_paths_3,&n_tplus_gplus_3,
				&tplus_gminus_paths_3,&n_tplus_gminus_3,
				&tminus_gminus_paths_3,&n_tminus_gminus_3,
				&tminus_gplus_paths_3,&n_tminus_gplus_3,
				
				&tplus_positions_5,&n_tplus_positions_5,&tplus_diagterm_5,
				&tminus_positions_5,&n_tminus_positions_5,&tminus_diagterm_5,
				&tplus_positions_3,&n_tplus_positions_3,&tplus_diagterm_3,
				&tminus_positions_3,&n_tminus_positions_3,&tminus_diagterm_3,
				
				queryuc_ptr_3,querylength3,
				query3_compress_fwd,query3_compress_rev,
				transcript_iit,transcriptome,transcriptomebits, 
				nmismatches_allowed_3,/*concordantp*/false) > 0) {
    /* Found hits */
  } else {
    /* Will reassign paths from search_transcriptome_complete */
    FREE(tplus_gplus_paths_3);
    FREE(tplus_gminus_paths_3);
    FREE(tminus_gplus_paths_3);
    FREE(tminus_gminus_paths_3);

    search_transcriptome_complete(&tplus_gplus_paths_3,&n_tplus_gplus_3,
				  &tplus_gminus_paths_3,&n_tplus_gminus_3,
				  &tminus_gminus_paths_3,&n_tminus_gminus_3,
				  &tminus_gplus_paths_3,&n_tminus_gplus_3,
				
				  tplus_positions_5,n_tplus_positions_5,tplus_diagterm_5,
				  tminus_positions_5,n_tminus_positions_5,tminus_diagterm_5,
				  tplus_positions_3,n_tplus_positions_3,tplus_diagterm_3,
				  tminus_positions_3,n_tminus_positions_3,tminus_diagterm_3,
				 
				  queryuc_ptr_3,querylength3,
				  tplus_stream_array_3,tplus_streamsize_array_3,tplus_diagterm_array_3,
				  tminus_stream_array_3,tminus_streamsize_array_3,tminus_diagterm_array_3,
				  query3_compress_fwd,query3_compress_rev,
				  transcript_iit,transcriptome,transcriptomebits, 
				  nmismatches_allowed_3,/*concordantp*/false);
  }


  debug(printf("Filtering tplus and tminus transcriptome hits for concordant pairs\n"));
  filter_concordant_paths(tplus_gplus_paths_5,n_tplus_gplus_5,
			  tplus_gplus_paths_3,n_tplus_gplus_3);
  filter_concordant_paths(tplus_gminus_paths_5,n_tplus_gminus_5,
			  tplus_gminus_paths_3,n_tplus_gminus_3);
  filter_concordant_paths(tminus_gminus_paths_5,n_tminus_gminus_5,
			  tminus_gminus_paths_3,n_tminus_gminus_3);
  filter_concordant_paths(tminus_gplus_paths_5,n_tminus_gplus_5,
			  tminus_gplus_paths_3,n_tminus_gplus_3);


  debug(printf("Converting tplus paths for queryseq5 to hits\n"));
  *hits5_gplus = single_hits_gplus(&(*found_score_5),*hits5_gplus,tplus_gplus_paths_5,n_tplus_gplus_5,
				   querylength5,query5_compress_fwd,nmismatches_allowed_5,
				   /*sensedir*/SENSE_FORWARD,listpool,hitlistpool,level);
  *hits5_gminus = single_hits_gminus(&(*found_score_5),*hits5_gminus,tplus_gminus_paths_5,n_tplus_gminus_5,
				     querylength5,query5_compress_rev,nmismatches_allowed_5,
				     /*sensedir*/SENSE_FORWARD,listpool,hitlistpool,level);
  debug(printf("Converting tplus paths for queryseq3 to hits\n"));
  *hits3_gplus = single_hits_gplus(&(*found_score_3),*hits3_gplus,tplus_gplus_paths_3,n_tplus_gplus_3,
				   querylength3,query3_compress_fwd,nmismatches_allowed_3,
				   /*sensedir*/SENSE_FORWARD,listpool,hitlistpool,level);
  *hits3_gminus = single_hits_gminus(&(*found_score_3),*hits3_gminus,tplus_gminus_paths_3,n_tplus_gminus_3,
				     querylength3,query3_compress_rev,nmismatches_allowed_3,
				     /*sensedir*/SENSE_FORWARD,listpool,hitlistpool,level);
  FREE(tplus_gplus_paths_5);
  FREE(tplus_gminus_paths_5);
  FREE(tplus_gplus_paths_3);
  FREE(tplus_gminus_paths_3);
  
  debug(printf("Converting tminus paths for queryseq5 to hits\n"));
  *hits5_gplus = single_hits_gplus(&(*found_score_5),*hits5_gplus,tminus_gplus_paths_5,n_tminus_gplus_5,
				   querylength5,query5_compress_fwd,nmismatches_allowed_5,
				   /*sensedir*/SENSE_ANTI,listpool,hitlistpool,level);
  *hits5_gminus = single_hits_gminus(&(*found_score_5),*hits5_gminus,tminus_gminus_paths_5,n_tminus_gminus_5,
				     querylength5,query5_compress_rev,nmismatches_allowed_5,
				     /*sensedir*/SENSE_ANTI,listpool,hitlistpool,level);
  debug(printf("Converting tminus paths for queryseq3 to hits\n"));
  *hits3_gplus = single_hits_gplus(&(*found_score_3),*hits3_gplus,tminus_gplus_paths_3,n_tminus_gplus_3,
				   querylength3,query3_compress_fwd,nmismatches_allowed_3,
				   /*sensedir*/SENSE_ANTI,listpool,hitlistpool,level);
  *hits3_gminus = single_hits_gminus(&(*found_score_3),*hits3_gminus,tminus_gminus_paths_3,n_tminus_gminus_3,
				     querylength3,query3_compress_rev,nmismatches_allowed_3,
				     /*sensedir*/SENSE_ANTI,listpool,hitlistpool,level);
  
  FREE(tminus_gplus_paths_5);
  FREE(tminus_gminus_paths_5);
  FREE(tminus_gplus_paths_3);
  FREE(tminus_gminus_paths_3);

  return;
}
#endif



/* Simplified from search_transcriptome_ends */
/* Appears to be unused currently */
List_T
Kmer_remap_transcriptome (char *remap_sequence, int remap_seqlength, 
			  Chrnum_T chrnum, Chrpos_T lowbound, Chrpos_T highbound,
			  Univ_IIT_T transcript_iit, 
			  Genome_T transcriptomebits, Transcriptome_T transcriptome) {
  List_T transcripts = NULL;
  Compress_T remap_compress_fwd, remap_compress_rev;

  UINT4 *tplus_positions_5 = NULL, *tminus_positions_5 = NULL,
    *tplus_positions_3 = NULL, *tminus_positions_3 = NULL;
  int n_tplus_positions_5 = 0, n_tminus_positions_5 = 0, n_tplus_positions_3 = 0, n_tminus_positions_3 = 0;
  int tplus_diagterm_5 = 0, tminus_diagterm_5 = 0, tplus_diagterm_3 = 0, tminus_diagterm_3 = 0;
				
  Trcoord_T tr_univdiagonal, tr_left;
  UINT4 *tplus_diagonals, *tminus_diagonals;
  int n_tplus_diagonals, n_tminus_diagonals, i;

  Reader_T reader;
  int querypos, query_lastpos, pos5;
  Oligospace_T forward, revcomp, forward_oligo, revcomp_oligo;

#if 0 && defined(HAVE_AVX2)
  __m256i _diff, _exonbounds, _trstart;
  int matchbits;
  int *ptr;
#elif 0 && defined(HAVE_SSE2)
  __m128i _diff, _exonbounds, _trstart;
  int matchbits;
  int *ptr;
#endif

#ifdef DEBUG2
  bool allocp;
#endif

  Chrnum_T trnum;
  int trstart, trend;
  int transcript_genestrand;
  Trcoord_T troffset, trhigh, trlength;
  Chrpos_T chrstart, chrend;
  int nmismatches, ref_nmismatches;
  int *exonbounds;
  Chrpos_T *exonstarts;
  int nexons, exoni;
  int exonpos;


  debug9(printf("%s\n",remap_sequence));
  remap_compress_fwd = Compress_new_fwd(remap_sequence,remap_seqlength);
  remap_compress_rev = Compress_new_rev(remap_sequence,remap_seqlength);

  reader = Reader_new(remap_sequence,/*querystart*/0,/*queryend*/remap_seqlength);
  forward = revcomp = 0;
  if (Oligo_next_5(/*last_state*/INIT,&querypos,&forward,&revcomp,reader,/*genestrand*/0) != DONE) {
    forward_oligo = forward & oligobase_mask;
    debug9(printf("%s\n",Oligo_one_nt(forward_oligo,index1part_tr)));

    tplus_diagterm_5 = remap_seqlength - querypos;
    n_tplus_positions_5 = Indexdb_ptr(&tplus_positions_5,/*plus_indexdb*/indexdb_tr,forward_oligo);

    revcomp_oligo = (revcomp >> leftreadshift) & oligobase_mask;
    tminus_diagterm_5 = querypos + index1part_tr;
    n_tminus_positions_5 = Indexdb_ptr(&tminus_positions_5,/*minus_indexdb*/indexdb_tr,revcomp_oligo);
  }
  Reader_free(&reader);

  query_lastpos = remap_seqlength - index1part_tr;
  reader = Reader_new(remap_sequence,/*querystart*/query_lastpos - /*index1interval*/1 + 1,/*queryend*/remap_seqlength);
  forward = revcomp = 0;
  if (Oligo_next_5(/*last_state*/INIT,&querypos,&forward,&revcomp,reader,/*genestrand*/0) != DONE) {
    forward_oligo = forward & oligobase_mask;
    debug9(printf("%s\n",Oligo_one_nt(forward_oligo,index1part_tr)));
    tplus_diagterm_3 = remap_seqlength - querypos;
    n_tplus_positions_3 = Indexdb_ptr(&tplus_positions_3,/*plus_indexdb*/indexdb_tr,forward_oligo);

    revcomp_oligo = (revcomp >> leftreadshift) & oligobase_mask;
    tminus_diagterm_3 = querypos + index1part_tr;
    n_tminus_positions_3 = Indexdb_ptr(&tminus_positions_3,/*minus_indexdb*/indexdb_tr,revcomp_oligo);
  }
  Reader_free(&reader);
	 
  tplus_diagonals = Intersect_exact(&n_tplus_diagonals,
				    tplus_positions_5,n_tplus_positions_5,tplus_diagterm_5,
				    tplus_positions_3,n_tplus_positions_3,tplus_diagterm_3);
  tminus_diagonals = Intersect_exact(&n_tminus_diagonals,
				     tminus_positions_5,n_tminus_positions_5,tminus_diagterm_5,
				     tminus_positions_3,n_tminus_positions_3,tminus_diagterm_3);

  debug9(printf("***Remap transcriptome: %d plus and %d minus diagonals\n",n_tplus_diagonals,n_tminus_diagonals));

  for (i = 0; i < n_tplus_diagonals; i++) {
    tr_univdiagonal = tplus_diagonals[i];
    tr_left = tr_univdiagonal - (Trcoord_T) remap_seqlength; /* NEW FORMULA for querystart of 0 */
    trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,/*low*/tr_left+pos5,
			       /*high*/tr_left+remap_seqlength);

    /* TRIM QUERY AT TRANSCRIPT BOUNDS */
    pos5 = (tr_univdiagonal >= troffset + (Trcoord_T) remap_seqlength) ? 0 : (int) (troffset - tr_left);

    debug9(printf("REMAP PLUS DIAGONAL %u\n",tr_univdiagonal));
    nmismatches = Genome_count_mismatches_substring(&ref_nmismatches,transcriptomebits,NULL,remap_compress_fwd,tr_left,
						    pos5,/*pos3*/remap_seqlength,/*plusp*/true,/*genestrand*/0);
    if (nmismatches > 0) {
      debug9(printf("nmismatches %d\n",nmismatches));
    } else {
      debug2(printf("(7) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));
#ifdef DEBUG9
      printf("chrnum = %d\n",Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum));
      printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand);
#endif
      if (Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum) == chrnum) {
	nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);
      
	trstart = tr_left - troffset;
	trend = trstart + remap_seqlength;
	debug9(printf("trstart %d, trend %d\n",trstart,trend));

	exoni = 0;
#if 0 && defined(HAVE_AVX2)
	_trstart = _mm256_set1_epi32(trstart);
	ptr = exonbounds;
	_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	while (all_zero_p(_diff)) {
	  exoni += 8;
	  ptr += 8;
	  _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	  _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	}
	matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	_trstart = _mm_set1_epi32(trstart);
	ptr = exonbounds;
	_exonbounds = _mm_loadu_si128((__m128i *) ptr);
	_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	while (all_zero_p(_diff)) {
	  exoni += 4;
	  ptr += 4;
	  _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	  _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	}
	matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	exoni += count_trailing_zeroes_32(matchbits);
#else
	while (exoni < nexons && exonbounds[exoni] <= trstart) {
	  exoni++;
	}
#endif

	if (exoni < nexons) {
	  if (exoni == 0) {
	    exonpos = trstart;
	  } else {
	    exonpos = trstart - exonbounds[exoni - 1];
	  }
	  if (transcript_genestrand > 0) {
	    chrstart = exonstarts[exoni] + exonpos - 1; /* gplus */
	  } else {
	    chrstart = exonstarts[exoni] - exonpos; /* gminus */
	  }
	    
	  while (exoni < nexons && exonbounds[exoni] <= trend) {
	    exoni++;
	  }

	  if (exoni < nexons) {
	    if (exoni == 0) {
	      exonpos = trend;
	    } else {
	      exonpos = trend - exonbounds[exoni - 1];
	    }
	    if (transcript_genestrand > 0) {
	      chrend = exonstarts[exoni] + exonpos - 1; /* gplus */
	      assert(chrstart < chrend);
	      debug9(printf("Checking if interval %u..%u overlaps with desired %u..%u",
			    chrstart,chrend,lowbound,highbound));
	      if (chrstart <= highbound && chrend >= lowbound) {
		debug9(printf(" => yes"));
		transcripts = List_push(transcripts,(void *) Transcript_new(trnum,trstart,trend));
	      }
	      debug9(printf("\n"));

	    } else {
	      chrend = exonstarts[exoni] - exonpos; /* gminus */
	      assert(chrend < chrstart);
	      debug9(printf("Checking if interval %u..%u overlaps with desired %u..%u",
			    chrend,chrstart,lowbound,highbound));
	      if (chrend <= highbound && chrstart >= lowbound) {
		debug9(printf(" => yes"));
		transcripts = List_push(transcripts,(void *) Transcript_new(trnum,trstart,trend));
	      }
	      debug9(printf("\n"));
	    }
	  }
	}
      }
    }
  }
  FREE(tplus_diagonals);	/* Occupies memory even if n_tminus_diagonals == 0 */

  for (i = 0; i < n_tminus_diagonals; i++) {
    tr_univdiagonal = tminus_diagonals[i];
    tr_left = tr_univdiagonal - (Trcoord_T) remap_seqlength; /* NEW FORMULA for querystart of 0 */
    trnum = Univ_IIT_get_trnum(&troffset,&trhigh,&trlength,transcript_iit,/*low*/tr_left+pos5,
			       /*high*/tr_left+remap_seqlength);
    debug2(printf("(8) trnum is %d: %s\n",trnum,Univ_IIT_label(transcript_iit,trnum,&allocp)));

    /* TRIM QUERY AT TRANSCRIPT BOUNDS */
    pos5 = (tr_univdiagonal >= troffset + (Trcoord_T) remap_seqlength) ? 0 : (int) (troffset - tr_left);

    debug9(printf("REMAP MINUS DIAGONAL %u\n",tr_univdiagonal));
    nmismatches = Genome_count_mismatches_substring(&ref_nmismatches,transcriptomebits,NULL,remap_compress_rev,tr_left,
						    pos5,/*pos3*/remap_seqlength,/*plusp*/false,/*genestrand*/0);
    if (nmismatches > 0) {
      debug9(printf("nmismatches %d\n",nmismatches));
    } else {
#ifdef DEBUG9
      printf("chrnum = %d\n",Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum));
      printf("trnum %d, transcript_genestrand %d\n",trnum,transcript_genestrand);
#endif
      if (Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum) == chrnum) {
	nexons = Transcriptome_exons(&exonbounds,&exonstarts,transcriptome,trnum);

	trend = tr_left - troffset;
	trstart = trend + remap_seqlength;
	debug9(printf("trstart %d, trend %d\n",trstart,trend));

	exoni = 0;
#if 0 && defined(HAVE_AVX2)
	_trstart = _mm256_set1_epi32(trend);
	ptr = exonbounds;
	_exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	_diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	while (all_zero_p(_diff)) {
	  exoni += 8;
	  ptr += 8;
	  _exonbounds = _mm256_loadu_si256((__m256i *) ptr);
	  _diff = _mm256_cmpgt_epi32(_exonbounds,_trstart);
	}
	matchbits = _mm256_movemask_ps(_mm256_castsi256_ps(_diff));
	exoni += count_trailing_zeroes_32(matchbits);
#elif 0 && defined(HAVE_SSE2)
	_trstart = _mm_set1_epi32(trend);
	ptr = exonbounds;
	_exonbounds = _mm_loadu_si128((__m128i *) ptr);
	_diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	while (all_zero_p(_diff)) {
	  exoni += 4;
	  ptr += 4;
	  _exonbounds = _mm_loadu_si128((__m128i *) ptr);
	  _diff = _mm_cmpgt_epi32(_exonbounds,_trstart);
	}
	matchbits = _mm_movemask_ps(_mm_castsi128_ps(_diff));
	exoni += count_trailing_zeroes_32(matchbits);
#else
	while (exoni < nexons && exonbounds[exoni] <= trend) {
	  exoni++;
	}
#endif

	if (exoni < nexons) {
	  if (exoni == 0) {
	    exonpos = trend;
	  } else {
	    exonpos = trend - exonbounds[exoni - 1];
	  }
	  if (transcript_genestrand > 0) {
	    chrend = exonstarts[exoni] + exonpos - 1; /* gplus */
	  } else {
	    chrend = exonstarts[exoni] - exonpos; /* gminus */
	  }
	  while (exoni < nexons && exonbounds[exoni] <= trstart) {
	    exoni++;
	  }

	  if (exoni < nexons) {
	    if (exoni == 0) {
	      exonpos = trstart;
	    } else {
	      exonpos = trstart - exonbounds[exoni - 1];
	    }
	    if (transcript_genestrand > 0) {
	      chrstart = exonstarts[exoni] + exonpos - 1; /* gplus */
	      assert(chrend < chrstart);
	      debug9(printf("Checking if interval %u..%u overlaps with desired %u..%u",
			    chrend,chrstart,lowbound,highbound));
	      if (chrend <= highbound && chrstart >= lowbound) {
		debug9(printf(" => yes"));
		transcripts = List_push(transcripts,(void *) Transcript_new(trnum,trstart,trend));
	      }
	      debug9(printf("\n"));
	    } else {
	      chrstart = exonstarts[exoni] - exonpos; /* gminus */
	      assert(chrstart < chrend);
	      debug9(printf("Checking if interval %u..%u overlaps with desired %u..%u",
			    chrstart,chrend,lowbound,highbound));
	      if (chrstart <= highbound && chrend >= lowbound) {
		debug9(printf(" => yes"));
		transcripts = List_push(transcripts,(void *) Transcript_new(trnum,trstart,trend));
	      }
	      debug9(printf("\n"));
	    }
	  }
	}
      }
    }
  }
  FREE(tminus_diagonals);	/* Occupies memory even if n_tminus_diagonals == 0 */

  Compress_free(&remap_compress_rev);
  Compress_free(&remap_compress_fwd);

  return transcripts;
}



#if 0
/* Divides hits into transcriptome (if remapped) and genome (otherwise) */
void
Kmer_divide_remapped_hits (List_T *transcriptome_hits, List_T *genome_hits, List_T hits,
			   Genome_T genomecomp,
			   Univ_IIT_T transcript_iit, Genome_T transcriptomebits, 
			   Transcriptome_T transcriptome, int genestrand) {
  Stage3end_T hit;
  List_T transcripts, p;

  *transcriptome_hits = *genome_hits = (List_T) NULL;

  for (p = hits; p != NULL; p = List_next(p)) {
    hit = (Stage3end_T) List_head(p);
    if (Stage3end_transcripts(hit) != NULL) {
      *transcriptome_hits = Hitlist_push(*transcriptome_hits,hitlistpool,(void *) hit);

    } else if ((transcripts = Kmer_remap_transcriptome(hit,transcript_iit,transcriptomebits,
						       transcriptome,genomecomp)) != NULL) {
      Stage3end_set_transcripts(hit,transcripts);
      *transcriptome_hits = Hitlist_push(*transcriptome_hits,hitlistpool,(void *) hit);

    } else {
      *genome_hits = Hitlist_push(*genome_hits,hitlistpool,(void *) hit);
    }
  }
  Hitlist_free(&hits);

  return;
}
#endif


static int
binary_search_univcoord (int lowi, int highi, Univcoord_T *positions, Univcoord_T goal) {
  int middlei;

  debug10(printf("entered binary search with lowi=%d, highi=%d, goal=%u\n",lowi,highi,goal));

  while (lowi < highi) {
    middlei = lowi + ((highi - lowi) / 2);
    debug10(printf("  binary: %d:%u %d:%u %d:%u   vs. %u\n",
		   lowi,positions[lowi],middlei,positions[middlei],
		   highi-1,positions[highi-1],goal));
    if (goal < positions[middlei]) {
      highi = middlei;
    } else if (goal > positions[middlei]) {
      lowi = middlei + 1;
    } else {
      debug10(printf("binary search returns %d\n",middlei));
      return middlei;
    }
  }

  debug10(printf("binary search returns %d\n",highi));
  return highi;
}


/* Genomic procedure */
/* univdiagonal0/left0 is for the beginning of the query; univdiagonal1/left1 is for the end */
/* Indel_resolve_middle_insertion is expecting left for the beginning of the query, or left0 */
static void
combine_ends_plus (int *found_score_overall, int *found_score_within_trims,
		   int *n_sense_hits, List_T *sense_hits, int *n_antisense_hits, List_T *antisense_hits,
		   int querylength, Univcoord_T univdiagonal0, Univcoord_T univdiagonal1,
		   int pos5_0, int pos3_0, int pos5_1, int pos3_1,
		   Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh, Chrpos_T chrlength,
		   int *mismatch_positions_alloc, Spliceinfo_T spliceinfo, Compress_T query_compress_fwd,
		   int genestrand, int nmismatches_allowed, int max_deletionlen,
		   Listpool_T listpool, Hitlistpool_T hitlistpool, int level) {
  Stage3end_T hit;
  Univcoord_T left, prev_left, left0, univdiagonal, prev_univdiagonal;
  Substring_T substring1, substring2;
  Univcoord_T deletionpos;
  int splice_distance;
  int pos5, pos3;
  bool abortp;
#ifdef DEBUG2
  Univcoord_T alignstart, alignend;
#endif

  int nmismatches1, nmismatches2, ref_nmismatches1, ref_nmismatches2,
    nmismatches_bothdiff, nmismatches_refdiff;
  int introntype, sensedir;

  int supporti, supportj;
  int best_knowni_i, best_knowni_j, best_nmismatches_i, best_nmismatches_j, best_nmismatches_indel;
  int best_ref_nmismatches_i, best_ref_nmismatches_j, best_ref_nmismatches_indel;
  double best_prob_i, best_prob_j, donor_prob, acceptor_prob;
  int j;

  List_T sense_junctions = NULL, antisense_junctions = NULL, sense_substrings, antisense_substrings;
  int ninserts = 0;
  int sense_splice_pos = -1, antisense_splice_pos = -1;
  int adj, nindels, indel_pos;

  debug2(printf("Entered combine_ends_plus with univdiagonal0 %u and univdiagonal1 %u\n",
		univdiagonal0,univdiagonal1));

  *n_sense_hits = *n_antisense_hits = 0;

  left0 = univdiagonal0 - (Univcoord_T) querylength;
#if 0
  chrnum = Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,
			       /*low*/left0+pos5_0,/*high*/left0+pos3_0,circular_typeint);
#endif

  if ((adj = (int) (univdiagonal1 - univdiagonal0)) == 0) {
    if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					  /*univdiagonal*/univdiagonal0,pos5_0,pos3_0,querylength,
					  mismatch_positions_alloc,query_compress_fwd,
					  /*plusp*/true,genestrand,/*sensedir*/SENSE_FORWARD,
					  nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					  listpool,/*method*/KMER_APPROX,level)) != NULL) {
      *sense_hits = Hitlist_push(*sense_hits,hitlistpool,(void *) hit);
      *n_sense_hits += 1;
    }

    if (splicingp == true) {
      if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					    /*univdiagonal*/univdiagonal0,pos5_0,pos3_0,querylength,
					    mismatch_positions_alloc,query_compress_fwd,
					    /*plusp*/true,genestrand,/*sensedir*/SENSE_ANTI,
					    nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					    listpool,/*method*/KMER_APPROX,level)) != NULL) {
	*antisense_hits = Hitlist_push(*antisense_hits,hitlistpool,(void *) hit);
	*n_antisense_hits += 1;
      }
    }

  } else if (adj < 0) {
    nindels = -adj;
    if (nindels > 3) {
      adj = 0;
    } else if ((indel_pos = Indel_resolve_middle_insertion(&best_nmismatches_i,&best_nmismatches_j,
							   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
							   /*left*/left0,/*indels*/-adj,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress_fwd,
							   pos5_0,pos3_1,querylength,nmismatches_allowed,/*plusp*/true,
							   genestrand,/*want_lowest_coordinate_p*/true)) <= 0) {
      adj = 0;
    } else {
      supporti = indel_pos - pos5_0;
      supportj = pos3_1 - (indel_pos + nindels);
      if (supporti - 3*best_nmismatches_i < MIN_SUPPORT_INDEL) {
	adj = 0;
      } else if (supportj - 3*best_nmismatches_j < MIN_SUPPORT_INDEL) {
	adj = 0;
      } else {
	ninserts = nindels;
	/* total_nmatches = querylength - best_nmismatches_i - best_nmismatches_j - ninserts; */
	sense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_insertion(nindels));
	antisense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_insertion(nindels));
      }
    }

  } else if (univdiagonal1 > univdiagonal0 + max_deletionlen) {
    if (splicingp == false) {
      /* Cannot be a splice */
      adj = 0;

    } else if (circularp[chrnum] == true) {
      /* Cannot be a splice */
      adj = 0;

    } else {
      /* Splice */
      debug2(printf("Appears to be a splice (plus)\n"));
    
      prev_univdiagonal = univdiagonal0;
      univdiagonal = univdiagonal1;
      assert(prev_univdiagonal < univdiagonal);

      spliceinfo->segmenti_donor_nknown = spliceinfo->segmenti_antiacceptor_nknown = 0;

      prev_left = prev_univdiagonal - (Univcoord_T) querylength;

      if (nsplicesites > 0 && Knownsplicing_splicesite_p(prev_left,pos5_0+1,pos3_0) == true) {
	j = binary_search_univcoord(0,nsplicesites,splicesites,prev_left+pos5_0);
	while (j < nsplicesites && splicesites[j] < prev_left+pos3_0) {
	  if (splicetypes[j] == DONOR) {
	    debug4s(printf("Setting known donor %d for segmenti at %u\n",j,splicesites[j]));
	    spliceinfo->segmenti_donor_knownpos[spliceinfo->segmenti_donor_nknown] = splicesites[j] - prev_left;
	    spliceinfo->segmenti_donor_knowni[spliceinfo->segmenti_donor_nknown++] = j;
	  } else if (splicetypes[j] == ANTIACCEPTOR) {
	    debug4s(printf("Setting known antiacceptor %d for segmenti at %u\n",j,splicesites[j]));
	    spliceinfo->segmenti_antiacceptor_knownpos[spliceinfo->segmenti_antiacceptor_nknown] = splicesites[j] - prev_left;
	    spliceinfo->segmenti_antiacceptor_knowni[spliceinfo->segmenti_antiacceptor_nknown++] = j;
	  }
	  j++;
	}
      }
      spliceinfo->segmenti_donor_knownpos[spliceinfo->segmenti_donor_nknown] = querylength + 100;
      spliceinfo->segmenti_antiacceptor_knownpos[spliceinfo->segmenti_antiacceptor_nknown] = querylength + 100;
      
      spliceinfo->segmentj_acceptor_nknown = spliceinfo->segmentj_antidonor_nknown = 0;

      left = univdiagonal - (Univcoord_T) querylength;

      if (nsplicesites > 0 && Knownsplicing_splicesite_p(left,pos5_1+1,pos3_1) == true) {
	j = binary_search_univcoord(0,nsplicesites,splicesites,left+pos5_1);
	while (j < nsplicesites && splicesites[j] < left+pos3_1) {
	  if (splicetypes[j] == ACCEPTOR) {
	    debug4s(printf("Setting known acceptor %d for segmentj at %u\n",j,splicesites[j]));
	    spliceinfo->segmentj_acceptor_knownpos[spliceinfo->segmentj_acceptor_nknown] = splicesites[j] - left;
	    spliceinfo->segmentj_acceptor_knowni[spliceinfo->segmentj_acceptor_nknown++] = j;
	  } else if (splicetypes[j] == ANTIDONOR) {
	    debug4s(printf("Setting known antidonor %d for segmentj at %u\n",j,splicesites[j]));
	    spliceinfo->segmentj_antidonor_knownpos[spliceinfo->segmentj_antidonor_nknown] = splicesites[j] - left;
	    spliceinfo->segmentj_antidonor_knowni[spliceinfo->segmentj_antidonor_nknown++] = j;
	  }
	  j++;
	}
      }
      spliceinfo->segmentj_acceptor_knownpos[spliceinfo->segmentj_acceptor_nknown] = querylength + 100;
      spliceinfo->segmentj_antidonor_knownpos[spliceinfo->segmentj_antidonor_nknown] = querylength + 100;
      
      splice_distance = (int) (univdiagonal - prev_univdiagonal);
      if ((sense_splice_pos = Splice_resolve_sense(&nindels,&indel_pos,&best_knowni_i,&best_knowni_j,
						   &best_nmismatches_i,&best_nmismatches_j,&best_nmismatches_indel,
						   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
						   &best_ref_nmismatches_indel,&best_prob_i,&best_prob_j,
						   /*segmenti_left*/prev_left,/*segmentj_left*/left,chroffset,
						   /*pos5*/pos5_0,/*pos3*/pos3_1,querylength,query_compress_fwd,
						   spliceinfo,nmismatches_allowed,/*plusp*/true,genestrand,
						   /*max_deletionlen*/0,/*max_insertionlen*/0,
						   /*allow_indel_p*/false)) > 0) {
	sense_junctions = Listpool_push(NULL,listpool,
					(void *) Junction_new_splice(splice_distance,/*sensedir*/SENSE_FORWARD,
								     /*donor_prob*/best_prob_i,/*acceptor_prob*/best_prob_j));
      }
      
      if ((antisense_splice_pos = Splice_resolve_antisense(&nindels,&indel_pos,&best_knowni_i,&best_knowni_j,
							   &best_nmismatches_i,&best_nmismatches_j,&best_nmismatches_indel,
							   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
							   &best_ref_nmismatches_indel,&best_prob_i,&best_prob_j,
							   /*segmenti_left*/prev_left,/*segmentj_left*/left,chroffset,
							   /*pos5*/pos5_0,/*pos3*/pos3_1,querylength,query_compress_fwd,
							   spliceinfo,nmismatches_allowed,/*plusp*/true,genestrand,
							   /*max_deletionlen*/0,/*max_insertionlen*/0,
							   /*allow_indel_p*/false)) > 0) {
	antisense_junctions = Listpool_push(NULL,listpool,
					    (void *) Junction_new_splice(splice_distance,/*sensedir*/SENSE_ANTI,
									 /*donor_prob*/best_prob_j,/*acceptor_prob*/best_prob_i));
      }

      adj = 0;
    }
      
  } else if ((indel_pos = Indel_resolve_middle_deletion_or_splice(&introntype,&best_nmismatches_i,&best_nmismatches_j,
								  &best_ref_nmismatches_i,&best_ref_nmismatches_j,
								  /*left*/left0,chroffset,chrhigh,/*indels*/-adj,
								  /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress_fwd,
								  pos5_0,pos3_1,querylength,nmismatches_allowed,/*plusp*/true,
								  genestrand,min_intronlength,/*want_lowest_coordinate_p*/true)) <= 0) {
    adj = 0;

  } else {
    supporti = indel_pos - pos5_0;
    supportj = pos3_1 - indel_pos;
    if (supporti - 3*best_nmismatches_i < MIN_SUPPORT_INDEL) {
      adj = 0;

    } else if (supportj - 3*best_nmismatches_j < MIN_SUPPORT_INDEL) {
      adj = 0;

    } else if ((Chrpos_T) (nindels = adj) < min_intronlength) {
      /* Cannot be an intron, so must be a deletion */
      deletionpos = (univdiagonal0 - querylength) + indel_pos; /* gplus */
      sense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
      antisense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
    
    } else if ((sensedir = Intron_canonical_sensedir(introntype)) == SENSE_NULL) {
      /* No intron dinucleotids found, so must be a deletion */
      deletionpos = (univdiagonal0 - querylength) + indel_pos; /* gplus */
      sense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
      antisense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
    
    } else if (splicingp == false) {
      /* Cannot be a splice */
      adj = 0;
      
    } else if (circularp[chrnum] == true) {
      /* Cannot be a splice */
      adj = 0;
      
    } else if (sensedir == SENSE_FORWARD) {
      /* Rare condition, so not tested */
      /* splice_distance = (int) (univdiagonal1 - univdiagonal0); */
      deletionpos = (univdiagonal0 - querylength) + indel_pos; /* gplus */
      donor_prob = Maxent_hr_donor_prob(deletionpos,chroffset);
      acceptor_prob = Maxent_hr_acceptor_prob(deletionpos+nindels,chroffset);
      sense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_splice(/*splice_distance*/nindels,SENSE_FORWARD,donor_prob,acceptor_prob));
      sense_splice_pos = indel_pos;
      adj = 0;
      
    } else {
      /* Rare condition, so not tested */
      /* splice_distance = (int) (univdiagonal1 - univdiagonal0); */
      deletionpos = (univdiagonal0 - querylength) + indel_pos; /* gplus */
      donor_prob = Maxent_hr_antidonor_prob(deletionpos+nindels,chroffset);
      acceptor_prob = Maxent_hr_antiacceptor_prob(deletionpos,chroffset);
      antisense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_splice(/*splice_distance*/nindels,SENSE_ANTI,donor_prob,acceptor_prob));
      antisense_splice_pos = indel_pos;
      adj = 0;
    }
  }
  
  debug2(printf("sense_splice_pos %d, antisense_splice_pos %d\n",sense_splice_pos,antisense_splice_pos));
  if (sense_splice_pos > 0) {
    abortp = false;
    sense_substrings = (List_T) NULL;
    
    /* second part */
    univdiagonal = univdiagonal1;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = sense_splice_pos;
    pos3 = pos3_1;

#ifdef DEBUG2
    alignstart = left + pos5;
    alignend = left + pos3;
    printf("(1) gplus 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches2 =
      Genome_count_mismatches_substring(&ref_nmismatches2,genomebits,genomebits_alt,query_compress_fwd,left,
					pos5,pos3,/*plusp*/true,genestrand);
    debug2(printf("nmismatches fwd from %d to %d: %d\n",pos5,pos3,nmismatches2));
    
    if ((substring2 = Substring_new(nmismatches2,ref_nmismatches2,left,pos5,pos3,querylength,
				    /*plusp*/true,genestrand,query_compress_fwd,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_FORWARD)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",pos5,pos3));
      sense_substrings = Listpool_push(NULL,listpool,(void *) substring2);
    }
    
    /* first part */
    univdiagonal = univdiagonal0;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = pos5_0;
    pos3 = sense_splice_pos;

#ifdef DEBUG2
    alignstart = left + pos5;
    alignend = left + pos3;
    printf("(1) gplus 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches1 =
      Genome_count_mismatches_substring(&ref_nmismatches1,genomebits,genomebits_alt,query_compress_fwd,left,
					pos5,pos3,/*plusp*/true,genestrand);
    debug2(printf("nmismatches fwd from %d to %d: %d\n",pos5,pos3,nmismatches1));
    
    if ((substring1 = Substring_new(nmismatches1,ref_nmismatches1,left,pos5,pos3,querylength,
				    /*plusp*/true,genestrand,query_compress_fwd,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_FORWARD)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",pos5,pos3));
      sense_substrings = Listpool_push(sense_substrings,listpool,(void *) substring1);
    }
    
    nmismatches_bothdiff = nmismatches1 + nmismatches2;
    nmismatches_refdiff = ref_nmismatches1 + ref_nmismatches2;
    if (abortp == true || 
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 sense_substrings,sense_junctions,
					 /*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/true,genestrand,/*sensedir*/SENSE_FORWARD,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&sense_junctions);
      Substring_list_gc(&sense_substrings);
    } else {
      *sense_hits = Hitlist_push(*sense_hits,hitlistpool,(void *) hit);
      *n_sense_hits += 1;
    }
  }
  
  if (antisense_splice_pos > 0) {
    abortp = false;
    antisense_substrings = (List_T) NULL;
    
    /* second part */
    univdiagonal = univdiagonal1;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = antisense_splice_pos;
    pos3 = pos3_1;

#ifdef DEBUG2
    alignstart = left + pos5;
    alignend = left + pos3;
    printf("(2) gplus 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches2 =
      Genome_count_mismatches_substring(&ref_nmismatches2,genomebits,genomebits_alt,query_compress_fwd,left,
					pos5,pos3,/*plusp*/true,genestrand);
    debug2(printf("nmismatches fwd from %d to %d: %d\n",pos5,pos3,nmismatches2));
    
    if ((substring2 = Substring_new(nmismatches2,ref_nmismatches2,left,pos5,pos3,querylength,
				    /*plusp*/true,genestrand,query_compress_fwd,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_ANTI)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",pos5,pos3));
      antisense_substrings = Listpool_push(NULL,listpool,(void *) substring2);
    }
    
    /* first part */
    univdiagonal = univdiagonal0;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = pos5_0;
    pos3 = antisense_splice_pos;

#ifdef DEBUG2
    alignstart = left + pos5;
    alignend = left + pos3;
    printf("(2) gplus 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches1 =
      Genome_count_mismatches_substring(&ref_nmismatches1,genomebits,genomebits_alt,query_compress_fwd,left,
					pos5,pos3,/*plusp*/true,genestrand);
    debug2(printf("nmismatches fwd from %d to %d: %d\n",pos5,pos3,nmismatches1));
    
    if ((substring1 = Substring_new(nmismatches1,ref_nmismatches1,left,pos5,pos3,querylength,
				    /*plusp*/true,genestrand,query_compress_fwd,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_ANTI)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",pos5,pos3));
      antisense_substrings = Listpool_push(antisense_substrings,listpool,(void *) substring1);
    }
    
    nmismatches_bothdiff = nmismatches1 + nmismatches2;
    nmismatches_refdiff = ref_nmismatches1 + ref_nmismatches2;
    if (abortp == true ||
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 antisense_substrings,antisense_junctions,
					 /*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/true,genestrand,/*sensedir*/SENSE_ANTI,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&antisense_junctions);
      Substring_list_gc(&antisense_substrings);
    } else {
      *antisense_hits = Hitlist_push(*antisense_hits,hitlistpool,(void *) hit);
      *n_antisense_hits += 1;
    }
  }
  
  if (adj != 0) {
    /* Indel (all splice options have set adj = 0) */
    abortp = false;
    sense_substrings = (List_T) NULL;
    antisense_substrings = (List_T) NULL;
    
    /* second part */
    univdiagonal = univdiagonal1;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = indel_pos + ninserts;
    pos3 = pos3_1;

#ifdef DEBUG2
    alignstart = left + pos5;
    alignend = left + pos3;
    printf("(3) gplus 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches2 =
      Genome_count_mismatches_substring(&ref_nmismatches2,genomebits,genomebits_alt,query_compress_fwd,left,
					pos5,pos3,/*plusp*/true,genestrand);
    debug2(printf("nmismatches fwd from %d to %d: %d\n",pos5,pos3,nmismatches2));
    
    if ((substring2 = Substring_new(nmismatches2,ref_nmismatches2,left,pos5,pos3,querylength,
				    /*plusp*/true,genestrand,query_compress_fwd,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_NULL)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",pos5,pos3));
      sense_substrings = Listpool_push(NULL,listpool,(void *) substring2);
      antisense_substrings = Listpool_push(NULL,listpool,(void *) Substring_copy(substring2));
    }
    
    /* first part */
    univdiagonal = univdiagonal0;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = pos5_0;
    pos3 = indel_pos;

#ifdef DEBUG2
    alignstart = left + pos5;
    alignend = left + pos3;
    printf("(3) gplus 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches1 =
      Genome_count_mismatches_substring(&ref_nmismatches1,genomebits,genomebits_alt,query_compress_fwd,left,
					pos5,pos3,/*plusp*/true,genestrand);
    debug2(printf("nmismatches fwd from %d to %d: %d\n",pos5,pos3,nmismatches1));
    
    if ((substring1 = Substring_new(nmismatches1,ref_nmismatches1,left,pos5,pos3,querylength,
				    /*plusp*/true,genestrand,query_compress_fwd,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_NULL)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",pos5,pos3));
      sense_substrings = Listpool_push(sense_substrings,listpool,(void *) substring1);
      antisense_substrings = Listpool_push(antisense_substrings,listpool,(void *) Substring_copy(substring1));
    }
    
    nmismatches_bothdiff = nmismatches1 + nmismatches2;
    nmismatches_refdiff = ref_nmismatches1 + ref_nmismatches2;

    /* SENSE_NULL: Put into both sense and antisense outputs */
    if (abortp == true ||
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 sense_substrings,sense_junctions,
					 /*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/true,genestrand,/*sensedir*/SENSE_NULL,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&sense_junctions);
      Substring_list_gc(&sense_substrings);
    } else {
      *sense_hits = Hitlist_push(*sense_hits,hitlistpool,(void *) hit);
      *n_sense_hits += 1;
    }

    if (splicingp == false ||
	abortp == true ||
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 antisense_substrings,antisense_junctions,
					 /*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/true,genestrand,/*sensedir*/SENSE_NULL,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&antisense_junctions);
      Substring_list_gc(&antisense_substrings);
    } else {
      *antisense_hits = Hitlist_push(*antisense_hits,hitlistpool,(void *) hit);
      *n_antisense_hits += 1;
    }
  }

  return;
}


/* Genomic procedure */
/* left0 is for the beginning of the query; left1 is for the end */
/* Indel_resolve_middle_insertion is expecting left for the beginning of the query, or left0 */
static void
combine_ends_minus (int *found_score_overall, int *found_score_within_trims,
		    int *n_sense_hits, List_T *sense_hits, int *n_antisense_hits, List_T *antisense_hits,
		    int querylength, Univcoord_T univdiagonal0, Univcoord_T univdiagonal1,
		    int pos5_0, int pos3_0, int pos5_1, int pos3_1,
		    Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh, Chrpos_T chrlength,
		    int *mismatch_positions_alloc, Spliceinfo_T spliceinfo, Compress_T query_compress_rev,
		    int genestrand, int nmismatches_allowed, int max_deletionlen,
		    Listpool_T listpool, Hitlistpool_T hitlistpool, int level) {
  Stage3end_T hit;
  Univcoord_T left, prev_left, left0, univdiagonal, prev_univdiagonal;
  Substring_T substring1, substring2;
  Univcoord_T deletionpos;
  int splice_distance;
  int pos5, pos3;
  bool abortp;
#ifdef DEBUG2
  Univcoord_T alignstart, alignend;
#endif

  int nmismatches1, nmismatches2, ref_nmismatches1, ref_nmismatches2,
    nmismatches_bothdiff, nmismatches_refdiff;
  int introntype, sensedir;

  int supporti, supportj;
  int best_knowni_i, best_knowni_j, best_nmismatches_i, best_nmismatches_j, best_nmismatches_indel;
  int best_ref_nmismatches_i, best_ref_nmismatches_j, best_ref_nmismatches_indel;
  double best_prob_i, best_prob_j, donor_prob, acceptor_prob;
  int j;

  List_T sense_junctions = NULL, antisense_junctions = NULL, sense_substrings, antisense_substrings;
  int ninserts = 0;
  int sense_splice_pos = -1, antisense_splice_pos = -1;
  int adj, nindels, indel_pos;

  debug2(printf("Entered combine_ends_minus with univdiagonal %u and univdiagonal %u\n",
		univdiagonal0,univdiagonal1));

  *n_sense_hits = *n_antisense_hits = 0;

  left0 = univdiagonal0 - (Univcoord_T) querylength;
#if 0
  chrnum = Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,
			       /*low*/left0+pos5_0,/*high*/left0+pos3_0,circular_typeint);
#endif

  if ((adj = (int) (univdiagonal1 - univdiagonal0)) == 0) {
    if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					  /*univdiagonal*/univdiagonal0,pos5_0,pos3_0,querylength,
					  mismatch_positions_alloc,query_compress_rev,
					  /*plusp*/false,genestrand,/*sensedir*/SENSE_FORWARD,
					  nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					  listpool,/*method*/KMER_APPROX,level)) != NULL) {
      *sense_hits = Hitlist_push(*sense_hits,hitlistpool,(void *) hit);
      *n_sense_hits += 1;
    }

    if (splicingp == true) {
      if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					    /*univdiagonal*/univdiagonal0,pos5_0,pos3_0,querylength,
					    mismatch_positions_alloc,query_compress_rev,
					    /*plusp*/false,genestrand,/*sensedir*/SENSE_ANTI,
					    nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					    listpool,/*method*/KMER_APPROX,level)) != NULL) {
	*antisense_hits = Hitlist_push(*antisense_hits,hitlistpool,(void *) hit);
	*n_antisense_hits += 1;
      }
    }
      
  } else if (adj < 0) {
    nindels = -adj;
    if (nindels > 3) {
      adj = 0;
    } else if ((indel_pos = Indel_resolve_middle_insertion(&best_nmismatches_i,&best_nmismatches_j,
							   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
							   /*left*/left0,/*indels*/-adj,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress_rev,
							   pos5_0,pos3_1,querylength,nmismatches_allowed,/*plusp*/false,
							   genestrand,/*want_lowest_coordinate_p*/true)) <= 0) {
      adj = 0;
    } else {
      supporti = indel_pos - pos5_0;
      supportj = pos3_1 - (indel_pos + nindels);
      if (supporti - 3*best_nmismatches_i < MIN_SUPPORT_INDEL) {
	adj = 0;
      } else if (supportj - 3*best_nmismatches_j < MIN_SUPPORT_INDEL) {
	adj = 0;
      } else {
	ninserts = nindels;
	/* total_nmatches = querylength - best_nmismatches_i - best_nmismatches_j - ninserts; */
	sense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_insertion(nindels));
	antisense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_insertion(nindels));
      }
    }
    
  } else if (univdiagonal1 > univdiagonal0 + max_deletionlen) {
    if (splicingp == false) {
      /* Cannot be a splice */
      adj = 0;

    } else if (circularp[chrnum] == true) {
      /* Cannot be a splice */
      adj = 0;

    } else {
      /* Splice */
      debug2(printf("Appears to be a splice (minus)\n"));
    
      prev_univdiagonal = univdiagonal0;
      univdiagonal = univdiagonal1;
      assert(prev_univdiagonal < univdiagonal);
    
      spliceinfo->segmenti_donor_nknown = spliceinfo->segmenti_antiacceptor_nknown = 0;

      prev_left = prev_univdiagonal - (Univcoord_T) querylength;

      if (nsplicesites > 0 && Knownsplicing_splicesite_p(prev_left,pos5_0+1,pos3_0) == true) {
	j = binary_search_univcoord(0,nsplicesites,splicesites,prev_left+pos5_0);
	while (j < nsplicesites && splicesites[j] < prev_left+pos3_0) {
	  if (splicetypes[j] == DONOR) {
	    debug4s(printf("Setting known donor %d for segmenti at %u\n",j,splicesites[j]));
	    spliceinfo->segmenti_donor_knownpos[spliceinfo->segmenti_donor_nknown] = splicesites[j] - prev_left;
	    spliceinfo->segmenti_donor_knowni[spliceinfo->segmenti_donor_nknown++] = j;
	  } else if (splicetypes[j] == ANTIACCEPTOR) {
	    debug4s(printf("Setting known antiacceptor %d for segmenti at %u\n",j,splicesites[j]));
	    spliceinfo->segmenti_antiacceptor_knownpos[spliceinfo->segmenti_antiacceptor_nknown] = splicesites[j] - prev_left;
	    spliceinfo->segmenti_antiacceptor_knowni[spliceinfo->segmenti_antiacceptor_nknown++] = j;
	  }
	  j++;
	}
      }
      spliceinfo->segmenti_donor_knownpos[spliceinfo->segmenti_donor_nknown] = querylength + 100;
      spliceinfo->segmenti_antiacceptor_knownpos[spliceinfo->segmenti_antiacceptor_nknown] = querylength + 100;
    
      spliceinfo->segmentj_acceptor_nknown = spliceinfo->segmentj_antidonor_nknown = 0;

      left = univdiagonal - (Univcoord_T) querylength;

      if (nsplicesites > 0 && Knownsplicing_splicesite_p(left,pos5_1+1,pos3_1) == true) {
	j = binary_search_univcoord(0,nsplicesites,splicesites,left+pos5_1);
	while (j < nsplicesites && splicesites[j] < left+pos3_1) {
	  if (splicetypes[j] == ACCEPTOR) {
	    debug4s(printf("Setting known acceptor %d for segmentj at %u\n",j,splicesites[j]));
	    spliceinfo->segmentj_acceptor_knownpos[spliceinfo->segmentj_acceptor_nknown] = splicesites[j] - left;
	    spliceinfo->segmentj_acceptor_knowni[spliceinfo->segmentj_acceptor_nknown++] = j;
	  } else if (splicetypes[j] == ANTIDONOR) {
	    debug4s(printf("Setting known antidonor %d for segmentj at %u\n",j,splicesites[j]));
	    spliceinfo->segmentj_antidonor_knownpos[spliceinfo->segmentj_antidonor_nknown] = splicesites[j] - left;
	    spliceinfo->segmentj_antidonor_knowni[spliceinfo->segmentj_antidonor_nknown++] = j;
	  }
	  j++;
	}
      }
      spliceinfo->segmentj_acceptor_knownpos[spliceinfo->segmentj_acceptor_nknown] = querylength + 100;
      spliceinfo->segmentj_antidonor_knownpos[spliceinfo->segmentj_antidonor_nknown] = querylength + 100;
    
      splice_distance = (int) (univdiagonal - prev_univdiagonal);
      if ((sense_splice_pos = Splice_resolve_sense(&nindels,&indel_pos,&best_knowni_i,&best_knowni_j,
						   &best_nmismatches_i,&best_nmismatches_j,&best_nmismatches_indel,
						   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
						   &best_ref_nmismatches_indel,&best_prob_i,&best_prob_j,
						   /*segmenti_left*/prev_left,/*segmentj_left*/left,chroffset,
						   /*pos5*/pos5_0,/*pos3*/pos3_1,querylength,query_compress_rev,
						   spliceinfo,nmismatches_allowed,/*plusp*/false,genestrand,
						   /*max_deletionlen*/0,/*max_insertionlen*/0,
						   /*allow_indel_p*/false)) > 0) {
	sense_junctions = Listpool_push(NULL,listpool,
					(void *) Junction_new_splice(splice_distance,/*sensedir*/SENSE_FORWARD,
								     /*donor_prob*/best_prob_j,/*acceptor_prob*/best_prob_i));
      }
    
      if ((antisense_splice_pos = Splice_resolve_antisense(&nindels,&indel_pos,&best_knowni_i,&best_knowni_j,
							   &best_nmismatches_i,&best_nmismatches_j,&best_nmismatches_indel,
							   &best_ref_nmismatches_i,&best_ref_nmismatches_j,
							   &best_ref_nmismatches_indel,&best_prob_i,&best_prob_j,
							   /*segmenti_left*/prev_left,/*segmentj_left*/left,chroffset,
							   /*pos5*/pos5_0,/*pos3*/pos3_1,querylength,query_compress_rev,
							   spliceinfo,nmismatches_allowed,/*plusp*/false,genestrand,
							   /*max_deletionlen*/0,/*max_insertionlen*/0,
							   /*allow_indel_p*/false)) > 0) {
	antisense_junctions = Listpool_push(NULL,listpool,
					    (void *) Junction_new_splice(splice_distance,/*sensedir*/SENSE_ANTI,
									 /*donor_prob*/best_prob_i,/*acceptor_prob*/best_prob_j));
      }
    
      adj = 0;
    }
    
  } else if ((indel_pos = Indel_resolve_middle_deletion_or_splice(&introntype,&best_nmismatches_i,&best_nmismatches_j,
								  &best_ref_nmismatches_i,&best_ref_nmismatches_j,
								  /*left*/left0,chroffset,chrhigh,/*indels*/-adj,
								  /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress_rev,
								  pos5_0,pos3_1,querylength,nmismatches_allowed,/*plusp*/false,
								  genestrand,min_intronlength,/*want_lowest_coordinate_p*/true)) <= 0) {
    adj = 0;

  } else {
    supporti = indel_pos - pos5_0;
    supportj = pos3_1 - indel_pos;
    if (supporti - 3*best_nmismatches_i < MIN_SUPPORT_INDEL) {
      adj = 0;

    } else if (supportj - 3*best_nmismatches_j < MIN_SUPPORT_INDEL) {
      adj = 0;

    } else if ((Chrpos_T) (nindels = adj) < min_intronlength) {
      /* Cannot be an intron, so must be a deletion */
      deletionpos = (univdiagonal1 - querylength) + indel_pos - nindels; /* gminus */
      sense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
      antisense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
    
    } else if ((sensedir = Intron_canonical_sensedir(introntype)) == SENSE_NULL) {
      /* No intron dinucleotids found, so must be a deletion */
      deletionpos = (univdiagonal1 - querylength) + indel_pos - nindels; /* gminus */
      sense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
      antisense_junctions = Listpool_push(NULL,listpool,(void *) Junction_new_deletion(nindels,deletionpos));
    
    } else if (splicingp == false) {
      /* Cannot be a splice */
      adj = 0;
      
    } else if (circularp[chrnum] == true) {
      /* Cannot be a splice */
      adj = 0;

    } else if (sensedir == SENSE_FORWARD) {
      /* Rare condition, so not tested */
      /* splice_distance = (int) (univdiagonal1 - univdiagonal0); */
      deletionpos = (univdiagonal1 - querylength) + indel_pos - nindels; /* gminus */
      donor_prob = Maxent_hr_antidonor_prob(deletionpos+nindels,chroffset);
      acceptor_prob = Maxent_hr_antiacceptor_prob(deletionpos,chroffset);
      sense_junctions = Listpool_push(NULL,listpool,
				      (void *) Junction_new_splice(/*splice_distance*/nindels,SENSE_FORWARD,donor_prob,acceptor_prob));
      sense_splice_pos = indel_pos;
      adj = 0;
    
    } else {
      /* Rare condition, so not tested */
      /* splice_distance = (int) (univdiagonal1 - univdiagonal0); */
      deletionpos = (univdiagonal1 - querylength) + indel_pos - nindels; /* gminus */
      donor_prob = Maxent_hr_donor_prob(deletionpos,chroffset);
      acceptor_prob = Maxent_hr_acceptor_prob(deletionpos+nindels,chroffset);
      antisense_junctions = Listpool_push(NULL,listpool,
					  (void *) Junction_new_splice(/*splice_distance*/nindels,SENSE_ANTI,donor_prob,acceptor_prob));
      antisense_splice_pos = indel_pos;
      adj = 0;
    }
  }
  
  debug2(printf("sense_splice_pos %d, antisense_splice_pos %d\n",sense_splice_pos,antisense_splice_pos));
  if (sense_splice_pos > 0) {
    abortp = false;
    sense_substrings = (List_T) NULL;
    
    /* first part */
    univdiagonal = univdiagonal0;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = pos5_0;
    pos3 = sense_splice_pos;

#ifdef DEBUG2
    alignstart = left + pos3;
    alignend = left + pos5;
    printf("(1) gminus 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches1 =
      Genome_count_mismatches_substring(&ref_nmismatches1,genomebits,genomebits_alt,query_compress_rev,left,
					pos5,pos3,/*plusp*/false,genestrand);
    debug2(printf("mismatches rev from %d to %d: %d\n",querylength - pos3,querylength - pos5,nmismatches1));

    if ((substring1 = Substring_new(nmismatches1,ref_nmismatches1,left,
				    querylength - pos3,querylength - pos5,querylength,
				    /*plusp*/false,genestrand,query_compress_rev,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_FORWARD)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",querylength - pos3,querylength - pos5));
      sense_substrings = Listpool_push(NULL,listpool,(void *) substring1);
    }
    
    /* second part */
    univdiagonal = univdiagonal1;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = sense_splice_pos;
    pos3 = pos3_1;

#ifdef DEBUG2
    alignstart = left + pos3;
    alignend = left + pos5;
    printf("(1) gminus 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches2 =
      Genome_count_mismatches_substring(&ref_nmismatches2,genomebits,genomebits_alt,query_compress_rev,left,
					pos5,pos3,/*plusp*/false,genestrand);
    debug2(printf("mismatches rev from %d to %d: %d\n",pos5,pos3,nmismatches2));
					
    if ((substring2 = Substring_new(nmismatches2,ref_nmismatches2,left,
				    querylength - pos3,querylength - pos5,querylength,
				    /*plusp*/false,genestrand,query_compress_rev,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_FORWARD)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",querylength - pos3,querylength - pos5));
      sense_substrings = Listpool_push(sense_substrings,listpool,(void *) substring2);
    }
    
    nmismatches_bothdiff = nmismatches1 + nmismatches2;
    nmismatches_refdiff = ref_nmismatches1 + ref_nmismatches2;
    if (abortp == true ||
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 sense_substrings,sense_junctions,/*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/false,genestrand,/*sensedir*/SENSE_FORWARD,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&sense_junctions);
      Substring_list_gc(&sense_substrings);
    } else {
      *sense_hits = Hitlist_push(*sense_hits,hitlistpool,(void *) hit);
      *n_sense_hits += 1;
    }
  }
  
  if (antisense_splice_pos > 0) {
    abortp = false;
    antisense_substrings = (List_T) NULL;
    
    /* first part */
    univdiagonal = univdiagonal0;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = pos5_0;
    pos3 = antisense_splice_pos;

#ifdef DEBUG2
    alignstart = left + pos3;
    alignend = left + pos5;
    printf("(2) gminus 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches1 =
      Genome_count_mismatches_substring(&ref_nmismatches1,genomebits,genomebits_alt,query_compress_rev,left,
					pos5,pos3,/*plusp*/false,genestrand);
    debug2(printf("mismatches rev from %d to %d: %d\n",querylength - pos3,querylength - pos5,nmismatches1));

    if ((substring1 = Substring_new(nmismatches1,ref_nmismatches1,left,
				    querylength - pos3,querylength - pos5,querylength,
				    /*plusp*/false,genestrand,query_compress_rev,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_ANTI)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",querylength - pos3,querylength - pos5));
      antisense_substrings = Listpool_push(NULL,listpool,(void *) substring1);
    }
    
    /* second part */
    univdiagonal = univdiagonal1;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = antisense_splice_pos;
    pos3 = pos3_1;

#ifdef DEBUG2
    alignstart = left + pos3;
    alignend = left + pos5;
    printf("(2) gminus 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches2 =
      Genome_count_mismatches_substring(&ref_nmismatches2,genomebits,genomebits_alt,query_compress_rev,left,
					pos5,pos3,/*plusp*/false,genestrand);
    debug2(printf("mismatches rev from %d to %d: %d\n",pos5,pos3,nmismatches2));

    if ((substring2 = Substring_new(nmismatches2,ref_nmismatches2,left,
				    querylength - pos3,querylength - pos5,querylength,
				    /*plusp*/false,genestrand,query_compress_rev,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_ANTI)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",querylength - pos3,querylength - pos5));
      antisense_substrings = Listpool_push(antisense_substrings,listpool,(void *) substring2);
    }
    
    nmismatches_bothdiff = nmismatches1 + nmismatches2;
    nmismatches_refdiff = ref_nmismatches1 + ref_nmismatches2;
    if (abortp == true ||
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 antisense_substrings,antisense_junctions,
					 /*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/false,genestrand,/*sensedir*/SENSE_ANTI,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&antisense_junctions);
      Substring_list_gc(&antisense_substrings);
    } else {
      *antisense_hits = Hitlist_push(*antisense_hits,hitlistpool,(void *) hit);
      *n_antisense_hits += 1;
    }
  }
  
  if (adj != 0) {
    /* Indel (all splice options have set adj = 0) */
    abortp = false;
    sense_substrings = (List_T) NULL;
    antisense_substrings = (List_T) NULL;
    
    /* first part */
    univdiagonal = univdiagonal0;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = pos5_0;
    pos3 = indel_pos;

#ifdef DEBUG2
    alignstart = left + pos3;
    alignend = left + pos5;
    printf("(3) gminus 1st part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches1 =
      Genome_count_mismatches_substring(&ref_nmismatches1,genomebits,genomebits_alt,query_compress_rev,left,
					pos5,pos3,/*plusp*/false,genestrand);
    debug2(printf("mismatches rev from %d to %d: %d\n",querylength - pos3,querylength - pos5,nmismatches1));


    if ((substring1 = Substring_new(nmismatches1,ref_nmismatches1,left,
				    querylength - pos3,querylength - pos5,querylength,
				    /*plusp*/false,genestrand,query_compress_rev,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_NULL)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",querylength - pos3,querylength - pos5));
      sense_substrings = Listpool_push(NULL,listpool,(void *) substring1);
      antisense_substrings = Listpool_push(NULL,listpool,(void *) Substring_copy(substring1));
    }
    
    /* second part */
    univdiagonal = univdiagonal1;
    left = univdiagonal - (Univcoord_T) querylength;
    pos5 = indel_pos + ninserts;
    pos3 = pos3_1;

#ifdef DEBUG2
    alignstart = left + pos3;
    alignend = left + pos5;
    printf("(3) gminus 2nd part query %d..%d, align %u..%u [%u..%u], left %u = %u - (%d - %d)\n",
	   pos5,pos3,alignstart,alignend,alignstart - chroffset,alignend - chroffset,
	   left - chroffset,alignend - chroffset,querylength,pos3);
#endif
    nmismatches2 =
      Genome_count_mismatches_substring(&ref_nmismatches2,genomebits,genomebits_alt,query_compress_rev,left,
					pos5,pos3,/*plusp*/false,genestrand);
    debug2(printf("mismatches rev from %d to %d: %d\n",pos5,pos3,nmismatches2));

    if ((substring2 = Substring_new(nmismatches2,ref_nmismatches2,left,
				    querylength - pos3,querylength - pos5,querylength,
				    /*plusp*/false,genestrand,query_compress_rev,
				    chrnum,chroffset,chrhigh,chrlength,
				    /*splice_querystart_p*/false,/*splicetype_querystart*/NO_SPLICE,/*ambig_prob_querystart*/0.0,
				    /*splice_queryend_p*/false,/*splicetype_queryend*/NO_SPLICE,/*ambig_prob_queryend*/0.0,
				    /*sensedir*/SENSE_NULL)) == NULL) {
      debug2(printf("ABORTING BECAUSE OF MISMATCHES\n"));
      abortp = true;
    } else {
      debug2(printf(" => Created substring for %d..%d\n",querylength - pos3,querylength - pos5));
      sense_substrings = Listpool_push(sense_substrings,listpool,(void *) substring2);
      antisense_substrings = Listpool_push(antisense_substrings,listpool,(void *) Substring_copy(substring2));
    }
    
    nmismatches_bothdiff = nmismatches1 + nmismatches2;
    nmismatches_refdiff = ref_nmismatches1 + ref_nmismatches2;

    /* SENSE_NULL: Put into both sense and antisense outputs */
    if (abortp == true ||
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 sense_substrings,sense_junctions,
					 /*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/false,genestrand,/*sensedir*/SENSE_FORWARD,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&sense_junctions);
      Substring_list_gc(&sense_substrings);
    } else {
      *sense_hits = Hitlist_push(*sense_hits,hitlistpool,(void *) hit);
      *n_sense_hits += 1;
    }

    if (splicingp == false ||
	abortp == true ||
	(hit = Stage3end_new_precomputed(&(*found_score_overall),&(*found_score_within_trims),
					 nmismatches_bothdiff,nmismatches_refdiff,
					 antisense_substrings,antisense_junctions,
					 /*transcripts*/NULL,/*transcripts_other*/NULL,
					 querylength,chrnum,chroffset,chrhigh,chrlength,
					 /*gplusp*/false,genestrand,/*sensedir*/SENSE_ANTI,
					 listpool,/*method*/KMER_APPROX,level)) == NULL) {
      Junction_list_gc(&antisense_junctions);
      Substring_list_gc(&antisense_substrings);
    } else {
      *antisense_hits = Hitlist_push(*antisense_hits,hitlistpool,(void *) hit);
      *n_antisense_hits += 1;
    }
  }

  return;
}


/* Searches the ends, just like ultrafast transcriptome */
/* Previously thought we needed max_hits because repetitive reads can give many exact matches in a genome */
void
Kmer_search_genome_ends_exact (bool *abort_exact_p, int *found_score_overall, int *found_score_within_trims,
			       List_T *sense_hits_gplus, List_T *sense_hits_gminus,
			       List_T *antisense_hits_gplus, List_T *antisense_hits_gminus,
			       Stage1_T stage1, int querylength, int *mismatch_positions_alloc,
			       Compress_T query_compress_fwd, Compress_T query_compress_rev,
			       int genestrand, int nmismatches_allowed,
			       Listpool_T listpool, Hitlistpool_T hitlistpool, int level) {
  int n_sense_hits = 0, n_antisense_hits = 0;
  Stage3end_T hit;
  int mod5, mod3;

  Univcoord_T *univdiagonals, left, low, high;
  int pos5, pos3, adj;
  int nunivdiagonals, i;

  Chrnum_T chrnum;
  Chrpos_T chrlength;
  Univcoord_T chroffset, chrhigh;


  /* max_hits = 1000000; */

  *abort_exact_p = false;

  /* gplus */
  for (mod5 = 0; mod5 < index1interval; mod5++) {
    adj = (querylength - index1part) % index1interval;
    mod3 = (index1interval + adj - mod5) % index1interval;

#ifdef LARGE_GENOMES
    univdiagonals = Intersect_exact_large(&nunivdiagonals,
					  stage1->plus_rawpositions_high_5[mod5],stage1->plus_rawpositions_5[mod5],
					  stage1->plus_nrawpositions_5[mod5],stage1->plus_diagterms_5[mod5],
					  stage1->plus_rawpositions_high_3[mod3],stage1->plus_rawpositions_3[mod3],
					  stage1->plus_nrawpositions_3[mod3],stage1->plus_diagterms_3[mod3]);
#else
    univdiagonals = Intersect_exact(&nunivdiagonals,
				    stage1->plus_rawpositions_5[mod5],stage1->plus_nrawpositions_5[mod5],stage1->plus_diagterms_5[mod5],
				    stage1->plus_rawpositions_3[mod3],stage1->plus_nrawpositions_3[mod3],stage1->plus_diagterms_3[mod3]);
#endif
    debug4(printf("plus mod5 %d, mod3 %d: %d univdiagonals\n",mod5,mod3,nunivdiagonals));

    i = 0;
    while (/*nhits <= max_hits && */ i < nunivdiagonals) {
      debug4a(printf("ULTRAFAST PLUS DIAGONAL %u\n",univdiagonals[i]));
      debug4a(printf(" => nmismatches %d\n",nmismatches));
      left = univdiagonals[i] - (Univcoord_T) querylength;

      /* TRIM COORDINATES AT GENOME BOUNDS */
      low = (univdiagonals[i] >= (Univcoord_T) querylength) ? left : 0;
      high = (univdiagonals[i] <= genomelength) ? univdiagonals[i] : genomelength;
      chrnum = Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,low,high,circular_typeint);

      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5 = (univdiagonals[i] >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - left);
      pos3 = (univdiagonals[i] <= chrhigh) ? querylength : (int) (chrhigh - left);

      if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					    /*univdiagonal*/univdiagonals[i],pos5,pos3,querylength,
					    mismatch_positions_alloc,query_compress_fwd,
					    /*plusp*/true,genestrand,/*sensedir*/SENSE_FORWARD,
					    nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					    listpool,/*method*/KMER_EXACT,level)) != NULL) {
	*sense_hits_gplus = Hitlist_push(*sense_hits_gplus,hitlistpool,(void *) hit);
	n_sense_hits++;
      }

      if (splicingp == true) {
	if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					      /*univdiagonal*/univdiagonals[i],pos5,pos3,querylength,
					      mismatch_positions_alloc,query_compress_fwd,
					      /*plusp*/true,genestrand,/*sensedir*/SENSE_ANTI,
					      nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					      listpool,/*method*/KMER_EXACT,level)) != NULL) {
	  *antisense_hits_gplus = Hitlist_push(*antisense_hits_gplus,hitlistpool,(void *) hit);
	  n_antisense_hits++;
	}
      }
      i++;
    }
    FREE(univdiagonals);


    /* gminus */
#ifdef LARGE_GENOMES
    univdiagonals = Intersect_exact_large(&nunivdiagonals,
					  stage1->minus_rawpositions_high_5[mod5],stage1->minus_rawpositions_5[mod5],
					  stage1->minus_nrawpositions_5[mod5],stage1->minus_diagterms_5[mod5],
					  stage1->minus_rawpositions_high_3[mod3],stage1->minus_rawpositions_3[mod3],
					  stage1->minus_nrawpositions_3[mod3],stage1->minus_diagterms_3[mod3]);
#else
    univdiagonals = Intersect_exact(&nunivdiagonals,
				    stage1->minus_rawpositions_5[mod5],stage1->minus_nrawpositions_5[mod5],stage1->minus_diagterms_5[mod5],
				    stage1->minus_rawpositions_3[mod3],stage1->minus_nrawpositions_3[mod3],stage1->minus_diagterms_3[mod3]);
#endif
    debug4(printf("minus mod5 %d, mod3 %d: %d univdiagonals\n",mod5,mod3,nunivdiagonals));

    i = 0;
    while (/*nhits <= max_hits && */ i < nunivdiagonals) {
      debug4a(printf("ULTRAFAST MINUS DIAGONAL %u\n",univdiagonals[i]));
      debug4a(printf(" => nmismatches %d\n",nmismatches));
      left = univdiagonals[i] - (Univcoord_T) querylength;

      /* TRIM COORDINATES AT GENOME BOUNDS */
      low = (univdiagonals[i] >= (Univcoord_T) querylength) ? left : 0;
      high = (univdiagonals[i] <= genomelength) ? univdiagonals[i] : genomelength;
      chrnum = Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,low,high,circular_typeint);
	    
      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5 = (univdiagonals[i] >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - left);
      pos3 = (univdiagonals[i] <= chrhigh) ? querylength : (int) (chrhigh - left);

      if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					    /*univdiagonal*/univdiagonals[i],pos5,pos3,querylength,
					    mismatch_positions_alloc,query_compress_rev,
					    /*plusp*/false,genestrand,/*sensedir*/SENSE_FORWARD,
					    nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					    listpool,/*method*/KMER_EXACT,level)) != NULL) {
	*sense_hits_gminus = Hitlist_push(*sense_hits_gminus,hitlistpool,(void *) hit);
	n_sense_hits++;
      }

      if (splicingp == true) {
	if ((hit = Stage3end_new_substitution(&(*found_score_overall),&(*found_score_within_trims),
					      /*univdiagonal*/univdiagonals[i],pos5,pos3,querylength,
					      mismatch_positions_alloc,query_compress_rev,
					      /*plusp*/false,genestrand,/*sensedir*/SENSE_ANTI,
					      nmismatches_allowed,chrnum,chroffset,chrhigh,chrlength,
					      listpool,/*method*/KMER_EXACT,level)) != NULL) {
	  *antisense_hits_gminus = Hitlist_push(*antisense_hits_gminus,hitlistpool,(void *) hit);
	  n_antisense_hits++;
	}
      }
      i++;
    }
    FREE(univdiagonals);

  }

#if 0
  if (nhits > max_hits) {
    debug4(printf("Kmer_search_ends_exact aborting because nhits %d > max_hits %d\n",nhits,max_hits));
    Stage3end_gc(*hits_gplus);
    Hitlist_free(&(*hits_gplus));
    Stage3end_gc(*hits_gminus);
    Hitlist_free(&(*hits_gminus));
    *abort_exact_p = true;	/* Indicates that we should not try to run approx algorithm */
  }
#endif

  debug4(printf("Kmer_search_genome_ends_exact returning sense %d plus and %d minus hits, antisense %d plus and %d minus hits\n",
		List_length(*sense_hits_gplus),List_length(*sense_hits_gminus),
		List_length(*antisense_hits_gplus),List_length(*antisense_hits_gminus)));

  return;
}


/* Performs a merge.  About 10% faster than the alternative of trying all combinations of mod5 and mod3. */
/* Need max_hits, because repetitive reads can give many exact matches in a genome */
void
Kmer_search_genome_ends_approx (int *found_score_overall, int *found_score_within_trims,
				List_T *sense_hits_gplus, List_T *sense_hits_gminus,
				List_T *antisense_hits_gplus, List_T *antisense_hits_gminus,
				Stage1_T stage1, int *mismatch_positions_alloc,
				Compress_T query_compress_fwd, Compress_T query_compress_rev,
				int querylength, int genestrand, int nmismatches_allowed,
				int max_deletionlen, Chrpos_T overall_max_distance, int sizelimit,
				Listpool_T listpool, Hitlistpool_T hitlistpool, int level) {
  int n_sense_hits = 0, n_antisense_hits = 0, n_sense_hits0, n_antisense_hits0;
  int gplus_streami_5 = 0, gminus_streami_5 = 0, gplus_streami_3 = 0, gminus_streami_3 = 0;


  int total_npositions_gplus_5 = 0, total_npositions_gminus_5 = 0,
    total_npositions_gplus_3 = 0, total_npositions_gminus_3 = 0;

  Univcoord_T *gplus_diagonals_5, *gminus_diagonals_5, *gplus_diagonals_3, *gminus_diagonals_3;
  int n_gplus_diagonals_5, n_gminus_diagonals_5, n_gplus_diagonals_3, n_gminus_diagonals_3;

  bool gplus_exactp, gminus_exactp;
  Univcoord_T *gplus_diagpairs, *gminus_diagpairs;
  int n_gplus_diagpairs, n_gminus_diagpairs;

  int i, k;
  Univcoord_T lefta, leftb, univdiagonala, univdiagonalb, univdiagonal0, univdiagonal1;

  int pos5_0, pos3_0, pos5_1, pos3_1, pos5a, pos3a, pos5b, pos3b, adj;
  int trim_a5, trim_a3, trim_b5, trim_b3;
  int nmismatches_a5, nmismatches_a3, nmismatches_b5, nmismatches_b3;

  Chrnum_T chrnum;
  Chrpos_T chrlength;
  Univcoord_T chroffset, chrhigh, low, high;


  /* max_hits = 1000000; */

  for (i = 0; i < index1interval; i++) {
    if (stage1->plus_nrawpositions_5[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gplus_stream_high_array_5[gplus_streami_5] = stage1->plus_rawpositions_high_5[i];
      stage1->gplus_stream_low_array_5[gplus_streami_5] = stage1->plus_rawpositions_5[i];
#else
      stage1->gplus_stream_array_5[gplus_streami_5] = stage1->plus_rawpositions_5[i];
#endif
      stage1->gplus_streamsize_array_5[gplus_streami_5] = stage1->plus_nrawpositions_5[i];
      stage1->gplus_diagterm_array_5[gplus_streami_5] = stage1->plus_diagterms_5[i];
      total_npositions_gplus_5 += stage1->plus_nrawpositions_5[i];
      gplus_streami_5++;
    }

    if (stage1->minus_nrawpositions_5[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gminus_stream_high_array_5[gminus_streami_5] = stage1->minus_rawpositions_high_5[i];
      stage1->gminus_stream_low_array_5[gminus_streami_5] = stage1->minus_rawpositions_5[i];
#else
      stage1->gminus_stream_array_5[gminus_streami_5] = stage1->minus_rawpositions_5[i];
#endif
      stage1->gminus_streamsize_array_5[gminus_streami_5] = stage1->minus_nrawpositions_5[i];
      stage1->gminus_diagterm_array_5[gminus_streami_5] = stage1->minus_diagterms_5[i];
      total_npositions_gminus_5 += stage1->minus_nrawpositions_5[i];
      gminus_streami_5++;
    }

    if (stage1->plus_nrawpositions_3[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gplus_stream_high_array_3[gplus_streami_3] = stage1->plus_rawpositions_high_3[i];
      stage1->gplus_stream_low_array_3[gplus_streami_3] = stage1->plus_rawpositions_3[i];
#else
      stage1->gplus_stream_array_3[gplus_streami_3] = stage1->plus_rawpositions_3[i];
#endif
      stage1->gplus_streamsize_array_3[gplus_streami_3] = stage1->plus_nrawpositions_3[i];
      stage1->gplus_diagterm_array_3[gplus_streami_3] = stage1->plus_diagterms_3[i];
      total_npositions_gplus_3 += stage1->plus_nrawpositions_3[i];
      gplus_streami_3++;
    }

    if (stage1->minus_nrawpositions_3[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gminus_stream_high_array_3[gminus_streami_3] = stage1->minus_rawpositions_high_3[i];
      stage1->gminus_stream_low_array_3[gminus_streami_3] = stage1->minus_rawpositions_3[i];
#else
      stage1->gminus_stream_array_3[gminus_streami_3] = stage1->minus_rawpositions_3[i];
#endif
      stage1->gminus_streamsize_array_3[gminus_streami_3] = stage1->minus_nrawpositions_3[i];
      stage1->gminus_diagterm_array_3[gminus_streami_3] = stage1->minus_diagterms_3[i];
      total_npositions_gminus_3 += stage1->minus_nrawpositions_3[i];
      gminus_streami_3++;
    }
  }

  debug4(printf("Comparing total_npositions_gplus_5 %d against sizelmit %d\n",total_npositions_gplus_5,sizelimit));
  if (0 && total_npositions_gplus_5 > sizelimit) {
    gplus_diagonals_5 = (Univcoord_T *) NULL;
    n_gplus_diagonals_5 = 0;
  } else {
#ifdef LARGE_GENOMES
    gplus_diagonals_5 = Merge_diagonals_large(&n_gplus_diagonals_5,
					      stage1->gplus_stream_high_array_5,stage1->gplus_stream_low_array_5,
					      stage1->gplus_streamsize_array_5,stage1->gplus_diagterm_array_5,
					      /*nstreams*/gplus_streami_5);
#else
    gplus_diagonals_5 = Merge_diagonals(&n_gplus_diagonals_5,stage1->gplus_stream_array_5,
					stage1->gplus_streamsize_array_5,stage1->gplus_diagterm_array_5,
					/*nstreams*/gplus_streami_5);
#endif
  }


  debug4(printf("Comparing total_npositions_gminus_5 %d against sizelmit %d\n",total_npositions_gminus_5,sizelimit));
  if (0 && total_npositions_gminus_5 > sizelimit) {
    gminus_diagonals_5 = (Univcoord_T *) NULL;
    n_gminus_diagonals_5 = 0;
  } else {
#ifdef LARGE_GENOMES
    gminus_diagonals_5 = Merge_diagonals_large(&n_gminus_diagonals_5,
					       stage1->gminus_stream_high_array_5,stage1->gminus_stream_low_array_5,
					       stage1->gminus_streamsize_array_5,stage1->gminus_diagterm_array_5,
					       /*nstreams*/gminus_streami_5);
#else
    gminus_diagonals_5 = Merge_diagonals(&n_gminus_diagonals_5,stage1->gminus_stream_array_5,
					 stage1->gminus_streamsize_array_5,stage1->gminus_diagterm_array_5,
					 /*nstreams*/gminus_streami_5);
#endif
  }


  debug4(printf("Comparing total_npositions_gplus_3 %d against sizelmit %d\n",total_npositions_gplus_3,sizelimit));
  if (0 && total_npositions_gplus_3 > sizelimit) {
    gplus_diagonals_3 = (Univcoord_T *) NULL;
    n_gplus_diagonals_3 = 0;
  } else {
#ifdef LARGE_GENOMES
    gplus_diagonals_3 = Merge_diagonals_large(&n_gplus_diagonals_3,
					      stage1->gplus_stream_high_array_3,stage1->gplus_stream_low_array_3,
					      stage1->gplus_streamsize_array_3,stage1->gplus_diagterm_array_3,
					      /*nstreams*/gplus_streami_3);
#else
    gplus_diagonals_3 = Merge_diagonals(&n_gplus_diagonals_3,stage1->gplus_stream_array_3,
					stage1->gplus_streamsize_array_3,stage1->gplus_diagterm_array_3,
					/*nstreams*/gplus_streami_3);
#endif

  }


  debug4(printf("Comparing total_npositions_gminus_3 %d against sizelmit %d\n",total_npositions_gminus_3,sizelimit));
  if (0 && total_npositions_gminus_3 > sizelimit) {
    gminus_diagonals_3 = (Univcoord_T *) NULL;
    n_gminus_diagonals_3 = 0;
  } else {
#ifdef LARGE_GENOMES
    gminus_diagonals_3 = Merge_diagonals_large(&n_gminus_diagonals_3,
					       stage1->gminus_stream_high_array_3,stage1->gminus_stream_low_array_3,
					       stage1->gminus_streamsize_array_3,stage1->gminus_diagterm_array_3,
					       /*nstreams*/gminus_streami_3);
#else
    gminus_diagonals_3 = Merge_diagonals(&n_gminus_diagonals_3,stage1->gminus_stream_array_3,
					 stage1->gminus_streamsize_array_3,stage1->gminus_diagterm_array_3,
					 /*nstreams*/gminus_streami_3);
#endif
  }

  gplus_diagpairs = Intersect_approx_simple(&gplus_exactp,&n_gplus_diagpairs,
					    gplus_diagonals_5,n_gplus_diagonals_5,
					    gplus_diagonals_3,n_gplus_diagonals_3,
					    /*maxdistance*/overall_max_distance);
  gminus_diagpairs = Intersect_approx_simple(&gminus_exactp,&n_gminus_diagpairs,
					     gminus_diagonals_5,n_gminus_diagonals_5,
					     gminus_diagonals_3,n_gminus_diagonals_3,
					     /*maxdistance*/overall_max_distance);
  debug4(printf("***Intersect ends approx: exactp %d and %d.  %d plus and %d minus diagpairs\n",
		gplus_exactp,gminus_exactp,n_gplus_diagpairs,n_gminus_diagpairs));

#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(gplus_diagonals_5);
  FREE_ALIGN(gminus_diagonals_5);
  FREE_ALIGN(gplus_diagonals_3);
  FREE_ALIGN(gminus_diagonals_3);
#else
  FREE(gplus_diagonals_5);
  FREE(gminus_diagonals_5);
  FREE(gplus_diagonals_3);
  FREE(gminus_diagonals_3);
#endif

  i = k = 0;
  while (/*nhits <= max_hits && */ i < n_gplus_diagpairs) {
    univdiagonala = gplus_diagpairs[k];
    univdiagonalb = gplus_diagpairs[k+1];

    lefta = univdiagonala - (Univcoord_T) querylength; /* NEW FORMULA for querystart of 0 */
    leftb = univdiagonalb - (Univcoord_T) querylength; /* NEW FORMULA for querystart of 0 */

    /* TRIM COORDINATES AT GENOME BOUNDS */
    low = (univdiagonala >= (Univcoord_T) querylength) ? lefta : 0;
    high = (univdiagonala <= genomelength) ? univdiagonala : genomelength;
    chrnum = Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,low,high,circular_typeint);

    low = (univdiagonalb >= (Univcoord_T) querylength) ? leftb : 0;
    high = (univdiagonalb <= genomelength) ? univdiagonalb : genomelength;
    if (Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,low,high,circular_typeint) != chrnum) {
      /* Skip further computation */
      adj = 0;
    } else {
      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5a = (univdiagonala >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - lefta);
      pos3a = (univdiagonala <= chrhigh) ? querylength : (int) (chrhigh - lefta);
      trim_a5 = Genome_first_kmer_left(&nmismatches_a5,genomebits,query_compress_fwd,lefta,pos5a,pos3a,
				       /*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      trim_a3 = querylength - Genome_first_kmer_right(&nmismatches_a3,genomebits,query_compress_fwd,lefta,pos5a,pos3a,
						      /*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5b = (univdiagonalb >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - leftb);
      pos3b = (univdiagonalb <= chrhigh) ? querylength : (int) (chrhigh - leftb);
      trim_b5 = Genome_first_kmer_left(&nmismatches_b5,genomebits,query_compress_fwd,leftb,pos5b,pos3b,
				       /*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      trim_b3 = querylength - Genome_first_kmer_right(&nmismatches_b3,genomebits,query_compress_fwd,leftb,pos5b,pos3b,
						      /*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
      debug4a(printf("trimmed a %d..%d, trimmed b %d..%d\n",
		     trim_a5,querylength - trim_a3,trim_b5,querylength - trim_b3));
      if (trim_a5 == pos5a || trim_b3 == pos3b) {
	univdiagonal0 = univdiagonala; pos5_0 = pos5a; pos3_0 = pos3a;
	univdiagonal1 = univdiagonalb; pos5_1 = pos5b; pos3_1 = pos3b;
	/* low0 = lefta + pos5a; high0 = lefta + pos3a; */
	/* low1 = leftb + pos5b; high1 = leftb + pos3b; */
	adj = (int) (univdiagonalb - univdiagonala);
	debug4a(printf("Setting a first, b second, adj %d\n",adj));
	
      } else if (trim_a3 == pos3a || trim_b5 == pos5b) {
	univdiagonal0 = univdiagonalb; pos5_0 = pos5b; pos3_0 = pos3b;
	univdiagonal1 = univdiagonala; pos5_1 = pos5a; pos3_1 = pos3a;
	/* low0 = leftb + pos5b; high0 = leftb + pos3b; */
	/* low1 = lefta + pos5a; high1 = lefta + pos3a; */
	adj = (int) (univdiagonala - univdiagonalb);
	debug4a(printf("Setting b first, a second, adj %d\n",adj));
	
      } else {
	adj = 0;
      }
    }
	  
    /* printf("plus adj %d, univdiagonal0 %llu, univdiagonal1 %llu\n",adj,univdiagonal0,univdiagonal1); */
    if (adj != 0) {
      if (univdiagonal1 > univdiagonal0) {
	/* assert(univdiagonal1 <= chrhigh); */
	combine_ends_plus(&(*found_score_overall),&(*found_score_within_trims),
			  &n_sense_hits0,&(*sense_hits_gplus),&n_antisense_hits0,&(*antisense_hits_gplus),
			  querylength,univdiagonal0,univdiagonal1,pos5_0,pos3_0,pos5_1,pos3_1,
			  chrnum,chroffset,chrhigh,chrlength,mismatch_positions_alloc,
			  stage1->spliceinfo,query_compress_fwd,genestrand,nmismatches_allowed,
			  max_deletionlen,listpool,hitlistpool,level);
	n_sense_hits += n_sense_hits0;
	n_antisense_hits += n_antisense_hits0;
	      
      } else {
	/* assert(univdiagonal0 <= chrhigh); */
	combine_ends_plus(&(*found_score_overall),&(*found_score_within_trims),
			  &n_sense_hits0,&(*sense_hits_gplus),&n_antisense_hits0,&(*antisense_hits_gplus),
			  querylength,univdiagonal0,univdiagonal1,pos5_0,pos3_0,pos5_1,pos3_1,
			  chrnum,chroffset,chrhigh,chrlength,mismatch_positions_alloc,
			  stage1->spliceinfo,query_compress_fwd,genestrand,nmismatches_allowed,
			  max_deletionlen,listpool,hitlistpool,level);
	n_sense_hits += n_sense_hits0;
	n_antisense_hits += n_antisense_hits0;
      }
    }
    i++;
    k += 2;
  }
      

  i = k = 0;
  while (/*nhits <= max_hits && */ i < n_gminus_diagpairs) {
    univdiagonala = gminus_diagpairs[k];
    univdiagonalb = gminus_diagpairs[k+1];

    lefta = univdiagonala - (Univcoord_T) querylength; /* NEW FORMULA for queryend of querylength */
    leftb = univdiagonalb - (Univcoord_T) querylength; /* NEW FORMULA for queryend of querylength */

    /* TRIM COORDINATES AT GENOME BOUNDS */
    low = (univdiagonala >= (Univcoord_T) querylength) ? lefta : 0;
    high = (univdiagonala <= genomelength) ? univdiagonala : genomelength;
    chrnum = Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,low,high,circular_typeint);

    low = (univdiagonalb >= (Univcoord_T) querylength) ? leftb : 0;
    high = (univdiagonalb <= genomelength) ? univdiagonalb : genomelength;
    if (Univ_IIT_get_chrnum(&chroffset,&chrhigh,&chrlength,chromosome_iit,low,high,circular_typeint) != chrnum) {
      /* Skip further computation */
      adj = 0;
    } else {
      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5a = (univdiagonala >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - lefta);
      pos3a = (univdiagonala <= chrhigh) ? querylength : (int) (chrhigh - lefta);
      trim_a5 = Genome_first_kmer_left(&nmismatches_a5,genomebits,query_compress_rev,lefta,pos5a,pos3a,
				       /*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      trim_a3 = querylength - Genome_first_kmer_right(&nmismatches_a3,genomebits,query_compress_rev,lefta,pos5a,pos3a,
						      /*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
      /* TRIM QUERY AT CHROMOSOME BOUNDS */
      pos5b = (univdiagonalb >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - leftb);
      pos3b = (univdiagonalb <= chrhigh) ? querylength : (int) (chrhigh - leftb);
      trim_b5 = Genome_first_kmer_left(&nmismatches_b5,genomebits,query_compress_rev,leftb,pos5b,pos3b,
				       /*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      trim_b3 = querylength - Genome_first_kmer_right(&nmismatches_b3,genomebits,query_compress_rev,leftb,pos5b,pos3b,
						      /*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
      debug4a(printf("trimmed a %d..%d, trimmed b %d..%d\n",
		     trim_a5,querylength - trim_a3,trim_b5,querylength - trim_b3));
      if (trim_a5 == pos5a || trim_b3 == pos3b) {
	univdiagonal0 = univdiagonala; pos5_0 = pos5a; pos3_0 = pos3a;
	univdiagonal1 = univdiagonalb; pos5_1 = pos5b; pos3_1 = pos3b;
	/* low0 = lefta + pos5a; high0 = lefta + pos3a; */
	/* low1 = leftb + pos5b; high1 = leftb + pos3b; */
	adj = (int) (univdiagonalb - univdiagonala);
	debug4a(printf("Setting a first, b second, adj %d\n",adj));
	
      } else if (trim_a3 == pos3a || trim_b5 == pos5b) {
	univdiagonal0 = univdiagonalb; pos5_0 = pos5b; pos3_0 = pos3b;
	univdiagonal1 = univdiagonala; pos5_1 = pos5a; pos3_1 = pos3a;
	/* low0 = leftb + pos5b; high0 = leftb + pos3b; */
	/* low1 = lefta + pos5a; high1 = lefta + pos3a; */
	adj = (int) (univdiagonala - univdiagonalb);
	debug4a(printf("Setting b first, a second, adj %d\n",adj));
	
      } else {
	adj = 0;
      }
    }
	  
    /* printf("minus adj %d, univdiagonal0 %llu, univdiagonal1 %llu\n",adj,univdiagonal0,univdiagonal1); */
    if (adj != 0) {
      if (univdiagonal1 > univdiagonal0) {
	/* assert(univdiagonal1 <= chrhigh); */
	combine_ends_minus(&(*found_score_overall),&(*found_score_within_trims),
			   &n_sense_hits0,&(*sense_hits_gminus),&n_antisense_hits0,&(*antisense_hits_gminus),
			   querylength,univdiagonal0,univdiagonal1,pos5_0,pos3_0,pos5_1,pos3_1,
			   chrnum,chroffset,chrhigh,chrlength,mismatch_positions_alloc,
			   stage1->spliceinfo,query_compress_rev,genestrand,nmismatches_allowed,
			   max_deletionlen,listpool,hitlistpool,level);
	n_sense_hits += n_sense_hits0;
	n_antisense_hits += n_antisense_hits0;
	      
      } else {
	/* assert(univdiagonal0 <= chrhigh); */
	combine_ends_minus(&(*found_score_overall),&(*found_score_within_trims),
			   &n_sense_hits0,&(*sense_hits_gminus),&n_antisense_hits0,&(*antisense_hits_gminus),
			   querylength,univdiagonal0,univdiagonal1,pos5_0,pos3_0,pos5_1,pos3_1,
			   chrnum,chroffset,chrhigh,chrlength,mismatch_positions_alloc,
			   stage1->spliceinfo,query_compress_rev,genestrand,nmismatches_allowed,
			   max_deletionlen,listpool,hitlistpool,level);
	n_sense_hits += n_sense_hits0;
	n_antisense_hits += n_antisense_hits0;
      }
    }

    i++;
    k += 2;
  }

  FREE(gminus_diagpairs);
  FREE(gplus_diagpairs);

#if 0
  if (nhits > max_hits) {
    Stage3end_gc(*hits_gplus);
    Hitlist_free(&(*hits_gplus));
    Stage3end_gc(*hits_gminus);
    Hitlist_free(&(*hits_gminus));
  }
#endif

  debug4(printf("Done with Kmer_search_genome_ends_approx\n"));
  return;
}


#if 0
static Oligospace_T
nt_oligo (char *query, int indexsize) {
  Oligospace_T oligo = 0U;
  int i;

  for (i = 0; i < indexsize; i++) {
    oligo *= 4;
    
    switch (query[i]) {
    case 'A': break;
    case 'C': oligo += 1; break;
    case 'G': oligo += 2; break;
    case 'T': oligo += 3; break;
    default:
      fprintf(stderr,"Saw N in nt_oligo\n");
      abort();
    }
  }

  return oligo;
}
#endif


void
Kmer_search_setup (Mode_T mode_in, int index1part_tr_in,
		   int index1part_in, int index1interval_in, int local1part_in,
		   Trcoord_T transcriptomelength_in, int min_intronlength_in,
		   Univ_IIT_T chromosome_iit_in, Univcoord_T genomelength_in,
		   int circular_typeint_in, bool *circularp_in, 
		   Genome_T genomebits_in, Genome_T genomebits_alt_in,
		   Indexdb_T indexdb_in, Indexdb_T indexdb2_in, Indexdb_T indexdb_tr_in,
		   bool splicingp_in, Univcoord_T *splicesites_in, Splicetype_T *splicetypes_in,
		   Chrpos_T *splicedists_in, int nsplicesites_in) {

  mode = mode_in;

  transcriptomelength = transcriptomelength_in;

  chromosome_iit = chromosome_iit_in;
  genomelength = genomelength_in;
  circular_typeint = circular_typeint_in;
  circularp = circularp_in;

  genomebits = genomebits_in;
  genomebits_alt = genomebits_alt_in;

  index1part_tr = index1part_tr_in;
  index1part = index1part_in;
  index1interval = index1interval_in;
  local1part = local1part_in;

#ifdef HAVE_64_BIT
  leftreadshift = 64 - index1part - index1part;
  oligobase_mask = ~(~ (Oligospace_T) 0 << 2*index1part);
#else
  leftreadshift = 32 - index1part - index1part;
  oligobase_mask = ~(~ (Oligospace_T) 0 << 2*index1part);
#endif

  indexdb = indexdb_in;
  indexdb2 = indexdb2_in;
  indexdb_tr = indexdb_tr_in;

  min_intronlength = min_intronlength_in;

  splicingp = splicingp_in;
  splicesites = splicesites_in;
  splicetypes = splicetypes_in;
  splicedists = splicedists_in;
  nsplicesites = nsplicesites_in;

  return;
}


