/* CSL - Common Sound Layer
 * Copyright (C) 2000-2001 Stefan Westerfeld and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef __CSL_MAIN_H__
#define __CSL_MAIN_H__


#include	<csl/csldefs.h>
#include	<csl/cslutils.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * @file
 * This file defines most of the CSL driver API functions.
 */

/* --- CSL Driver API --- */

/**
 * Opaque type used as a handle to a CSL driver.
 * @short CSL Driver handle.
 */
typedef struct _CslDriver        CslDriver;

/**
 * Data type used for CSL mutual exclusion/locking functions.
 * @see csl_driver_init_mutex
 * @short CSL Mutex.
 */
typedef struct _CslMutex         CslMutex;

/**
 * Function prototype for callback function to mutex lock function.
 * @short Mutex lock callback.
 */
typedef void (*CslMutexLock)    (CslMutex *mutex);

/**
 * Function prototype for callback function to mutex unlock function.
 * @short Mutex unlock callback.
 */
typedef void (*CslMutexUnlock)  (CslMutex *mutex);

/**
 * Function prototype for callback function to mutex destroy function.
 * @short Mutex destroy callback.
 */
typedef void (*CslMutexDestroy) (CslMutex *mutex);

/**
 * Data structure for a user-supplied mutex object. Has three function
 * pointers to the lock, unlock, and destroy functions and a pointer to
 * user data.
 * @short User-defined mutex object.
 */
struct _CslMutex
{
  void           *user_data;
  CslMutexLock    lock;
  CslMutexUnlock  unlock;
  CslMutexDestroy destroy;
};

/**
 * CSL error values, returned by most CSL function calls.
 * @see csl_strerror
 * @short CSL return codes.
 */
typedef enum
{
  CSL_ENONE,
  CSL_EINTERN,
  CSL_ENODRIVER,
  CSL_ENOIMPL,
  CSL_EBUSY,
  CSL_EPERMS,
  CSL_EIO,
  CSL_EFMTINVAL,
  CSL_EGETCAPS,
  CSL_ECAPSUPPORT,
  CSL_ESETCAPS
} CslErrorType;

/**
 * CSL driver capabilities. These are used when selecting a driver to
 * be used when calling csl_driver_init_mutex.
 * @see csl_driver_init_mutex
 * @short Driver capabilities.
 */
typedef enum	/* driver capabilities */
{
  CSL_DRIVER_CAP_PCM	= (1 << 0),
  CSL_DRIVER_CAP_SAMPLE	= (1 << 1),
  CSL_DRIVER_CAP_MIXER	= (1 << 2),
  CSL_DRIVER_CAP_MASK	= 0x07
} CslDriverCaps;

/* standard drivers */

/**
 * Can use this value when calling csl_driver_init or csl_driver_init_mutex.
 * @see csl_driver_init
 * @see csl_driver_init_mutex.
 * @short Symbolic name for aRts driver.
 */
#define	CSL_DRIVER_NAME_ARTS	"arts"

/**
 * Can use this value when calling csl_driver_init or csl_driver_init_mutex.
 * @see csl_driver_init
 * @see csl_driver_init_mutex.
 * @short Symbolic name for OSS driver.
 */
#define	CSL_DRIVER_NAME_OSS	"oss"


/* --- main API --- */

const char**	csl_list_drivers	(unsigned int	  *n_backends);

CslErrorType	csl_driver_init		(const char	  *driver_name,
					 CslDriver	 **driver);

CslErrorType	csl_driver_init_mutex	(const char	  *driver_name,
					 CslDriverCaps	   required_caps,
					 CslMutex         *mutex,
					 CslDriver	 **driver);

void		csl_driver_shutdown	(CslDriver	  *driver);

const char*	csl_strerror		(CslErrorType	   error_type);


/* --- convenience functions --- */

/**
 * Data type for storing CSL command line options.
 * @short Command line options.
 */
typedef struct _CslOptions CslOptions;

/**
 * Data type for storing CSL command line options.
 * @short Command line options.
 */
struct _CslOptions
{
  /** 1 if samples are of type float (-F option) */
  unsigned int  float_samples : 1;	/* whether samples are of type float (-F option) */
  /** 1 if samples are signed values (-s option) */
  unsigned int  signed_samples : 1;
  /** 1 if samples are unsigned values (-u option) */
  unsigned int  unsigned_samples : 1;
  /** 1 if samples are in big-endian format (-B option) */
  unsigned int  big_endian : 1;
  /** 1 if samples are in little-endian format (-L option) */
  unsigned int  little_endian : 1;
  /** sample size in bits (-w option) */
  unsigned int  n_bits;
  /** number of channels (-c option) */
  unsigned int  n_channels;
  /** sampling rate in samples/second (-r option) */
  unsigned int  rate;
  /** debug flags (-D option) */
  CslDebugFlags debug_flags;
  /** endianness format, either 1234 or 4321 (composed construct) */
  unsigned int  endianess;
  /** PCM format (composed construct) */
  unsigned int  pcm_format; /** PCM format */
};

void	csl_options_parse	(CslOptions	 *options,
				 int		 *argc_p,
				 char           **argv_p[]);

char*	csl_options_dump	(CslOptions	 *options);

char*	csl_options_describe	(unsigned int	  indent);

char*	csl_describe_pcm_format	(unsigned int     pcm_format);


#if 0	/* Proposed */
typedef struct _CslPollFD {
  int fd;           /* file descriptor */
  unsigned short events;     /* requested events */
  unsigned short revents;    /* returned events */
} CslPollFD;

unsigned int csl_poll_count_fds (void);
unsigned int csl_poll_get_fds (unsigned int n_fds, CslPollFD *fds);
void csl_poll_handle_fds (unsigned int n_fds, CslPollFD *fds);
void csl_poll_wait (unsigned int time_in_ms);
#endif


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __CSL_MAIN_H__ */
