/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "libgimpwidgets/gimpwidgets.h"

#include "tools-types.h"

#include "core/gimptoolinfo.h"

#include "paint/gimpdodgeburnoptions.h"

#include "widgets/gimphelp-ids.h"
#include "widgets/gimppropwidgets.h"
#include "widgets/gimpwidgets-utils.h"

#include "gimpdodgeburntool.h"
#include "gimppaintoptions-gui.h"
#include "gimptoolcontrol.h"

#include "gimp-intl.h"


static void   gimp_dodge_burn_tool_class_init (GimpDodgeBurnToolClass *klass);
static void   gimp_dodge_burn_tool_init       (GimpDodgeBurnTool      *dodgeburn);

static void   gimp_dodge_burn_tool_modifier_key  (GimpTool        *tool,
                                                  GdkModifierType  key,
                                                  gboolean         press,
                                                  GdkModifierType  state,
                                                  GimpDisplay     *gdisp);
static void   gimp_dodge_burn_tool_cursor_update (GimpTool        *tool,
                                                  GimpCoords      *coords,
                                                  GdkModifierType  state,
                                                  GimpDisplay     *gdisp);

static GtkWidget * gimp_dodge_burn_options_gui   (GimpToolOptions *tool_options);


static GimpPaintToolClass *parent_class = NULL;


void
gimp_dodge_burn_tool_register (GimpToolRegisterCallback  callback,
                               gpointer                  data)
{
  (* callback) (GIMP_TYPE_DODGE_BURN_TOOL,
                GIMP_TYPE_DODGE_BURN_OPTIONS,
                gimp_dodge_burn_options_gui,
                GIMP_PAINT_OPTIONS_CONTEXT_MASK,
                "gimp-dodge-burn-tool",
                _("Dodge/Burn"),
                _("Dodge or Burn strokes"),
                N_("Dod_geBurn"), "<shift>D",
                NULL, GIMP_HELP_TOOL_DODGE_BURN,
                GIMP_STOCK_TOOL_DODGE,
                data);
}

GType
gimp_dodge_burn_tool_get_type (void)
{
  static GType tool_type = 0;

  if (!tool_type)
    {
      static const GTypeInfo tool_info =
      {
        sizeof (GimpDodgeBurnToolClass),
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) gimp_dodge_burn_tool_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data     */
        sizeof (GimpDodgeBurnTool),
        0,              /* n_preallocs    */
        (GInstanceInitFunc) gimp_dodge_burn_tool_init,
      };

      tool_type = g_type_register_static (GIMP_TYPE_PAINT_TOOL,
                                          "GimpDodgeBurnTool",
                                          &tool_info, 0);
    }

  return tool_type;
}

static void
gimp_dodge_burn_tool_class_init (GimpDodgeBurnToolClass *klass)
{
  GimpToolClass	*tool_class = GIMP_TOOL_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);

  tool_class->modifier_key  = gimp_dodge_burn_tool_modifier_key;
  tool_class->cursor_update = gimp_dodge_burn_tool_cursor_update;
}

static void
gimp_dodge_burn_tool_init (GimpDodgeBurnTool *dodgeburn)
{
  GimpTool *tool = GIMP_TOOL (dodgeburn);

  gimp_tool_control_set_tool_cursor        (tool->control,
                                            GIMP_DODGE_TOOL_CURSOR);
  gimp_tool_control_set_toggle_tool_cursor (tool->control,
                                            GIMP_BURN_TOOL_CURSOR);
}

static void
gimp_dodge_burn_tool_modifier_key (GimpTool        *tool,
                                   GdkModifierType  key,
                                   gboolean         press,
                                   GdkModifierType  state,
                                   GimpDisplay     *gdisp)
{
  GimpDodgeBurnOptions *options;

  options = GIMP_DODGE_BURN_OPTIONS (tool->tool_info->tool_options);

  if (key == GDK_CONTROL_MASK &&
      ! (state & GDK_SHIFT_MASK)) /* leave stuff untouched in line draw mode */
    {
      switch (options->type)
        {
        case GIMP_DODGE:
          g_object_set (options, "type", GIMP_BURN, NULL);
          break;

        case GIMP_BURN:
          g_object_set (options, "type", GIMP_DODGE, NULL);
          break;

        default:
          break;
        }
    }

  GIMP_TOOL_CLASS (parent_class)->modifier_key (tool, key, press, state, gdisp);
}

static void
gimp_dodge_burn_tool_cursor_update (GimpTool        *tool,
                                    GimpCoords      *coords,
                                    GdkModifierType  state,
                                    GimpDisplay     *gdisp)
{
  GimpDodgeBurnOptions *options;

  options = GIMP_DODGE_BURN_OPTIONS (tool->tool_info->tool_options);

  gimp_tool_control_set_toggle (tool->control, (options->type == GIMP_BURN));

  GIMP_TOOL_CLASS (parent_class)->cursor_update (tool, coords, state, gdisp);
}


/*  tool options stuff  */

static GtkWidget *
gimp_dodge_burn_options_gui (GimpToolOptions *tool_options)
{
  GObject   *config;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *frame;
  gchar     *str;

  config = G_OBJECT (tool_options);

  vbox = gimp_paint_options_gui (tool_options);

  /* the type (dodge or burn) */
  str = g_strdup_printf (_("Type  %s"), gimp_get_mod_name_control ());

  frame = gimp_prop_enum_radio_frame_new (config, "type",
                                          str, 0, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  g_free (str);

  /*  mode (highlights, midtones, or shadows)  */
  frame = gimp_prop_enum_radio_frame_new (config, "mode",
                                          _("Mode"), 0, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  /*  the exposure scale  */
  table = gtk_table_new (1, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  gimp_prop_scale_entry_new (config, "exposure",
                             GTK_TABLE (table), 0, 0,
                             _("Exposure:"),
                             1.0, 10.0, 1,
                             FALSE, 0.0, 0.0);

  return vbox;
}
