/*
   This file is part of TALER
   Copyright (C) 2024, 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_select_all_donau_instances.c
 * @brief Implementation of the select_donau_instance function for Postgres
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_select_all_donau_instances.h"
#include "taler_merchant_donau.h"
#include "pg_helper.h"

/**
 * Context for select_donau_instances().
 */
struct SelectDonauInstanceContext
{
  /**
   * Function to call with the results.
   */
  TALER_MERCHANTDB_DonauInstanceCallback cb;

  /**
   * Closure for @e cb.
   */
  void *cb_cls;

  /**
   * Did database result extraction fail?
   */
  bool extract_failed;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about donau instances.
 *
 * @param[in, out] cls of type `struct SelectDonauInstanceContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
select_donau_instance_cb (void *cls,
                          PGresult *result,
                          unsigned int num_results)
{
  struct SelectDonauInstanceContext *sdc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    uint64_t donau_instance_serial;
    char *donau_url;
    char *charity_name;
    struct DONAU_CharityPublicKeyP charity_pub_key;
    uint64_t charity_id;
    struct TALER_Amount charity_max_per_year;
    struct TALER_Amount charity_receipts_to_date;
    int64_t current_year;
    json_t *donau_keys_json = NULL;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_uint64 ("donau_instances_serial",
                                    &donau_instance_serial),
      GNUNET_PQ_result_spec_string ("donau_url",
                                    &donau_url),
      GNUNET_PQ_result_spec_string ("charity_name",
                                    &charity_name),
      GNUNET_PQ_result_spec_auto_from_type ("charity_pub_key",
                                            &charity_pub_key),
      GNUNET_PQ_result_spec_uint64 ("charity_id",
                                    &charity_id),
      TALER_PQ_result_spec_amount_with_currency ("charity_max_per_year",
                                                 &charity_max_per_year),
      TALER_PQ_result_spec_amount_with_currency ("charity_receipts_to_date",
                                                 &charity_receipts_to_date),
      GNUNET_PQ_result_spec_int64 ("current_year",
                                   &current_year),
      GNUNET_PQ_result_spec_allow_null (
        TALER_PQ_result_spec_json ("keys_json",
                                   &donau_keys_json),
        NULL),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      sdc->extract_failed = true;
      return;
    }
    sdc->cb (sdc->cb_cls,
             donau_instance_serial,
             donau_url,
             charity_name,
             &charity_pub_key,
             charity_id,
             &charity_max_per_year,
             &charity_receipts_to_date,
             current_year,
             donau_keys_json);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_select_all_donau_instances (void *cls,
                                   TALER_MERCHANTDB_DonauInstanceCallback cb,
                                   void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct SelectDonauInstanceContext sdc = {
    .cb = cb,
    .cb_cls = cb_cls,
    /* Can be overwritten by the select_donau_instance_cb */
    .extract_failed = false,
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "select_all_donau_instances",
           "SELECT"
           " di.donau_instances_serial"
           ",di.donau_url"
           ",di.charity_name"
           ",mi.merchant_pub AS charity_pub_key"
           ",di.charity_id"
           ",di.charity_max_per_year"
           ",di.charity_receipts_to_date"
           ",di.current_year"
           ",dk.keys_json"
           " FROM merchant_donau_instances di"
           " LEFT JOIN merchant_donau_keys dk"
           "   ON di.donau_url = dk.donau_url"
           " JOIN merchant_instances mi"
           "   ON di.merchant_instance_serial = mi.merchant_serial");
  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             "select_all_donau_instances",
                                             params,
                                             &select_donau_instance_cb,
                                             &sdc);

  /* If there was an error inside select_donau_instance_cb, return a hard error. */
  if (sdc.extract_failed)
    return GNUNET_DB_STATUS_HARD_ERROR;

  return qs;
}
