/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { j2s } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { createSimpleTestkudosEnvironmentV3 } from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

/**
 * Test if the wallet handles outdated exchange versions correctly.
 */
export async function runExchangeManagementTest(
  t: GlobalTestState,
): Promise<void> {
  // Set up test environment

  const { walletClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t);

  // Since the default exchanges can change, we start the wallet in tests
  // with no built-in defaults. Thus the list of exchanges is empty here.
  const exchangesListResult = await walletClient.call(
    WalletApiOperation.ListExchanges,
    {},
  );

  t.assertDeepEqual(exchangesListResult.exchanges.length, 0);

  await walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
    exchangeBaseUrl: exchange.baseUrl,
    force: true,
  });

  const exchangesListResult2 = await walletClient.call(
    WalletApiOperation.ListExchanges,
    {},
  );

  t.assertDeepEqual(exchangesListResult2.exchanges.length, 1);

  await walletClient.call(WalletApiOperation.DeleteExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  const exchangesListResult3 = await walletClient.call(
    WalletApiOperation.ListExchanges,
    {},
  );

  t.assertDeepEqual(exchangesListResult3.exchanges.length, 0);

  // Check for regression: Can we re-add a deleted exchange?

  await walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
    exchangeBaseUrl: exchange.baseUrl,
    force: true,
  });

  const exchangesListResult4 = await walletClient.call(
    WalletApiOperation.ListExchanges,
    {},
  );

  t.assertDeepEqual(exchangesListResult4.exchanges.length, 1);

  await walletClient.call(WalletApiOperation.DeleteExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  const url = new URL(exchange.baseUrl);

  {
    await t.assertThrowsAsync(async () => {
      await walletClient.call(WalletApiOperation.AddExchange, {
        uri: `${url.hostname}:${url.port}`,
      });
    });
  }

  {
    const err = await t.assertThrowsTalerErrorAsync(async () => {
      const res = await walletClient.call(WalletApiOperation.AddExchange, {
        uri: `${url.hostname}:${url.port}`,
        allowCompletion: true,
      });
    });
    console.log(j2s(err));
    // Tries to complete to https://, but we use http in the local test.
    t.assertDeepEqual(
      (err.errorDetail as any).innerError.requestUrl,
      "https://localhost:8081/keys",
    );
  }
}

runExchangeManagementTest.suites = ["wallet", "exchange"];
