/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  HttpStatusCode,
  TalerError,
  TalerMerchantApi,
  TokenInfo,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../../components/exception/loading.js";
import { useInstanceAccessTokens } from "../../../../hooks/access-tokens.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { CardTable } from "./Table.js";
import { NotificationCard } from "../../../../components/menu/index.js";
import { ConfirmModal } from "../../../../components/modal/index.js";
import { useSessionContext } from "../../../../context/session.js";
import { Notification } from "../../../../utils/types.js";
import { useTranslationContext } from "@gnu-taler/web-util/browser";

interface Props {
  onCreate: () => void;
}

export default function AccessTokenListPage({ onCreate }: Props): VNode {
  const result = useInstanceAccessTokens();
  const { state, lib } = useSessionContext();
  const [deleting, setDeleting] = useState<TokenInfo | null>(null);
  const { i18n } = useTranslationContext();
  const [notif, setNotif] = useState<Notification | undefined>(undefined);

  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      case HttpStatusCode.Forbidden: {
        return <LoginPage />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  return (
    <Fragment>
      <section class="section is-main-section">
        <NotificationCard notification={notif} />

        <CardTable
          tokens={result.body.map((o) => ({
            ...o,
            id: String(o.serial),
          }))}
          onLoadMoreBefore={result.loadFirst}
          onLoadMoreAfter={result.loadNext}
          onCreate={onCreate}
          onDelete={(d) => {
            setDeleting(d);
          }}
        />
        {deleting && (
          <ConfirmModal
            label={`Delete access token`}
            description={deleting.description}
            danger
            active
            onCancel={() => setDeleting(null)}
            onConfirm={async (): Promise<void> => {
              try {
                const resp = await lib.instance.deleteAccessToken(
                  state.token,
                  deleting.serial,
                );
                if (resp.type === "ok") {
                  setNotif({
                    message: i18n.str`Access token has been deleted`,
                    type: "SUCCESS",
                  });
                } else {
                  setNotif({
                    message: i18n.str`Could not delete the access token`,
                    type: "ERROR",
                    description: resp.detail?.hint,
                  });
                }
              } catch (error) {
                setNotif({
                  message: i18n.str`Could not delete the access token`,
                  type: "ERROR",
                  description:
                    error instanceof Error ? error.message : undefined,
                });
              }
              setDeleting(null);
            }}
          >
            <p class="warning">
              <i18n.Translate>
                Deleting an access token cannot be undone.
              </i18n.Translate>
            </p>
          </ConfirmModal>
        )}
      </section>
    </Fragment>
  );
}
