/*
** Copyright (c) 1998 by Timothy Gerard Endres
** <mailto:time@ice.com>  <http://www.ice.com>
** 
** This program is free software.
** 
** You may redistribute it and/or modify it under the terms of the GNU
** General Public License as published by the Free Software Foundation.
** Version 2 of the license should be included with this distribution in
** the file LICENSE, as well as License.html. If the license is not
** included	with this distribution, you may find a copy at the FSF web
** site at 'www.gnu.org' or 'www.fsf.org', or you may write to the
** Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139 USA.
**
** THIS SOFTWARE IS PROVIDED AS-IS WITHOUT WARRANTY OF ANY KIND,
** NOT EVEN THE IMPLIED WARRANTY OF MERCHANTABILITY. THE AUTHOR
** OF THIS SOFTWARE, ASSUMES _NO_ RESPONSIBILITY FOR ANY
** CONSEQUENCE RESULTING FROM THE USE, MODIFICATION, OR
** REDISTRIBUTION OF THIS SOFTWARE. 
** 
*/

package com.ice.sqlclient;

import java.sql.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;
import com.sun.java.swing.*;
import com.sun.java.swing.border.*;
import com.sun.java.swing.event.ChangeEvent;
import com.sun.java.swing.event.ChangeListener;

import com.ice.util.AWTUtilities;
import com.ice.util.StringUtilities;
import com.ice.util.UserProperties;


public class
HostInfoPanel extends JPanel
		implements ActionListener, ChangeListener, ItemListener
	{
	// name:hostname:port:user:pass:database:handler
	private static final int		PROFILE_NAME = 0;
	private static final int		PROFILE_HOST = 1;
	private static final int		PROFILE_PORT = 2;
	private static final int		PROFILE_USER = 3;
	private static final int		PROFILE_PASS = 4;
	private static final int		PROFILE_DB = 5;
	private static final int		PROFILE_HAND = 6;

	private JTextField		hostText;
	private JTextField		dbText;
	private JTextField		portText;
	private JTextField		userText;
	private JTextField		passText;
	private JTextArea		statusText;
	private JComboBox		profileList;
	private JComboBox		handlerList;

	private JButton			connButton;
	private JButton			disconnButton;

	private MainPanel		mainPanel;

	private Vector			profileNames;
	private Hashtable		profileTable;


	public
	HostInfoPanel( MainPanel mainPanel )
		{
		this.mainPanel = mainPanel;

		this.loadProfiles();

		this.setDoubleBuffered( true );

		this.establishContents();
		}

	public void
	stateChanged( ChangeEvent event )
		{
		}

	public void
	actionPerformed( ActionEvent event )
		{
		String command = event.getActionCommand();

		if ( command.equals( "CONNECT" ) )
			{
			boolean ok = true;
			this.clearStatusMessage();

			String handlerName = (String)
				this.handlerList.getSelectedItem();

			SQLClientHandler handler =
				SQLClientHandler.getHandler( handlerName );

			if ( handler == null )
				{
				this.appendStatusMessage
					( "Could not find handler '"
						+ handlerName + "'." );
				return;
				}

			if ( handler.isOpen() )
				{
				this.appendStatusMessage
					( "Connection is already open." );
				return;
				}

			String qResult;
			String port = this.portText.getText();
			String hostname = this.hostText.getText();
			String database = this.dbText.getText();
			String username = this.userText.getText();

			String password = this.passText.getText();
			/*
			if ( ! this.passCheck.isSelected() )
				password = null;
			*/

			this.appendStatusMessage( "** Input Parameters:" );
			this.appendStatusMessage( "-- DB       '" + database + "'" );
			this.appendStatusMessage( "-- Host     '" + hostname + "'" );
			this.appendStatusMessage( "-- Port     '" + port + "'" );
			this.appendStatusMessage( "-- Username '" + username + "'" );
			this.appendStatusMessage( "-- Password '" + password + "'" );
			this.appendStatusMessage( "-- Handler  '" + handlerName + "'" );
			this.appendStatusMessage( "" );

			handler.setConnectInfo
				( hostname, port, username, password, database );

			this.appendStatusMessage
				( "Opening the '" + handler.getName()
					+ "' connection..." );

			try {
				handler.openConnection();
				this.appendHandlerDetails( handler );
				SQLClientHandler.setCurrentHandler( handler );
				this.mainPanel.adjustConnectedTabs( true );

				String[] dbList = handler.getDatabaseList();
				MainFrame.showDatabaseMenu( dbList, database );

				this.connButton.setEnabled( false );
				this.disconnButton.setEnabled( true );
				}
			catch ( SQLException ex )
				{
				ok = false;
				this.appendStatusMessage
					( "SQLException: while opening connection:\n"
						+ "\t" + ex.getMessage() );
				}

			if ( ok )
				{
				this.mainPanel.dbHasChanged( database );
				this.appendStatusMessage
					( "The connection has been "
						+ "successfully estabilshed." );
				}
			else
				{
				this.appendStatusMessage
					( "The connection has not been estabilshed." );
				}
			}
		else if ( command.equals( "DISCONNECT" ) )
			{
			MainFrame.hideDatabaseMenu();
			this.connButton.setEnabled( true );
			this.disconnButton.setEnabled( false );

			this.mainPanel.adjustConnectedTabs( false );

			this.clearStatusMessage();

			SQLClientHandler handler =
				SQLClientHandler.getCurrentHandler();

			if ( handler == null )
				{
				this.appendStatusMessage
					( "Current handler is null." );
				}
			else if ( ! handler.isOpen() )
				{
				this.appendStatusMessage
					( "The connection is not currently open." );
				}
			else
				{
				handler.closeConnection();

				this.appendStatusMessage
					( "The handler was successfully closed." );
				}
			}
		else
			{
			this.appendStatusMessage
				( "INTERNAL ERROR: UNKNOWN Command '"
					+ command + "'" );
			}
		}

	public void
	clearStatusMessage()
		{
		this.statusText.setText( "" );
		}

	public void
	setStatusMessage( String message )
		{
		this.statusText.setText( message );
		}

	public void
	appendStatusMessage( String message )
		{
		this.statusText.append( message );

		if ( ! message.endsWith( "\n" ) )
			this.statusText.append( "\n" );
		}

	private void
	appendHandlerDetails( SQLClientHandler handler )
		{
		this.appendStatusMessage
			( "** JDBC Driver and Datbase Information:" );

		try {
			DatabaseMetaData dma = handler.getMetaData();

			this.appendStatusMessage
				( "-- URL      " + dma.getURL() );
			this.appendStatusMessage
				( "-- Driver   " + dma.getDriverName() );
			this.appendStatusMessage
				( "-- Version  " + dma.getDriverVersion() );
			this.appendStatusMessage
				( "-- Database " + dma.getDatabaseProductName() );
			this.appendStatusMessage
				( "-- Version  " + dma.getDatabaseProductVersion() );
			this.appendStatusMessage
				( "-- ReadOnly " +
					( dma.isReadOnly() ? "Yes" : "No" ) );
			}
		catch ( SQLException ex )
			{
			this.appendStatusMessage
				( "-- Driver does not support Database Meta Data." );
			}
		}

	private void
	establishContents()
		{
		JLabel	lbl;
		int		row = 0;

		this.setLayout( new GridBagLayout() );

		JPanel infoPanel = new JPanel();
		infoPanel.setLayout( new GridBagLayout() );
		infoPanel.setBorder(
			new CompoundBorder(
				new EtchedBorder( EtchedBorder.RAISED ),
				new EmptyBorder( 6, 6, 6, 6 ) ) );
		AWTUtilities.constrain(
			this, infoPanel,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.NORTH,
			0, 0, 1, 1, 1.0, 0.0 );

		lbl = new JLabel( "Hostname:" );
		AWTUtilities.constrain(
			infoPanel, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			0, row, 1, 1, 0.0, 0.0 );
		
		String sqlHostname =
			UserProperties.getProperty( "sqlHostname", "" );

		this.hostText = new JTextField( sqlHostname );
		AWTUtilities.constrain(
			infoPanel, this.hostText,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			1, row, 1, 1, 1.0, 0.0 );
		
		++row;

		lbl = new JLabel( "Port:" );
		AWTUtilities.constrain(
			infoPanel, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			0, row, 1, 1, 0.0, 0.0 );
		
		String sqlPort =
			UserProperties.getProperty( "sqlPort", "3306" );

		this.portText = new JTextField( sqlPort );
		AWTUtilities.constrain(
			infoPanel, this.portText,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			1, row, 1, 1, 1.0, 0.0 );
		
		++row;

		lbl = new JLabel( "Username:" );
		AWTUtilities.constrain(
			infoPanel, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			0, row, 1, 1, 0.0, 0.0 );
		
		String sqlUsername =
			UserProperties.getProperty( "sqlUsername", null );

		if ( sqlUsername == null )
			sqlUsername =
				UserProperties.getProperty
					( "user.name", "" );

		this.userText = new JTextField( sqlUsername );
		AWTUtilities.constrain(
			infoPanel, this.userText,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			1, row, 1, 1, 1.0, 0.0 );
		
		++row;

		lbl = new JLabel( "Password:" );
		AWTUtilities.constrain(
			infoPanel, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			0, row, 1, 1, 0.0, 0.0 );
		
		/*
		JPanel passPan = new JPanel();
		passPan.setLayout( new GridBagLayout() );
		AWTUtilities.constrain(
			infoPanel, passPan,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.WEST,
			1, row, 1, 1, 1.0, 0.0 );
		*/

		String sqlPassword =
			UserProperties.getProperty( "sqlPassword", "" );

		this.passText = new JTextField( sqlPassword );
//		this.passText.setEnabled( false );
		AWTUtilities.constrain(
			infoPanel, this.passText,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.WEST,
			1, row, 1, 1, 1.0, 0.0 );
		/*
		this.passCheck = new JCheckBox();
		this.passCheck.setSelected( false );
		this.passCheck.addChangeListener( this );
		AWTUtilities.constrain(
			passPan, this.passCheck,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			2, 0, 1, 1, 0.0, 0.0 );
		*/
		++row;

		lbl = new JLabel( "Database:" );
		AWTUtilities.constrain(
			infoPanel, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			0, row, 1, 1, 0.0, 0.0 );
		
		String sqlDatabase =
			UserProperties.getProperty( "sqlDatabase", "mysql" );

		this.dbText = new JTextField( sqlDatabase );
		AWTUtilities.constrain(
			infoPanel, this.dbText,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			1, row, 1, 1, 1.0, 0.0 );
		
		++row;

		lbl = new JLabel( "SQL Handler:" );
		AWTUtilities.constrain(
			infoPanel, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.WEST,
			0, row, 1, 1, 0.0, 0.0 );
		
		this.handlerList = new JComboBox();
		String[] names = SQLClientHandler.getAllHandlerNames();
		for ( int i = 0 ; i < names.length ; ++i )
			{
			this.handlerList.addItem( names[i] );
			}

		AWTUtilities.constrain(
			infoPanel, this.handlerList,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			1, row, 1, 1, 1.0, 0.0 );
		
		++row;

		JPanel buttonPan = new JPanel();
		buttonPan.setLayout( new GridBagLayout() );
		AWTUtilities.constrain(
			this, buttonPan,
			GridBagConstraints.VERTICAL,
			GridBagConstraints.NORTH,
			1, 0, 1, 1, 0.0, 0.0 );

		buttonPan.setBorder
			( new CompoundBorder(
				new EtchedBorder( EtchedBorder.RAISED ),
				new EmptyBorder( 6, 6, 6, 6 ) ) );

		JPanel subPan;

		row = 0;

		lbl = new JLabel( "Profile:" );
		AWTUtilities.constrain(
			buttonPan, lbl,
			GridBagConstraints.NONE,
			GridBagConstraints.NORTHWEST,
			0, row++, 1, 1, 0.0, 0.0 );
		
		this.profileList = new JComboBox();
		this.profileList.setEditable( false );
		this.profileList.addItemListener( this );
		this.addProfiles( this.profileList );

		AWTUtilities.constrain(
			buttonPan, this.profileList,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.NORTH,
			0, row++, 1, 1, 1.0, 0.0 );

		subPan = new JPanel();
		subPan.setLayout( new GridBagLayout() );
		subPan.setBorder( new EmptyBorder( 6, 10, 4, 8 ) );
		AWTUtilities.constrain(
			buttonPan, subPan,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.SOUTH,
			0, row++, 1, 1, 1.0, 0.0 );

		this.connButton =
			new JButton( "Connect" );
		this.connButton.setActionCommand( "CONNECT" );
		this.connButton.addActionListener( this );
		this.connButton.setEnabled( true );
		AWTUtilities.constrain(
			subPan, this.connButton,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			0, 0, 1, 1, 1.0, 0.0 );

		subPan = new JPanel();
		subPan.setLayout( new GridBagLayout() );
		subPan.setBorder( new EmptyBorder( 6, 10, 4, 8 ) );
		AWTUtilities.constrain(
			buttonPan, subPan,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.SOUTH,
			0, row++, 1, 1, 1.0, 0.0 );

		this.disconnButton =
			new JButton( "DisConnect" );
		this.disconnButton.setActionCommand( "DISCONNECT" );
		this.disconnButton.addActionListener( this );
		this.disconnButton.setEnabled( false );
		AWTUtilities.constrain(
			subPan, this.disconnButton,
			GridBagConstraints.HORIZONTAL,
			GridBagConstraints.CENTER,
			0, 0, 1, 1, 1.0, 0.0 );

		this.statusText = new JTextArea();
		JScrollPane scroller = new JScrollPane();
		scroller.getViewport().add( this.statusText );
		scroller.setBorder( new EtchedBorder() );
		AWTUtilities.constrain(
			this, scroller,
			GridBagConstraints.BOTH,
			GridBagConstraints.CENTER,
			0, 1, 2, 1, 1.0, 1.0 );
		}

	public void
	itemStateChanged( ItemEvent event )
		{
		if ( event.getStateChange() == ItemEvent.SELECTED )
			{
			String profileName = (String)
				this.profileList.getSelectedItem();

			Vector profile = (Vector)
				this.profileTable.get( profileName );

			if ( profile == null )
				{
				System.err.println
					( "ERROR, why is profile '"
						+ profileName + "' null?" );
				}
			else
				{
				this.hostText.setText
					( (String)profile.elementAt( PROFILE_HOST ) );
				this.portText.setText
					( (String)profile.elementAt( PROFILE_PORT ) );
				this.userText.setText
					( (String)profile.elementAt( PROFILE_USER ) );
				this.passText.setText
					( (String)profile.elementAt( PROFILE_PASS ) );
				this.dbText.setText
					( (String)profile.elementAt( PROFILE_DB ) );

				if ( profile.size() > PROFILE_HAND )
					{
					this.handlerList.setSelectedItem
						( (String)profile.elementAt( PROFILE_HAND ) );
					}
				}
			}
		}

	private void
	addProfiles( JComboBox combo )
		{
		if ( this.profileNames.size() < 1 )
			{
			combo.setEnabled( false );
			return;
			}

		for ( int i = 0 ; i < profileNames.size() ; ++i )
			{
			String profileName = (String)
				this.profileNames.elementAt(i);

			combo.addItem( profileName );
			}

		String defaultProfile =
			UserProperties.getProperty( "sqlProfile", null );

		if ( defaultProfile == null )
			combo.setSelectedIndex( 0 );
		else
			combo.setSelectedItem( defaultProfile );
		}


	private void
	loadProfiles()
		{
		this.profileNames = new Vector();
		this.profileTable = new Hashtable();

		String profProp =
			UserProperties.getProperty( "profiles", null );

		if ( profProp == null )
			return;

		String[] profList =
			StringUtilities.splitString( profProp, ":" );

		if ( profList.length < 1 )
			return;

		for ( int i = 0 ; i < profList.length ; ++i )
			{
			String profileName = profList[i];

			String profStr = 
				UserProperties.getProperty
					( "profile." + profileName, null );
			
			if ( profStr == null )
				{
				System.err.println
					( "ERROR profile definition '" + profileName
						+ "' not found." );
				continue;
				}

			Vector fields =
				StringUtilities.vectorString( profStr, ":" );

			if ( fields == null || fields.size() < PROFILE_HAND )
				{
				System.err.println
					( "ERROR profile definition '" + profileName
						+ "' has only " + fields.size() + " fields." );
				continue;
				}

			this.profileNames.addElement( profileName );
			this.profileTable.put( profileName, fields );
			}
		}

	}


