package Net::DNS::SEC::Private;

#
# $Id: Private.pm 1705 2018-08-23 10:24:02Z willem $
#
our $VERSION = (qw$LastChangedRevision: 1705 $)[1];


=head1 NAME

Net::DNS::SEC::Private - DNSSEC Private key object


=head1 SYNOPSIS

    use Net::DNS::SEC::Private;

    $private = new Net::DNS::SEC::Private( $keypath );

    $private = new Net::DNS::SEC::Private(
	'algorithm'  => '13',
	'keytag'     => '26512',
	'privatekey' => 'h/mc+iq9VDUbNAjQgi8S8JzlEX29IALchwJmNM3QYKk=',
	'signame'    => 'example.com.'
	);


=head1 DESCRIPTION

Class representing private keys as read from a keyfile generated by BIND
dnssec-keygen. The class is written to be used only in the context of the
Net::DNS::RR::RRSIG create method. This class is not designed to interact
with any other system.

=cut


use strict;
use integer;
use warnings;
use File::Spec;
use IO::File;


sub new { scalar(@_) > 2 ? &_new_params : &_new_keyfile }

sub _new_keyfile {
	my ( $class, $file ) = @_;

	my ( $vol, $dir, $keyname ) = File::Spec->splitpath($file);

	# Format something like: /Kbla.foo.+001+12345.private'
	# as created by BIND keygen.
	# We determine the algorithm from the filename.
	die "$keyname does not appear to be a BIND private key"
			unless $keyname =~ /^K(.*\.)\+(\d+)\+(\d+)\.private$/;
	my @identifier = ( signame => $1, algorithm => 0 + $2, keytag => 0 + $3 );

	my $handle = new IO::File($file) or die qq(open: "$file" $!);

	my @content;
	local $_;
	while (<$handle>) {
		chomp;
		next if /^$/;
		next if /^\s*[;]/;
		s/\(.+\)//;
		my ( $name, $value ) = split;
		push @content, $name, $value;
	}

	return $class->_new_params( @content, @identifier );
}


sub _new_params {
	my $class = shift;
	my $param = {};
	while (@_) {
		my $name = shift @_;
		$name =~ tr/A-Za-z0-9\000-\377/a-za-z0-9/d;
		$param->{$name} = shift @_;
	}

	my $self = bless sub { $param->{shift()} }, $class;
	die 'no algorithm specified' unless $self->algorithm;
	die 'no signame specified'   unless $self->signame;
	return $self;
}


our $AUTOLOAD;

sub AUTOLOAD {				## Default method
	my ($self) = @_;
	no strict q/refs/;

	my ($attribute) = $AUTOLOAD =~ m/::([^:]*)$/;
	$attribute =~ tr/A-Za-z0-9\000-\377/a-za-z0-9/d;

	# Build a method in the class
	*{$AUTOLOAD} = sub { &{shift()}($attribute) };

	# and jump to it
	goto &{$AUTOLOAD};
}


1;
__END__


=head1 METHODS

=head2 new (from private keyfile)

    $keypath = '/home/foo/Kexample.com.+013+26512.private';
    $private = new Net::DNS::SEC::Private( $keypath );

The argument is the full path to a private key file generated by the
BIND dnssec-keygen tool.  Note that the filename contains information
about the algorithm and keytag.


=head2 new (from private key parameters)

    $private = new Net::DNS::SEC::Private(
	'algorithm'  => '13',
	'keytag'     => '26512',
	'privatekey' => 'h/mc+iq9VDUbNAjQgi8S8JzlEX29IALchwJmNM3QYKk=',
	'signame'    => 'example.com.'
	);

The arguments define the private key parameters as (name,value) pairs.
The name and data representation are identical to that used in a BIND
private keyfile.


=head2 private_key_format

    $format = $private->private_key_format;

Returns a string which identifies the format of the private key file.


=head2 algorithm, keytag, signame
 
    $algorithm = $private->algorithm;
    $keytag    = $private->keytag;
    $signame   = $private->signame;

Returns the corresponding attribute determined from the filename.


=head2 Private key attributes

    $attribute = $private->attribute;

Returns the value as it appears in the private key file.
The attribute names correspond to the tag in the key file, modified
to form an acceptable Perl subroutine name.


=head2 created, publish, activate

    $created  = $private->created;
    $publish  = $private->publish;
    $activate = $private->activate;

Returns a string which represents a date in the form 20141212123456.
Returns undefined value for key formats older than v1.3.


=head1 COPYRIGHT

Copyright (c)2014,2018 Dick Franks

All Rights Reserved


=head1 LICENSE

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the author not be used in advertising
or publicity pertaining to distribution of the software without specific
prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.


=head1 SEE ALSO

L<perl>, L<Net::DNS>, L<Net::DNS::SEC>,
L<Net::DNS::RR::DNSKEY>, L<Net::DNS::RR::RRSIG>

=cut

