/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util;

import ghidra.app.util.importer.MemoryConflictHandler;
import ghidra.framework.store.LockException;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.AddressRange;
import ghidra.program.model.address.AddressRangeIterator;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.ProgramFragment;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.mem.MemoryBlockException;
import ghidra.program.model.mem.MemoryConflictException;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolIterator;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.util.AddressLabelInfo;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.NotFoundException;
import ghidra.util.task.TaskMonitor;
import ghidra.util.task.TaskMonitorAdapter;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

public class MemoryBlockUtil {
    private static final byte MEMORY_CONFLICT_UNKNOWN = 0;
    private static final byte MEMORY_CONFLICT_IGNORE = 1;
    private static final byte MEMORY_CONFLICT_REMOVE_WHOLE = 2;
    private static final byte MEMORY_CONFLICT_REMOVE_1ST_HALF = 3;
    private static final byte MEMORY_CONFLICT_REMOVE_2ND_HALF = 4;
    private static final byte MEMORY_CONFLICT_REMOVE_MIDDLE = 5;
    private Listing listing;
    private Memory memory;
    private SymbolTable symbolTable;
    private MemoryConflictHandler handler;
    private StringBuffer messages;

    public MemoryBlockUtil(Program program, MemoryConflictHandler handler) {
        this.listing = program.getListing();
        this.memory = program.getMemory();
        this.symbolTable = program.getSymbolTable();
        this.handler = handler;
        this.messages = new StringBuffer();
    }

    public String getMessages() {
        return this.messages.toString();
    }

    public void dispose() {
        this.listing = null;
        this.memory = null;
        this.handler = null;
    }

    public MemoryBlock createMappedBlock(boolean isBitMapped, String name, Address start, Address base, int length, String comment, String source, boolean r, boolean w, boolean x) {
        try {
            MemoryBlock block = isBitMapped ? this.memory.createBitMappedBlock(name, start, base, (long)length) : this.memory.createByteMappedBlock(name, start, base, (long)length);
            block.setComment(comment);
            block.setSourceName(source);
            block.setRead(r);
            block.setWrite(w);
            block.setExecute(x);
            this.renameFragment(start, name);
            return block;
        }
        catch (LockException e) {
            this.appendMessage("Failed to create '" + name + "' mapped memory block: exclusive lock/checkout required");
        }
        catch (MemoryConflictException e) {
            this.appendMessage("Failed to create '" + name + "' mapped memory block: " + e.getMessage());
        }
        catch (AddressOverflowException e) {
            this.appendMessage("Failed to create '" + name + "' mapped memory block: " + e.getMessage());
        }
        return null;
    }

    public MemoryBlock createUninitializedBlock(boolean isOverlay, String name, Address start, long length, String comment, String source, boolean r, boolean w, boolean x) {
        try {
            MemoryBlock block = this.memory.createUninitializedBlock(name, start, length, isOverlay);
            this.setBlockAttributes(block, comment, source, r, w, x);
            this.renameFragment(start, name);
            return block;
        }
        catch (LockException e) {
            this.appendMessage("Failed to create '" + name + "' memory block: exclusive lock/checkout required");
        }
        catch (DuplicateNameException e) {
            this.appendMessage("Failed to create '" + name + "' memory block: " + e.getMessage());
        }
        catch (MemoryConflictException e) {
            this.appendMessage("Failed to create '" + name + "' memory block: " + e.getMessage());
        }
        catch (AddressOverflowException e) {
            this.appendMessage("Failed to create '" + name + "' memory block: " + e.getMessage());
        }
        return null;
    }

    public MemoryBlock createInitializedBlock(String name, Address start, byte[] data, String comment, String source, boolean r, boolean w, boolean x, TaskMonitor monitor) throws AddressOverflowException {
        return this.createInitializedBlock(name, start, new ByteArrayInputStream(data), data.length, comment, source, r, w, x, monitor);
    }

    public MemoryBlock createInitializedBlock(String name, Address start, InputStream dataInput, long dataLength, String comment, String source, boolean r, boolean w, boolean x, TaskMonitor monitor) throws AddressOverflowException {
        if (!this.memory.getProgram().hasExclusiveAccess()) {
            this.appendMessage("Failed to create '" + name + "' memory block: exclusive access/checkout required");
            return null;
        }
        MemoryBlock firstBlock = null;
        try {
            int blockNum = 0;
            long blockLength = 0L;
            while (dataLength > 0L) {
                start = start.add(blockLength);
                blockLength = Math.min(dataLength, 0x40000000L);
                String blockName = this.getBlockName(name, blockNum);
                monitor.setMessage("Creating memory block \"" + blockName + "\" at 0x" + start + "...");
                try {
                    MemoryBlock block = this.memory.createInitializedBlock(blockName, start, dataInput, blockLength, monitor, false);
                    this.setBlockAttributes(block, comment, source, r, w, x);
                    this.renameFragment(start, blockName);
                    firstBlock = firstBlock == null ? block : firstBlock;
                }
                catch (MemoryConflictException memExc) {
                    this.handleMemoryConflict(blockName, start, dataInput, blockLength, comment, source, r, w, x);
                }
                ++blockNum;
                dataLength -= blockLength;
            }
            return firstBlock;
        }
        catch (LockException e) {
            throw new RuntimeException(e);
        }
        catch (CancelledException e) {
        }
        catch (DuplicateNameException e) {
            throw new RuntimeException(e);
        }
        return null;
    }

    private String getBlockName(String name, int blockNum) {
        if (blockNum == 0) {
            return name;
        }
        return name + "." + blockNum;
    }

    private void handleMemoryConflict(String name, Address start, InputStream dataInput, long dataLength, String comment, String source, boolean r, boolean w, boolean x) throws AddressOverflowException {
        Address end = start.addNoWrap(dataLength - 1L);
        List<AddressLabelInfo> removedSymbolList = this.resolveConflicts(start, end);
        AddressSet set = new AddressSet(start, end);
        MemoryBlock[] existingBlocks = this.memory.getBlocks();
        for (int i = 0; i < existingBlocks.length; ++i) {
            set.deleteRange(existingBlocks[i].getStart(), existingBlocks[i].getEnd());
        }
        this.appendMessage("WARNING!!\n\tMemory block [" + name + "] has caused an address collision.\n\tAddress range automatically changed from [" + start + "," + end + "] to " + set.toString());
        ArrayList<MemoryBlock> newBlocks = this.createNeededBlocks(name, comment, source, r, w, x, set);
        this.restoreSymbols(removedSymbolList);
        boolean shouldOverwriteBlock = true;
        if (existingBlocks.length > 0) {
            shouldOverwriteBlock = this.handler.allowOverwrite(start, end);
        }
        byte[] data = new byte[(int)dataLength];
        try {
            dataInput.read(data);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        this.restoreBytes(start, end, newBlocks, shouldOverwriteBlock, data);
    }

    private void restoreBytes(Address start, Address end, ArrayList<MemoryBlock> newBlocks, boolean shouldOverwriteBlock, byte[] data) {
        if (shouldOverwriteBlock) {
            this.listing.clearCodeUnits(start, end, false);
            this.appendMessage("Cleared code units from start=0x" + start.toString() + " to end=0x" + end.toString());
            try {
                this.memory.setBytes(start, data);
                this.appendMessage("Overwrote memory from start=" + start + " length=" + data.length);
            }
            catch (MemoryAccessException exc) {
                this.appendMessage("Error Overwriting Bytes[1]: " + exc);
            }
        } else {
            try {
                for (int i = 0; i < newBlocks.size(); ++i) {
                    MemoryBlock mb = newBlocks.get(i);
                    this.memory.setBytes(mb.getStart(), data, (int)mb.getStart().subtract(start), (int)mb.getSize());
                }
            }
            catch (MemoryAccessException exc) {
                this.appendMessage("Error Overwriting Bytes[2]: " + exc);
            }
        }
    }

    private void restoreSymbols(List<AddressLabelInfo> removedSymbolList) {
        for (AddressLabelInfo info : removedSymbolList) {
            try {
                Symbol symbol = this.symbolTable.createLabel(info.getAddress(), info.getLabel(), info.getScope(), info.getSource());
                if (!info.isPrimary()) continue;
                symbol.setPrimary();
            }
            catch (Exception exception) {}
        }
    }

    private ArrayList<MemoryBlock> createNeededBlocks(String name, String comment, String source, boolean r, boolean w, boolean x, AddressSet set) {
        ArrayList<MemoryBlock> newBlocks = new ArrayList<MemoryBlock>();
        AddressRangeIterator iter = set.getAddressRanges();
        while (iter.hasNext()) {
            AddressRange range = (AddressRange)iter.next();
            byte[] bytes = new byte[(int)range.getLength()];
            MemoryBlock newBlock = null;
            try {
                newBlock = this.memory.createInitializedBlock(name, range.getMinAddress(), (InputStream)new ByteArrayInputStream(bytes), (long)bytes.length, null, false);
                this.setBlockAttributes(newBlock, comment, source, r, w, x);
            }
            catch (CancelledException cancelledException) {
            }
            catch (LockException e) {
                throw new RuntimeException(e);
            }
            catch (DuplicateNameException e) {
                throw new RuntimeException(e);
            }
            catch (MemoryConflictException e) {
                throw new RuntimeException(e);
            }
            catch (AddressOverflowException e) {
                throw new RuntimeException(e);
            }
            if (newBlock == null) continue;
            newBlock.setRead(r);
            newBlock.setWrite(w);
            newBlock.setExecute(x);
            newBlocks.add(newBlock);
        }
        return newBlocks;
    }

    private List<AddressLabelInfo> resolveConflicts(Address start, Address end) {
        ArrayList<AddressLabelInfo> symbolList = new ArrayList<AddressLabelInfo>();
        MemoryBlock[] blocks = this.memory.getBlocks();
        for (int i = 0; i < blocks.length; ++i) {
            if (blocks[i].isInitialized()) continue;
            int caseVal = this.getMemoryConflictCase(start, end, blocks[i]);
            try {
                switch (caseVal) {
                    case 3: {
                        this.memory.split(blocks[i], end.add(1L));
                        MemoryBlock secondHalf = this.memory.getBlock(end.add(1L));
                        try {
                            secondHalf.setName(blocks[i].getName());
                        }
                        catch (DuplicateNameException e) {
                            throw new AssertException((Throwable)e);
                        }
                        this.loadSymbolsFromBlock(symbolList, blocks[i]);
                        this.memory.removeBlock(blocks[i], TaskMonitorAdapter.DUMMY_MONITOR);
                        blocks[i] = secondHalf;
                        break;
                    }
                    case 4: {
                        this.memory.split(blocks[i], start);
                        MemoryBlock secondHalf = this.memory.getBlock(start);
                        this.loadSymbolsFromBlock(symbolList, secondHalf);
                        this.memory.removeBlock(secondHalf, TaskMonitorAdapter.DUMMY_MONITOR);
                        blocks[i] = this.memory.getBlock(blocks[i].getStart());
                        break;
                    }
                    case 2: {
                        this.loadSymbolsFromBlock(symbolList, blocks[i]);
                        this.memory.removeBlock(blocks[i], TaskMonitorAdapter.DUMMY_MONITOR);
                        break;
                    }
                    case 5: {
                        this.memory.split(blocks[i], start);
                        MemoryBlock middleBlock = this.memory.getBlock(start);
                        this.memory.split(middleBlock, end.add(1L));
                        MemoryBlock endBlock = this.memory.getBlock(end.add(1L));
                        try {
                            endBlock.setName(blocks[i].getName());
                        }
                        catch (DuplicateNameException e) {
                            throw new AssertException((Throwable)e);
                        }
                        this.loadSymbolsFromBlock(symbolList, middleBlock);
                        this.memory.removeBlock(middleBlock, TaskMonitorAdapter.DUMMY_MONITOR);
                        break;
                    }
                    case 1: {
                        break;
                    }
                    default: {
                        throw new RuntimeException("Unable to resolve memory confliction exception.");
                    }
                }
                continue;
            }
            catch (MemoryBlockException memoryBlockException) {
                continue;
            }
            catch (LockException lockException) {
                continue;
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
        }
        return symbolList;
    }

    private void loadSymbolsFromBlock(List<AddressLabelInfo> symbolList, MemoryBlock block) {
        Symbol symbol;
        SymbolIterator it = this.symbolTable.getSymbolIterator(block.getStart(), true);
        Address end = block.getEnd();
        while (it.hasNext() && (symbol = it.next()).getAddress().compareTo((Object)end) <= 0) {
            symbolList.add(new AddressLabelInfo(symbol));
        }
    }

    public MemoryBlock createOverlayBlock(String name, Address start, InputStream dataInput, long dataLength, String comment, String source, boolean r, boolean w, boolean x, TaskMonitor monitor) throws AddressOverflowException, DuplicateNameException {
        if (!this.memory.getProgram().hasExclusiveAccess()) {
            this.appendMessage("Failed to create '" + name + "' overlay memory block, exclusive access/checkout required");
            return null;
        }
        try {
            MemoryBlock block = this.memory.createInitializedBlock(name, start, dataInput, dataLength, monitor, true);
            this.setBlockAttributes(block, comment, source, r, w, x);
            this.renameFragment(block.getStart(), name);
            return block;
        }
        catch (LockException e) {
            throw new AssertException((Throwable)e);
        }
        catch (CancelledException e) {
        }
        catch (MemoryConflictException e) {
            throw new AssertException((Throwable)e);
        }
        return null;
    }

    private void setBlockAttributes(MemoryBlock block, String comment, String source, boolean r, boolean w, boolean x) {
        block.setComment(comment);
        block.setSourceName(source);
        block.setRead(r);
        block.setWrite(w);
        block.setExecute(x);
    }

    private int getMemoryConflictCase(Address start, Address end, MemoryBlock block) {
        if (block.isInitialized()) {
            return 1;
        }
        Address blockStart = block.getStart();
        Address blockEnd = block.getEnd();
        AddressSpace startSpace = start.getAddressSpace();
        if (!startSpace.equals(end.getAddressSpace())) {
            return 1;
        }
        if (!startSpace.equals(blockStart.getAddressSpace())) {
            return 1;
        }
        if (start.compareTo((Object)blockStart) <= 0 && end.compareTo((Object)blockEnd) >= 0) {
            return 2;
        }
        if (blockStart.compareTo((Object)start) < 0 && blockEnd.compareTo((Object)end) > 0) {
            return 5;
        }
        if (start.compareTo((Object)blockStart) <= 0 && end.compareTo((Object)blockStart) >= 0 && end.compareTo((Object)blockEnd) < 0) {
            return 3;
        }
        if (start.compareTo((Object)blockStart) > 0 && start.compareTo((Object)blockEnd) <= 0 && end.compareTo((Object)blockEnd) >= 0) {
            return 4;
        }
        if (start.compareTo((Object)blockEnd) > 0 || end.compareTo((Object)blockStart) < 0) {
            return 1;
        }
        return 0;
    }

    private void appendMessage(String msg) {
        if (this.messages.length() != 0) {
            this.messages.append('\n');
        }
        this.messages.append(msg);
    }

    private void renameFragment(Address blockStart, String blockName) {
        String[] treeNames = this.listing.getTreeNames();
        for (int i = 0; i < treeNames.length; ++i) {
            try {
                ProgramFragment frag = this.listing.getFragment(treeNames[i], blockStart);
                frag.setName(blockName);
                continue;
            }
            catch (DuplicateNameException duplicateNameException) {
                // empty catch block
            }
        }
    }
}

