/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.javaclass.format;

/**
 * A utility class defining access flags and access utility methods.
 * Access flags are as defined by the Java Virtual Machine Specification
 * Second Edition, tables 4.1, 4.4, 4.5 and 4.7.
 */
public abstract class AccessFlagsJava {

    /** Declared public, may be accessed from outside its package. */
    public static final int PUBLIC       = 0x0001;

    /** Declared private, usable only within the defining class. */
    public static final int PRIVATE      = 0x0002;

    /** Declared protected, may be accessed within subclasses. */
    public static final int PROTECTED    = 0x0004;

    /** Declared static. */
    public static final int STATIC       = 0x0008;

    /**
     * Declared final.  For classes this means no subclassing allowed.
     * For fields it means no further assignment allowed after initialization.
     * For methods it means that the method cannot be overridden.
     */
    public static final int FINAL        = 0x0010;

    /** Declared synchronized; invocation is wrapped in a monitor lock. */
    public static final int SYNCHRONIZED = 0x0020;

    /**
     * Treat superclass methods specially when invoked by the
     * <i>invokespecial</i> instruction.  This access only applies to
     * classes, and shares the same value as SYNCHRONIZED.
     */
    public static final int SUPER        = 0x0020;

    /** Declared volatile; cannot be cached. */
    public static final int VOLATILE     = 0x0040;

    /** A bridge method, generated by the compiler. */
    public static final int BRIDGE       = 0x0040;

    /** 
     * Declared transient; not written or read by a persistent object 
     * manager 
     */
    public static final int TRANSIENT    = 0x0080;

    /** Declared with a variable number of arguments. */
    public static final int VARARGS      = 0x0080;

    /** Declared native; implemented in a language other than Java. */
    public static final int NATIVE       = 0x0100;

    /** Is an interface, not a class. */
    public static final int INTERFACE    = 0x0200;

    /** Declared abstract; must not be instantiated. */
    public static final int ABSTRACT     = 0x0400;

    /** Declared strictfp; floating point mode is FP-strict. */
    public static final int STRICT       = 0x0800;

    /** Declared synthetic, not present in the source file. */
    public static final int SYNTHETIC    = 0x1000;

    /** Declared as an annotation type. */
    public static final int ANNOTATION   = 0x2000;

    /** 
     * For classes, declared as an enum type.  For fields, declared as
     * an element of an enum.
     */
    public static final int ENUM         = 0x4000;

    /**
     * Return a text representation for a given set of access flags.  
     * Here are some examples:
     * <DL>
     *  <DD><CODE>"public static final"</CODE>,</DD>
     *  <DD><CODE>"package private"</CODE>, or</DD>
     *  <DD><CODE>"protected transient"</CODE>.</DD>
     * </DL>
     * Note: only access flags that map to Java modifier keywords are returned.
     * @param access the mask of flags denoting access permission.
     * @return a text representation of the access flags.
     */
    public static String toString( int access, boolean isClass ) {
        StringBuffer stringBuffer = new StringBuffer();
        if ( ( access & PUBLIC ) == PUBLIC ) {
            stringBuffer.append( "public " );
        }
        if ( ( access & PRIVATE ) == PRIVATE ) {
            stringBuffer.append( "private " );
        }
        if ( ( access & PROTECTED ) == PROTECTED ) {
            stringBuffer.append( "protected " );
        }
        if ( ( access & STATIC ) == STATIC ) {
            stringBuffer.append( "static " );
        }
        if ( ( access & FINAL ) == FINAL ) {
            stringBuffer.append( "final " );
        }
        if ( !isClass && ( access & SYNCHRONIZED ) == SYNCHRONIZED ) {
            stringBuffer.append( "synchronized " );
        }
        if ( ( access & VOLATILE ) == VOLATILE ) {
            stringBuffer.append( "volatile ");
        }
        if ( ( access & TRANSIENT ) == TRANSIENT ) {
            stringBuffer.append( "transient " );
        }
        if ( ( access & NATIVE) == NATIVE ) {
            stringBuffer.append( "native " );
        }
        if ( ( access & ABSTRACT ) == ABSTRACT && ( access & INTERFACE ) == 0) {//interfaces are always abstract, so drop the abstract keyword
            stringBuffer.append( "abstract " );
        }
        // trim trailing space
        return stringBuffer.toString();
    }

    public static boolean isStatic( int access ) {
        return ( ( access & STATIC ) == STATIC );
    }

    public static final boolean isPublic( int access ) {
        return ( ( access & PUBLIC ) == PUBLIC );
    }

    public static final boolean isProtected( int access ) {
        return ( ( access & PROTECTED ) == PROTECTED );
    }

    public static final boolean isPackagePrivate( int access ) {
        return ( ( access & ( PUBLIC | PRIVATE | PROTECTED ) ) == 0 );
    }

    public static final boolean isPrivate( int access ) {
        return ( ( access & PRIVATE ) == PRIVATE );
    }

    public static final boolean isInterface( int access ) {
        return ( ( access & INTERFACE ) == INTERFACE );
    }
    
}
