" The clipboard provider uses shell commands to communicate with the clipboard.
" The provider function will only be registered if a supported command is
" available.
let s:copy = {}
let s:paste = {}
let s:clipboard = {}

" When caching is enabled, store the jobid of the xclip/xsel process keeping
" ownership of the selection, so we know how long the cache is valid.
let s:selection = { 'owner': 0, 'data': [], 'stderr_buffered': v:true }

function! s:selection.on_exit(jobid, data, event) abort
  " At this point this nvim instance might already have launched
  " a new provider instance. Don't drop ownership in this case.
  if self.owner == a:jobid
    let self.owner = 0
  endif
  if a:data != 0
    echohl WarningMsg
    echomsg 'clipboard: error invoking '.get(self.argv, 0, '?').': '.join(self.stderr)
    echohl None
  endif
endfunction

let s:selections = { '*': s:selection, '+': copy(s:selection) }

function! s:try_cmd(cmd, ...) abort
  let argv = split(a:cmd, " ")
  let out = systemlist(argv, (a:0 ? a:1 : ['']), 1)
  if v:shell_error
    if !exists('s:did_error_try_cmd')
      echohl WarningMsg
      echomsg "clipboard: error: ".(len(out) ? out[0] : v:shell_error)
      echohl None
      let s:did_error_try_cmd = 1
    endif
    return 0
  endif
  return out
endfunction

" Returns TRUE if `cmd` exits with success, else FALSE.
function! s:cmd_ok(cmd) abort
  call system(a:cmd)
  return v:shell_error == 0
endfunction

let s:cache_enabled = 1
let s:err = ''

function! provider#clipboard#Error() abort
  return s:err
endfunction

function! provider#clipboard#Executable() abort
  if exists('g:clipboard')
    if type({}) isnot# type(g:clipboard)
          \ || type({}) isnot# type(get(g:clipboard, 'copy', v:null))
          \ || type({}) isnot# type(get(g:clipboard, 'paste', v:null))
      let s:err = 'clipboard: invalid g:clipboard'
      return ''
    endif

    let s:copy = get(g:clipboard, 'copy', { '+': v:null, '*': v:null })
    let s:paste = get(g:clipboard, 'paste', { '+': v:null, '*': v:null })
    let s:cache_enabled = get(g:clipboard, 'cache_enabled', 0)
    return get(g:clipboard, 'name', 'g:clipboard')
  elseif has('mac') && executable('pbpaste') && s:cmd_ok('pbpaste')
    let s:copy['+'] = 'pbcopy'
    let s:paste['+'] = 'pbpaste'
    let s:copy['*'] = s:copy['+']
    let s:paste['*'] = s:paste['+']
    let s:cache_enabled = 0
    return 'pbcopy'
  elseif exists('$WAYLAND_DISPLAY') && executable('wl-copy') && executable('wl-paste')
    let s:copy['+'] = 'wl-copy --foreground'
    let s:paste['+'] = 'wl-paste --no-newline'
    let s:copy['*'] = 'wl-copy --foreground --primary'
    let s:paste['*'] = 'wl-paste --no-newline --primary'
    return 'wl-copy'
  elseif exists('$DISPLAY') && executable('xclip')
    let s:copy['+'] = 'xclip -quiet -i -selection clipboard'
    let s:paste['+'] = 'xclip -o -selection clipboard'
    let s:copy['*'] = 'xclip -quiet -i -selection primary'
    let s:paste['*'] = 'xclip -o -selection primary'
    return 'xclip'
  elseif exists('$DISPLAY') && executable('xsel') && s:cmd_ok('xsel -o -b')
    let s:copy['+'] = 'xsel --nodetach -i -b'
    let s:paste['+'] = 'xsel -o -b'
    let s:copy['*'] = 'xsel --nodetach -i -p'
    let s:paste['*'] = 'xsel -o -p'
    return 'xsel'
  elseif executable('lemonade')
    let s:copy['+'] = 'lemonade copy'
    let s:paste['+'] = 'lemonade paste'
    let s:copy['*'] = 'lemonade copy'
    let s:paste['*'] = 'lemonade paste'
    return 'lemonade'
  elseif executable('doitclient')
    let s:copy['+'] = 'doitclient wclip'
    let s:paste['+'] = 'doitclient wclip -r'
    let s:copy['*'] = s:copy['+']
    let s:paste['*'] = s:paste['+']
    return 'doitclient'
  elseif executable('win32yank.exe')
    let s:copy['+'] = 'win32yank.exe -i --crlf'
    let s:paste['+'] = 'win32yank.exe -o --lf'
    let s:copy['*'] = s:copy['+']
    let s:paste['*'] = s:paste['+']
    return 'win32yank'
  elseif exists('$TMUX') && executable('tmux')
    let s:copy['+'] = 'tmux load-buffer -'
    let s:paste['+'] = 'tmux save-buffer -'
    let s:copy['*'] = s:copy['+']
    let s:paste['*'] = s:paste['+']
    return 'tmux'
  endif

  let s:err = 'clipboard: No clipboard tool. :help clipboard'
  return ''
endfunction

if empty(provider#clipboard#Executable())
  " provider#clipboard#Call() *must not* be defined if the provider is broken.
  " Otherwise eval_has_provider() thinks the clipboard provider is
  " functioning, and eval_call_provider() will happily call it.
  finish
endif

function! s:clipboard.get(reg) abort
  if type(s:paste[a:reg]) == v:t_func
    return s:paste[a:reg]()
  elseif s:selections[a:reg].owner > 0
    return s:selections[a:reg].data
  end
  return s:try_cmd(s:paste[a:reg])
endfunction

function! s:clipboard.set(lines, regtype, reg) abort
  if a:reg == '"'
    call s:clipboard.set(a:lines,a:regtype,'+')
    if s:copy['*'] != s:copy['+']
      call s:clipboard.set(a:lines,a:regtype,'*')
    end
    return 0
  end

  if type(s:copy[a:reg]) == v:t_func
    call s:copy[a:reg](a:lines, a:regtype)
    return 0
  end

  if s:cache_enabled == 0
    call s:try_cmd(s:copy[a:reg], a:lines)
    return 0
  end

  if s:selections[a:reg].owner > 0
    " The previous provider instance should exit when the new one takes
    " ownership, but kill it to be sure we don't fill up the job table.
    call jobstop(s:selections[a:reg].owner)
  end
  let s:selections[a:reg] = copy(s:selection)
  let selection = s:selections[a:reg]
  let selection.data = [a:lines, a:regtype]
  let argv = split(s:copy[a:reg], " ")
  let selection.argv = argv
  let selection.detach = s:cache_enabled
  let selection.cwd = "/"
  let jobid = jobstart(argv, selection)
  if jobid > 0
    call jobsend(jobid, a:lines)
    call jobclose(jobid, 'stdin')
    let selection.owner = jobid
  else
    echohl WarningMsg
    echomsg 'clipboard: failed to execute: '.(s:copy[a:reg])
    echohl None
    return 0
  endif
  return 1
endfunction

function! provider#clipboard#Call(method, args) abort
  if get(s:, 'here', v:false)  " Clipboard provider must not recurse. #7184
    return 0
  endif
  let s:here = v:true
  try
    return call(s:clipboard[a:method],a:args,s:clipboard)
  finally
    let s:here = v:false
  endtry
endfunction
