/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package docking.widgets.list;

import java.awt.*;

import javax.swing.*;

import docking.widgets.AbstractGCellRenderer;

/**
 * Provides a common implementation of a list renderer, for use in both JList and JComboBox.
 *
 * @param <E> the element-type this list models.
 */
public class GListCellRenderer<E> extends AbstractGCellRenderer implements ListCellRenderer<E> {

	/**
	 * Constructs a new GListCellRenderer.
	 */
	public GListCellRenderer() {
	}

	/**
	 * Constructs a new GListCellRenderer using the specified font.
	 * @param f the font to use when rendering text in the lists' cells
	 */
	public GListCellRenderer(Font f) {
		this();
		setFont(f);
	}

	/**
	 * Return the cell renderer text
	 * @param value Cell object value
	 * @return A string interpretation of value; generated by calling value.toString()
	 */
	protected String getItemText(E value) {
		return value == null ? "" : value.toString();
	}

	@Override
	public Component getListCellRendererComponent(JList<? extends E> list, E value, int index,
			boolean isSelected, boolean hasFocus) {

		setText(getItemText(value));
		setHorizontalAlignment(
			value instanceof Number ? SwingConstants.RIGHT : SwingConstants.LEFT);
		ListModel<? extends E> model = list.getModel();
		configureFont(list, model, index);

		if (isSelected) {
			setForeground(list.getSelectionForeground());
			setBackground(list.getSelectionBackground());
			setOpaque(true);
		}
		else {
			setForegroundColor(list, model, value);

			JList.DropLocation dropLocation = list.getDropLocation();
			// @formatter:off
			boolean isDropRow = (dropLocation != null && 
								dropLocation.isInsert() &&
								dropLocation.getIndex() == index);
			// @formatter:on
			if (isDropRow) {
				setBackground(Color.CYAN);
			}
			else {
				setBackground(getOSDependentBackgroundColor(list, index));
			}
		}

		setBorder(hasFocus ? focusBorder : noFocusBorder);
		return this;
	}

	protected void setForegroundColor(JList<? extends E> list, ListModel<? extends E> model,
			Object value) {
		setForeground(list.getForeground());
	}

	protected void configureFont(JList<? extends E> list, ListModel<? extends E> model, int index) {
		setFont(defaultFont);
	}
}
