"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Server = void 0;

var _config = require("./config");

var _core_app = require("./core_app");

var _audit_trail = require("./audit_trail");

var _elasticsearch = require("./elasticsearch");

var _http = require("./http");

var _http_resources = require("./http_resources");

var _rendering = require("./rendering");

var _legacy = require("./legacy");

var _logging = require("./logging");

var _ui_settings = require("./ui_settings");

var _plugins = require("./plugins");

var _saved_objects = require("../server/saved_objects");

var _metrics = require("./metrics");

var _capabilities = require("./capabilities");

var _uuid = require("./uuid");

var _status_service = require("./status/status_service");

var _csp = require("./csp");

var _dev = require("./dev");

var _path = require("./path");

var _kibana_config = require("./kibana_config");

var _saved_objects2 = require("./saved_objects");

var _utils = require("../utils");

var _context = require("./context");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function _classPrivateFieldGet(receiver, privateMap) { var descriptor = privateMap.get(receiver); if (!descriptor) { throw new TypeError("attempted to get private field on non-instance"); } if (descriptor.get) { return descriptor.get.call(receiver); } return descriptor.value; }

function _classPrivateFieldSet(receiver, privateMap, value) { var descriptor = privateMap.get(receiver); if (!descriptor) { throw new TypeError("attempted to set private field on non-instance"); } if (descriptor.set) { descriptor.set.call(receiver, value); } else { if (!descriptor.writable) { throw new TypeError("attempted to set read only private field"); } descriptor.value = value; } return value; }

const coreId = Symbol('core');
const rootConfigPath = '';

class Server {
  constructor(rawConfigProvider, env, loggingSystem) {
    this.env = env;
    this.loggingSystem = loggingSystem;

    _defineProperty(this, "configService", void 0);

    _defineProperty(this, "capabilities", void 0);

    _defineProperty(this, "context", void 0);

    _defineProperty(this, "elasticsearch", void 0);

    _defineProperty(this, "http", void 0);

    _defineProperty(this, "rendering", void 0);

    _defineProperty(this, "legacy", void 0);

    _defineProperty(this, "log", void 0);

    _defineProperty(this, "plugins", void 0);

    _defineProperty(this, "savedObjects", void 0);

    _defineProperty(this, "uiSettings", void 0);

    _defineProperty(this, "uuid", void 0);

    _defineProperty(this, "metrics", void 0);

    _defineProperty(this, "httpResources", void 0);

    _defineProperty(this, "status", void 0);

    _defineProperty(this, "logging", void 0);

    _defineProperty(this, "coreApp", void 0);

    _defineProperty(this, "auditTrail", void 0);

    _pluginsInitialized.set(this, {
      writable: true,
      value: void 0
    });

    _defineProperty(this, "coreStart", void 0);

    _defineProperty(this, "logger", void 0);

    this.logger = this.loggingSystem.asLoggerFactory();
    this.log = this.logger.get('server');
    this.configService = new _config.ConfigService(rawConfigProvider, env, this.logger);
    const core = {
      coreId,
      configService: this.configService,
      env,
      logger: this.logger
    };
    this.context = new _context.ContextService(core);
    this.http = new _http.HttpService(core);
    this.rendering = new _rendering.RenderingService(core);
    this.plugins = new _plugins.PluginsService(core);
    this.legacy = new _legacy.LegacyService(core);
    this.elasticsearch = new _elasticsearch.ElasticsearchService(core);
    this.savedObjects = new _saved_objects.SavedObjectsService(core);
    this.uiSettings = new _ui_settings.UiSettingsService(core);
    this.capabilities = new _capabilities.CapabilitiesService(core);
    this.uuid = new _uuid.UuidService(core);
    this.metrics = new _metrics.MetricsService(core);
    this.status = new _status_service.StatusService(core);
    this.coreApp = new _core_app.CoreApp(core);
    this.httpResources = new _http_resources.HttpResourcesService(core);
    this.auditTrail = new _audit_trail.AuditTrailService(core);
    this.logging = new _logging.LoggingService(core);
  }

  async setup() {
    this.log.debug('setting up server'); // Discover any plugins before continuing. This allows other systems to utilize the plugin dependency graph.

    const {
      pluginTree,
      uiPlugins
    } = await this.plugins.discover();
    const legacyPlugins = await this.legacy.discoverPlugins(); // Immediately terminate in case of invalid configuration

    await this.configService.validate();
    await (0, _legacy.ensureValidConfiguration)(this.configService, legacyPlugins);
    const contextServiceSetup = this.context.setup({
      // We inject a fake "legacy plugin" with dependencies on every plugin so that legacy plugins:
      // 1) Can access context from any NP plugin
      // 2) Can register context providers that will only be available to other legacy plugins and will not leak into
      //    New Platform plugins.
      pluginDependencies: new Map([...pluginTree, [this.legacy.legacyId, [...pluginTree.keys()]]])
    });
    const auditTrailSetup = this.auditTrail.setup();
    const uuidSetup = await this.uuid.setup();
    const httpSetup = await this.http.setup({
      context: contextServiceSetup
    });
    const capabilitiesSetup = this.capabilities.setup({
      http: httpSetup
    });
    const elasticsearchServiceSetup = await this.elasticsearch.setup({
      http: httpSetup
    });
    const savedObjectsSetup = await this.savedObjects.setup({
      http: httpSetup,
      elasticsearch: elasticsearchServiceSetup,
      legacyPlugins
    });
    const uiSettingsSetup = await this.uiSettings.setup({
      http: httpSetup,
      savedObjects: savedObjectsSetup
    });
    await this.metrics.setup({
      http: httpSetup
    });
    const renderingSetup = await this.rendering.setup({
      http: httpSetup,
      legacyPlugins,
      uiPlugins
    });
    const httpResourcesSetup = this.httpResources.setup({
      http: httpSetup,
      rendering: renderingSetup
    });
    const statusSetup = this.status.setup({
      elasticsearch: elasticsearchServiceSetup,
      savedObjects: savedObjectsSetup
    });
    const loggingSetup = this.logging.setup({
      loggingSystem: this.loggingSystem
    });
    const coreSetup = {
      capabilities: capabilitiesSetup,
      context: contextServiceSetup,
      elasticsearch: elasticsearchServiceSetup,
      http: httpSetup,
      savedObjects: savedObjectsSetup,
      status: statusSetup,
      uiSettings: uiSettingsSetup,
      uuid: uuidSetup,
      rendering: renderingSetup,
      httpResources: httpResourcesSetup,
      auditTrail: auditTrailSetup,
      logging: loggingSetup
    };
    const pluginsSetup = await this.plugins.setup(coreSetup);

    _classPrivateFieldSet(this, _pluginsInitialized, pluginsSetup.initialized);

    await this.legacy.setup({
      core: { ...coreSetup,
        plugins: pluginsSetup,
        rendering: renderingSetup
      },
      plugins: (0, _utils.mapToObject)(pluginsSetup.contracts),
      uiPlugins
    });
    this.registerCoreContext(coreSetup);
    this.coreApp.setup(coreSetup);
    return coreSetup;
  }

  async start() {
    this.log.debug('starting server');
    const auditTrailStart = this.auditTrail.start();
    const elasticsearchStart = await this.elasticsearch.start({
      auditTrail: auditTrailStart
    });
    const savedObjectsStart = await this.savedObjects.start({
      elasticsearch: elasticsearchStart,
      pluginsInitialized: _classPrivateFieldGet(this, _pluginsInitialized)
    });
    const capabilitiesStart = this.capabilities.start();
    const uiSettingsStart = await this.uiSettings.start();
    const metricsStart = await this.metrics.start();
    const httpStart = this.http.getStartContract();
    this.coreStart = {
      capabilities: capabilitiesStart,
      elasticsearch: elasticsearchStart,
      http: httpStart,
      metrics: metricsStart,
      savedObjects: savedObjectsStart,
      uiSettings: uiSettingsStart,
      auditTrail: auditTrailStart
    };
    const pluginsStart = await this.plugins.start(this.coreStart);
    await this.legacy.start({
      core: { ...this.coreStart,
        plugins: pluginsStart
      },
      plugins: (0, _utils.mapToObject)(pluginsStart.contracts)
    });
    await this.http.start();
    await this.rendering.start({
      legacy: this.legacy
    });
    return this.coreStart;
  }

  async stop() {
    this.log.debug('stopping server');
    await this.legacy.stop();
    await this.plugins.stop();
    await this.savedObjects.stop();
    await this.elasticsearch.stop();
    await this.http.stop();
    await this.uiSettings.stop();
    await this.rendering.stop();
    await this.metrics.stop();
    await this.status.stop();
    await this.logging.stop();
    await this.auditTrail.stop();
  }

  registerCoreContext(coreSetup) {
    coreSetup.http.registerRouteHandlerContext(coreId, 'core', async (context, req, res) => {
      const coreStart = this.coreStart;
      const savedObjectsClient = coreStart.savedObjects.getScopedClient(req);
      return {
        savedObjects: {
          client: savedObjectsClient,
          typeRegistry: coreStart.savedObjects.getTypeRegistry()
        },
        elasticsearch: {
          legacy: {
            client: coreStart.elasticsearch.legacy.client.asScoped(req)
          }
        },
        uiSettings: {
          client: coreStart.uiSettings.asScopedToClient(savedObjectsClient)
        },
        auditor: coreStart.auditTrail.asScoped(req)
      };
    });
  }

  async setupCoreConfig() {
    const schemas = [[_path.config.path, _path.config.schema], [_csp.config.path, _csp.config.schema], [_elasticsearch.config.path, _elasticsearch.config.schema], [_logging.config.path, _logging.config.schema], [_http.config.path, _http.config.schema], [_plugins.config.path, _plugins.config.schema], [_dev.config.path, _dev.config.schema], [_kibana_config.config.path, _kibana_config.config.schema], [_saved_objects2.savedObjectsConfig.path, _saved_objects2.savedObjectsConfig.schema], [_saved_objects2.savedObjectsMigrationConfig.path, _saved_objects2.savedObjectsMigrationConfig.schema], [_ui_settings.config.path, _ui_settings.config.schema], [_metrics.opsConfig.path, _metrics.opsConfig.schema]];
    this.configService.addDeprecationProvider(rootConfigPath, _config.coreDeprecationProvider);
    this.configService.addDeprecationProvider(_elasticsearch.config.path, _elasticsearch.config.deprecations);
    this.configService.addDeprecationProvider(_ui_settings.config.path, _ui_settings.config.deprecations);

    for (const [path, schema] of schemas) {
      await this.configService.setSchema(path, schema);
    }
  }

}

exports.Server = Server;

var _pluginsInitialized = new WeakMap();