/*
 * Decompiled with CFR 0.152.
 */
package db;

import db.Buffer;
import db.buffers.BufferMgr;
import db.buffers.DataBuffer;
import ghidra.util.exception.AssertException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

public class ChainedBuffer
implements Buffer {
    private static final byte[] XOR_MASK_BYTES = new byte[]{89, -22, 103, 35, -38, -72, 0, -72, -61, 72, -35, -117, 33, -42, -108, 120, 53, -85, 43, 126, -78, 79, -126, 78, 14, 22, -60, 87, 18, -114, 126, -26, -74, -67, 86, -111, 87, 114, -26, -111, -36, 82, 46, -14, 26, -73, -42, 111, -38, -34, -24, 72, -79, -69, 80, 111, -12, -35, 17, -18, -14, 103, -2, 72, -115, -82, 105, 26, -32, 38, -116, 36, -114, 23, 118, 81, -30, 96, -41, -26, -125, 101, -43, -16, 127, -14, -96, -42, 75, -67, 36, -40, -85, -22, -98, -90, 72, -108, 62, 123, 44, -12, -50, -36, 105, 17, -8, 60, -89, 63, 93, 119, -108, 63, -28, -114, 72, 32, -37, 86, 50, -63, -121, 1, 46, -29, 127, 64};
    private static final int NODE_TYPE_SIZE = 1;
    private static final int DATA_LENGTH_SIZE = 4;
    private static final int ID_SIZE = 4;
    private static final int NODE_TYPE_OFFSET = 0;
    private static final int DATA_LENGTH_OFFSET = 1;
    private static final int NEXT_INDEX_ID_OFFSET = 5;
    private static final int INDEX_BASE_OFFSET = 9;
    private static final int DATA_BASE_OFFSET_NONINDEXED = 5;
    private static final int DATA_BASE_OFFSET_INDEXED = 1;
    private BufferMgr bufferMgr;
    private int size;
    private int firstBufferId;
    private byte[] xorData;
    private boolean useXORMask = false;
    private boolean readOnly = false;
    private Buffer uninitializedDataSource;
    private int uninitializedDataSourceOffset;
    private int[] indexBufferIdTable;
    private int[] dataBufferIdTable;
    private int indexesPerBuffer;
    private int dataBaseOffset;
    private int dataSpace;

    public ChainedBuffer(int size, boolean enableObfuscation, Buffer uninitializedDataSource, int unintializedDataSourceOffset, BufferMgr bufferMgr) throws IOException {
        this.bufferMgr = bufferMgr;
        this.size = size;
        this.useXORMask = enableObfuscation;
        if (size == 0) {
            throw new IllegalArgumentException("Zero length buffer not permitted");
        }
        if (size < 0) {
            throw new IllegalArgumentException("Maximum bufer size is 2147483647; given size of " + size);
        }
        if (uninitializedDataSource != null) {
            this.setUnintializedDataSource(uninitializedDataSource, unintializedDataSourceOffset);
        }
        DataBuffer firstBuffer = bufferMgr.createBuffer();
        this.firstBufferId = firstBuffer.getId();
        this.dataBaseOffset = 5;
        this.dataSpace = bufferMgr.getBufferSize() - this.dataBaseOffset;
        if (size <= this.dataSpace) {
            this.dataBufferIdTable = new int[]{this.firstBufferId};
            this.initializeAllocatedBuffer(0, firstBuffer);
            firstBuffer.putByte(0, (byte)9);
            firstBuffer.putInt(1, this.getObfuscationDataLengthFieldValue());
            bufferMgr.releaseBuffer(firstBuffer);
        } else {
            this.createIndex(firstBuffer);
        }
    }

    public ChainedBuffer(int size, boolean enableObfuscation, BufferMgr bufferMgr) throws IOException {
        this(size, enableObfuscation, null, 0, bufferMgr);
    }

    public ChainedBuffer(int size, BufferMgr bufferMgr) throws IOException {
        this(size, false, null, 0, bufferMgr);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ChainedBuffer(BufferMgr bufferMgr, int bufferId, Buffer uninitializedDataSource, int unintializedDataSourceOffset) throws IOException {
        byte bufType;
        this.bufferMgr = bufferMgr;
        this.firstBufferId = bufferId;
        DataBuffer firstBuffer = bufferMgr.getBuffer(bufferId);
        this.size = firstBuffer.getInt(1);
        if (this.size < 0) {
            this.useXORMask = true;
            this.size &= Integer.MAX_VALUE;
        }
        if ((bufType = firstBuffer.getByte(0)) == 8) {
            this.buildIndex(firstBuffer);
        } else if (bufType == 9) {
            try {
                this.dataBaseOffset = 5;
                this.dataSpace = firstBuffer.length() - this.dataBaseOffset;
                this.dataBufferIdTable = new int[]{bufferId};
            }
            finally {
                bufferMgr.releaseBuffer(firstBuffer);
            }
        } else {
            throw new IOException("Invalid Buffer");
        }
        if (uninitializedDataSource != null) {
            this.setUnintializedDataSource(uninitializedDataSource, unintializedDataSourceOffset);
        }
    }

    public ChainedBuffer(BufferMgr bufferMgr, int bufferId) throws IOException {
        this(bufferMgr, bufferId, null, 0);
    }

    private int getObfuscationDataLengthFieldValue() {
        return this.size | (this.useXORMask ? Integer.MIN_VALUE : 0);
    }

    private byte xorMaskByte(int bufferOffset, byte byteValue) {
        byte maskByte = XOR_MASK_BYTES[bufferOffset % XOR_MASK_BYTES.length];
        return (byte)(byteValue ^ maskByte);
    }

    private long getXorMask(int bufferOffset, int len) {
        long mask = 0L;
        for (int i = 0; i < len; ++i) {
            mask = mask << 8 | (long)this.xorMaskByte(bufferOffset++, (byte)0) & 0xFFL;
        }
        return mask;
    }

    private void setUnintializedDataSource(Buffer dataSource, int dataSourceOffset) {
        if (dataSourceOffset < 0) {
            throw new IllegalArgumentException("Invalid data source offset: " + dataSourceOffset);
        }
        int maxOffset = dataSourceOffset - 1 + this.size;
        if (maxOffset < 0 || maxOffset >= dataSource.length()) {
            throw new IllegalArgumentException("Data source has insufficient bytes (need " + this.size + " bytes at offset " + dataSourceOffset);
        }
        this.uninitializedDataSource = dataSource;
        this.uninitializedDataSourceOffset = dataSourceOffset;
    }

    public boolean hasObfuscatedStorage() {
        return this.useXORMask;
    }

    public void setReadOnly() {
        this.readOnly = true;
    }

    int getBufferCount() {
        return this.dataBufferIdTable.length + (this.indexBufferIdTable != null ? this.indexBufferIdTable.length : 0);
    }

    public synchronized void setSize(int size, boolean preserveData) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        if (this.uninitializedDataSource != null) {
            throw new UnsupportedOperationException("Buffer size may not be changed when using unintialized data source");
        }
        if (this.dataBufferIdTable == null) {
            throw new AssertException("Invalid Buffer");
        }
        if (size > this.size) {
            this.grow(size, preserveData);
        } else {
            this.shrink(size, preserveData);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void grow(int newSize, boolean preserveData) throws IOException {
        int oldSize = this.size;
        this.size = newSize;
        if (this.dataBufferIdTable.length == 1) {
            if (newSize > this.dataSpace) {
                DataBuffer firstBuffer = this.bufferMgr.getBuffer(this.firstBufferId);
                DataBuffer newFirstDataBuf = null;
                try {
                    if (preserveData) {
                        newFirstDataBuf = this.bufferMgr.createBuffer();
                        newFirstDataBuf.copy(1, firstBuffer, 5, oldSize);
                    }
                    this.createIndex(firstBuffer);
                    firstBuffer = null;
                    if (preserveData) {
                        this.addBuffer(0, newFirstDataBuf);
                    }
                }
                finally {
                    if (firstBuffer != null) {
                        this.bufferMgr.releaseBuffer(firstBuffer);
                    }
                    if (newFirstDataBuf != null) {
                        this.bufferMgr.releaseBuffer(newFirstDataBuf);
                    }
                }
                return;
            }
            DataBuffer buffer = this.bufferMgr.getBuffer(this.firstBufferId);
            buffer.putInt(1, this.getObfuscationDataLengthFieldValue());
            this.bufferMgr.releaseBuffer(buffer);
        } else {
            byte[] emptyIndexData = new byte[this.indexesPerBuffer * 4];
            Arrays.fill(emptyIndexData, (byte)-1);
            int newIndexCount = (newSize - 1) / this.dataSpace + 1;
            int newIndexBufferCount = (newIndexCount - 1) / this.indexesPerBuffer + 1;
            int[] newDataBufferIdTable = new int[newIndexCount];
            System.arraycopy(this.dataBufferIdTable, 0, newDataBufferIdTable, 0, this.dataBufferIdTable.length);
            Arrays.fill(newDataBufferIdTable, this.dataBufferIdTable.length, newIndexCount, -1);
            this.dataBufferIdTable = newDataBufferIdTable;
            int oldIndexBufferCount = this.indexBufferIdTable.length;
            if (oldIndexBufferCount < newIndexBufferCount) {
                int[] newIndexBufferIdTable = new int[newIndexBufferCount];
                System.arraycopy(this.indexBufferIdTable, 0, newIndexBufferIdTable, 0, oldIndexBufferCount);
                Arrays.fill(newIndexBufferIdTable, oldIndexBufferCount, newIndexBufferCount, -1);
                this.indexBufferIdTable = newIndexBufferIdTable;
                DataBuffer indexBuffer = this.bufferMgr.getBuffer(this.indexBufferIdTable[oldIndexBufferCount - 1]);
                for (int i = oldIndexBufferCount; i < newIndexBufferCount; ++i) {
                    indexBuffer = this.appendIndexBuffer(indexBuffer);
                    indexBuffer.put(9, emptyIndexData);
                    this.indexBufferIdTable[i] = indexBuffer.getId();
                }
                this.bufferMgr.releaseBuffer(indexBuffer);
            }
            DataBuffer buffer = this.bufferMgr.getBuffer(this.firstBufferId);
            buffer.putInt(1, this.getObfuscationDataLengthFieldValue());
            this.bufferMgr.releaseBuffer(buffer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean shrinkToSingleBuffer(boolean preserveData) throws IOException {
        int singleDataSpace = this.bufferMgr.getBufferSize() - 5;
        if (this.size > singleDataSpace) {
            return false;
        }
        DataBuffer firstBuffer = this.bufferMgr.getBuffer(this.firstBufferId);
        DataBuffer oldFirstDataBuf = null;
        try {
            firstBuffer.putByte(0, (byte)9);
            firstBuffer.putInt(1, this.getObfuscationDataLengthFieldValue());
            if (preserveData && this.dataBufferIdTable[0] >= 0) {
                oldFirstDataBuf = this.bufferMgr.getBuffer(this.dataBufferIdTable[0]);
                firstBuffer.copy(5, oldFirstDataBuf, 1, this.size);
            }
        }
        finally {
            this.bufferMgr.releaseBuffer(firstBuffer);
            if (oldFirstDataBuf != null) {
                this.bufferMgr.releaseBuffer(oldFirstDataBuf);
            }
        }
        for (int element : this.dataBufferIdTable) {
            if (element < 0) continue;
            this.bufferMgr.deleteBuffer(element);
        }
        for (int i = 1; i < this.indexBufferIdTable.length; ++i) {
            this.bufferMgr.deleteBuffer(this.indexBufferIdTable[i]);
        }
        this.dataBaseOffset = 5;
        this.dataSpace = singleDataSpace;
        this.dataBufferIdTable = new int[1];
        this.dataBufferIdTable[0] = this.firstBufferId;
        this.indexBufferIdTable = null;
        return true;
    }

    private void shrink(int newSize, boolean preserveData) throws IOException {
        this.size = newSize;
        if (this.dataBufferIdTable.length == 1) {
            DataBuffer buffer = this.bufferMgr.getBuffer(this.firstBufferId);
            buffer.putInt(1, this.getObfuscationDataLengthFieldValue());
            this.bufferMgr.releaseBuffer(buffer);
        } else if (!this.shrinkToSingleBuffer(preserveData)) {
            int newIndexCount = (newSize - 1) / this.dataSpace + 1;
            int newIndexBufferCount = (newIndexCount - 1) / this.indexesPerBuffer + 1;
            int oldIndexCount = this.dataBufferIdTable.length;
            for (int i = newIndexCount; i < oldIndexCount; ++i) {
                if (this.dataBufferIdTable[i] < 0) continue;
                this.bufferMgr.deleteBuffer(this.dataBufferIdTable[i]);
            }
            int[] newDataBufferIdTable = new int[newIndexCount];
            System.arraycopy(this.dataBufferIdTable, 0, newDataBufferIdTable, 0, newDataBufferIdTable.length);
            this.dataBufferIdTable = newDataBufferIdTable;
            int oldIndexBufferCount = this.indexBufferIdTable.length;
            if (oldIndexBufferCount > newIndexBufferCount) {
                for (int i = newIndexBufferCount; i < oldIndexBufferCount; ++i) {
                    this.bufferMgr.deleteBuffer(this.indexBufferIdTable[i]);
                }
                int[] newIndexBufferIdTable = new int[newIndexBufferCount];
                System.arraycopy(this.indexBufferIdTable, 0, newIndexBufferIdTable, 0, newIndexBufferCount);
                this.indexBufferIdTable = newIndexBufferIdTable;
            }
            DataBuffer buffer = this.bufferMgr.getBuffer(this.firstBufferId);
            buffer.putInt(1, this.getObfuscationDataLengthFieldValue());
            this.bufferMgr.releaseBuffer(buffer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized ChainedBuffer split(int offset) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        if (this.firstBufferId < 0) {
            throw new AssertException("Invalid Buffer");
        }
        if (offset < 0 || offset >= this.size) {
            throw new ArrayIndexOutOfBoundsException();
        }
        int cnt = this.size - offset;
        ChainedBuffer newDBBuf = new ChainedBuffer(cnt, this.useXORMask, this.uninitializedDataSource, this.uninitializedDataSourceOffset + offset, this.bufferMgr);
        int newOffset = 0;
        byte[] data = new byte[this.dataSpace];
        int bufferDataOffset = offset % this.dataSpace;
        int dataSize = this.dataSpace - bufferDataOffset;
        for (int index = offset / this.dataSpace; index < this.dataBufferIdTable.length; ++index) {
            if (cnt < dataSize) {
                dataSize = cnt;
            }
            if (this.dataBufferIdTable[index] >= 0) {
                DataBuffer dataBuf = this.bufferMgr.getBuffer(this.dataBufferIdTable[index]);
                try {
                    dataBuf.get(bufferDataOffset + this.dataBaseOffset, data, 0, dataSize);
                    if (this.useXORMask) {
                        for (int i = 0; i < dataSize; ++i) {
                            data[i] = this.xorMaskByte(bufferDataOffset + i, data[i]);
                        }
                    }
                    newDBBuf.put(newOffset, data, 0, dataSize);
                }
                finally {
                    this.bufferMgr.releaseBuffer(dataBuf);
                }
                if (bufferDataOffset == 0) {
                    this.bufferMgr.deleteBuffer(this.dataBufferIdTable[index]);
                    this.dataBufferIdTable[index] = -1;
                }
            }
            cnt -= dataSize;
            newOffset += dataSize;
            bufferDataOffset = 0;
            dataSize = data.length;
        }
        this.shrink(offset, true);
        return newDBBuf;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void append(ChainedBuffer dbBuf) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        if (this.uninitializedDataSource != null) {
            throw new UnsupportedOperationException("Buffer size may not be changed when using unintialized data source");
        }
        if (this.firstBufferId < 0) {
            throw new AssertException("Invalid Buffer");
        }
        if (dbBuf.firstBufferId < 0 || this.firstBufferId == dbBuf.firstBufferId) {
            throw new IllegalArgumentException("Illegal DBBuffer argument");
        }
        int offset = this.size;
        int newSize = this.size + dbBuf.size;
        this.grow(newSize, true);
        BufferMgr otherBufMgr = dbBuf.bufferMgr;
        if (dbBuf.indexBufferIdTable != null) {
            for (int element : dbBuf.indexBufferIdTable) {
                otherBufMgr.deleteBuffer(element);
            }
        }
        dbBuf.indexBufferIdTable = null;
        dbBuf.firstBufferId = -1;
        int cnt = 0;
        int dataSize = dbBuf.dataSpace;
        byte[] data = new byte[dataSize];
        for (int element : dbBuf.dataBufferIdTable) {
            if ((cnt += dataSize) > dbBuf.size) {
                dataSize -= cnt - dbBuf.size;
            }
            if (element >= 0) {
                DataBuffer otherDataBuf = otherBufMgr.getBuffer(element);
                try {
                    otherDataBuf.get(dbBuf.dataBaseOffset, data, 0, dataSize);
                    if (dbBuf.useXORMask) {
                        for (int n = 0; n < dataSize; ++n) {
                            data[n] = this.xorMaskByte(n, data[n]);
                        }
                    }
                    this.put(offset, data, 0, dataSize);
                }
                finally {
                    otherBufMgr.releaseBuffer(otherDataBuf);
                }
                otherBufMgr.deleteBuffer(element);
            }
            offset += dataSize;
        }
        dbBuf.dataBufferIdTable = null;
        dbBuf.size = 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DataBuffer appendIndexBuffer(DataBuffer indexBuffer) throws IOException {
        try {
            DataBuffer nextBuf = this.bufferMgr.createBuffer();
            nextBuf.putByte(0, (byte)8);
            nextBuf.putInt(1, -1);
            nextBuf.putInt(5, -1);
            indexBuffer.putInt(5, nextBuf.getId());
            DataBuffer dataBuffer = nextBuf;
            return dataBuffer;
        }
        finally {
            this.bufferMgr.releaseBuffer(indexBuffer);
        }
    }

    private int allocateIndex(DataBuffer buffer) {
        this.dataBaseOffset = 1;
        this.dataSpace = buffer.length() - this.dataBaseOffset;
        int indexCount = (this.size - 1) / this.dataSpace + 1;
        this.indexesPerBuffer = (buffer.length() - 9) / 4;
        this.dataBufferIdTable = new int[indexCount];
        return indexCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createIndex(DataBuffer indexBuffer) throws IOException {
        try {
            indexBuffer.putByte(0, (byte)8);
            indexBuffer.putInt(1, this.getObfuscationDataLengthFieldValue());
            indexBuffer.putInt(5, -1);
            int indexCount = this.allocateIndex(indexBuffer);
            Arrays.fill(this.dataBufferIdTable, -1);
            int indexBufferCount = (indexCount - 1) / this.indexesPerBuffer + 1;
            this.indexBufferIdTable = new int[indexBufferCount];
            byte[] emptyIndexData = new byte[this.indexesPerBuffer * 4];
            Arrays.fill(emptyIndexData, (byte)-1);
            this.indexBufferIdTable[0] = indexBuffer.getId();
            indexBuffer.put(9, emptyIndexData);
            for (int i = 1; i < indexBufferCount; ++i) {
                indexBuffer = this.appendIndexBuffer(indexBuffer);
                this.indexBufferIdTable[i] = indexBuffer.getId();
                indexBuffer.put(9, emptyIndexData);
            }
        }
        finally {
            this.bufferMgr.releaseBuffer(indexBuffer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void buildIndex(DataBuffer indexBuffer) throws IOException {
        int indexCount = this.allocateIndex(indexBuffer);
        int indexBufferCount = (indexCount - 1) / this.indexesPerBuffer + 1;
        this.indexBufferIdTable = new int[indexBufferCount];
        try {
            int index = 0;
            this.indexBufferIdTable[index] = indexBuffer.getId();
            int ix = 0;
            int offset = 9;
            for (int i = 0; i < indexCount; ++i) {
                if (ix == this.indexesPerBuffer) {
                    int nextId = indexBuffer.getInt(5);
                    if (nextId < 0) {
                        throw new AssertException();
                    }
                    this.bufferMgr.releaseBuffer(indexBuffer);
                    indexBuffer = this.bufferMgr.getBuffer(nextId);
                    this.indexBufferIdTable[++index] = indexBuffer.getId();
                    ix = 0;
                    offset = 9;
                }
                this.dataBufferIdTable[i] = indexBuffer.getInt(offset);
                offset += 4;
                ++ix;
            }
        }
        finally {
            this.bufferMgr.releaseBuffer(indexBuffer);
        }
    }

    @Override
    public int getId() {
        return this.firstBufferId;
    }

    public synchronized void delete() throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        if (this.firstBufferId < 0) {
            throw new AssertException("Invalid Buffer");
        }
        for (int element : this.dataBufferIdTable) {
            if (element < 0) continue;
            this.bufferMgr.deleteBuffer(element);
        }
        this.dataBufferIdTable = null;
        if (this.indexBufferIdTable != null) {
            for (int element : this.indexBufferIdTable) {
                this.bufferMgr.deleteBuffer(element);
            }
        }
        this.indexBufferIdTable = null;
        this.size = 0;
        this.firstBufferId = -1;
    }

    private int getBytes(int offset, int index, int bufferDataOffset, byte[] data, int dataOffset, int length) throws IOException {
        int availableData = this.dataSpace - bufferDataOffset;
        int len = availableData < length ? availableData : length;
        int id = this.dataBufferIdTable[index];
        if (id < 0) {
            if (this.uninitializedDataSource != null) {
                this.uninitializedDataSource.get(this.uninitializedDataSourceOffset + offset, data, dataOffset, len);
            } else {
                Arrays.fill(data, dataOffset, dataOffset + len, (byte)0);
            }
        } else {
            DataBuffer buffer = this.bufferMgr.getBuffer(id);
            buffer.get(this.dataBaseOffset + bufferDataOffset, data, dataOffset, len);
            this.bufferMgr.releaseBuffer(buffer);
            if (this.useXORMask) {
                int dataIndex = dataOffset;
                int i = 0;
                while (i < len) {
                    data[dataIndex] = this.xorMaskByte(bufferDataOffset++, data[dataIndex]);
                    ++i;
                    ++dataIndex;
                }
            }
        }
        return len;
    }

    @Override
    public synchronized void get(int offset, byte[] data, int dataOffset, int length) throws IOException {
        int n;
        if (this.dataBufferIdTable == null) {
            throw new AssertException("Invalid Buffer");
        }
        if (offset < 0 || offset + length - 1 >= this.size) {
            throw new ArrayIndexOutOfBoundsException();
        }
        if (data.length < dataOffset + length) {
            throw new ArrayIndexOutOfBoundsException();
        }
        int index = offset / this.dataSpace;
        int bufferDataOffset = offset % this.dataSpace;
        for (int len = length; len > 0; len -= n) {
            n = this.getBytes(offset, index++, bufferDataOffset, data, dataOffset, len);
            dataOffset += n;
            offset += n;
            bufferDataOffset = 0;
        }
    }

    @Override
    public synchronized void get(int offset, byte[] data) throws IOException {
        this.get(offset, data, 0, data.length);
    }

    @Override
    public synchronized byte[] get(int offset, int length) throws IOException {
        byte[] data = new byte[length];
        this.get(offset, data, 0, length);
        return data;
    }

    @Override
    public synchronized byte getByte(int offset) throws IOException {
        if (this.dataBufferIdTable == null) {
            throw new AssertException("Invalid Buffer");
        }
        if (offset < 0 || offset >= this.size) {
            throw new ArrayIndexOutOfBoundsException();
        }
        int index = offset / this.dataSpace;
        int bufferDataOffset = offset % this.dataSpace;
        int id = this.dataBufferIdTable[index];
        if (id < 0) {
            if (this.uninitializedDataSource != null) {
                return this.uninitializedDataSource.getByte(this.uninitializedDataSourceOffset + offset);
            }
            return 0;
        }
        DataBuffer buffer = this.bufferMgr.getBuffer(id);
        byte b = buffer.getByte(this.dataBaseOffset + bufferDataOffset);
        this.bufferMgr.releaseBuffer(buffer);
        if (this.useXORMask) {
            b = this.xorMaskByte(bufferDataOffset, b);
        }
        return b;
    }

    @Override
    public synchronized int getInt(int offset) throws IOException {
        int bufferOffset = this.dataBaseOffset + offset % this.dataSpace;
        if (bufferOffset + 3 <= this.dataSpace) {
            if (this.dataBufferIdTable == null) {
                throw new AssertException("Invalid Buffer");
            }
            if (offset < 0 || offset + 3 >= this.size) {
                throw new ArrayIndexOutOfBoundsException();
            }
            int index = offset / this.dataSpace;
            int id = this.dataBufferIdTable[index];
            if (id < 0) {
                if (this.uninitializedDataSource != null) {
                    return this.uninitializedDataSource.getInt(this.uninitializedDataSourceOffset + offset);
                }
                return 0;
            }
            DataBuffer buffer = this.bufferMgr.getBuffer(id);
            int value = buffer.getInt(bufferOffset);
            this.bufferMgr.releaseBuffer(buffer);
            if (this.useXORMask) {
                value ^= (int)this.getXorMask(offset % this.dataSpace, 4);
            }
            return value;
        }
        byte[] data = this.get(offset, 4);
        return (data[0] & 0xFF) << 24 | (data[1] & 0xFF) << 16 | (data[2] & 0xFF) << 8 | data[3] & 0xFF;
    }

    @Override
    public synchronized long getLong(int offset) throws IOException {
        int bufferOffset = this.dataBaseOffset + offset % this.dataSpace;
        if (bufferOffset + 7 <= this.dataSpace) {
            if (this.dataBufferIdTable == null) {
                throw new AssertException("Invalid Buffer");
            }
            if (offset < 0 || offset + 7 >= this.size) {
                throw new ArrayIndexOutOfBoundsException();
            }
            int index = offset / this.dataSpace;
            int id = this.dataBufferIdTable[index];
            if (id < 0) {
                if (this.uninitializedDataSource != null) {
                    return this.uninitializedDataSource.getLong(this.uninitializedDataSourceOffset + offset);
                }
                return 0L;
            }
            DataBuffer buffer = this.bufferMgr.getBuffer(id);
            long value = buffer.getLong(bufferOffset);
            this.bufferMgr.releaseBuffer(buffer);
            if (this.useXORMask) {
                value ^= this.getXorMask(offset % this.dataSpace, 8);
            }
            return value;
        }
        byte[] data = this.get(offset, 8);
        return ((long)data[0] & 0xFFL) << 56 | ((long)data[1] & 0xFFL) << 48 | ((long)data[2] & 0xFFL) << 40 | ((long)data[3] & 0xFFL) << 32 | ((long)data[4] & 0xFFL) << 24 | ((long)data[5] & 0xFFL) << 16 | ((long)data[6] & 0xFFL) << 8 | (long)data[7] & 0xFFL;
    }

    @Override
    public synchronized short getShort(int offset) throws IOException {
        int bufferOffset = this.dataBaseOffset + offset % this.dataSpace;
        if (bufferOffset + 1 <= this.dataSpace) {
            if (this.dataBufferIdTable == null) {
                throw new AssertException("Invalid Buffer");
            }
            if (offset < 0 || offset + 1 >= this.size) {
                throw new ArrayIndexOutOfBoundsException();
            }
            int index = offset / this.dataSpace;
            int id = this.dataBufferIdTable[index];
            if (id < 0) {
                if (this.uninitializedDataSource != null) {
                    return this.uninitializedDataSource.getShort(this.uninitializedDataSourceOffset + offset);
                }
                return 0;
            }
            DataBuffer buffer = this.bufferMgr.getBuffer(id);
            short value = buffer.getShort(bufferOffset);
            this.bufferMgr.releaseBuffer(buffer);
            if (this.useXORMask) {
                value = (short)(value ^ (int)this.getXorMask(offset % this.dataSpace, 2));
            }
            return value;
        }
        byte[] data = this.get(offset, 2);
        return (short)((data[0] & 0xFF) << 8 | data[1] & 0xFF);
    }

    @Override
    public int length() {
        return this.size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void fill(int startOffset, int endOffset, byte fillByte) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        if (endOffset <= startOffset) {
            throw new IllegalArgumentException();
        }
        if (startOffset < 0 || endOffset > this.size) {
            throw new ArrayIndexOutOfBoundsException();
        }
        byte[] fillData = new byte[this.dataSpace];
        Arrays.fill(fillData, fillByte);
        int index = startOffset / this.dataSpace;
        int bufferDataOffset = startOffset % this.dataSpace;
        int len = endOffset - startOffset + 1;
        if (this.useXORMask) {
            this.xorData = new byte[this.dataSpace];
        }
        try {
            while (len > 0) {
                int n = this.putBytes(index++, bufferDataOffset, fillData, 0, len);
                len -= n;
                bufferDataOffset = 0;
            }
        }
        finally {
            this.xorData = null;
        }
    }

    private int putBytes(int index, int bufferDataOffset, byte[] data, int dataOffset, int length) throws IOException {
        int len;
        int availableSpace = this.dataSpace - bufferDataOffset;
        int n = len = availableSpace < length ? availableSpace : length;
        if (this.xorData != null) {
            int offset = bufferDataOffset;
            int dataIndex = dataOffset;
            int i = 0;
            while (i < len) {
                this.xorData[i] = this.xorMaskByte(offset, data[dataIndex]);
                ++i;
                ++dataIndex;
                ++offset;
            }
            data = this.xorData;
            dataOffset = 0;
        }
        DataBuffer buffer = this.getBuffer(index);
        buffer.put(bufferDataOffset + this.dataBaseOffset, data, dataOffset, len);
        this.bufferMgr.releaseBuffer(buffer);
        return len;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void fill(InputStream in) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        byte[] data = new byte[this.dataSpace];
        int index = 0;
        int offset = 0;
        int length = this.size;
        int readSpace = this.dataSpace;
        if (this.useXORMask) {
            this.xorData = new byte[this.dataSpace];
        }
        try {
            while (length > 0) {
                int readLen = Math.min(length, readSpace);
                int cnt = in.read(data, 0, readLen);
                if (cnt < 0) {
                    break;
                }
                this.putBytes(index, offset, data, 0, readLen);
                offset += cnt;
                length -= cnt;
                if ((readSpace -= cnt) != 0) continue;
                ++index;
                readSpace = this.dataSpace;
                offset = 0;
            }
        }
        finally {
            this.xorData = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized int put(int offset, byte[] data, int dataOffset, int length) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        if (this.dataBufferIdTable == null) {
            throw new AssertException("Invalid Buffer");
        }
        if (offset < 0 || offset + length - 1 >= this.size) {
            throw new ArrayIndexOutOfBoundsException();
        }
        int index = offset / this.dataSpace;
        int bufferDataOffset = offset % this.dataSpace;
        int len = length;
        if (this.useXORMask) {
            this.xorData = new byte[this.dataSpace];
        }
        try {
            while (len > 0) {
                int n = this.putBytes(index++, bufferDataOffset, data, dataOffset, len);
                dataOffset += n;
                len -= n;
                bufferDataOffset = 0;
            }
            int n = offset + length;
            return n;
        }
        finally {
            this.xorData = null;
        }
    }

    @Override
    public synchronized int put(int offset, byte[] bytes) throws IOException {
        return this.put(offset, bytes, 0, bytes.length);
    }

    @Override
    public synchronized int putByte(int offset, byte b) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        if (this.dataBufferIdTable == null) {
            throw new AssertException("Invalid Buffer");
        }
        if (offset < 0 || offset >= this.size) {
            throw new ArrayIndexOutOfBoundsException();
        }
        DataBuffer buffer = this.getBuffer(offset / this.dataSpace);
        int bufferDataOffset = offset % this.dataSpace;
        if (this.useXORMask) {
            b = this.xorMaskByte(bufferDataOffset, b);
        }
        buffer.putByte(this.dataBaseOffset + bufferDataOffset, b);
        this.bufferMgr.releaseBuffer(buffer);
        return offset + 1;
    }

    @Override
    public synchronized int putInt(int offset, int v) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        int bufferOffset = this.dataBaseOffset + offset % this.dataSpace;
        if (bufferOffset + 3 <= this.dataSpace) {
            if (this.dataBufferIdTable == null) {
                throw new AssertException("Invalid Buffer");
            }
            if (offset < 0 || offset + 3 >= this.size) {
                throw new ArrayIndexOutOfBoundsException();
            }
            if (this.useXORMask) {
                v ^= (int)this.getXorMask(offset % this.dataSpace, 4);
            }
            DataBuffer buffer = this.getBuffer(offset / this.dataSpace);
            buffer.putInt(bufferOffset, v);
            this.bufferMgr.releaseBuffer(buffer);
        } else {
            byte[] data = new byte[]{(byte)(v >> 24), (byte)(v >> 16), (byte)(v >> 8), (byte)v};
            this.put(offset, data);
        }
        return offset + 4;
    }

    @Override
    public synchronized int putLong(int offset, long v) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        int bufferOffset = this.dataBaseOffset + offset % this.dataSpace;
        if (bufferOffset + 7 <= this.dataSpace) {
            if (this.dataBufferIdTable == null) {
                throw new AssertException("Invalid Buffer");
            }
            if (offset < 0 || offset + 7 >= this.size) {
                throw new ArrayIndexOutOfBoundsException();
            }
            if (this.useXORMask) {
                v ^= this.getXorMask(offset % this.dataSpace, 8);
            }
            DataBuffer buffer = this.getBuffer(offset / this.dataSpace);
            buffer.putLong(bufferOffset, v);
            this.bufferMgr.releaseBuffer(buffer);
        } else {
            byte[] data = new byte[]{(byte)(v >> 56), (byte)(v >> 48), (byte)(v >> 40), (byte)(v >> 32), (byte)(v >> 24), (byte)(v >> 16), (byte)(v >> 8), (byte)v};
            this.put(offset, data);
        }
        return offset + 8;
    }

    @Override
    public synchronized int putShort(int offset, short v) throws IOException {
        if (this.readOnly) {
            throw new UnsupportedOperationException("Read-only buffer");
        }
        int bufferOffset = this.dataBaseOffset + offset % this.dataSpace;
        if (bufferOffset + 1 <= this.dataSpace) {
            if (this.dataBufferIdTable == null) {
                throw new AssertException("Invalid Buffer");
            }
            if (offset < 0 || offset + 1 >= this.size) {
                throw new ArrayIndexOutOfBoundsException();
            }
            if (this.useXORMask) {
                v = (short)(v ^ (short)this.getXorMask(offset % this.dataSpace, 2));
            }
            DataBuffer buffer = this.getBuffer(offset / this.dataSpace);
            buffer.putShort(bufferOffset, v);
            this.bufferMgr.releaseBuffer(buffer);
        } else {
            byte[] data = new byte[]{(byte)(v >> 8), (byte)v};
            this.put(offset, data);
        }
        return offset + 2;
    }

    private DataBuffer getBuffer(int index) throws IOException {
        int bufferId;
        int n = bufferId = this.dataBufferIdTable == null && index == 0 ? this.firstBufferId : this.dataBufferIdTable[index];
        if (bufferId < 0) {
            DataBuffer buf = this.bufferMgr.createBuffer();
            this.initializeAllocatedBuffer(index, buf);
            this.addBuffer(index, buf);
            return buf;
        }
        return this.bufferMgr.getBuffer(bufferId);
    }

    private void initializeAllocatedBuffer(int chainBufferIndex, DataBuffer buf) throws IOException {
        int offset = chainBufferIndex * this.dataSpace;
        int len = this.size - offset;
        if (len >= this.dataSpace) {
            len = this.dataSpace;
        } else {
            buf.clear();
        }
        byte[] data = new byte[len];
        if (this.uninitializedDataSource != null) {
            this.uninitializedDataSource.get(this.uninitializedDataSourceOffset + offset, data, 0, len);
        }
        if (this.useXORMask) {
            for (int i = 0; i < len; ++i) {
                data[i] = this.xorMaskByte(i, data[i]);
            }
        }
        buf.put(this.dataBaseOffset, data);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addBuffer(int index, DataBuffer buf) throws IOException {
        buf.putByte(0, (byte)9);
        this.dataBufferIdTable[index] = buf.getId();
        int indexBufferId = this.indexBufferIdTable[index / this.indexesPerBuffer];
        int indexOffset = 9 + index % this.indexesPerBuffer * 4;
        DataBuffer indexBuffer = null;
        try {
            indexBuffer = this.bufferMgr.getBuffer(indexBufferId);
            indexBuffer.putInt(indexOffset, buf.getId());
            this.bufferMgr.releaseBuffer(indexBuffer);
        }
        finally {
            if (indexBuffer == null) {
                this.bufferMgr.releaseBuffer(buf);
            }
        }
    }
}

