/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.storage.managers;

import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.stream.Stream;
import org.jboss.logging.Logger;
import org.keycloak.cluster.ClusterEvent;
import org.keycloak.cluster.ClusterListener;
import org.keycloak.cluster.ClusterProvider;
import org.keycloak.cluster.ExecutionResult;
import org.keycloak.common.util.Time;
import org.keycloak.component.ComponentModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.KeycloakSessionTask;
import org.keycloak.models.LegacyRealmModel;
import org.keycloak.models.RealmModel;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.storage.UserStorageProvider;
import org.keycloak.storage.UserStorageProviderFactory;
import org.keycloak.storage.UserStorageProviderModel;
import org.keycloak.storage.user.ImportSynchronization;
import org.keycloak.storage.user.SynchronizationResult;
import org.keycloak.timer.TimerProvider;

public class UserStorageSyncManager {
    private static final String USER_STORAGE_TASK_KEY = "user-storage";
    private static final Logger logger = Logger.getLogger(UserStorageSyncManager.class);

    public static void bootstrapPeriodic(final KeycloakSessionFactory sessionFactory, final TimerProvider timer) {
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                Stream realms = session.realms().getRealmsWithProviderTypeStream(UserStorageProvider.class);
                realms.forEach(realm -> {
                    Stream providers = ((LegacyRealmModel)realm).getUserStorageProvidersStream();
                    providers.forEachOrdered(provider -> {
                        UserStorageProviderFactory factory = (UserStorageProviderFactory)session.getKeycloakSessionFactory().getProviderFactory(UserStorageProvider.class, provider.getProviderId());
                        if (factory instanceof ImportSynchronization && provider.isImportEnabled()) {
                            UserStorageSyncManager.refreshPeriodicSyncForProvider(sessionFactory, timer, provider, realm.getId());
                        }
                    });
                });
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                clusterProvider.registerListener(UserStorageSyncManager.USER_STORAGE_TASK_KEY, (ClusterListener)new UserStorageClusterListener(sessionFactory));
            }
        });
    }

    public static SynchronizationResult syncAllUsers(final KeycloakSessionFactory sessionFactory, final String realmId, final UserStorageProviderModel provider) {
        final UserStorageProviderFactory factory = (UserStorageProviderFactory)sessionFactory.getProviderFactory(UserStorageProvider.class, provider.getProviderId());
        if (!(factory instanceof ImportSynchronization && provider.isImportEnabled() && provider.isEnabled())) {
            return SynchronizationResult.ignored();
        }
        final Holder holder = new Holder();
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                String taskKey = provider.getId() + "::sync";
                int timeout = Math.max(30, provider.getFullSyncPeriod());
                holder.result = clusterProvider.executeIfNotExecuted(taskKey, timeout, (Callable)new Callable<SynchronizationResult>(){

                    @Override
                    public SynchronizationResult call() throws Exception {
                        int lastSync = Time.currentTime();
                        SynchronizationResult result = ((ImportSynchronization)factory).sync(sessionFactory, realmId, provider);
                        if (!result.isIgnored()) {
                            UserStorageSyncManager.updateLastSyncInterval(sessionFactory, provider, realmId, lastSync);
                        }
                        return result;
                    }
                });
            }
        });
        if (holder.result == null || !holder.result.isExecuted()) {
            logger.debugf("syncAllUsers for federation provider %s was ignored as it's already in progress", (Object)provider.getName());
            return SynchronizationResult.ignored();
        }
        return (SynchronizationResult)holder.result.getResult();
    }

    public static SynchronizationResult syncChangedUsers(final KeycloakSessionFactory sessionFactory, final String realmId, final UserStorageProviderModel provider) {
        final UserStorageProviderFactory factory = (UserStorageProviderFactory)sessionFactory.getProviderFactory(UserStorageProvider.class, provider.getProviderId());
        if (!(factory instanceof ImportSynchronization && provider.isImportEnabled() && provider.isEnabled())) {
            return SynchronizationResult.ignored();
        }
        final Holder holder = new Holder();
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                String taskKey = provider.getId() + "::sync";
                int timeout = Math.max(30, provider.getChangedSyncPeriod());
                holder.result = clusterProvider.executeIfNotExecuted(taskKey, timeout, (Callable)new Callable<SynchronizationResult>(){

                    @Override
                    public SynchronizationResult call() throws Exception {
                        int oldLastSync = provider.getLastSync();
                        int lastSync = Time.currentTime();
                        SynchronizationResult result = ((ImportSynchronization)factory).syncSince(Time.toDate((int)oldLastSync), sessionFactory, realmId, provider);
                        if (!result.isIgnored()) {
                            UserStorageSyncManager.updateLastSyncInterval(sessionFactory, provider, realmId, lastSync);
                        }
                        return result;
                    }
                });
            }
        });
        if (holder.result == null || !holder.result.isExecuted()) {
            logger.debugf("syncChangedUsers for federation provider %s was ignored as it's already in progress", (Object)provider.getName());
            return SynchronizationResult.ignored();
        }
        return (SynchronizationResult)holder.result.getResult();
    }

    public static void notifyToRefreshPeriodicSyncAll(KeycloakSession session, RealmModel realm, boolean removed) {
        ((LegacyRealmModel)realm).getUserStorageProvidersStream().forEachOrdered(fedProvider -> UserStorageSyncManager.notifyToRefreshPeriodicSync(session, realm, fedProvider, removed));
    }

    public static void notifyToRefreshPeriodicSyncSingle(KeycloakSession session, RealmModel realm, ComponentModel component, boolean removed) {
        UserStorageSyncManager.notifyToRefreshPeriodicSync(session, realm, new UserStorageProviderModel(component), removed);
    }

    public static void notifyToRefreshPeriodicSync(KeycloakSession session, RealmModel realm, UserStorageProviderModel provider, boolean removed) {
        UserStorageProviderFactory factory = (UserStorageProviderFactory)session.getKeycloakSessionFactory().getProviderFactory(UserStorageProvider.class, provider.getProviderId());
        if (!(factory instanceof ImportSynchronization) || !provider.isImportEnabled()) {
            return;
        }
        ClusterProvider cp = (ClusterProvider)session.getProvider(ClusterProvider.class);
        if (cp != null) {
            UserStorageProviderClusterEvent event = UserStorageProviderClusterEvent.createEvent(removed, realm.getId(), provider);
            cp.notify(USER_STORAGE_TASK_KEY, (ClusterEvent)event, false, ClusterProvider.DCNotify.ALL_DCS);
        }
    }

    protected static void refreshPeriodicSyncForProvider(final KeycloakSessionFactory sessionFactory, TimerProvider timer, final UserStorageProviderModel provider, final String realmId) {
        logger.debugf("Going to refresh periodic sync for provider '%s' . Full sync period: %d , changed users sync period: %d", (Object)provider.getName(), (Object)provider.getFullSyncPeriod(), (Object)provider.getChangedSyncPeriod());
        if (provider.getFullSyncPeriod() > 0) {
            timer.schedule(new Runnable(){

                @Override
                public void run() {
                    try {
                        boolean shouldPerformSync = UserStorageSyncManager.shouldPerformNewPeriodicSync(provider.getLastSync(), provider.getChangedSyncPeriod());
                        if (shouldPerformSync) {
                            UserStorageSyncManager.syncAllUsers(sessionFactory, realmId, provider);
                        } else {
                            logger.debugf("Ignored periodic full sync with storage provider %s due small time since last sync", (Object)provider.getName());
                        }
                    }
                    catch (Throwable t) {
                        logger.error((Object)"Error occurred during full sync of users", t);
                    }
                }
            }, (long)(provider.getFullSyncPeriod() * 1000), provider.getId() + "-FULL");
        } else {
            timer.cancelTask(provider.getId() + "-FULL");
        }
        if (provider.getChangedSyncPeriod() > 0) {
            timer.schedule(new Runnable(){

                @Override
                public void run() {
                    try {
                        boolean shouldPerformSync = UserStorageSyncManager.shouldPerformNewPeriodicSync(provider.getLastSync(), provider.getChangedSyncPeriod());
                        if (shouldPerformSync) {
                            UserStorageSyncManager.syncChangedUsers(sessionFactory, realmId, provider);
                        } else {
                            logger.debugf("Ignored periodic changed-users sync with storage provider %s due small time since last sync", (Object)provider.getName());
                        }
                    }
                    catch (Throwable t) {
                        logger.error((Object)"Error occurred during sync of changed users", t);
                    }
                }
            }, (long)(provider.getChangedSyncPeriod() * 1000), provider.getId() + "-CHANGED");
        } else {
            timer.cancelTask(provider.getId() + "-CHANGED");
        }
    }

    private static boolean shouldPerformNewPeriodicSync(int lastSyncTime, int period) {
        if (lastSyncTime <= 0) {
            return true;
        }
        int currentTime = Time.currentTime();
        int timeSinceLastSync = currentTime - lastSyncTime;
        return timeSinceLastSync * 2 > period;
    }

    protected static void removePeriodicSyncForProvider(TimerProvider timer, UserStorageProviderModel fedProvider) {
        logger.debugf("Removing periodic sync for provider %s", (Object)fedProvider.getName());
        timer.cancelTask(fedProvider.getId() + "-FULL");
        timer.cancelTask(fedProvider.getId() + "-CHANGED");
    }

    private static void updateLastSyncInterval(KeycloakSessionFactory sessionFactory, final UserStorageProviderModel provider, final String realmId, final int lastSync) {
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                RealmModel persistentRealm = session.realms().getRealm(realmId);
                ((LegacyRealmModel)persistentRealm).getUserStorageProvidersStream().filter(persistentFedProvider -> Objects.equals(provider.getId(), persistentFedProvider.getId())).forEachOrdered(persistentFedProvider -> {
                    persistentFedProvider.setLastSync(lastSync);
                    persistentRealm.updateComponent((ComponentModel)persistentFedProvider);
                    provider.setLastSync(lastSync);
                });
            }
        });
    }

    public static class UserStorageProviderClusterEvent
    implements ClusterEvent {
        private boolean removed;
        private String realmId;
        private UserStorageProviderModel storageProvider;

        public boolean isRemoved() {
            return this.removed;
        }

        public void setRemoved(boolean removed) {
            this.removed = removed;
        }

        public String getRealmId() {
            return this.realmId;
        }

        public void setRealmId(String realmId) {
            this.realmId = realmId;
        }

        public UserStorageProviderModel getStorageProvider() {
            return this.storageProvider;
        }

        public void setStorageProvider(UserStorageProviderModel federationProvider) {
            this.storageProvider = federationProvider;
        }

        public static UserStorageProviderClusterEvent createEvent(boolean removed, String realmId, UserStorageProviderModel provider) {
            UserStorageProviderClusterEvent notification = new UserStorageProviderClusterEvent();
            notification.setRemoved(removed);
            notification.setRealmId(realmId);
            notification.setStorageProvider(provider);
            return notification;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            UserStorageProviderClusterEvent that = (UserStorageProviderClusterEvent)o;
            return this.removed == that.removed && Objects.equals(this.realmId, that.realmId) && Objects.equals(this.storageProvider.getId(), that.storageProvider.getId());
        }

        public int hashCode() {
            return Objects.hash(this.removed, this.realmId, this.storageProvider.getId());
        }
    }

    private static class UserStorageClusterListener
    implements ClusterListener {
        private final KeycloakSessionFactory sessionFactory;

        public UserStorageClusterListener(KeycloakSessionFactory sessionFactory) {
            this.sessionFactory = sessionFactory;
        }

        public void eventReceived(ClusterEvent event) {
            final UserStorageProviderClusterEvent fedEvent = (UserStorageProviderClusterEvent)event;
            KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)this.sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

                public void run(KeycloakSession session) {
                    TimerProvider timer = (TimerProvider)session.getProvider(TimerProvider.class);
                    if (fedEvent.isRemoved()) {
                        UserStorageSyncManager.removePeriodicSyncForProvider(timer, fedEvent.getStorageProvider());
                    } else {
                        UserStorageSyncManager.refreshPeriodicSyncForProvider(sessionFactory, timer, fedEvent.getStorageProvider(), fedEvent.getRealmId());
                    }
                }
            });
        }
    }

    private static class Holder {
        ExecutionResult<SynchronizationResult> result;

        private Holder() {
        }
    }
}

