#!/usr/bin/python
#
# Copyright (c) 2019 Liu Qingyi, (@smile37773)
#
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function
__metaclass__ = type


DOCUMENTATION = '''
---
module: azure_rm_gallery_info
version_added: "0.1.2"
short_description: Get Azure Shared Image Gallery info
description:
    - Get info of Azure Shared Image Gallery.
options:
    resource_group:
        description:
            - The name of the resource group.
        type: str
    name:
        description:
            - Resource name
        type: str
extends_documentation_fragment:
    - azure.azcollection.azure
author:
    - Liu Qingyi (@smile37773)

'''

EXAMPLES = '''
- name: List galleries in a subscription.
  azure_rm_gallery_info:
- name: List galleries in a resource group.
  azure_rm_gallery_info:
    resource_group: myResourceGroup
- name: Get a gallery.
  azure_rm_gallery_info:
    resource_group: myResourceGroup
    name: myGallery
'''

RETURN = '''
galleries:
    description:
        - A list of dict results where the key is the name of the gallery and the values are the info for that gallery.
    returned: always
    type: complex
    contains:
        id:
            description:
                - Resource ID.
            returned: always
            type: str
            sample: "/subscriptions/xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx/resourceGroups/myResourceGroup/providers/Microsoft.Compute/galleries/myGallery"
        name:
            description:
                - Resource name.
            returned: always
            type: str
            sample: "myGallery"
        location:
            description:
                - Resource location.
            returned: always
            type: str
            sample: "eastus"
        tags:
            description:
                - Resource tags.
            returned: always
            type: dict
            sample: { "tag": "value" }
        description:
            description:
                - This is the gallery description.
            returned: always
            type: str
            sample: "This is the gallery description."
        provisioning_state:
            description:
                - The current state of the gallery.
            returned: always
            type: str
            sample: "Succeeded"

'''

import json
from ansible_collections.azure.azcollection.plugins.module_utils.azure_rm_common import AzureRMModuleBase
from ansible_collections.azure.azcollection.plugins.module_utils.azure_rm_common_rest import GenericRestClient


class AzureRMGalleriesInfo(AzureRMModuleBase):
    def __init__(self):
        self.module_arg_spec = dict(
            resource_group=dict(
                type='str'
            ),
            name=dict(
                type='str'
            )
        )

        self.resource_group = None
        self.name = None

        self.results = dict(changed=False)
        self.mgmt_client = None
        self.state = None
        self.url = None
        self.status_code = [200]

        self.query_parameters = {}
        self.query_parameters['api-version'] = '2019-03-01'
        self.header_parameters = {}
        self.header_parameters['Content-Type'] = 'application/json; charset=utf-8'

        self.mgmt_client = None
        super(AzureRMGalleriesInfo, self).__init__(self.module_arg_spec, supports_check_mode=True, supports_tags=False)

    def exec_module(self, **kwargs):

        for key in self.module_arg_spec:
            setattr(self, key, kwargs[key])

        self.mgmt_client = self.get_mgmt_svc_client(GenericRestClient,
                                                    base_url=self._cloud_environment.endpoints.resource_manager)

        if (self.resource_group is not None and self.name is not None):
            # self.results['galleries'] = self.format_item(self.get())
            self.results['galleries'] = self.get()
        elif (self.resource_group is not None):
            # self.results['galleries'] = self.format_item(self.listbyresourcegroup())
            self.results['galleries'] = self.listbyresourcegroup()
        else:
            # self.results['galleries'] = [self.format_item(self.list())]
            self.results['galleries'] = self.list()
        return self.results

    def get(self):
        response = None
        results = {}
        # prepare url
        self.url = ('/subscriptions' +
                    '/{{ subscription_id }}' +
                    '/resourceGroups' +
                    '/{{ resource_group }}' +
                    '/providers' +
                    '/Microsoft.Compute' +
                    '/galleries' +
                    '/{{ gallery_name }}')
        self.url = self.url.replace('{{ subscription_id }}', self.subscription_id)
        self.url = self.url.replace('{{ resource_group }}', self.resource_group)
        self.url = self.url.replace('{{ gallery_name }}', self.name)

        try:
            response = self.mgmt_client.query(self.url,
                                              'GET',
                                              self.query_parameters,
                                              self.header_parameters,
                                              None,
                                              self.status_code,
                                              600,
                                              30)
            results = json.loads(response.body())
            # self.log('Response : {0}'.format(response))
        except Exception as e:
            self.log('Could not get info for @(Model.ModuleOperationNameUpper).')

        return self.format_item(results)

    def listbyresourcegroup(self):
        response = None
        results = {}
        # prepare url
        self.url = ('/subscriptions' +
                    '/{{ subscription_id }}' +
                    '/resourceGroups' +
                    '/{{ resource_group }}' +
                    '/providers' +
                    '/Microsoft.Compute' +
                    '/galleries')
        self.url = self.url.replace('{{ subscription_id }}', self.subscription_id)
        self.url = self.url.replace('{{ resource_group }}', self.resource_group)

        try:
            response = self.mgmt_client.query(self.url,
                                              'GET',
                                              self.query_parameters,
                                              self.header_parameters,
                                              None,
                                              self.status_code,
                                              600,
                                              30)
            results = json.loads(response.body())
            # self.log('Response : {0}'.format(response))
        except Exception as e:
            self.log('Could not get info for @(Model.ModuleOperationNameUpper).')

        return [self.format_item(x) for x in results['value']] if results['value'] else []

    def list(self):
        response = None
        results = {}
        # prepare url
        self.url = ('/subscriptions' +
                    '/{{ subscription_id }}' +
                    '/providers' +
                    '/Microsoft.Compute' +
                    '/galleries')
        self.url = self.url.replace('{{ subscription_id }}', self.subscription_id)

        try:
            response = self.mgmt_client.query(self.url,
                                              'GET',
                                              self.query_parameters,
                                              self.header_parameters,
                                              None,
                                              self.status_code,
                                              600,
                                              30)
            results = json.loads(response.body())
            # self.log('Response : {0}'.format(response))
        except Exception as e:
            self.log('Could not get info for @(Model.ModuleOperationNameUpper).')

        return [self.format_item(x) for x in results['value']] if results['value'] else []

    def format_item(self, item):
        if not item:
            return
        d = {
            'id': item['id'],
            'name': item['name'],
            'location': item['location'],
            'tags': item.get('tags'),
            'description': item['properties']['description'],
            'provisioning_state': item['properties']['provisioningState']
        }
        return d


def main():
    AzureRMGalleriesInfo()


if __name__ == '__main__':
    main()
