"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var vega_util_1 = require("vega-util");
var aggregate_1 = require("./aggregate");
var channel_1 = require("./channel");
var encoding_1 = require("./encoding");
var fielddef_1 = require("./fielddef");
var log = require("./log");
var mark_1 = require("./mark");
var scale_1 = require("./scale");
var util_1 = require("./util");
var STACK_OFFSET_INDEX = {
    zero: 1,
    center: 1,
    normalize: 1
};
function isStackOffset(s) {
    return !!STACK_OFFSET_INDEX[s];
}
exports.isStackOffset = isStackOffset;
exports.STACKABLE_MARKS = [mark_1.BAR, mark_1.AREA, mark_1.RULE, mark_1.POINT, mark_1.CIRCLE, mark_1.SQUARE, mark_1.LINE, mark_1.TEXT, mark_1.TICK];
exports.STACK_BY_DEFAULT_MARKS = [mark_1.BAR, mark_1.AREA];
function potentialStackedChannel(encoding) {
    var xDef = encoding.x;
    var yDef = encoding.y;
    if (fielddef_1.isFieldDef(xDef) && fielddef_1.isFieldDef(yDef)) {
        if (xDef.type === 'quantitative' && yDef.type === 'quantitative') {
            if (xDef.stack) {
                return 'x';
            }
            else if (yDef.stack) {
                return 'y';
            }
            // if there is no explicit stacking, only apply stack if there is only one aggregate for x or y
            if ((!!xDef.aggregate) !== (!!yDef.aggregate)) {
                return xDef.aggregate ? 'x' : 'y';
            }
        }
        else if (xDef.type === 'quantitative') {
            return 'x';
        }
        else if (yDef.type === 'quantitative') {
            return 'y';
        }
    }
    else if (fielddef_1.isFieldDef(xDef) && xDef.type === 'quantitative') {
        return 'x';
    }
    else if (fielddef_1.isFieldDef(yDef) && yDef.type === 'quantitative') {
        return 'y';
    }
    return undefined;
}
// Note: CompassQL uses this method and only pass in required properties of each argument object.
// If required properties change, make sure to update CompassQL.
function stack(m, encoding, stackConfig) {
    var mark = mark_1.isMarkDef(m) ? m.type : m;
    // Should have stackable mark
    if (!util_1.contains(exports.STACKABLE_MARKS, mark)) {
        return null;
    }
    var fieldChannel = potentialStackedChannel(encoding);
    if (!fieldChannel) {
        return null;
    }
    var stackedFieldDef = encoding[fieldChannel];
    var stackedField = fielddef_1.isStringFieldDef(stackedFieldDef) ? fielddef_1.vgField(stackedFieldDef, {}) : undefined;
    var dimensionChannel = fieldChannel === 'x' ? 'y' : 'x';
    var dimensionDef = encoding[dimensionChannel];
    var dimensionField = fielddef_1.isStringFieldDef(dimensionDef) ? fielddef_1.vgField(dimensionDef, {}) : undefined;
    // Should have grouping level of detail that is different from the dimension field
    var stackBy = channel_1.NONPOSITION_CHANNELS.reduce(function (sc, channel) {
        if (encoding_1.channelHasField(encoding, channel)) {
            var channelDef = encoding[channel];
            (vega_util_1.isArray(channelDef) ? channelDef : [channelDef]).forEach(function (cDef) {
                var fieldDef = fielddef_1.getFieldDef(cDef);
                if (fieldDef.aggregate) {
                    return;
                }
                // Check whether the channel's field is identical to x/y's field or if the channel is a repeat
                var f = fielddef_1.isStringFieldDef(fieldDef) ? fielddef_1.vgField(fieldDef, {}) : undefined;
                if (
                // if fielddef is a repeat, just include it in the stack by
                !f ||
                    // otherwise, the field must be different from x and y fields.
                    (f !== dimensionField && f !== stackedField)) {
                    sc.push({ channel: channel, fieldDef: fieldDef });
                }
            });
        }
        return sc;
    }, []);
    if (stackBy.length === 0) {
        return null;
    }
    // Automatically determine offset
    var offset = undefined;
    if (stackedFieldDef.stack !== undefined) {
        offset = stackedFieldDef.stack;
    }
    else if (util_1.contains(exports.STACK_BY_DEFAULT_MARKS, mark)) {
        // Bar and Area with sum ops are automatically stacked by default
        offset = stackConfig === undefined ? 'zero' : stackConfig;
    }
    else {
        offset = stackConfig;
    }
    if (!offset || !isStackOffset(offset)) {
        return null;
    }
    // If stacked, check scale type if it is linear
    if (stackedFieldDef.scale && stackedFieldDef.scale.type && stackedFieldDef.scale.type !== scale_1.ScaleType.LINEAR) {
        log.warn(log.message.cannotStackNonLinearScale(stackedFieldDef.scale.type));
        return null;
    }
    // Check if it is a ranged mark
    if (encoding_1.channelHasField(encoding, fieldChannel === channel_1.X ? channel_1.X2 : channel_1.Y2)) {
        log.warn(log.message.cannotStackRangedMark(fieldChannel));
        return null;
    }
    // Warn if stacking summative aggregate
    if (stackedFieldDef.aggregate && !util_1.contains(aggregate_1.SUM_OPS, stackedFieldDef.aggregate)) {
        log.warn(log.message.stackNonSummativeAggregate(stackedFieldDef.aggregate));
    }
    return {
        groupbyChannel: dimensionDef ? dimensionChannel : undefined,
        fieldChannel: fieldChannel,
        impute: mark_1.isPathMark(mark),
        stackBy: stackBy,
        offset: offset
    };
}
exports.stack = stack;
//# sourceMappingURL=data:application/json;base64,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