"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var vega_util_1 = require("vega-util");
var selection_1 = require("./compile/selection/selection");
var datetime_1 = require("./datetime");
var fielddef_1 = require("./fielddef");
var timeunit_1 = require("./timeunit");
var util_1 = require("./util");
function isSelectionPredicate(predicate) {
    return predicate && predicate['selection'];
}
exports.isSelectionPredicate = isSelectionPredicate;
function isFieldEqualPredicate(predicate) {
    return predicate && !!predicate.field && predicate.equal !== undefined;
}
exports.isFieldEqualPredicate = isFieldEqualPredicate;
function isFieldRangePredicate(predicate) {
    if (predicate && predicate.field) {
        if (vega_util_1.isArray(predicate.range) && predicate.range.length === 2) {
            return true;
        }
    }
    return false;
}
exports.isFieldRangePredicate = isFieldRangePredicate;
function isFieldOneOfPredicate(predicate) {
    return predicate && !!predicate.field && (vega_util_1.isArray(predicate.oneOf) ||
        vega_util_1.isArray(predicate.in) // backward compatibility
    );
}
exports.isFieldOneOfPredicate = isFieldOneOfPredicate;
function isFieldPredicate(predicate) {
    return isFieldOneOfPredicate(predicate) || isFieldEqualPredicate(predicate) || isFieldRangePredicate(predicate);
}
exports.isFieldPredicate = isFieldPredicate;
/**
 * Converts a predicate into an expression.
 */
// model is only used for selection filters.
function expression(model, filterOp, node) {
    return util_1.logicalExpr(filterOp, function (predicate) {
        if (vega_util_1.isString(predicate)) {
            return predicate;
        }
        else if (isSelectionPredicate(predicate)) {
            return selection_1.selectionPredicate(model, predicate.selection, node);
        }
        else { // Filter Object
            return fieldFilterExpression(predicate);
        }
    });
}
exports.expression = expression;
// This method is used by Voyager.  Do not change its behavior without changing Voyager.
function fieldFilterExpression(predicate, useInRange) {
    if (useInRange === void 0) { useInRange = true; }
    var fieldExpr = predicate.timeUnit ?
        // For timeUnit, cast into integer with time() so we can use ===, inrange, indexOf to compare values directly.
        // TODO: We calculate timeUnit on the fly here. Consider if we would like to consolidate this with timeUnit pipeline
        // TODO: support utc
        ('time(' + timeunit_1.fieldExpr(predicate.timeUnit, predicate.field) + ')') :
        fielddef_1.vgField(predicate, { expr: 'datum' });
    if (isFieldEqualPredicate(predicate)) {
        return fieldExpr + '===' + valueExpr(predicate.equal, predicate.timeUnit);
    }
    else if (isFieldOneOfPredicate(predicate)) {
        // "oneOf" was formerly "in" -- so we need to add backward compatibility
        var oneOf = predicate.oneOf || predicate['in'];
        return 'indexof([' +
            oneOf.map(function (v) { return valueExpr(v, predicate.timeUnit); }).join(',') +
            '], ' + fieldExpr + ') !== -1';
    }
    else if (isFieldRangePredicate(predicate)) {
        var lower = predicate.range[0];
        var upper = predicate.range[1];
        if (lower !== null && upper !== null && useInRange) {
            return 'inrange(' + fieldExpr + ', [' +
                valueExpr(lower, predicate.timeUnit) + ', ' +
                valueExpr(upper, predicate.timeUnit) + '])';
        }
        var exprs = [];
        if (lower !== null) {
            exprs.push(fieldExpr + " >= " + valueExpr(lower, predicate.timeUnit));
        }
        if (upper !== null) {
            exprs.push(fieldExpr + " <= " + valueExpr(upper, predicate.timeUnit));
        }
        return exprs.length > 0 ? exprs.join(' && ') : 'true';
    }
    /* istanbul ignore next: it should never reach here */
    throw new Error("Invalid field predicate: " + JSON.stringify(predicate));
}
exports.fieldFilterExpression = fieldFilterExpression;
function valueExpr(v, timeUnit) {
    if (datetime_1.isDateTime(v)) {
        var expr = datetime_1.dateTimeExpr(v, true);
        return 'time(' + expr + ')';
    }
    if (timeunit_1.isLocalSingleTimeUnit(timeUnit)) {
        var datetime = {};
        datetime[timeUnit] = v;
        var expr = datetime_1.dateTimeExpr(datetime, true);
        return 'time(' + expr + ')';
    }
    else if (timeunit_1.isUtcSingleTimeUnit(timeUnit)) {
        return valueExpr(v, timeunit_1.getLocalTimeUnit(timeUnit));
    }
    return JSON.stringify(v);
}
function normalizePredicate(f) {
    if (isFieldPredicate(f) && f.timeUnit) {
        return tslib_1.__assign({}, f, { timeUnit: timeunit_1.normalizeTimeUnit(f.timeUnit) });
    }
    return f;
}
exports.normalizePredicate = normalizePredicate;
//# sourceMappingURL=data:application/json;base64,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