/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.cloud;

import java.lang.invoke.MethodHandles;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.solr.common.util.ObjectReleaseTracker;
import org.apache.solr.common.util.Utils;
import org.apache.solr.core.CoreDescriptor;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkShardTerms
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final Object writingLock = new Object();
    private final String collection;
    private final String shard;
    private final String znodePath;
    private final SolrZkClient zkClient;
    private final Set<CoreTermWatcher> listeners = new HashSet<CoreTermWatcher>();
    private final AtomicBoolean isClosed = new AtomicBoolean(false);
    private Terms terms;

    public ZkShardTerms(String collection, String shard, SolrZkClient zkClient) {
        this.znodePath = "/collections/" + collection + "/terms/" + shard;
        this.collection = collection;
        this.shard = shard;
        this.zkClient = zkClient;
        this.ensureTermNodeExist();
        this.refreshTerms();
        this.retryRegisterWatcher();
        ObjectReleaseTracker.track((Object)this);
    }

    public void ensureTermsIsHigher(String leader, Set<String> replicasNeedingRecovery) {
        Terms newTerms;
        if (replicasNeedingRecovery.isEmpty()) {
            return;
        }
        while ((newTerms = this.terms.increaseTerms(leader, replicasNeedingRecovery)) != null) {
            if (!this.forceSaveTerms(newTerms)) continue;
            return;
        }
    }

    public boolean canBecomeLeader(String coreNodeName) {
        return this.terms.canBecomeLeader(coreNodeName);
    }

    public boolean skipSendingUpdatesTo(String coreNodeName) {
        return !this.terms.haveHighestTermValue(coreNodeName);
    }

    public boolean registered(String coreNodeName) {
        return this.terms.getTerm(coreNodeName) != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.isClosed.set(true);
        Set<CoreTermWatcher> set = this.listeners;
        synchronized (set) {
            this.listeners.clear();
        }
        ObjectReleaseTracker.release((Object)this);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Map<String, Long> getTerms() {
        Object object = this.writingLock;
        synchronized (object) {
            return new HashMap<String, Long>(this.terms.values);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addListener(CoreTermWatcher listener) {
        Set<CoreTermWatcher> set = this.listeners;
        synchronized (set) {
            this.listeners.add(listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean removeTerm(CoreDescriptor cd) {
        int numListeners;
        Set<CoreTermWatcher> set = this.listeners;
        synchronized (set) {
            this.listeners.removeIf(coreTermWatcher -> !coreTermWatcher.onTermChanged(this.terms));
            numListeners = this.listeners.size();
        }
        return this.removeTerm(cd.getCloudDescriptor().getCoreNodeName()) || numListeners == 0;
    }

    boolean removeTerm(String coreNodeName) {
        Terms newTerms;
        while ((newTerms = this.terms.removeTerm(coreNodeName)) != null) {
            try {
                if (!this.saveTerms(newTerms)) continue;
                return false;
            }
            catch (KeeperException.NoNodeException e) {
                return true;
            }
        }
        return true;
    }

    void registerTerm(String coreNodeName) {
        Terms newTerms;
        while ((newTerms = this.terms.registerTerm(coreNodeName)) != null && !this.forceSaveTerms(newTerms)) {
        }
    }

    public void setTermEqualsToLeader(String coreNodeName) {
        Terms newTerms;
        while ((newTerms = this.terms.setTermEqualsToLeader(coreNodeName)) != null && !this.forceSaveTerms(newTerms)) {
        }
    }

    public void setTermToZero(String coreNodeName) {
        Terms newTerms;
        while ((newTerms = this.terms.setTermToZero(coreNodeName)) != null && !this.forceSaveTerms(newTerms)) {
        }
    }

    public void startRecovering(String coreNodeName) {
        Terms newTerms;
        while ((newTerms = this.terms.startRecovering(coreNodeName)) != null && !this.forceSaveTerms(newTerms)) {
        }
    }

    public void doneRecovering(String coreNodeName) {
        Terms newTerms;
        while ((newTerms = this.terms.doneRecovering(coreNodeName)) != null && !this.forceSaveTerms(newTerms)) {
        }
    }

    public boolean isRecovering(String name) {
        return this.terms.values.containsKey(name + "_recovering");
    }

    public void ensureHighestTermsAreNotZero() {
        Terms newTerms;
        while ((newTerms = this.terms.ensureHighestTermsAreNotZero()) != null && !this.forceSaveTerms(newTerms)) {
        }
    }

    public long getHighestTerm() {
        return this.terms.getMaxTerm();
    }

    public long getTerm(String coreNodeName) {
        Long term = this.terms.getTerm(coreNodeName);
        return term == null ? -1L : term;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int getNumListeners() {
        Set<CoreTermWatcher> set = this.listeners;
        synchronized (set) {
            return this.listeners.size();
        }
    }

    private boolean forceSaveTerms(Terms newTerms) {
        try {
            return this.saveTerms(newTerms);
        }
        catch (KeeperException.NoNodeException e) {
            this.ensureTermNodeExist();
            return false;
        }
    }

    private boolean saveTerms(Terms newTerms) throws KeeperException.NoNodeException {
        byte[] znodeData = Utils.toJSON((Object)newTerms.values);
        try {
            Stat stat = this.zkClient.setData(this.znodePath, znodeData, newTerms.version, true);
            this.setNewTerms(new Terms(newTerms.values, stat.getVersion()));
            log.info("Successful update of terms at {} to {}", (Object)this.znodePath, (Object)newTerms);
            return true;
        }
        catch (KeeperException.BadVersionException e) {
            log.info("Failed to save terms, version is not a match, retrying");
            this.refreshTerms();
        }
        catch (KeeperException.NoNodeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error while saving shard term for collection: " + this.collection, (Throwable)e);
        }
        return false;
    }

    private void ensureTermNodeExist() {
        String path = "/collections/" + this.collection + "/terms";
        try {
            path = path + "/" + this.shard;
            try {
                HashMap initialTerms = new HashMap();
                this.zkClient.makePath(path, Utils.toJSON(initialTerms), CreateMode.PERSISTENT, true);
            }
            catch (KeeperException.NodeExistsException initialTerms) {}
        }
        catch (InterruptedException e) {
            Thread.interrupted();
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error creating shard term node in Zookeeper for collection: " + this.collection, (Throwable)e);
        }
        catch (KeeperException e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error creating shard term node in Zookeeper for collection: " + this.collection, (Throwable)e);
        }
    }

    public void refreshTerms() {
        Terms newTerms;
        try {
            Stat stat = new Stat();
            byte[] data = this.zkClient.getData(this.znodePath, null, stat, true);
            newTerms = new Terms((Map)Utils.fromJSON((byte[])data), stat.getVersion());
        }
        catch (KeeperException e) {
            Thread.interrupted();
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error updating shard term for collection: " + this.collection, (Throwable)e);
        }
        catch (InterruptedException e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error updating shard term for collection: " + this.collection, (Throwable)e);
        }
        this.setNewTerms(newTerms);
    }

    private void retryRegisterWatcher() {
        while (!this.isClosed.get()) {
            try {
                this.registerWatcher();
                return;
            }
            catch (KeeperException.AuthFailedException | KeeperException.SessionExpiredException e) {
                this.isClosed.set(true);
                log.error("Failed watching shard term for collection: {} due to unrecoverable exception", (Object)this.collection, (Object)e);
                return;
            }
            catch (KeeperException e) {
                log.warn("Failed watching shard term for collection: {}, retrying!", (Object)this.collection, (Object)e);
                try {
                    this.zkClient.getConnectionManager().waitForConnected((long)this.zkClient.getZkClientTimeout());
                }
                catch (TimeoutException te) {
                    if (!Thread.interrupted()) continue;
                    throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error watching shard term for collection: " + this.collection, (Throwable)te);
                }
            }
        }
    }

    private void registerWatcher() throws KeeperException {
        Watcher watcher = event -> {
            if (Watcher.Event.EventType.None == event.getType()) {
                return;
            }
            this.retryRegisterWatcher();
            this.refreshTerms();
        };
        try {
            this.zkClient.exists(this.znodePath, watcher, true);
        }
        catch (InterruptedException e) {
            Thread.interrupted();
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error watching shard term for collection: " + this.collection, (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setNewTerms(Terms newTerms) {
        boolean isChanged = false;
        Object object = this.writingLock;
        synchronized (object) {
            if (this.terms == null || newTerms.version > this.terms.version) {
                this.terms = newTerms;
                isChanged = true;
            }
        }
        if (isChanged) {
            this.onTermUpdates(newTerms);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onTermUpdates(Terms newTerms) {
        Set<CoreTermWatcher> set = this.listeners;
        synchronized (set) {
            this.listeners.removeIf(coreTermWatcher -> !coreTermWatcher.onTermChanged(newTerms));
        }
    }

    static class Terms {
        private final Map<String, Long> values;
        private final long maxTerm;
        private final int version;

        public Terms() {
            this(new HashMap<String, Long>(), 0);
        }

        public Terms(Map<String, Long> values, int version) {
            this.values = values;
            this.version = version;
            this.maxTerm = values.isEmpty() ? 0L : Collections.max(values.values());
        }

        boolean canBecomeLeader(String coreNodeName) {
            return this.haveHighestTermValue(coreNodeName) && !this.values.containsKey(coreNodeName + "_recovering");
        }

        boolean haveHighestTermValue(String coreNodeName) {
            if (this.values.isEmpty()) {
                return true;
            }
            long maxTerm = Collections.max(this.values.values());
            return this.values.getOrDefault(coreNodeName, 0L) == maxTerm;
        }

        Long getTerm(String coreNodeName) {
            return this.values.get(coreNodeName);
        }

        Terms increaseTerms(String leader, Set<String> replicasNeedingRecovery) {
            if (!this.values.containsKey(leader)) {
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Can not find leader's term " + leader);
            }
            boolean changed = false;
            boolean foundReplicasInLowerTerms = false;
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            long leaderTerm = newValues.get(leader);
            for (String key : newValues.keySet()) {
                if (replicasNeedingRecovery.contains(key)) {
                    foundReplicasInLowerTerms = true;
                }
                if (!Objects.equals(newValues.get(key), leaderTerm)) continue;
                if (this.skipIncreaseTermOf(key, replicasNeedingRecovery)) {
                    changed = true;
                    continue;
                }
                newValues.put(key, leaderTerm + 1L);
            }
            if (!changed && foundReplicasInLowerTerms) {
                return null;
            }
            return new Terms(newValues, this.version);
        }

        private boolean skipIncreaseTermOf(String key, Set<String> replicasNeedingRecovery) {
            if (key.endsWith("_recovering")) {
                key = key.substring(0, key.length() - "_recovering".length());
                return replicasNeedingRecovery.contains(key);
            }
            return replicasNeedingRecovery.contains(key);
        }

        Terms ensureHighestTermsAreNotZero() {
            if (this.maxTerm > 0L) {
                return null;
            }
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            for (String replica : this.values.keySet()) {
                newValues.put(replica, 1L);
            }
            return new Terms(newValues, this.version);
        }

        Terms removeTerm(String coreNodeName) {
            if (!this.values.containsKey(coreNodeName)) {
                return null;
            }
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            newValues.remove(coreNodeName);
            return new Terms(newValues, this.version);
        }

        Terms registerTerm(String coreNodeName) {
            if (this.values.containsKey(coreNodeName)) {
                return null;
            }
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            newValues.put(coreNodeName, 0L);
            return new Terms(newValues, this.version);
        }

        Terms setTermToZero(String coreNodeName) {
            if (this.values.getOrDefault(coreNodeName, -1L) == 0L) {
                return null;
            }
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            newValues.put(coreNodeName, 0L);
            return new Terms(newValues, this.version);
        }

        Terms setTermEqualsToLeader(String coreNodeName) {
            long maxTerm = this.getMaxTerm();
            if (this.values.get(coreNodeName) == maxTerm) {
                return null;
            }
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            newValues.put(coreNodeName, maxTerm);
            newValues.remove(coreNodeName + "_recovering");
            return new Terms(newValues, this.version);
        }

        long getMaxTerm() {
            return this.maxTerm;
        }

        Terms startRecovering(String coreNodeName) {
            long maxTerm = this.getMaxTerm();
            if (this.values.get(coreNodeName) == maxTerm) {
                return null;
            }
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            if (!newValues.containsKey(coreNodeName + "_recovering")) {
                long currentTerm = newValues.getOrDefault(coreNodeName, 0L);
                newValues.put(coreNodeName + "_recovering", currentTerm);
            }
            newValues.put(coreNodeName, maxTerm);
            return new Terms(newValues, this.version);
        }

        Terms doneRecovering(String coreNodeName) {
            if (!this.values.containsKey(coreNodeName + "_recovering")) {
                return null;
            }
            HashMap<String, Long> newValues = new HashMap<String, Long>(this.values);
            newValues.remove(coreNodeName + "_recovering");
            return new Terms(newValues, this.version);
        }

        public String toString() {
            return "Terms{values=" + this.values + ", version=" + this.version + '}';
        }
    }

    static interface CoreTermWatcher {
        public boolean onTermChanged(Terms var1);
    }
}

