# -*- coding: utf-8 -*-
#
# Copyright © Spyder Project Contributors
# Licensed under the terms of the MIT License
#

"""
Tests for stringmatching.py
"""

# Standard library imports
import os

# Test library imports
import pytest

# Local imports
from spyder.utils.stringmatching import get_search_scores

TEST_FILE = os.path.join(os.path.dirname(__file__), 'data/example.py')

def test_stringmatching_full():
    """Test stringmatching full results."""
    template = '<b>{0}</b>'
    names = ['close pane', 'debug continue', 'debug exit', 'debug step into',
             'debug step over', 'debug step return', 'fullscreen mode',
             'layout preferences', 'lock unlock panes', 'maximize pane',
             'preferences', 'quit', 'restart', 'save current layout',
             'switch to breakpoints', 'switch to console', 'switch to editor',
             'switch to explorer', 'switch to find_in_files',
             'switch to historylog', 'switch to help',
             'switch to ipython_console', 'switch to onlinehelp',
             'switch to outline_explorer', 'switch to project_explorer',
             'switch to variable_explorer',
             'use next layout', 'use previous layout', 'clear line',
             'clear shell', 'inspect current object', 'blockcomment',
             'breakpoint', 'close all', 'code completion',
             'conditional breakpoint', 'configure', 'copy', 'copy line', 'cut',
             'debug', 'debug with winpdb', 'delete', 'delete line',
             'duplicate line', 'end of document', 'end of line',
             'file list management', 'find next', 'find previous', 'find text',
             'go to definition', 'go to line', 'go to next file',
             'go to previous file', 'inspect current object', 'kill next word',
             'kill previous word', 'kill to line end', 'kill to line start',
             'last edit location', 'move line down', 'move line up',
             'new file', 'next char', 'next cursor position', 'next line',
             'next word', 'open file', 'paste', 'previous char',
             'previous cursor position', 'previous line', 'previous word',
             'print', 're-run last script', 'redo', 'replace text',
             'rotate kill ring', 'run', 'run selection', 'save all', 'save as',
             'save file', 'select all', 'show/hide outline',
             'show/hide project explorer', 'start of document',
             'start of line', 'toggle comment', 'unblockcomment', 'undo',
             'yank', 'run profiler', 'run analysis']

    full_results = get_search_scores('lay', names, template=template, )

    assert full_results == [('close pane', 'close pane', -1),
                            ('debug continue', 'debug continue', -1),
                            ('debug exit', 'debug exit', -1),
                            ('debug step into', 'debug step into', -1),
                            ('debug step over', 'debug step over', -1),
                            ('debug step return', 'debug step return', -1),
                            ('fullscreen mode', 'fullscreen mode', -1),
                            ('layout preferences', '<b>lay</b>out preferences',
                             400100),
                            ('lock unlock panes', 'lock unlock panes', -1),
                            ('maximize pane', 'maximize pane', -1),
                            ('preferences', 'preferences', -1),
                            ('quit', 'quit', -1),
                            ('restart', 'restart', -1),
                            ('save current layout',
                             'save current <b>lay</b>out', 400113),
                            ('switch to breakpoints',
                             'switch to breakpoints', -1),
                            ('switch to console', 'switch to console', -1),
                            ('switch to editor', 'switch to editor', -1),
                            ('switch to explorer', 'switch to explorer', -1),
                            ('switch to find_in_files',
                             'switch to find_in_files', -1),
                            ('switch to historylog',
                             'switch to historylog', -1),
                            ('switch to help', 'switch to help', -1),
                            ('switch to ipython_console',
                             'switch to ipython_console', -1),
                            ('switch to onlinehelp',
                             'switch to onlinehelp', -1),
                            ('switch to outline_explorer',
                             'switch to outline_explorer', -1),
                            ('switch to project_explorer',
                             'switch to project_explorer', -1),
                            ('switch to variable_explorer',
                             'switch to variable_explorer', -1),
                            ('use next layout', 'use next <b>lay</b>out',
                             400109),
                            ('use previous layout',
                             'use previous <b>lay</b>out', 400113),
                            ('clear line', 'clear line', -1),
                            ('clear shell', 'clear shell', -1),
                            ('inspect current object',
                             'inspect current object', -1),
                            ('blockcomment', 'blockcomment', -1),
                            ('breakpoint', 'breakpoint', -1),
                            ('close all', 'close all', -1),
                            ('code completion', 'code completion', -1),
                            ('conditional breakpoint',
                            'conditional breakpoint', -1),
                            ('configure', 'configure', -1),
                            ('copy', 'copy', -1),
                            ('copy line', 'copy line', -1),
                            ('cut', 'cut', -1),
                            ('debug', 'debug', -1),
                            ('debug with winpdb', 'debug with winpdb', -1),
                            ('delete', 'delete', -1),
                            ('delete line', 'delete line', -1),
                            ('duplicate line', 'duplicate line', -1),
                            ('end of document', 'end of document', -1),
                            ('end of line', 'end of line', -1),
                            ('file list management',
                             'file list management', -1),
                            ('find next', 'find next', -1),
                            ('find previous', 'find previous', -1),
                            ('find text', 'find text', -1),
                            ('go to definition', 'go to definition', -1),
                            ('go to line', 'go to line', -1),
                            ('go to next file', 'go to next file', -1),
                            ('go to previous file', 'go to previous file', -1),
                            ('inspect current object',
                             'inspect current object', -1),
                            ('kill next word', 'kill next word', -1),
                            ('kill previous word', 'kill previous word', -1),
                            ('kill to line end', 'kill to line end', -1),
                            ('kill to line start', 'kill to line start', -1),
                            ('last edit location', 'last edit location', -1),
                            ('move line down', 'move line down', -1),
                            ('move line up', 'move line up', -1),
                            ('new file', 'new file', -1),
                            ('next char', 'next char', -1),
                            ('next cursor position',
                             'next cursor position', -1),
                            ('next line', 'next line', -1),
                            ('next word', 'next word', -1),
                            ('open file', 'open file', -1),
                            ('paste', 'paste', -1),
                            ('previous char', 'previous char', -1),
                            ('previous cursor position',
                             'previous cursor position', -1),
                            ('previous line', 'previous line', -1),
                            ('previous word', 'previous word', -1),
                            ('print', 'print', -1),
                            ('re-run last script', 're-run last script', -1),
                            ('redo', 'redo', -1),
                            ('replace text', 'replace text', -1),
                            ('rotate kill ring', 'rotate kill ring', -1),
                            ('run', 'run', -1),
                            ('run selection', 'run selection', -1),
                            ('save all', 'save all', -1),
                            ('save as', 'save as', -1),
                            ('save file', 'save file', -1),
                            ('select all', 'select all', -1),
                            ('show/hide outline', 'show/hide outline', -1),
                            ('show/hide project explorer',
                             'show/hide project explorer', -1),
                            ('start of document', 'start of document', -1),
                            ('start of line', 'start of line', -1),
                            ('toggle comment', 'toggle comment', -1),
                            ('unblockcomment', 'unblockcomment', -1),
                            ('undo', 'undo', -1), ('yank', 'yank', -1),
                            ('run profiler', 'run profiler', -1),
                            ('run analysis', 'run analysis', -1)]

def test_stringmatching_order_filter():
    """Test stringmatching ordered and filtered."""
    template = '<b>{0}</b>'
    names = ['close pane', 'debug continue', 'debug exit', 'debug step into',
             'debug step over', 'debug step return', 'fullscreen mode',
             'layout preferences', 'lock unlock panes', 'maximize pane',
             'preferences', 'quit', 'restart', 'save current layout',
             'switch to breakpoints', 'switch to console', 'switch to editor',
             'switch to explorer', 'switch to find_in_files',
             'switch to historylog', 'switch to help',
             'switch to ipython_console', 'switch to onlinehelp',
             'switch to outline_explorer', 'switch to project_explorer',
             'switch to variable_explorer',
             'use next layout', 'use previous layout', 'clear line',
             'clear shell', 'inspect current object', 'blockcomment',
             'breakpoint', 'close all', 'code completion',
             'conditional breakpoint', 'configure', 'copy', 'copy line', 'cut',
             'debug', 'debug with winpdb', 'delete', 'delete line',
             'duplicate line', 'end of document', 'end of line',
             'file list management', 'find next', 'find previous', 'find text',
             'go to definition', 'go to line', 'go to next file',
             'go to previous file', 'inspect current object', 'kill next word',
             'kill previous word', 'kill to line end', 'kill to line start',
             'last edit location', 'move line down', 'move line up',
             'new file', 'next char', 'next cursor position', 'next line',
             'next word', 'open file', 'paste', 'previous char',
             'previous cursor position', 'previous line', 'previous word',
             'print', 're-run last script', 'redo', 'replace text',
             'rotate kill ring', 'run', 'run selection', 'save all', 'save as',
             'save file', 'select all', 'show/hide outline',
             'show/hide project explorer', 'start of document',
             'start of line', 'toggle comment', 'unblockcomment', 'undo',
             'yank', 'run profiler', 'run analysis']

    order_filter_results = get_search_scores('lay', names, template=template,
                                             valid_only=True, sort=True)

    assert order_filter_results == [('layout preferences',
                                     '<b>lay</b>out preferences', 400100),
                                    ('use next layout',
                                     'use next <b>lay</b>out', 400109),
                                    ('save current layout',
                                     'save current <b>lay</b>out', 400113),
                                    ('use previous layout',
                                     'use previous <b>lay</b>out', 400113)]


if __name__ == "__main__":
    pytest.main()
