/*	$OpenBSD: db_disasm.c,v 1.18 2017/12/07 20:38:15 deraadt Exp $	*/
/*	$NetBSD: db_disasm.c,v 1.11 1996/05/03 19:41:58 christos Exp $	*/

/* 
 * Mach Operating System
 * Copyright (c) 1991,1990 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 *
 *	Id: db_disasm.c,v 2.6  92/01/03  20:05:00  dbg (CMU)
 */

/*
 * Instruction disassembler.
 */
#include <sys/param.h>
#include <machine/db_machdep.h>

#include <ddb/db_access.h>
#include <ddb/db_sym.h>
#include <ddb/db_output.h>

/*
 * Size attributes
 */
#define	BYTE	0
#define	WORD	1
#define	LONG	2
#define	QUAD	3
#define	SNGL	4
#define	DBLR	5
#define	EXTR	6
#define	SDEP	7
#define	NONE	8
#define	RDEP	9

/*
 * Addressing modes
 */
#define	E	1			/* general effective address */
#define	Eind	2			/* indirect address (jump, call) */
#define	Ew	3			/* address, word size */
#define	Eb	4			/* address, byte size */
#define	R	5			/* register, in 'reg' field */
#define	Rw	6			/* word register, in 'reg' field */
#define	Ri	7			/* register in instruction */
#define	S	8			/* segment reg, in 'reg' field */
#define	Si	9			/* segment reg, in instruction */
#define	A	10			/* accumulator */
#define	BX	11			/* (bx) */
#define	CL	12			/* cl, for shifts */
#define	DX	13			/* dx, for IO */
#define	SI	14			/* si */
#define	DI	15			/* di */
#define	CR	16			/* control register */
#define	DR	17			/* debug register */
#define	TR	18			/* test register */
#define	I	19			/* immediate, unsigned */
#define	Is	20			/* immediate, signed */
#define	Ib	21			/* byte immediate, unsigned */
#define	Ibs	22			/* byte immediate, signed */
#define	Iw	23			/* word immediate, unsigned */
#define	Iq	24			/* quad immediate, unsigned */
#define	O	25			/* direct address */
#define	Db	26			/* byte displacement from EIP */
#define	Dl	27			/* long displacement from EIP */
#define	o1	28			/* constant 1 */
#define	o3	29			/* constant 3 */
#define	OS	30			/* immediate offset/segment */
#define	ST	31			/* FP stack top */
#define	STI	32			/* FP stack */
#define	X	33			/* extended FP op */
#define	XA	34			/* for 'fstcw %ax' */
#define	Ril	36			/* long register in instruction */
#define	Iba	37			/* byte immediate, don't print if 0xa */
#define	MEx	38			/* memory, or an extension op */

struct inst {
	char *	i_name;			/* name */
	short	i_has_modrm;		/* has regmodrm byte */
	short	i_size;			/* operand size */
	int	i_mode;			/* addressing modes */
	void *	i_extra;		/* pointer to extra opcode table */
};

#define	op1(x)		(x)
#define	op2(x,y)	((x)|((y)<<8))
#define	op3(x,y,z)	((x)|((y)<<8)|((z)<<16))

struct finst {
	char *	f_name;			/* name for memory instruction */
	int	f_size;			/* size for memory instruction */
	int	f_rrmode;		/* mode for rr instruction */
	void *	f_rrname;		/* name for rr instruction
					   (or pointer to table) */
};

char *	db_Grp6[] = {
	"sldt",		"str",		"lldt",		"ltr",
	"verr",		"verw",		"",		""
};

struct inst db_Grp7[] = {
	{ "sgdt",   FALSE, NONE, op2(MEx,5), "\0vmcall\0vmlaunch\0vmresume\0vmxoff"},
	{ "sidt",   FALSE, NONE, op2(MEx,4), "monitor\0mwait\0clac\0stac"},
	{ "lgdt",   FALSE, NONE, op2(MEx,7), "xgetbv\0xsetbv\0\0\0vmfunc\0xend\0xtest" },
	{ "lidt",   FALSE, NONE, op1(E),     0 },
	{ "smsw",   FALSE, NONE, op1(E),     0 },
	{ "",       FALSE, NONE, 0,          0 },
	{ "lmsw",   FALSE, NONE, op1(E),     0 },
	{ "invlpg", FALSE, NONE, op2(MEx,2), "swapgs\0rdtscp" },
};

char *	db_Grp8[] = {
	"",		"",		"",		"",
	"bt",		"bts",		"btr",		"btc"
};

struct inst db_Grp9[] = {
	{ "fxsave",   FALSE, NONE, op2(MEx,1), "rdfsbase" },
	{ "fxrstor",  FALSE, NONE, op2(MEx,1), "rdgsbase" },
	{ "ldmxcsr",  FALSE, NONE, op2(MEx,1), "wrfsbase" },
	{ "stmxcsr",  FALSE, NONE, op2(MEx,1), "wrgsbase" },
	{ "xsave",    FALSE, NONE, op1(E),     0 },
	{ "xrstor",   FALSE, NONE, op2(MEx,1), "lfence" },
	{ "xsaveopt", FALSE, NONE, op2(MEx,1), "mfence" },
	{ "clflush",  FALSE, NONE, op2(MEx,1), "sfence" },
};

char *	db_GrpA[] = {
	"",		"cmpxchg8b",	"",		"",
	"",		"",		"rdrand",	"rdseed"
};

char *	db_GrpB[] = {
	"xstore-rng",	"xcrypt-ecb",	"xcrypt-cbc",	"xcrypt-ctr",
	"xcrypt-cfb",	"xcrypt-ofb",	"",		""
};

char * db_GrpC[] = {
	"montmul",	"xsha1",	"xsha256",	"",
	"",		"",		"",		""
};

struct inst db_inst_0f0x[] = {
/*00*/	{ NULL,	   TRUE,  NONE,  op1(Ew),     db_Grp6 },
/*01*/	{ "",	   TRUE,  RDEP,  0,           db_Grp7 },
/*02*/	{ "lar",   TRUE,  LONG,  op2(E,R),    0 },
/*03*/	{ "lsl",   TRUE,  LONG,  op2(E,R),    0 },
/*04*/	{ "",      FALSE, NONE,  0,	      0 },
/*05*/	{ "",      FALSE, NONE,  0,	      0 },
/*06*/	{ "clts",  FALSE, NONE,  0,	      0 },
/*07*/	{ "",      FALSE, NONE,  0,	      0 },

/*08*/	{ "invd",  FALSE, NONE,  0,	      0 },
/*09*/	{ "wbinvd",FALSE, NONE,  0,	      0 },
/*0a*/	{ "",      FALSE, NONE,  0,	      0 },
/*0b*/	{ "",      FALSE, NONE,  0,	      0 },
/*0c*/	{ "",      FALSE, NONE,  0,	      0 },
/*0d*/	{ "",      FALSE, NONE,  0,	      0 },
/*0e*/	{ "",      FALSE, NONE,  0,	      0 },
/*0f*/	{ "",      FALSE, NONE,  0,	      0 },
};

struct inst	db_inst_0f2x[] = {
/*20*/	{ "mov",   TRUE,  LONG,  op2(CR,E),   0 }, /* use E for reg */
/*21*/	{ "mov",   TRUE,  LONG,  op2(DR,E),   0 }, /* since mod == 11 */
/*22*/	{ "mov",   TRUE,  LONG,  op2(E,CR),   0 },
/*23*/	{ "mov",   TRUE,  LONG,  op2(E,DR),   0 },
/*24*/	{ "mov",   TRUE,  LONG,  op2(TR,E),   0 },
/*25*/	{ "",      FALSE, NONE,  0,	      0 },
/*26*/	{ "mov",   TRUE,  LONG,  op2(E,TR),   0 },
/*27*/	{ "",      FALSE, NONE,  0,	      0 },

/*28*/	{ "",      FALSE, NONE,  0,	      0 },
/*29*/	{ "",      FALSE, NONE,  0,	      0 },
/*2a*/	{ "",      FALSE, NONE,  0,	      0 },
/*2b*/	{ "",      FALSE, NONE,  0,	      0 },
/*2c*/	{ "",      FALSE, NONE,  0,	      0 },
/*2d*/	{ "",      FALSE, NONE,  0,	      0 },
/*2e*/	{ "",      FALSE, NONE,  0,	      0 },
/*2f*/	{ "",      FALSE, NONE,  0,	      0 },
};

struct inst	db_inst_0f3x[] = {
/*30*/	{ "wrmsr", FALSE, NONE,  0,           0 },
/*31*/	{ "rdtsc", FALSE, NONE,  0,           0 },
/*32*/	{ "rdmsr", FALSE, NONE,  0,           0 },
/*33*/	{ "rdpmc", FALSE, NONE,  0,           0 },
/*34*/	{ "",      FALSE, NONE,  0,           0 },
/*35*/	{ "",      FALSE, NONE,  0,           0 },
/*36*/	{ "",      FALSE, NONE,  0,           0 },
/*37*/	{ "",      FALSE, NONE,  0,           0 },

/*38*/	{ "",      FALSE, NONE,  0,           0 },
/*39*/	{ "",      FALSE, NONE,  0,           0 },
/*3a*/	{ "",      FALSE, NONE,  0,           0 },
/*3b*/	{ "",      FALSE, NONE,  0,           0 },
/*3c*/	{ "",      FALSE, NONE,  0,           0 },
/*3d*/	{ "",      FALSE, NONE,  0,           0 },
/*3e*/	{ "",      FALSE, NONE,  0,           0 },
/*3f*/	{ "",      FALSE, NONE,  0,           0 },
};

struct inst	db_inst_0f8x[] = {
/*80*/	{ "jo",    FALSE, NONE,  op1(Dl),     0 },
/*81*/	{ "jno",   FALSE, NONE,  op1(Dl),     0 },
/*82*/	{ "jb",    FALSE, NONE,  op1(Dl),     0 },
/*83*/	{ "jnb",   FALSE, NONE,  op1(Dl),     0 },
/*84*/	{ "jz",    FALSE, NONE,  op1(Dl),     0 },
/*85*/	{ "jnz",   FALSE, NONE,  op1(Dl),     0 },
/*86*/	{ "jbe",   FALSE, NONE,  op1(Dl),     0 },
/*87*/	{ "jnbe",  FALSE, NONE,  op1(Dl),     0 },

/*88*/	{ "js",    FALSE, NONE,  op1(Dl),     0 },
/*89*/	{ "jns",   FALSE, NONE,  op1(Dl),     0 },
/*8a*/	{ "jp",    FALSE, NONE,  op1(Dl),     0 },
/*8b*/	{ "jnp",   FALSE, NONE,  op1(Dl),     0 },
/*8c*/	{ "jl",    FALSE, NONE,  op1(Dl),     0 },
/*8d*/	{ "jnl",   FALSE, NONE,  op1(Dl),     0 },
/*8e*/	{ "jle",   FALSE, NONE,  op1(Dl),     0 },
/*8f*/	{ "jnle",  FALSE, NONE,  op1(Dl),     0 },
};

struct inst	db_inst_0f9x[] = {
/*90*/	{ "seto",  TRUE,  NONE,  op1(Eb),     0 },
/*91*/	{ "setno", TRUE,  NONE,  op1(Eb),     0 },
/*92*/	{ "setb",  TRUE,  NONE,  op1(Eb),     0 },
/*93*/	{ "setnb", TRUE,  NONE,  op1(Eb),     0 },
/*94*/	{ "setz",  TRUE,  NONE,  op1(Eb),     0 },
/*95*/	{ "setnz", TRUE,  NONE,  op1(Eb),     0 },
/*96*/	{ "setbe", TRUE,  NONE,  op1(Eb),     0 },
/*97*/	{ "setnbe",TRUE,  NONE,  op1(Eb),     0 },

/*98*/	{ "sets",  TRUE,  NONE,  op1(Eb),     0 },
/*99*/	{ "setns", TRUE,  NONE,  op1(Eb),     0 },
/*9a*/	{ "setp",  TRUE,  NONE,  op1(Eb),     0 },
/*9b*/	{ "setnp", TRUE,  NONE,  op1(Eb),     0 },
/*9c*/	{ "setl",  TRUE,  NONE,  op1(Eb),     0 },
/*9d*/	{ "setnl", TRUE,  NONE,  op1(Eb),     0 },
/*9e*/	{ "setle", TRUE,  NONE,  op1(Eb),     0 },
/*9f*/	{ "setnle",TRUE,  NONE,  op1(Eb),     0 },
};

struct inst	db_inst_0fax[] = {
/*a0*/	{ "push",  FALSE, QUAD,  op1(Si),     0 },
/*a1*/	{ "pop",   FALSE, QUAD,  op1(Si),     0 },
/*a2*/	{ "cpuid", FALSE, NONE,  0,	      0 },
/*a3*/	{ "bt",    TRUE,  LONG,  op2(R,E),    0 },
/*a4*/	{ "shld",  TRUE,  LONG,  op3(Ib,R,E), 0 },
/*a5*/	{ "shld",  TRUE,  LONG,  op3(CL,R,E), 0 },
/*a6*/	{ NULL,    TRUE,  NONE,  0,	      db_GrpC },
/*a7*/	{ NULL,    TRUE,  NONE,  0,	      db_GrpB },

/*a8*/	{ "push",  FALSE, QUAD,  op1(Si),     0 },
/*a9*/	{ "pop",   FALSE, QUAD,  op1(Si),     0 },
/*aa*/	{ "",      FALSE, NONE,  0,	      0 },
/*ab*/	{ "bts",   TRUE,  LONG,  op2(R,E),    0 },
/*ac*/	{ "shrd",  TRUE,  LONG,  op3(Ib,E,R), 0 },
/*ad*/	{ "shrd",  TRUE,  LONG,  op3(CL,E,R), 0 },
/*ae*/	{ "",      TRUE,  RDEP,  op1(E),      db_Grp9 },
/*af*/	{ "imul",  TRUE,  LONG,  op2(E,R),    0 },
};

struct inst	db_inst_0fbx[] = {
/*b0*/	{ "cmpxchg",TRUE, BYTE,	 op2(R, E),   0 },
/*b1*/	{ "cmpxchg",TRUE, LONG,	 op2(R, E),   0 },
/*b2*/	{ "lss",   TRUE,  LONG,  op2(E, R),   0 },
/*b3*/	{ "btr",   TRUE,  LONG,  op2(R, E),   0 },
/*b4*/	{ "lfs",   TRUE,  LONG,  op2(E, R),   0 },
/*b5*/	{ "lgs",   TRUE,  LONG,  op2(E, R),   0 },
/*b6*/	{ "movzb", TRUE,  LONG,  op2(Eb, R),  0 },
/*b7*/	{ "movzw", TRUE,  LONG,  op2(Ew, R),  0 },

/*b8*/	{ "",      FALSE, NONE,  0,	      0 },
/*b9*/	{ "",      FALSE, NONE,  0,	      0 },
/*ba*/	{ NULL,    TRUE,  LONG,  op2(Ib, E),  db_Grp8 },
/*bb*/	{ "btc",   TRUE,  LONG,  op2(R, E),   0 },
/*bc*/	{ "bsf",   TRUE,  LONG,  op2(E, R),   0 },
/*bd*/	{ "bsr",   TRUE,  LONG,  op2(E, R),   0 },
/*be*/	{ "movsb", TRUE,  LONG,  op2(Eb, R),  0 },
/*bf*/	{ "movsw", TRUE,  LONG,  op2(Ew, R),  0 },
};

struct inst	db_inst_0fcx[] = {
/*c0*/	{ "xadd",  TRUE,  BYTE,	 op2(R, E),   0 },
/*c1*/	{ "xadd",  TRUE,  LONG,	 op2(R, E),   0 },
/*c2*/	{ "",	   FALSE, NONE,	 0,	      0 },
/*c3*/	{ "",	   FALSE, NONE,	 0,	      0 },
/*c4*/	{ "",	   FALSE, NONE,	 0,	      0 },
/*c5*/	{ "",	   FALSE, NONE,	 0,	      0 },
/*c6*/	{ "",	   FALSE, NONE,	 0,	      0 },
/*c7*/	{ NULL,    TRUE,  NONE,  op1(E),      db_GrpA },

/*c8*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
/*c9*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
/*ca*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
/*cb*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
/*cc*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
/*cd*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
/*ce*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
/*cf*/	{ "bswap", FALSE, LONG,  op1(Ril),    0 },
};

struct inst *db_inst_0f[] = {
	db_inst_0f0x,
	NULL,
	db_inst_0f2x,
	db_inst_0f3x,
	NULL,
	NULL,
	NULL,
	NULL,
	db_inst_0f8x,
	db_inst_0f9x,
	db_inst_0fax,
	db_inst_0fbx,
	db_inst_0fcx,
	NULL,
	NULL,
	NULL
};

char *	db_Esc92[] = {
	"fnop",		"",		"",		"",
	"",		"",		"",		""
};
char *	db_Esc94[] = {
	"fchs",		"fabs",		"",		"",
	"ftst",		"fxam",		"",		""
};
char *	db_Esc95[] = {
	"fld1",		"fldl2t",	"fldl2e",	"fldpi",
	"fldlg2",	"fldln2",	"fldz",		""
};
char *	db_Esc96[] = {
	"f2xm1",	"fyl2x",	"fptan",	"fpatan",
	"fxtract",	"fprem1",	"fdecstp",	"fincstp"
};
char *	db_Esc97[] = {
	"fprem",	"fyl2xp1",	"fsqrt",	"fsincos",
	"frndint",	"fscale",	"fsin",		"fcos"
};

char *	db_Esca5[] = {
	"",		"fucompp",	"",		"",
	"",		"",		"",		""
};

char *	db_Escb4[] = {
	"fneni",	"fndisi",       "fnclex",	"fninit",
	"fsetpm",	"",		"",		""
};

char *	db_Esce3[] = {
	"",		"fcompp",	"",		"",
	"",		"",		"",		""
};

char *	db_Escf4[] = {
	"fnstsw",	"",		"",		"",
	"",		"",		"",		""
};

struct finst db_Esc8[] = {
/*0*/	{ "fadd",   SNGL,  op2(STI,ST),	0 },
/*1*/	{ "fmul",   SNGL,  op2(STI,ST),	0 },
/*2*/	{ "fcom",   SNGL,  op2(STI,ST),	0 },
/*3*/	{ "fcomp",  SNGL,  op2(STI,ST),	0 },
/*4*/	{ "fsub",   SNGL,  op2(STI,ST),	0 },
/*5*/	{ "fsubr",  SNGL,  op2(STI,ST),	0 },
/*6*/	{ "fdiv",   SNGL,  op2(STI,ST),	0 },
/*7*/	{ "fdivr",  SNGL,  op2(STI,ST),	0 },
};

struct finst db_Esc9[] = {
/*0*/	{ "fld",    SNGL,  op1(STI),	0 },
/*1*/	{ "",       NONE,  op1(STI),	"fxch" },
/*2*/	{ "fst",    SNGL,  op1(X),	db_Esc92 },
/*3*/	{ "fstp",   SNGL,  op1(X),	0 },
/*4*/	{ "fldenv", NONE,  op1(X),	db_Esc94 },
/*5*/	{ "fldcw",  NONE,  op1(X),	db_Esc95 },
/*6*/	{ "fnstenv",NONE,  op1(X),	db_Esc96 },
/*7*/	{ "fnstcw", NONE,  op1(X),	db_Esc97 },
};

struct finst db_Esca[] = {
/*0*/	{ "fiadd",  LONG,  0,		0 },
/*1*/	{ "fimul",  LONG,  0,		0 },
/*2*/	{ "ficom",  LONG,  0,		0 },
/*3*/	{ "ficomp", LONG,  0,		0 },
/*4*/	{ "fisub",  LONG,  op1(X),	0 },
/*5*/	{ "fisubr", LONG,  0,		0 },
/*6*/	{ "fidiv",  LONG,  0,		0 },
/*7*/	{ "fidivr", LONG,  0,		0 }
};

struct finst db_Escb[] = {
/*0*/	{ "fild",   LONG,  0,		0 },
/*1*/	{ "",       NONE,  0,		0 },
/*2*/	{ "fist",   LONG,  0,		0 },
/*3*/	{ "fistp",  LONG,  0,		0 },
/*4*/	{ "",       WORD,  op1(X),	db_Escb4 },
/*5*/	{ "fld",    EXTR,  0,		0 },
/*6*/	{ "",       WORD,  0,		0 },
/*7*/	{ "fstp",   EXTR,  0,		0 },
};

struct finst db_Escc[] = {
/*0*/	{ "fadd",   DBLR,  op2(ST,STI),	0 },
/*1*/	{ "fmul",   DBLR,  op2(ST,STI),	0 },
/*2*/	{ "fcom",   DBLR,  0,	 	0 },
/*3*/	{ "fcomp",  DBLR,  0,		0 },
/*4*/	{ "fsub",   DBLR,  op2(ST,STI),	"fsubr" },
/*5*/	{ "fsubr",  DBLR,  op2(ST,STI),	"fsub" },
/*6*/	{ "fdiv",   DBLR,  op2(ST,STI),	"fdivr" },
/*7*/	{ "fdivr",  DBLR,  op2(ST,STI),	"fdiv" },
};

struct finst db_Escd[] = {
/*0*/	{ "fld",    DBLR,  op1(STI),	"ffree" },
/*1*/	{ "",       NONE,  0,		0 },
/*2*/	{ "fst",    DBLR,  op1(STI),	0 },
/*3*/	{ "fstp",   DBLR,  op1(STI),	0 },
/*4*/	{ "frstor", NONE,  op1(STI),	"fucom" },
/*5*/	{ "",       NONE,  op1(STI),	"fucomp" },
/*6*/	{ "fnsave", NONE,  0,		0 },
/*7*/	{ "fnstsw", NONE,  0,		0 },
};

struct finst db_Esce[] = {
/*0*/	{ "fiadd",  WORD,  op2(ST,STI),	"faddp" },
/*1*/	{ "fimul",  WORD,  op2(ST,STI),	"fmulp" },
/*2*/	{ "ficom",  WORD,  0,		0 },
/*3*/	{ "ficomp", WORD,  op1(X),	db_Esce3 },
/*4*/	{ "fisub",  WORD,  op2(ST,STI),	"fsubrp" },
/*5*/	{ "fisubr", WORD,  op2(ST,STI),	"fsubp" },
/*6*/	{ "fidiv",  WORD,  op2(ST,STI),	"fdivrp" },
/*7*/	{ "fidivr", WORD,  op2(ST,STI),	"fdivp" },
};

struct finst db_Escf[] = {
/*0*/	{ "fild",   WORD,  0,		0 },
/*1*/	{ "",       WORD,  0,		0 },
/*2*/	{ "fist",   WORD,  0,		0 },
/*3*/	{ "fistp",  WORD,  0,		0 },
/*4*/	{ "fbld",   NONE,  op1(XA),	db_Escf4 },
/*5*/	{ "fild",   QUAD,  0,		0 },
/*6*/	{ "fbstp",  NONE,  0,		0 },
/*7*/	{ "fistp",  QUAD,  0,		0 },
};

struct finst *db_Esc_inst[] = {
	db_Esc8, db_Esc9, db_Esca, db_Escb,
	db_Escc, db_Escd, db_Esce, db_Escf
};

char *	db_Grp1[] = {
	"add",
	"or",
	"adc",
	"sbb",
	"and",
	"sub",
	"xor",
	"cmp"
};

char *	db_Grp2[] = {
	"rol",
	"ror",
	"rcl",
	"rcr",
	"shl",
	"shr",
	"shl",
	"sar"
};

struct inst db_Grp3[] = {
	{ "test",  TRUE, NONE, op2(I,E), 0 },
	{ "test",  TRUE, NONE, op2(I,E), 0 },
	{ "not",   TRUE, NONE, op1(E),   0 },
	{ "neg",   TRUE, NONE, op1(E),   0 },
	{ "mul",   TRUE, NONE, op2(E,A), 0 },
	{ "imul",  TRUE, NONE, op2(E,A), 0 },
	{ "div",   TRUE, NONE, op2(E,A), 0 },
	{ "idiv",  TRUE, NONE, op2(E,A), 0 },
};

struct inst	db_Grp4[] = {
	{ "inc",   TRUE, BYTE, op1(E),   0 },
	{ "dec",   TRUE, BYTE, op1(E),   0 },
	{ "",      TRUE, NONE, 0,	 0 },
	{ "",      TRUE, NONE, 0,	 0 },
	{ "",      TRUE, NONE, 0,	 0 },
	{ "",      TRUE, NONE, 0,	 0 },
	{ "",      TRUE, NONE, 0,	 0 },
	{ "",      TRUE, NONE, 0,	 0 }
};

struct inst	db_Grp5[] = {
	{ "inc",   TRUE, LONG, op1(E),   0 },
	{ "dec",   TRUE, LONG, op1(E),   0 },
	{ "call",  TRUE, QUAD, op1(Eind),0 },
	{ "lcall", TRUE, NONE, op1(Eind),0 },
	{ "jmp",   TRUE, NONE, op1(Eind),0 },
	{ "ljmp",  TRUE, NONE, op1(Eind),0 },
	{ "push",  TRUE, QUAD, op1(E),   0 },
	{ "",      TRUE, NONE, 0,	 0 }
};

struct inst db_inst_table[256] = {
/*00*/	{ "add",   TRUE,  BYTE,  op2(R, E),  0 },
/*01*/	{ "add",   TRUE,  LONG,  op2(R, E),  0 },
/*02*/	{ "add",   TRUE,  BYTE,  op2(E, R),  0 },
/*03*/	{ "add",   TRUE,  LONG,  op2(E, R),  0 },
/*04*/	{ "add",   FALSE, BYTE,  op2(I, A),  0 },
/*05*/	{ "add",   FALSE, LONG,  op2(Is, A), 0 },
/*06*/	{ "",      FALSE, NONE,  op1(Si),    0 },
/*07*/	{ "",      FALSE, NONE,  op1(Si),    0 },

/*08*/	{ "or",    TRUE,  BYTE,  op2(R, E),  0 },
/*09*/	{ "or",    TRUE,  LONG,  op2(R, E),  0 },
/*0a*/	{ "or",    TRUE,  BYTE,  op2(E, R),  0 },
/*0b*/	{ "or",    TRUE,  LONG,  op2(E, R),  0 },
/*0c*/	{ "or",    FALSE, BYTE,  op2(I, A),  0 },
/*0d*/	{ "or",    FALSE, LONG,  op2(I, A),  0 },
/*0e*/	{ "",      FALSE, NONE,  op1(Si),    0 },
/*0f*/	{ "",      FALSE, NONE,  0,	     0 },

/*10*/	{ "adc",   TRUE,  BYTE,  op2(R, E),  0 },
/*11*/	{ "adc",   TRUE,  LONG,  op2(R, E),  0 },
/*12*/	{ "adc",   TRUE,  BYTE,  op2(E, R),  0 },
/*13*/	{ "adc",   TRUE,  LONG,  op2(E, R),  0 },
/*14*/	{ "adc",   FALSE, BYTE,  op2(I, A),  0 },
/*15*/	{ "adc",   FALSE, LONG,  op2(Is, A), 0 },
/*16*/	{ "",      FALSE, NONE,  op1(Si),    0 },
/*17*/	{ "",      FALSE, NONE,  op1(Si),    0 },

/*18*/	{ "sbb",   TRUE,  BYTE,  op2(R, E),  0 },
/*19*/	{ "sbb",   TRUE,  LONG,  op2(R, E),  0 },
/*1a*/	{ "sbb",   TRUE,  BYTE,  op2(E, R),  0 },
/*1b*/	{ "sbb",   TRUE,  LONG,  op2(E, R),  0 },
/*1c*/	{ "sbb",   FALSE, BYTE,  op2(I, A),  0 },
/*1d*/	{ "sbb",   FALSE, LONG,  op2(Is, A), 0 },
/*1e*/	{ "",      FALSE, NONE,  op1(Si),    0 },
/*1f*/	{ "",      FALSE, NONE,  op1(Si),    0 },

/*20*/	{ "and",   TRUE,  BYTE,  op2(R, E),  0 },
/*21*/	{ "and",   TRUE,  LONG,  op2(R, E),  0 },
/*22*/	{ "and",   TRUE,  BYTE,  op2(E, R),  0 },
/*23*/	{ "and",   TRUE,  LONG,  op2(E, R),  0 },
/*24*/	{ "and",   FALSE, BYTE,  op2(I, A),  0 },
/*25*/	{ "and",   FALSE, LONG,  op2(I, A),  0 },
/*26*/	{ "",      FALSE, NONE,  0,	     0 },
/*27*/	{ "",      FALSE, NONE,  0,	     0 },

/*28*/	{ "sub",   TRUE,  BYTE,  op2(R, E),  0 },
/*29*/	{ "sub",   TRUE,  LONG,  op2(R, E),  0 },
/*2a*/	{ "sub",   TRUE,  BYTE,  op2(E, R),  0 },
/*2b*/	{ "sub",   TRUE,  LONG,  op2(E, R),  0 },
/*2c*/	{ "sub",   FALSE, BYTE,  op2(I, A),  0 },
/*2d*/	{ "sub",   FALSE, LONG,  op2(Is, A), 0 },
/*2e*/	{ "",      FALSE, NONE,  0,	     0 },
/*2f*/	{ "",      FALSE, NONE,  0,	     0 },

/*30*/	{ "xor",   TRUE,  BYTE,  op2(R, E),  0 },
/*31*/	{ "xor",   TRUE,  LONG,  op2(R, E),  0 },
/*32*/	{ "xor",   TRUE,  BYTE,  op2(E, R),  0 },
/*33*/	{ "xor",   TRUE,  LONG,  op2(E, R),  0 },
/*34*/	{ "xor",   FALSE, BYTE,  op2(I, A),  0 },
/*35*/	{ "xor",   FALSE, LONG,  op2(I, A),  0 },
/*36*/	{ "",      FALSE, NONE,  0,	     0 },
/*37*/	{ "",      FALSE, NONE,  0,	     0 },

/*38*/	{ "cmp",   TRUE,  BYTE,  op2(R, E),  0 },
/*39*/	{ "cmp",   TRUE,  LONG,  op2(R, E),  0 },
/*3a*/	{ "cmp",   TRUE,  BYTE,  op2(E, R),  0 },
/*3b*/	{ "cmp",   TRUE,  LONG,  op2(E, R),  0 },
/*3c*/	{ "cmp",   FALSE, BYTE,  op2(I, A),  0 },
/*3d*/	{ "cmp",   FALSE, LONG,  op2(Is, A), 0 },
/*3e*/	{ "",      FALSE, NONE,  0,	     0 },
/*3f*/	{ "",      FALSE, NONE,  0,	     0 },

/*40*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*41*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*42*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*43*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*44*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*45*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*46*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*47*/	{ "",      FALSE, LONG,  op1(Ri),    0 },

/*48*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*49*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*4a*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*4b*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*4c*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*4d*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*4e*/	{ "",      FALSE, LONG,  op1(Ri),    0 },
/*4f*/	{ "",      FALSE, LONG,  op1(Ri),    0 },

/*50*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },
/*51*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },
/*52*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },
/*53*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },
/*54*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },
/*55*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },
/*56*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },
/*57*/	{ "push",  FALSE, QUAD,  op1(Ri),    0 },

/*58*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },
/*59*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },
/*5a*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },
/*5b*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },
/*5c*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },
/*5d*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },
/*5e*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },
/*5f*/	{ "pop",   FALSE, QUAD,  op1(Ri),    0 },

/*60*/	{ "",      FALSE, LONG,  0,	     0 },
/*61*/	{ "",      FALSE, LONG,  0,	     0 },
/*62*/	{ "",      TRUE,  LONG,  op2(E, R),  0 },
/*63*/	{ "movsxd",TRUE,  LONG,  op2(E, R),  0 },
/*64*/	{ "",      FALSE, NONE,  0,	     0 },
/*65*/	{ "",      FALSE, NONE,  0,	     0 },
/*66*/	{ "",      FALSE, NONE,  0,	     0 },
/*67*/	{ "",      FALSE, NONE,  0,	     0 },

/*68*/	{ "push",  FALSE, QUAD,  op1(I),     0 },
/*69*/	{ "imul",  TRUE,  LONG,  op3(I,E,R), 0 },
/*6a*/	{ "push",  FALSE, QUAD,  op1(Ibs),   0 },
/*6b*/	{ "imul",  TRUE,  LONG,  op3(Ibs,E,R),0 },
/*6c*/	{ "ins",   FALSE, BYTE,  op2(DX, DI), 0 },
/*6d*/	{ "ins",   FALSE, LONG,  op2(DX, DI), 0 },
/*6e*/	{ "outs",  FALSE, BYTE,  op2(SI, DX), 0 },
/*6f*/	{ "outs",  FALSE, LONG,  op2(SI, DX), 0 },

/*70*/	{ "jo",    FALSE, NONE,  op1(Db),     0 },
/*71*/	{ "jno",   FALSE, NONE,  op1(Db),     0 },
/*72*/	{ "jb",    FALSE, NONE,  op1(Db),     0 },
/*73*/	{ "jnb",   FALSE, NONE,  op1(Db),     0 },
/*74*/	{ "jz",    FALSE, NONE,  op1(Db),     0 },
/*75*/	{ "jnz",   FALSE, NONE,  op1(Db),     0 },
/*76*/	{ "jbe",   FALSE, NONE,  op1(Db),     0 },
/*77*/	{ "jnbe",  FALSE, NONE,  op1(Db),     0 },

/*78*/	{ "js",    FALSE, NONE,  op1(Db),     0 },
/*79*/	{ "jns",   FALSE, NONE,  op1(Db),     0 },
/*7a*/	{ "jp",    FALSE, NONE,  op1(Db),     0 },
/*7b*/	{ "jnp",   FALSE, NONE,  op1(Db),     0 },
/*7c*/	{ "jl",    FALSE, NONE,  op1(Db),     0 },
/*7d*/	{ "jnl",   FALSE, NONE,  op1(Db),     0 },
/*7e*/	{ "jle",   FALSE, NONE,  op1(Db),     0 },
/*7f*/	{ "jnle",  FALSE, NONE,  op1(Db),     0 },

/*80*/	{ NULL,	   TRUE,  BYTE,  op2(I, E),   db_Grp1 },
/*81*/	{ NULL,	   TRUE,  LONG,  op2(I, E),   db_Grp1 },
/*82*/	{ NULL,	   TRUE,  BYTE,  op2(I, E),   db_Grp1 },
/*83*/	{ NULL,	   TRUE,  LONG,  op2(Ibs,E),  db_Grp1 },
/*84*/	{ "test",  TRUE,  BYTE,  op2(R, E),   0 },
/*85*/	{ "test",  TRUE,  LONG,  op2(R, E),   0 },
/*86*/	{ "xchg",  TRUE,  BYTE,  op2(R, E),   0 },
/*87*/	{ "xchg",  TRUE,  LONG,  op2(R, E),   0 },

/*88*/	{ "mov",   TRUE,  BYTE,  op2(R, E),   0 },
/*89*/	{ "mov",   TRUE,  LONG,  op2(R, E),   0 },
/*8a*/	{ "mov",   TRUE,  BYTE,  op2(E, R),   0 },
/*8b*/	{ "mov",   TRUE,  LONG,  op2(E, R),   0 },
/*8c*/	{ "mov",   TRUE,  NONE,  op2(S, Ew),  0 },
/*8d*/	{ "lea",   TRUE,  LONG,  op2(E, R),   0 },
/*8e*/	{ "mov",   TRUE,  NONE,  op2(Ew, S),  0 },
/*8f*/	{ "pop",   TRUE,  QUAD,  op1(Dl),     0 },

/*90*/	{ "nop",   FALSE, NONE,  0,	      0 },
/*91*/	{ "xchg",  FALSE, LONG,  op2(A, Ri),  0 },
/*92*/	{ "xchg",  FALSE, LONG,  op2(A, Ri),  0 },
/*93*/	{ "xchg",  FALSE, LONG,  op2(A, Ri),  0 },
/*94*/	{ "xchg",  FALSE, LONG,  op2(A, Ri),  0 },
/*95*/	{ "xchg",  FALSE, LONG,  op2(A, Ri),  0 },
/*96*/	{ "xchg",  FALSE, LONG,  op2(A, Ri),  0 },
/*97*/	{ "xchg",  FALSE, LONG,  op2(A, Ri),  0 },

/*98*/	{ "cbw",   FALSE, SDEP,  0,	      "cwde\0cdqe" },
/*99*/	{ "cwd",   FALSE, SDEP,  0,	      "cdq\0cqo"  },
/*9a*/	{ "",      FALSE, NONE,  op1(OS),     0 },
/*9b*/	{ "wait",  FALSE, NONE,  0,	      0 },
/*9c*/	{ "pushf", FALSE, QUAD,  0,	      0 },
/*9d*/	{ "popf",  FALSE, QUAD,  0,	      0 },
/*9e*/	{ "sahf",  FALSE, NONE,  0,	      0 },
/*9f*/	{ "lahf",  FALSE, NONE,  0,	      0 },

/*a0*/	{ "mov",   FALSE, BYTE,  op2(O, A),   0 },
/*a1*/	{ "mov",   FALSE, LONG,  op2(O, A),   0 },
/*a2*/	{ "mov",   FALSE, BYTE,  op2(A, O),   0 },
/*a3*/	{ "mov",   FALSE, LONG,  op2(A, O),   0 },
/*a4*/	{ "movs",  FALSE, BYTE,  op2(SI,DI),  0 },
/*a5*/	{ "movs",  FALSE, LONG,  op2(SI,DI),  0 },
/*a6*/	{ "cmps",  FALSE, BYTE,  op2(SI,DI),  0 },
/*a7*/	{ "cmps",  FALSE, LONG,  op2(SI,DI),  0 },

/*a8*/	{ "test",  FALSE, BYTE,  op2(I, A),   0 },
/*a9*/	{ "test",  FALSE, LONG,  op2(I, A),   0 },
/*aa*/	{ "stos",  FALSE, BYTE,  op1(DI),     0 },
/*ab*/	{ "stos",  FALSE, LONG,  op1(DI),     0 },
/*ac*/	{ "lods",  FALSE, BYTE,  op1(SI),     0 },
/*ad*/	{ "lods",  FALSE, LONG,  op1(SI),     0 },
/*ae*/	{ "scas",  FALSE, BYTE,  op1(SI),     0 },
/*af*/	{ "scas",  FALSE, LONG,  op1(SI),     0 },

/*b0*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },
/*b1*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },
/*b2*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },
/*b3*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },
/*b4*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },
/*b5*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },
/*b6*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },
/*b7*/	{ "mov",   FALSE, BYTE,  op2(I, Ri),  0 },

/*b8*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },
/*b9*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },
/*ba*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },
/*bb*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },
/*bc*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },
/*bd*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },
/*be*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },
/*bf*/	{ "mov",   FALSE, LONG,  op2(I, Ri),  0 },

/*c0*/	{ NULL,	   TRUE,  BYTE,  op2(Ib, E),  db_Grp2 },
/*c1*/	{ NULL,	   TRUE,  LONG,  op2(Ib, E),  db_Grp2 },
/*c2*/	{ "ret",   FALSE, NONE,  op1(Iw),     0 },
/*c3*/	{ "ret",   FALSE, NONE,  0,	      0 },
/*c4*/	{ "",      TRUE,  LONG,  op2(E, R),   0 },
/*c5*/	{ "",      TRUE,  LONG,  op2(E, R),   0 },
/*c6*/	{ "mov",   TRUE,  BYTE,  op2(I, E),   0 },
/*c7*/	{ "mov",   TRUE,  LONG,  op2(I, E),   0 },

/*c8*/	{ "enter", FALSE, NONE,  op2(Iq, Ib), 0 },
/*c9*/	{ "leave", FALSE, NONE,  0,           0 },
/*ca*/	{ "lret",  FALSE, NONE,  op1(Iw),     0 },
/*cb*/	{ "lret",  FALSE, NONE,  0,	      0 },
/*cc*/	{ "int",   FALSE, NONE,  op1(o3),     0 },
/*cd*/	{ "int",   FALSE, NONE,  op1(Ib),     0 },
/*ce*/	{ "",      FALSE, NONE,  0,	      0 },
/*cf*/	{ "iret",  FALSE, NONE,  0,	      0 },

/*d0*/	{ NULL,	   TRUE,  BYTE,  op2(o1, E),  db_Grp2 },
/*d1*/	{ NULL,	   TRUE,  LONG,  op2(o1, E),  db_Grp2 },
/*d2*/	{ NULL,	   TRUE,  BYTE,  op2(CL, E),  db_Grp2 },
/*d3*/	{ NULL,	   TRUE,  LONG,  op2(CL, E),  db_Grp2 },
/*d4*/	{ "",      TRUE,  NONE,  op1(Iba),    0 },
/*d5*/	{ "",      TRUE,  NONE,  op1(Iba),    0 },
/*d6*/	{ ".byte\t0xd6",FALSE, NONE, 0,       0 },
/*d7*/	{ "xlat",  FALSE, BYTE,  op1(BX),     0 },

/* d8 to df block is ignored: direct test in code handles them */
/*d8*/	{ "",      TRUE,  NONE,  0,	      db_Esc8 },
/*d9*/	{ "",      TRUE,  NONE,  0,	      db_Esc9 },
/*da*/	{ "",      TRUE,  NONE,  0,	      db_Esca },
/*db*/	{ "",      TRUE,  NONE,  0,	      db_Escb },
/*dc*/	{ "",      TRUE,  NONE,  0,	      db_Escc },
/*dd*/	{ "",      TRUE,  NONE,  0,	      db_Escd },
/*de*/	{ "",      TRUE,  NONE,  0,	      db_Esce },
/*df*/	{ "",      TRUE,  NONE,  0,	      db_Escf },

/*e0*/	{ "loopne",FALSE, NONE,  op1(Db),     0 },
/*e1*/	{ "loope", FALSE, NONE,  op1(Db),     0 },
/*e2*/	{ "loop",  FALSE, NONE,  op1(Db),     0 },
/*e3*/	{ "jcxz",  FALSE, SDEP,  op1(Db),     "jecxz\0jrcxz" },
/*e4*/	{ "in",    FALSE, BYTE,  op2(Ib, A),  0 },
/*e5*/	{ "in",    FALSE, LONG,  op2(Ib, A) , 0 },
/*e6*/	{ "out",   FALSE, BYTE,  op2(A, Ib),  0 },
/*e7*/	{ "out",   FALSE, LONG,  op2(A, Ib) , 0 },

/*e8*/	{ "call",  FALSE, QUAD,  op1(Dl),     0 },
/*e9*/	{ "jmp",   FALSE, NONE,  op1(Dl),     0 },
/*ea*/	{ "",      FALSE, NONE,  op1(OS),     0 },
/*eb*/	{ "jmp",   FALSE, NONE,  op1(Db),     0 },
/*ec*/	{ "in",    FALSE, BYTE,  op2(DX, A),  0 },
/*ed*/	{ "in",    FALSE, LONG,  op2(DX, A) , 0 },
/*ee*/	{ "out",   FALSE, BYTE,  op2(A, DX),  0 },
/*ef*/	{ "out",   FALSE, LONG,  op2(A, DX) , 0 },

/*f0*/	{ "",      FALSE, NONE,  0,	     0 },
/*f1*/	{ "",      FALSE, NONE,  0,	     0 },
/*f2*/	{ "",      FALSE, NONE,  0,	     0 },
/*f3*/	{ "",      FALSE, NONE,  0,	     0 },
/*f4*/	{ "hlt",   FALSE, NONE,  0,	     0 },
/*f5*/	{ "cmc",   FALSE, NONE,  0,	     0 },
/*f6*/	{ "",      TRUE,  BYTE,  0,	     db_Grp3 },
/*f7*/	{ "",	   TRUE,  LONG,  0,	     db_Grp3 },

/*f8*/	{ "clc",   FALSE, NONE,  0,	     0 },
/*f9*/	{ "stc",   FALSE, NONE,  0,	     0 },
/*fa*/	{ "cli",   FALSE, NONE,  0,	     0 },
/*fb*/	{ "sti",   FALSE, NONE,  0,	     0 },
/*fc*/	{ "cld",   FALSE, NONE,  0,	     0 },
/*fd*/	{ "std",   FALSE, NONE,  0,	     0 },
/*fe*/	{ "",	   TRUE,  RDEP,  0,	     db_Grp4 },
/*ff*/	{ "",	   TRUE,  RDEP,  0,	     db_Grp5 },
};

struct inst	db_bad_inst =
	{ "???",   FALSE, NONE,  0,	      0 }
;

#define REX_W(rex)	(((rex) & 0x8) == 0x8)	/* 64-bit operand size */
#define REX_R(rex)	(((rex) & 0x4) << 1)	/* ModRM reg field extension */
#define REX_X(rex)	(((rex) & 0x2) << 2)	/* SIB index field extension */
#define REX_B(rex)	(((rex) & 0x1) << 3)	/* ModRM r/m and SIB base ext */

#define	f_mod(byte)		((byte)>>6)
#define	f_reg(byte, rex)	((((byte)>>3)&0x7) + REX_R(rex))
#define	f_rm(byte, rex)		(((byte)&0x7) + REX_B(rex))

#define	sib_ss(byte)		((byte)>>6)
#define	sib_index(byte, rex)	((((byte)>>3)&0x7) + REX_X(rex))
#define	sib_base(byte, rex)	(((byte)&0x7) + REX_B(rex))

struct i_addr {
	int		is_reg;	/* if reg, reg number is in 'disp' */
	int		disp;
	char *		base;
	char *		index;
	int		ss;
};

char *	db_index_reg_16[8] = {
	"%bx,%si",
	"%bx,%di",
	"%bp,%si",
	"%bp,%di",
	"%si",
	"%di",
	"%bp",
	"%bx"
};

char *	db_reg[4][16] = {
	{ "%al",  "%cl",  "%dl",  "%bl",  "%ah",  "%ch",  "%dh",  "%bh",
	  "%r8b", "%r9b", "%r10b", "%r11b", "%r12b", "%r13b", "%r14b", "%r15b"},
	{ "%ax",  "%cx",  "%dx",  "%bx",  "%sp",  "%bp",  "%si",  "%di",
	  "%r8w", "%r9w", "%r10w", "%r11w", "%r12w", "%r13w", "%r14w", "%r15w"},
	{ "%eax", "%ecx", "%edx", "%ebx", "%esp", "%ebp", "%esi", "%edi",
	  "%r8d", "%r9d", "%r10d", "%r11d", "%r12d", "%r13d", "%r14d", "%r15d"},
	{ "%rax", "%rcx", "%rdx", "%rbx", "%rsp", "%rbp", "%rsi", "%rdi",
	  "%r8",  "%r9",  "%r10", "%r11", "%r12", "%r13", "%r14", "%r15" }
};

char *	db_seg_reg[8] = {
	"%es", "%cs", "%ss", "%ds", "%fs", "%gs", "", ""
};

/*
 * lengths for size attributes
 */
int db_lengths[] = {
	1,	/* BYTE */
	2,	/* WORD */
	4,	/* LONG */
	8,	/* QUAD */
	4,	/* SNGL */
	8,	/* DBLR */
	10,	/* EXTR */
};

#define	get_value_inc(result, loc, size, is_signed) \
	do { \
		result = db_get_value((loc), (size), (is_signed)); \
		(loc) += (size); \
	} while (0)


db_addr_t db_read_address(db_addr_t, int, int, int, struct i_addr *);
void db_print_address(char *, int, struct i_addr *);
db_addr_t db_disasm_esc(db_addr_t, int, int, int, int, char *);
db_addr_t db_disasm_3dnow(db_addr_t, int, int, int, char *);

/*
 * Read address at location and return updated location.
 */
db_addr_t
db_read_address(db_addr_t loc, int short_addr, int regmodrm, int rex,
    struct i_addr *addrp)
{
	int		mod, rm, sib, index, disp, size;

	size = (short_addr ? LONG : QUAD);
	mod = f_mod(regmodrm);
	rm  = f_rm(regmodrm, rex);

	if (mod == 3) {
		addrp->is_reg = TRUE;
		addrp->disp = rm;
		return (loc);
	}
	addrp->is_reg = FALSE;
	addrp->index = 0;

	if (rm == 4 || rm == 12) {
		get_value_inc(sib, loc, 1, FALSE);
		rm = sib_base(sib, rex);
		index = sib_index(sib, rex);
		if (index != 4)
			addrp->index = db_reg[size][index];
		addrp->ss = sib_ss(sib);
	}

	switch (mod) {
	case 0:
		if (rm == 5) {
			get_value_inc(addrp->disp, loc, 4, FALSE);
			addrp->base = 0;
		} else {
			addrp->disp = 0;
			addrp->base = db_reg[size][rm];
		}
		break;
	case 1:
		get_value_inc(disp, loc, 1, TRUE);
		addrp->disp = disp;
		addrp->base = db_reg[size][rm];
		break;
	case 2:
		get_value_inc(disp, loc, 4, FALSE);
		addrp->disp = disp;
		addrp->base = db_reg[size][rm];
		break;
	}
	return (loc);
}

void
db_print_address(char *seg, int size, struct i_addr *addrp)
{
	if (addrp->is_reg) {
		db_printf("%s", db_reg[size][addrp->disp]);
		return;
	}

	if (seg)
		db_printf("%s:", seg);

	db_printsym((db_addr_t)addrp->disp, DB_STGY_ANY, db_printf);
	if (addrp->base != 0 || addrp->index != 0) {
		db_printf("(");
		if (addrp->base)
			db_printf("%s", addrp->base);
		if (addrp->index)
			db_printf(",%s,%d", addrp->index, 1<<addrp->ss);
		db_printf(")");
	}
}

/*
 * Disassemble 3DNow! instruction and return updated location.
 */
db_addr_t
db_disasm_3dnow(db_addr_t loc, int short_addr, int size, int rex, char *seg)
{
	int regmodrm, sib, displacement, opcode;

	get_value_inc(regmodrm, loc, 1, FALSE);
	get_value_inc(sib, loc, 1, FALSE);
	get_value_inc(displacement, loc, 1, FALSE);
	get_value_inc(opcode, loc, 1, FALSE);

	/* XXX fix later... */ 
	db_printf("<3DNow! instruction>");

	return (loc);
}

/*
 * Disassemble floating-point ("escape") instruction
 * and return updated location.
 */
db_addr_t
db_disasm_esc(db_addr_t loc, int inst, int short_addr, int size, int rex,
    char *seg)
{
	int		regmodrm;
	struct finst	*fp;
	int		mod;
	struct i_addr	address;
	char *		name;

	get_value_inc(regmodrm, loc, 1, FALSE);
	fp = &db_Esc_inst[inst - 0xd8][f_reg(regmodrm, 0)];
	mod = f_mod(regmodrm);
	if (mod != 3) {
		if (*fp->f_name == '\0') {
			db_printf("<bad instruction>");
			return (loc);
		}

		/*
		 * Normal address modes.
		 */
		loc = db_read_address(loc, short_addr, regmodrm, rex, &address);
		db_printf("%s", fp->f_name);
		switch (fp->f_size) {
		case SNGL:
			db_printf("s");
			break;
		case DBLR:
			db_printf("l");
			break;
		case EXTR:
			db_printf("t");
			break;
		case WORD:
			db_printf("s");
			break;
		case LONG:
			db_printf("l");
			break;
		case QUAD:
			db_printf("q");
			break;
		default:
			break;
		}
		db_printf("\t");
		db_print_address(seg, BYTE, &address);
	} else {
		/*
		 * 'reg-reg' - special formats
		 */
		switch (fp->f_rrmode) {
		case op2(ST,STI):
			name = (fp->f_rrname) ? fp->f_rrname : fp->f_name;
			db_printf("%s\t%%st,%%st(%d)",name, f_rm(regmodrm, 0));
			break;
		case op2(STI,ST):
			name = (fp->f_rrname) ? fp->f_rrname : fp->f_name;
			db_printf("%s\t%%st(%d),%%st",name, f_rm(regmodrm, 0));
			break;
		case op1(STI):
			name = (fp->f_rrname) ? fp->f_rrname : fp->f_name;
			db_printf("%s\t%%st(%d)",name, f_rm(regmodrm, 0));
			break;
		case op1(X):
			name = ((char * const *)fp->f_rrname)[f_rm(regmodrm,0)];
			if (*name == '\0')
				goto bad;
			db_printf("%s", name);
			break;
		case op1(XA):
			name = ((char * const *)fp->f_rrname)[f_rm(regmodrm,0)];
			if (*name == '\0')
				goto bad;
			db_printf("%s\t%%ax", name);
			break;
		default:
		bad:
			db_printf("<bad instruction>");
			break;
		}
	}

	return (loc);
}

/*
 * Disassemble instruction at 'loc'.  'altfmt' specifies an
 * (optional) alternate format.  Return address of start of
 * next instruction.
 */
db_addr_t
db_disasm(db_addr_t loc, boolean_t altfmt)
{
	int	inst;
	int	size;
	int	short_addr;
	char *	seg;
	struct inst *	ip;
	char *	i_name;
	int	i_size;
	int	i_mode;
	int	regmodrm = 0;
	boolean_t	first;
	int	displ;
	int	prefix;
	long	imm;
	int	imm2;
	int	len;
	int	rex = 0;
	int	segovr_grp;
	int	repe, repne;
	struct i_addr	address;
	db_addr_t	loc_orig = loc;
	char	tmpfmt[28];

	get_value_inc(inst, loc, 1, FALSE);
	short_addr = FALSE;
	size = LONG;
	seg = 0;
	segovr_grp = 0;
	repe = 0;
	repne = 0;

	/*
	 * Get prefixes
	 */
	prefix = TRUE;
	do {
		switch (inst) {
		case 0x66:		/* data16 */
			size = WORD;
			break;
		case 0x67:
			short_addr = TRUE;
			break;
		case 0x26:
			segovr_grp++;
			db_printf(" <segment override prefix ignored>");
			break;
		case 0x36:
			db_printf(" <segment override prefix ignored>");
			segovr_grp++;
			break;
		case 0x2e:
			db_printf(" <segment override prefix ignored>");
			segovr_grp++;
			break;
		case 0x3e:
			db_printf(" <segment override prefix ignored>");
			segovr_grp++;
			break;
		case 0x64:
			segovr_grp++;
			seg = "%fs";
			break;
		case 0x65:
			segovr_grp++;
			seg = "%gs";
			break;
		case 0xf0:
			db_printf("lock ");
			break;
		case 0xf2:
			repne++;
			break;
		case 0xf3:
			repe++;
			break;
		default:
			prefix = FALSE;
			break;
		}
		if (prefix)
			get_value_inc(inst, loc, 1, FALSE);
	} while (prefix);
	if (segovr_grp > 1)
		seg = "<bad segment override prefix combination> ";
	if (repe > 0 && repne > 0)
		db_printf("<bad repeat prefex combination> ");
	else if (repe > 0)
		db_printf("repe ");	/* XXX "rep" if not CMPSx or SCASx */
	else if (repne > 0)
		db_printf("repne ");

	if (inst >= 0x40 && inst <= 0x4f) {
		// rex page 14
		rex = inst;
		if (REX_W(rex))
			size = QUAD;
		get_value_inc(inst, loc, 1, FALSE);
	}

	if (inst >= 0xd8 && inst <= 0xdf) {
		loc = db_disasm_esc(loc, inst, short_addr, size, rex, seg);
		goto done;
	}

	if (inst == 0x0f) {
		get_value_inc(inst, loc, 1, FALSE);
		if (inst == 0x0f) {
			loc = db_disasm_3dnow(loc, short_addr, size, rex, seg);
			goto done;
		}
		ip = db_inst_0f[inst>>4];
		if (ip == 0)
			ip = &db_bad_inst;
		else
			ip = &ip[inst&0xf];
	} else {
		ip = &db_inst_table[inst];
	}

	if (ip->i_has_modrm) {
		get_value_inc(regmodrm, loc, 1, FALSE);
		loc = db_read_address(loc, short_addr, regmodrm, rex, &address);
	}

	i_name = ip->i_name;
	i_size = ip->i_size;
	i_mode = ip->i_mode;

	if (i_size == RDEP) {
		/* sub-table to handle dependency on reg from ModR/M byte */
		ip = (struct inst *)ip->i_extra;
		ip = &ip[f_reg(regmodrm, 0)];
		i_name = ip->i_name;
		i_mode = ip->i_mode;
		i_size = ip->i_size;
	} else if (i_name == NULL) {
		i_name = ((char **)ip->i_extra)[f_reg(regmodrm, 0)];
	} else if (ip->i_extra == db_Grp3) {
		ip = (struct inst *)ip->i_extra;
		ip = &ip[f_reg(regmodrm, 0)];
		i_name = ip->i_name;
		i_mode = ip->i_mode;
	}

	/* ModR/M-specific operation? */
	if ((i_mode & 0xFF) == MEx) {
		if (f_mod(regmodrm) != 3)
			i_mode = op1(E);
		else {
			/* unknown extension? */
			if (f_rm(regmodrm, 0) > (i_mode >> 8))
				i_name = "";
			else {
				/* skip to the specific op */
				int i = f_rm(regmodrm, 0);
				i_name = ip->i_extra;
				while (i-- > 0)
					while (*i_name++)
						;
			}
			i_mode = 0;
		}
	}

	if (i_size == SDEP) {
		if (size == WORD)
			db_printf("%s", i_name);
		else if (size == LONG)
			db_printf("%s", (char *)ip->i_extra);
		else {
			char *cp = ip->i_extra;
			while (*cp)
				cp++;
			cp++;
			db_printf("%s", cp);
		}
	} else {
		db_printf("%s", i_name);
		if (i_size != NONE) {
			if (i_size == BYTE) {
				db_printf("b");
				size = BYTE;
			} else if (REX_W(rex)) {
				db_printf("q");
				size = QUAD;
			} else if (i_size == WORD) {
				db_printf("w");
				size = WORD;
			} else if (i_size == QUAD) {
				size = QUAD;
				db_printf("q");
			} else if (size == WORD) {
				db_printf("w");
			} else {
				db_printf("l");
			}
		}
	}
	db_printf("\t");
	for (first = TRUE; i_mode != 0; i_mode >>= 8, first = FALSE) {
		if (!first)
			db_printf(",");

		switch (i_mode & 0xFF) {
		case E: //XXX
			db_print_address(seg, size, &address);
			break;
		case Eind: //XXX
			db_printf("*");
			db_print_address(seg, size, &address);
			break;
		case Ew: //XXX
			db_print_address(seg, WORD, &address);
			break;
		case Eb: //XXX
			db_print_address(seg, BYTE, &address);
			break;
		case R:
			db_printf("%s", db_reg[size][f_reg(regmodrm, rex)]);
			break;
		case Rw:
			db_printf("%s", db_reg[WORD][f_reg(regmodrm, rex)]);
			break;
		case Ri:
			db_printf("%s", db_reg[size][f_rm(inst, rex)]);
			break;
		case Ril:
			db_printf("%s", db_reg[LONG][f_rm(inst, rex)]);
			break;
		case S:
			db_printf("%s", db_seg_reg[f_reg(regmodrm, 0)]);
			break;
		case Si:
			db_printf("%s", db_seg_reg[f_reg(inst, 0)]);
			break;
		case A:
			db_printf("%s", db_reg[size][0]);	/* acc */
			break;
		case BX:
			if (seg)
				db_printf("%s:", seg);
			db_printf("(%s)", short_addr ? "%ebx" : "%rbx");
			break;
		case CL:
			db_printf("%%cl");
			break;
		case DX:
			db_printf("%%dx");
			break;
		case SI:
			if (seg)
				db_printf("%s:", seg);
			db_printf("(%s)", short_addr ? "%esi" : "%rsi");
			break;
		case DI:
			db_printf("%%es:(%s)", short_addr ? "%edi" : "%rdi");
			break;
		case CR:
			db_printf("%%cr%d", f_reg(regmodrm, rex));
			break;
		case DR:
			db_printf("%%dr%d", f_reg(regmodrm, rex));
			break;
		case TR: //XXX
			db_printf("%%tr%d", f_reg(regmodrm, rex));
			break;
		case I:
			len = db_lengths[size];
			get_value_inc(imm, loc, len, FALSE);
			db_printf("$%s", db_format(tmpfmt, sizeof tmpfmt,
			    (int)imm, DB_FORMAT_N, 1, 0));
			break;
		case Is:
			len = db_lengths[size];
			get_value_inc(imm, loc, len, TRUE);
			db_printf("$%s", db_format(tmpfmt, sizeof tmpfmt,
			    (int)imm, DB_FORMAT_R, 1, 0));
			break;
		case Ib:
			get_value_inc(imm, loc, 1, FALSE);
			db_printf("$%s", db_format(tmpfmt, sizeof tmpfmt,
			    (int)imm, DB_FORMAT_N, 1, 0));
			break;
		case Iba:
			get_value_inc(imm, loc, 1, FALSE);
			if (imm != 0x0a)
				db_printf("$%s", db_format(tmpfmt, sizeof tmpfmt,
				    (int)imm, DB_FORMAT_N, 1, 0));
			break;
		case Ibs: //XXX
			get_value_inc(imm, loc, 1, TRUE);
			if (size == WORD)
				imm &= 0xFFFF;
			db_printf("$%s", db_format(tmpfmt, sizeof tmpfmt,
			    (int)imm, DB_FORMAT_R, 1, 0));
			break;
		case Iw:
			get_value_inc(imm, loc, 2, FALSE);
			db_printf("$%s", db_format(tmpfmt, sizeof tmpfmt,
			    (int)imm, DB_FORMAT_N, 1, 0));
			break;
		case Iq:
			get_value_inc(imm, loc, 8, TRUE);
			db_printf("$%s", db_format(tmpfmt, sizeof tmpfmt,
			   (int)((imm >> 32) & 0xffffffff), DB_FORMAT_R, 1, 0));
			db_printf(" %s", db_format(tmpfmt, sizeof tmpfmt,
			   (int)(imm & 0xffffffff), DB_FORMAT_R, 1, 0));
			break;
		case O: //XXX
			if (short_addr)
				get_value_inc(displ, loc, 2, TRUE);
			else
				get_value_inc(displ, loc, 4, TRUE);
			if (seg)
				db_printf("%s:%s", seg, db_format(tmpfmt,
				    sizeof tmpfmt, displ, DB_FORMAT_R, 1, 0));
			else
				db_printsym((db_addr_t)displ, DB_STGY_ANY,
				    db_printf);
			break;
		case Db: //XXX
			get_value_inc(displ, loc, 1, TRUE);
			displ += loc;
			if (size == WORD)
				displ &= 0xFFFF;
			db_printsym((db_addr_t)displ, DB_STGY_XTRN, db_printf);
			break;
		case Dl: //XXX
			len = db_lengths[size];
			get_value_inc(displ, loc, len, FALSE);
			displ += loc;
			if (size == WORD)
				displ &= 0xFFFF;
			db_printsym((db_addr_t)displ, DB_STGY_XTRN, db_printf);
			break;
		case o1: //XXX
			db_printf("$1");
			break;
		case o3: //XXX
			db_printf("$3");
			break;
		case OS: //XXX
			len = db_lengths[size];
			get_value_inc(imm, loc, len, FALSE);	/* offset */
			get_value_inc(imm2, loc, 2, FALSE);	/* segment */
			db_printf("$0x%#x", imm2);
			break;
		}
	}

	if (altfmt == 0 && (inst == 0xe9 || inst == 0xeb)) {
		/*
		 * GAS pads to longword boundary after unconditional jumps.
		 */
		loc = (loc + (4-1)) & ~(4-1);
	}
done:
	if (loc - loc_orig > 15)
		db_printf(" <instruction too long>");
	if (altfmt) {
		db_printf("\n\t");
		while (loc_orig < loc) {
			get_value_inc(imm, loc_orig, 1, FALSE);
			if (imm < 0x10)
				db_printf("0");
			db_printf("%x ", (int)imm);
		}
	}
	db_printf("\n");
	return (loc);
}
