# -*- coding: utf-8 -*- #
#
# Copyright 2018 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""A list of regex that can be compiled lazily to improve gcloud performance.

Some regex need to be compiled immediately so that they will raise an
re.error if the pattern is not valid, but the below list are known to be
valid, so they can be compiled lazily. Most of these regex don't end up being
compiled for running any given gcloud command.

These patterns are kept in a Python source file to minimize loading time.

They should be updated periodically with //cloud/sdk:update_lazy_regex.
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

# These lines don't need to be readable, so bother don't breaking them up.
# pylint: disable=line-too-long
PATTERNS = frozenset((
    "'(?:[^'\\n\\r\\\\]|(?:'')|(?:\\\\(?:[^x]|x[0-9a-fA-F]+)))*",
    "[8901234567\\*\\+\\-\\.\\!\\#\\$\\%\\&\\'xyz\\|\\~pqrstuvwhijklmno\\`abcdefgXYZ\\^\\_PQRSTUVWHIJKLMNOABCDEFG]+$",
    "[abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789\\!\\#\\$\\%\\&\\'\\*\\+\\-\\.\\^_\\`\\|\\~\\:]+",
    "[bLx9tdE\\|17SR3GiQMYhaT6\\&\\#0\\*\\%8kz42H\\`U\\^ZN\\-qpAguwr\\+v\\~sPIeyXlom\\.fBFV_CW5j\\$\\!ncODK\\'J]+$",
    "\\'(?:[^'\\n\\r])*\\'",
    "\n        (?P<operator>(~=|==|!=|<=|>=|<|>|===))\n        (?P<version>\n            (?:\n                # The identity operators allow for an escape hatch that will\n                # do an exact string match of the version you wish to install.\n                # This will not be parsed by PEP 440 and we cannot determine\n                # any semantic meaning from it. This operator is discouraged\n                # but included entirely as an escape hatch.\n                (?<====)  # Only match for the identity operator\n                \\s*\n                [^\\s]*    # We just match everything, except for whitespace\n                          # since we are only testing for strict identity.\n            )\n            |\n            (?:\n                # The (non)equality operators allow for wild card and local\n                # versions to be specified so we have to define these two\n                # operators separately to enable that.\n                (?<===|!=)            # Only match for equals and not equals\n\n                \\s*\n                v?\n                (?:[0-9]+!)?          # epoch\n                [0-9]+(?:\\.[0-9]+)*   # release\n                (?:                   # pre release\n                    [-_\\.]?\n                    (a|b|c|rc|alpha|beta|pre|preview)\n                    [-_\\.]?\n                    [0-9]*\n                )?\n                (?:                   # post release\n                    (?:-[0-9]+)|(?:[-_\\.]?(post|rev|r)[-_\\.]?[0-9]*)\n                )?\n\n                # You cannot use a wild card and a dev or local version\n                # together so group them with a | and make them optional.\n                (?:\n                    (?:[-_\\.]?dev[-_\\.]?[0-9]*)?         # dev release\n                    (?:\\+[a-z0-9]+(?:[-_\\.][a-z0-9]+)*)? # local\n                    |\n                    \\.\\*  # Wild card syntax of .*\n                )?\n            )\n            |\n            (?:\n                # The compatible operator requires at least two digits in the\n                # release segment.\n                (?<=~=)               # Only match for the compatible operator\n\n                \\s*\n                v?\n                (?:[0-9]+!)?          # epoch\n                [0-9]+(?:\\.[0-9]+)+   # release  (We have a + instead of a *)\n                (?:                   # pre release\n                    [-_\\.]?\n                    (a|b|c|rc|alpha|beta|pre|preview)\n                    [-_\\.]?\n                    [0-9]*\n                )?\n                (?:                                   # post release\n                    (?:-[0-9]+)|(?:[-_\\.]?(post|rev|r)[-_\\.]?[0-9]*)\n                )?\n                (?:[-_\\.]?dev[-_\\.]?[0-9]*)?          # dev release\n            )\n            |\n            (?:\n                # All other operators only allow a sub set of what the\n                # (non)equality operators do. Specifically they do not allow\n                # local versions to be specified nor do they allow the prefix\n                # matching wild cards.\n                (?<!==|!=|~=)         # We have special cases for these\n                                      # operators so we want to make sure they\n                                      # don't match here.\n\n                \\s*\n                v?\n                (?:[0-9]+!)?          # epoch\n                [0-9]+(?:\\.[0-9]+)*   # release\n                (?:                   # pre release\n                    [-_\\.]?\n                    (a|b|c|rc|alpha|beta|pre|preview)\n                    [-_\\.]?\n                    [0-9]*\n                )?\n                (?:                                   # post release\n                    (?:-[0-9]+)|(?:[-_\\.]?(post|rev|r)[-_\\.]?[0-9]*)\n                )?\n                (?:[-_\\.]?dev[-_\\.]?[0-9]*)?          # dev release\n            )\n        )\n        ",
    "\n# Single star or\n^([*]|\n# Host prefix with no .,  Ex '*-a' or\n[*][a-z0-9\\-]*[a-z0-9]|\n# Host prefix with ., Ex '*-a.b-c.d'\n[*](\\.|[a-z0-9\\-]*[a-z0-9]\\.)([a-z0-9]([a-z0-9\\-]*[a-z0-9])*\\.)*\n([a-z0-9]([a-z0-9\\-]*[a-z0-9])*))$\n",
    "^\\s*\n        (?P<operator>(~=|==|!=|<=|>=|<|>|===))\n        (?P<version>\n            (?:\n                # The identity operators allow for an escape hatch that will\n                # do an exact string match of the version you wish to install.\n                # This will not be parsed by PEP 440 and we cannot determine\n                # any semantic meaning from it. This operator is discouraged\n                # but included entirely as an escape hatch.\n                (?<====)  # Only match for the identity operator\n                \\s*\n                [^\\s]*    # We just match everything, except for whitespace\n                          # since we are only testing for strict identity.\n            )\n            |\n            (?:\n                # The (non)equality operators allow for wild card and local\n                # versions to be specified so we have to define these two\n                # operators separately to enable that.\n                (?<===|!=)            # Only match for equals and not equals\n\n                \\s*\n                v?\n                (?:[0-9]+!)?          # epoch\n                [0-9]+(?:\\.[0-9]+)*   # release\n                (?:                   # pre release\n                    [-_\\.]?\n                    (a|b|c|rc|alpha|beta|pre|preview)\n                    [-_\\.]?\n                    [0-9]*\n                )?\n                (?:                   # post release\n                    (?:-[0-9]+)|(?:[-_\\.]?(post|rev|r)[-_\\.]?[0-9]*)\n                )?\n\n                # You cannot use a wild card and a dev or local version\n                # together so group them with a | and make them optional.\n                (?:\n                    (?:[-_\\.]?dev[-_\\.]?[0-9]*)?         # dev release\n                    (?:\\+[a-z0-9]+(?:[-_\\.][a-z0-9]+)*)? # local\n                    |\n                    \\.\\*  # Wild card syntax of .*\n                )?\n            )\n            |\n            (?:\n                # The compatible operator requires at least two digits in the\n                # release segment.\n                (?<=~=)               # Only match for the compatible operator\n\n                \\s*\n                v?\n                (?:[0-9]+!)?          # epoch\n                [0-9]+(?:\\.[0-9]+)+   # release  (We have a + instead of a *)\n                (?:                   # pre release\n                    [-_\\.]?\n                    (a|b|c|rc|alpha|beta|pre|preview)\n                    [-_\\.]?\n                    [0-9]*\n                )?\n                (?:                                   # post release\n                    (?:-[0-9]+)|(?:[-_\\.]?(post|rev|r)[-_\\.]?[0-9]*)\n                )?\n                (?:[-_\\.]?dev[-_\\.]?[0-9]*)?          # dev release\n            )\n            |\n            (?:\n                # All other operators only allow a sub set of what the\n                # (non)equality operators do. Specifically they do not allow\n                # local versions to be specified nor do they allow the prefix\n                # matching wild cards.\n                (?<!==|!=|~=)         # We have special cases for these\n                                      # operators so we want to make sure they\n                                      # don't match here.\n\n                \\s*\n                v?\n                (?:[0-9]+!)?          # epoch\n                [0-9]+(?:\\.[0-9]+)*   # release\n                (?:                   # pre release\n                    [-_\\.]?\n                    (a|b|c|rc|alpha|beta|pre|preview)\n                    [-_\\.]?\n                    [0-9]*\n                )?\n                (?:                                   # post release\n                    (?:-[0-9]+)|(?:[-_\\.]?(post|rev|r)[-_\\.]?[0-9]*)\n                )?\n                (?:[-_\\.]?dev[-_\\.]?[0-9]*)?          # dev release\n            )\n        )\n        \\s*$",
    '        # A numeric string consists of:\n#    \\s*\n    (?P<sign>[-+])?              # an optional sign, followed by either...\n    (\n        (?=\\d|\\.\\d)              # ...a number (with at least one digit)\n        (?P<int>\\d*)             # having a (possibly empty) integer part\n        (\\.(?P<frac>\\d*))?       # followed by an optional fractional part\n        (E(?P<exp>[-+]?\\d+))?    # followed by an optional exponent, or...\n    |\n        Inf(inity)?              # ...an infinity, or...\n    |\n        (?P<signal>s)?           # ...an (optionally signaling)\n        NaN                      # NaN\n        (?P<diag>\\d*)            # with (possibly empty) diagnostic info.\n    )\n#    \\s*\n    \\Z\n',
    '"(?:[^"\\n\\r\\\\]|(?:"")|(?:\\\\(?:[^x]|x[0-9a-fA-F]+)))*',
    '# End of Google Compute Engine Section',
    '# Google Compute Engine Section',
    '#( Netscape)? HTTP Cookie File',
    '#.*',
    '#.*coding[:=]\\s*([-\\w.]+).*\\r?\\n',
    '%([0-9a-fA-F][0-9a-fA-F])',
    '& (?:\n                                          \\#(\\d+)\n                                          | \\#x([\\da-f]+)\n                                          | ( (?!\\d) [:\\w] [-.:\\w]+ )\n                                          ) ;',
    '&(#[0-9]+;?|#[xX][0-9a-fA-F]+;?|[^\\t\\n\\f <&#;]{1,32};?)',
    '&(?P<entity>gt|amp|quot|nbsp|apos|lt);',
    '',
    '(.*[\\\\/:])?(main|patch)\\.\\d+(\\.[a-zA-Z]\\w*)+\\.obb$',
    '(.+)=(.*)',
    '(25[0-5]|2[0-4][0-9]|1?[0-9]{1,2})(\\.(25[0-5]|2[0-4][0-9]|1?[0-9]{1,2})){3}',
    '(?!)',
    '(?P<Y>\\d\\d\\d\\d)\\.(?P<m>1[0-2]|0[1-9]|[1-9])\\.(?P<d>3[0-1]|[1-2]\\d|0[1-9]|[1-9]| [1-9])',
    '(?P<collection>[a-zA-Z_]+(?:\\.[a-zA-Z0-9_]+)+)\\.get',
    '(?P<year>\\d\\d\\d\\d)(?:-(?P<month>\\d\\d)(?:-(?P<day>\\d\\d)(?:T(?P<hour>\\d\\d)(?::(?P<minute>\\d\\d)(?::(?P<second>\\d\\d))?)?)?)?)?Z',
    '(?P<year>\\d{4})(?:-(?P<month>\\d\\d)(?:-(?P<day>\\d\\d))?)?',
    '(?P<year>\\d{4})-(?P<month>\\d\\d)-(?P<day>\\d\\d)[T ](?P<hour>\\d\\d):(?P<minute>\\d\\d)(:(?P<second>\\d\\d(\\.\\d*)?)?)?(?P<tz>Z|[+-]\\d\\d:?\\d\\d)?',
    '(?i)[a-z0-9_*-]+/[a-z0-9_*-]+',
    '(?s:.*)\\Z',
    '(?x)\\s*(?P<key>[\\w\\d!#%&\'~_`><@,:/\\$\\*\\+\\-\\.\\^\\|\\)\\(\\?\\}\\{\\=]+?)(\\s*=\\s*(?P<val>"(?:[^\\\\"]|\\\\.)*"|\\w{3},\\s[\\s\\w\\d-]{9,11}\\s[\\d:]{8}\\sGMT|[\\w\\d!#%&\'~_`><@,:/\\$\\*\\+\\-\\.\\^\\|\\)\\(\\?\\}\\{\\=\\[\\]]*))?\\s*(\\s+|;|$)',
    '([\\"\\\\])',
    '([\\(,])\\s*(\\d.*?)\\s*([,\\)])',
    '([\\\\.^$*+?\\(\\){}\\[\\]|])',
    '([\t ]+)',
    '([^\\:\\=]+)([\\:\\=]?)(.*)',
    '([^\\:]+)([\\:]?)(.*)',
    '([^\\=\\:]+)([\\=\\:]?)(.*)',
    '([^\\=]+)([\\=]?)(.*)',
    '([^\\s]+)',
    '([a-zA-Z0-9_]+|[^a-zA-Z0-9_\\s]+)',
    '(\\d+ | [a-z]+ | \\.)',
    '(\\d+ | [a-z]+ | \\.| -)',
    '(\\s+)',
    '(\\s+|".*?"|\'.*?\')',
    '(\\s+|[^\\s\\w]*\\w+[^0-9\\W]-(?=\\w+[^0-9\\W])|(?<=[\\w\\!\\"\\\'\\&\\.\\,\\?])-{2,}(?=\\w))',
    '(\r\n|\r|\n)',
    '(\r\n|\r|\n)\\Z',
    '(^# Google Compute Engine Section\n#\n# The following has been auto-generated by "gcloud compute config-ssh"\n# to make accessing your Google Compute Engine virtual machines easier.\n#\n# To remove this blob, run:\n#\n#   gcloud compute config-ssh --remove\n#\n# You can also manually remove this blob by deleting everything from\n# here until the comment that contains the string "End of Google Compute\n# Engine Section".\n#\n# You should not hand-edit this section, unless you are deleting it.\n#\n\\n*?(Host.*?)\\n+# End of Google Compute Engine Section\\n?)',
    '(^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\\.[a-zA-Z0-9-.]+$)',
    '(^|^.*[^\\\\ ]|^.*\\\\ ) *$',
    '(go|go1\\..+)$',
    '(go|go1\\..+|go1\\d\\d(\\w+\\d)?)$',
    '(python|python-.+)$',
    '.*',
    '.*?\n',
    '//(?:\\\\\\n|[^\\n])*',
    '//|/$',
    '/\\*(?:[^*]|\\*(?!/))*',
    '0*$',
    '50*$',
    ':\\d+$',
    '<!--[\\s\\S]*?-->',
    '<(.+?)(\\s.*?)?>.*?</.+?>',
    'CLASS([0-9]+)$',
    'Host (.*?)\\n',
    'TYPE([0-9]+)$',
    '[ \\(\\)<>@,;:\\\\"/\\[\\]\\?=]',
    '[ \\t\\f]* \\# .* coding[=:][ \\t]*([-\\w.]+)',
    '["&<>]|[^\\x00-\\x7f]',
    '[+-]?\\d+',
    '[+-]?\\d+([eE][+-]?\\d+|\\.\\d*([eE][+-]?\\d+)?)',
    '[+-]?\\d+\\.?\\d*([eE][+-]?\\d+)?',
    '[+-]?\\d+\\.\\d*',
    '[.\u3002\uff0e\uff61]',
    '[.。．｡]',
    '[/\\\\ ]',
    '[0-9a-fA-F]{1,4}',
    '[0-9a-fA-F]{2}([:.-])[0-9a-fA-F]{2}(?:\\1[0-9a-fA-F]{2}){4}',
    '[0-9a-fA-F]{8}(-[0-9a-fA-F]{4}){3}-[0-9a-fA-F]{12}',
    '[0123456789ABCDEFabcdef]+',
    '[0123456789]+',
    '[0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!"#$%&\'()*+,\\-./:;<=?@[\\\\\\]\\^_`{|}~]+',
    '[0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!"#$%&\'()*+\\-./:;<=>?@[\\\\\\]\\^_`{|}~]+',
    '[A-Z]',
    '[ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz][ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_:]*',
    '[ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz][ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_\\-:]*',
    '[ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_][ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_]*',
    '[\\-_.]+',
    '[\\._-]',
    '[\\041-\\176]+:$',
    '[\\\\].',
    '[\\^\\\\\\-\\]]',
    '[\\x00-\\x20\\x7F]',
    '[\\x00\\x01\\x02\\x03\\x04\\x05\\x06\\x07\\x08\\x0b\\x0c\\x0e\\x0f\\x10\\x11\\x12\\x13\\x14\\x15\\x16\\x17\\x18\\x19\\x1a\\x1b\\x1c\\x1d\\x1e\\x1f]',
    '[][\\\\()"]',
    '[][\\\\()<>@,:;".]',
    '[^ !-<>-~\\t]',
    '[^ ]+',
    '[^*\\](\\\\ )<\t=,?[/:@";>\']+',
    '[^*\\](\\\\% )<\t=,?[/:@";>\']+',
    '[^-a-zA-Z0-9!*+/ ]',
    '[^.]*',
    '[^[,/:\\](\\\\ )@"<;\t=>?]+',
    '[^[:\\](\\\\ )@"<;\t>.,]+',
    '[a-z][a-z0-9\\-]{0,29}\\Z',
    '[abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789]+',
    '\\"(?:[^"\\n\\r])*\\"',
    '\\.?[^.]*',
    '\\.\\.|^\\./|\\.$|/\\./|^-|^_ah/|^/',
    '\\.\\d+$',
    '\\A[\\x00-\\x7f]*\\Z',
    '\\A\n(?:\n   (?P<fill>.)?\n   (?P<align>[<>=^])\n)?\n(?P<sign>[-+ ])?\n(?P<zeropad>0)?\n(?P<minimumwidth>(?!0)\\d+)?\n(?P<thousands_sep>,)?\n(?:\\.(?P<precision>0|(?!0)\\d+))?\n(?P<type>[eEfFgGn%])?\n\\Z\n',
    '\\W',
    '\\[(.*)\\]:(\\d*)',
    '\\\\(.)',
    '\\\\0?[xX][0-9a-fA-F]+',
    '\\\\0[0-7]+',
    '\\\\[0-3][0-7][0-7]',
    '\\belse\\s+do\\s*$',
    '\\n[^ \\t]+:',
    '\\n\\n\\n+',
    '\\n|\\r',
    '\\r\\n|\\r|\\n',
    '\\s*(?P<name>.+?)\\s*=\\s*(?P<module>[\\w.]+)\\s*(:\\s*(?P<attr>[\\w.]+))?\\s*(?P<extras>\\[.*\\])?\\s*$',
    '\\s*<\\?xml[^>]*\\?>',
    '\\s+',
    '\\w',
    '\\w+(\\.\\w+)*$',
    '\n            # A column definition has a name and a type, with some additional\n            # properties.\n            # Some examples:\n            #    Foo INT64 NOT NULL\n            #    Bar STRING(1024)\n            #    Baz ARRAY<FLOAT32>\n            (?P<name>\\w+)\\s+\n            (?P<type>[\\w<>]+)\n            # We don\'t care about "NOT NULL", and the length number after STRING\n            # or BYTES (e.g.STRING(MAX), BYTES(1024)).\n        ',
    '\n          # Every table starts with "CREATE TABLE" followed by name and column\n          # definitions, in a big set of parenthesis.\n          # For example:\n          #    CREATE TABLE Foos (\n          #        Bar INT64 NOT NULL,\n          #        Baz INT64 NOT NULL,\n          #        Qux STRING(MAX),\n          #    )\n          CREATE\\s+TABLE\\s+\n          (?P<name>\\w+)\\s+\\(\\s+\n          (?P<columns>.*)\\)\\s+\n          # Then, it has "PRIMARY KEY" and a list of primary keys, in parens:\n          # PRIMARY KEY ( Bar, Qux )\n          PRIMARY\\s+KEY\\s*\\(\n          (?P<primary_keys>.*)\\)\n          # It may have extra instructions on the end (e.g. INTERLEAVE) to\n          # tell Spanner how to store the data, but we don\'t really care.\n      ',
    '\n        (?P<operator>(==|!=|<=|>=|<|>))\n        \\s*\n        (?P<version>\n            [^,;\\s)]* # Since this is a "legacy" specifier, and the version\n                      # string can be just about anything, we match everything\n                      # except for whitespace, a semi-colon for marker support,\n                      # a closing paren since versions can be enclosed in\n                      # them, and a comma since it\'s a version separator.\n        )\n        ',
    '\n    (<\\?.*?\\?>)?\\s*\n    <!DOCTYPE\\s+(\n     [a-zA-Z_][a-zA-Z0-9]*\n     (?: \\s+      # optional in HTML5\n     [a-zA-Z_][a-zA-Z0-9]*\\s+\n     "[^"]*")?\n     )\n     [^>]*>\n',
    '\n    (?: vi | vim | ex ) (?: [<=>]? \\d* )? :\n    .* (?: ft | filetype | syn | syntax ) = ( [^:\\s]+ )\n',
    '\n    (?P<name>[^-]+) (\n        -(?P<ver>[^-]+) (\n            -py(?P<pyver>[^-]+) (\n                -(?P<plat>.+)\n            )?\n        )?\n    )?\n    ',
    '\n    \\s*                            # Optional whitespace at start of cookie\n    (?P<key>                       # Start of group \'key\'\n    [\\w\\d!#%&\'~_`><@,:/\\$\\*\\+\\-\\.\\^\\|\\)\\(\\?\\}\\{\\=]+?   # Any word of at least one letter\n    )                              # End of group \'key\'\n    (                              # Optional group: there may not be a value.\n    \\s*=\\s*                          # Equal Sign\n    (?P<val>                         # Start of group \'val\'\n    "(?:[^\\\\"]|\\\\.)*"                  # Any doublequoted string\n    |                                  # or\n    \\w{3},\\s[\\w\\d\\s-]{9,11}\\s[\\d:]{8}\\sGMT  # Special case for "expires" attr\n    |                                  # or\n    [\\w\\d!#%&\'~_`><@,:/\\$\\*\\+\\-\\.\\^\\|\\)\\(\\?\\}\\{\\=\\[\\]]*      # Any word or empty string\n    )                                # End of group \'val\'\n    )?                             # End of optional value group\n    \\s*                            # Any number of spaces.\n    (\\s+|;|$)                      # Ending either at space, semicolon, or EOS.\n    ',
    '\n  =\\?                   # literal =?\n  (?P<charset>[^?]*?)   # non-greedy up to the next ? is the charset\n  \\?                    # literal ?\n  (?P<encoding>[qb])    # either a "q" or a "b", case insensitive\n  \\?                    # literal ?\n  (?P<atom>.*?)         # non-greedy up to the next ?= is the atom\n  \\?=                   # literal ?=\n  ',
    '\n  =\\?                   # literal =?\n  (?P<charset>[^?]*?)   # non-greedy up to the next ? is the charset\n  \\?                    # literal ?\n  (?P<encoding>[qb])    # either a "q" or a "b", case insensitive\n  \\?                    # literal ?\n  (?P<encoded>.*?)      # non-greedy up to the next ?= is the encoded string\n  \\?=                   # literal ?=\n  (?=[ \\t]|$)           # whitespace or the end of the string\n  ',
    '\n# 0 or more . terminated hostname segments (may not start or end in -).\n^([a-z0-9]([a-z0-9\\-]*[a-z0-9])*\\.)*\n# followed by a host name segment.\n([a-z0-9]([a-z0-9\\-]*[a-z0-9])*)$',
    '\n#4 1-3 digit numbers separated by .\n^([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})$',
    '\n\n    (?P<protocol>[a-zA-Z0-9+.-]+) # The protocol group.\n\n    (:(?P<ports>\\d+(-\\d+)?))?     # The optional ports group.\n                                  # May specify a range.\n\n    $                             # End of input marker.\n    ',
    '\r\n|\r|\n',
    '^ *$',
    '^ | $|/ | /|\\r|\\n',
    '^$',
    '^(([a-zA-Z0-9_]+|[^a-zA-Z0-9_\\s]+)\\s*)',
    '^(.*)-windmill-(.*)-watermark',
    '^(.*/)?((#.*#)|(.*~)|(.*\\.py[co])|(.*/RCS/.*)|(\\..*)|)$',
    '^(?:((gs://[a-z0-9\\-\\._/]+)|([a-z][a-z0-9\\-\\.]{0,29})))$',
    '^(?:(1|2|standard|flex|flexible))$',
    '^(?:(?!\\^).*(?!\\$).{1,256})$',
    '^(?:(?!\\^)/.*|\\..*|(\\(.).*(?!\\$).)$',
    '^(?:(?:[a-z\\d\\-]{1,100}\\~)?(?:(?!\\-)[a-z\\d\\-\\.]{1,100}:)?(?!-)[a-z\\d\\-]{0,99}[a-z\\d])$',
    '^(?:([^:@]+)@)?([^.:/\\\\@][^:/\\\\@]*)$',
    '^(?:(default|over_quota|dos_api_denial|timeout))$',
    '^(?:(mail|mail_bounce|xmpp_message|xmpp_subscribe|xmpp_presence|xmpp_error|channel_presence|rest|warmup))$',
    '^(?:(pull)|(push))$',
    '^(?:.+)$',
    '^(?:Sun|Mon|Tue|Wed|Thu|Fri|Sat)[a-z]*,?\\s*',
    '^(?:[0-9]+(:[0-9]+)?(/(udp|tcp))?)$',
    '^(?:[\\w.\\-]{1,32})$',
    '^(?:[\\w.]{1,32})$',
    '^(?:[a-zA-Z_][a-zA-Z0-9_]*)$',
    '^(?:\\s*(([0-9]+)([DdHhMm]|[sS]?))(\\s+([0-9]+)([DdHhMm]|[sS]?))*\\s*)$',
    '^(?:^(0|[0-9]+(\\.[0-9]*)?/[smhd]))$',
    '^(?:^(?!-)[a-z\\d\\-]{0,62}[a-z\\d]$)$',
    '^(?:^(?:(?:((?!-)[a-z\\d\\-]{1,100})\\.)?)((?!-)[a-z\\d\\-]{1,63})$)$',
    '^(?:^(?:(?:((?!-)[a-z\\d\\-]{1,63}):)?)((?!-)[a-z\\d\\-]{1,100})$)$',
    '^(?:^(GAUGE|DELTA_PER_SECOND|DELTA_PER_MINUTE)$)$',
    '^(?:^(START|STOP|DISABLED)$)$',
    '^(?:^([0-9]+(\\.[0-9]*)?[BKMGT]?))$',
    '^(?:^([1-9]\\d*)$)$',
    '^(?:^([\\d]+|automatic)$)$',
    '^(?:^(\\d+((\\.\\d{1,3})?s|ms)|automatic)$)$',
    '^(?:^.{0,499}$)$',
    '^(?:^[1-9][\\d]*$)$',
    '^(?:^[\\d]+(s|m)$)$',
    '^(?:^[a-z, ]*$)$',
    '^(?:^[a-z\\d-]+(/[a-z\\d-]+)*$)$',
    '^(?:^[a-z]([a-z\\d-]{0,61}[a-z\\d])?$)$',
    '^(?:^[bB](1|2|4|8|4_1G)$)$',
    '^(?:^\\S+$)$',
    '^(?=.{1,128}$)[_a-zA-Z0-9]+(?:\\.[_a-zA-Z0-9]+)*$',
    '^(?P<name>\\w+)\\*((?P<num>[0-9]+)\\*?)?$',
    '^(?P<protocol>\\w+):',
    '^(From |[\\041-\\071\\073-\\176]{1,}:|[\\t ])',
    '^([-+])?(\\d\\d?):?(\\d\\d)?$',
    '^([0-9]+)((?:a|b|c|rc)[0-9]+)$',
    '^([^/]+)(/.*)$',
    '^([^\\s]+)',
    '^([^\\s]+\\s*)',
    '^([a-f0-9][a-f0-9]:){19}[a-f0-9][a-f0-9]$',
    '^([a-z0-9]\\.?-?)+[a-z0-9]$',
    '^([a-zA-Z0-9_]+|[^a-zA-Z0-9_\\s]+)',
    '^(\\d+) \\. (\\d+) (\\. (\\d+))? ([ab](\\d+))?$',
    '^(https?)://([^/]+)(/.*)$',
    '^-\\d+$|^-\\d*\\.\\d+$',
    '^.+@([^.@][^@]+)$',
    '^//[a-z#$@][a-z0-9#$@]{0,7}\\s+job(\\s+.*)?$',
    '^/?/(?:sdcard|data/local/tmp)(?:/[\\w\\-\\.\\+ /]+)*$',
    '^From ',
    '^[0-9A-Za-z._-]{0,100}$',
    '^[A-Za-z0-9][A-Za-z0-9\\.\\-]{1,255}$',
    '^[A-Za-z](?:[-_A-Za-z0-9]{0,61}[A-Za-z0-9])?$',
    '^[SMTWF][a-z][a-z], (\\d\\d) ([JFMASOND][a-z][a-z]) (\\d\\d\\d\\d) (\\d\\d):(\\d\\d):(\\d\\d) GMT$',
    '^[a-zA-Z0-9._\\-]+\\Z',
    '^[a-zA-Z][\\-\\._~%\\+a-zA-Z0-9]{2,254}$',
    '^[a-zA-Z]\\w+$',
    '^[a-z](?:[-0-9a-z]{0,62}[0-9a-z])?$',
    '^\\#LWP-Cookies-(\\d+\\.\\d+)',
    '^\\.+',
    '^\\\x1b\\[(<?[\\d;]*|M.{0,2})\\Z',
    '^\\\x1b\\[(<?[\\d;]+[mM]|M...)\\Z',
    '^\\\x1b\\[[\\d;]*\\Z',
    '^\\\x1b\\[\\d+;\\d+R\\Z',
    '^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$',
    '^\\s*([^=\\s;,]+)',
    '^\\s*([a-z_]\\w*)(\\s*)(:)(\\s*)(procedure)\\b',
    '^\\s*=\\s*([^\\s;,]*)',
    '^\\s*=\\s*\\"([^\\"\\\\]*(?:\\\\.[^\\"\\\\]*)*)\\"',
    '^\\s*MACRO',
    '^\\s*\\*',
    '^\\s*\n        (?P<operator>(==|!=|<=|>=|<|>))\n        \\s*\n        (?P<version>\n            [^,;\\s)]* # Since this is a "legacy" specifier, and the version\n                      # string can be just about anything, we match everything\n                      # except for whitespace, a semi-colon for marker support,\n                      # a closing paren since versions can be enclosed in\n                      # them, and a comma since it\'s a version separator.\n        )\n        \\s*$',
    '^\\s*\n    v?\n    (?:\n        (?:(?P<epoch>[0-9]+)!)?                           # epoch\n        (?P<release>[0-9]+(?:\\.[0-9]+)*)                  # release segment\n        (?P<pre>                                          # pre-release\n            [-_\\.]?\n            (?P<pre_l>(a|b|c|rc|alpha|beta|pre|preview))\n            [-_\\.]?\n            (?P<pre_n>[0-9]+)?\n        )?\n        (?P<post>                                         # post release\n            (?:-(?P<post_n1>[0-9]+))\n            |\n            (?:\n                [-_\\.]?\n                (?P<post_l>post|rev|r)\n                [-_\\.]?\n                (?P<post_n2>[0-9]+)?\n            )\n        )?\n        (?P<dev>                                          # dev release\n            [-_\\.]?\n            (?P<dev_l>dev)\n            [-_\\.]?\n            (?P<dev_n>[0-9]+)?\n        )?\n    )\n    (?:\\+(?P<local>[a-z0-9]+(?:[-_\\.][a-z0-9]+)*))?       # local version\n\\s*$',
    '^\\s*address\\s+',
    '^\\s*address\\s+command\\b',
    '^\\s*do\\s+while\\b',
    '^\\s*if\\b.+\\bthen\\s+do\\s*$',
    '^\\s*parse\\s+(upper\\s+)?(arg|value)\\b',
    '^\\w+://([^/]*[.@])?(?P<domain>\\w+\\.\\w+)[/:]',
    '^\n    (\\d\\d?)            # day\n       (?:\\s+|[-\\/])\n    (\\w+)              # month\n        (?:\\s+|[-\\/])\n    (\\d+)              # year\n    (?:\n          (?:\\s+|:)    # separator before clock\n       (\\d\\d?):(\\d\\d)  # hour:min\n       (?::(\\d\\d))?    # optional seconds\n    )?                 # optional clock\n       \\s*\n    ([-+]?\\d{2,4}|(?![APap][Mm]\\b)[A-Za-z]+)? # timezone\n       \\s*\n    (?:\\(\\w+\\))?       # ASCII representation of timezone in parens.\n       \\s*$',
    '^\n    (\\d{4})              # year\n       [-\\/]?\n    (\\d\\d?)              # numerical month\n       [-\\/]?\n    (\\d\\d?)              # day\n   (?:\n         (?:\\s+|[-:Tt])  # separator before clock\n      (\\d\\d?):?(\\d\\d)    # hour:min\n      (?::?(\\d\\d(?:\\.\\d*)?))?  # optional seconds (and fractional)\n   )?                    # optional clock\n      \\s*\n   ([-+]?\\d\\d?:?(:?\\d\\d)?\n    |Z|z)?               # timezone  (Z is "zero meridian", i.e. GMT)\n      \\s*$',
    '^_[A-Z0-9_]+$',
    '^__.*__$',
    '^dataflow',
    '^for\\s+((?:\\(?)\\s*[A-Za-z_][A-Za-z_0-9]*(?:\\s*,\\s*(?:[A-Za-z_][A-Za-z0-9_]*),??)*\\s*(?:\\)?))\\s+in\\s+(.*):',
    '^gs://',
    '^projects/_/buckets/.{3,222}$',
    'darwin-(\\d+)\\.(\\d+)\\.(\\d+)-(.*)',
    'go1\\d\\d(\\w+\\d)?',
    'irb\\([a-zA-Z_]\\w*\\):\\d{3}:\\d+[>*"\'] |>> |\\?> ',
    'macosx-(\\d+)\\.(\\d+)-(.*)',
    'nodejs\\d*',
    'php[789]\\d*',
    'python3\\d*',
    'remote origin\\n.*Fetch URL: (?P<url>.+)\\n',
    'test-re-[ab]',
))
