# Upgrading from the OpenID 1.x series library

## Consumer Upgrade

The flow is largely the same, however there are a number of significant
changes. The consumer example is helpful to look at:
`examples/rails_openid/app/controllers/consumer_controller.rb`

### Stores

You will need to require the file for the store that you are using.
For the filesystem store, this is 'openid/stores/filesystem'
They are also now in modules. The filesystem store is
  `OpenID::Store::Filesystem`
The format has changed, and you should remove your old store directory.

The ActiveRecord store (`examples/active_record_openid_store`) still needs
to be put in a plugin directory for your rails app.  There's a migration
that needs to be run; examine the `README` in that directory.

Also, note that the stores now can be garbage collected with the method
  `store.cleanup`

### Starting the OpenID transaction

The OpenIDRequest object no longer has status codes.  Instead,
consumer.begin raises an OpenID::OpenIDError if there is a problem
initiating the transaction, so you'll want something along the lines of:

    begin
      openid_request = consumer.begin(params[:openid_identifier])
    rescue OpenID::OpenIDError => e
      # display error e
      return
    end
    #success case

Data regarding the OpenID server once lived in
  `openid_request.service`

The corresponding object in the 2.0 lib can be retrieved with
  `openid_request.endpoint`

Getting the unverified identifier: Where you once had
  `openid_request.identity_url`
you will now want
  `openid_request.endpoint.claimed_id`
which might be different from what you get at the end of the transaction,
since it is now possible for users to enter their server's url directly.

Arguments on the return_to URL are now verified, so if you want to add
additional arguments to the return_to url, use
  `openid_request.return_to_args['param'] = value`

Generating the redirect is the same as before, but add any extensions
first.

If you need to set up an SSL certificate authority list for the fetcher,
use the 'ca_file' attr_accessor on the `OpenID::StandardFetcher`. This has
changed from 'ca_path' in the 1.x.x series library. That is, set
`OpenID.fetcher.ca_file = '/path/to/ca.list'`
before calling consumer.begin.

### Requesting Simple Registration Data

You'll need to require the code for the extension

    require 'openid/extensions/sreg'

The new code for adding an SReg request now looks like:

    sreg_request = OpenID::SReg::Request.new
    sreg_request.request_fields(['email', 'dob'], true) # required
    sreg_request.request_fields(['nickname', 'fullname'], false) # optional
    sreg_request.policy_url = policy_url
    openid_request.add_extension(sreg_request)

The code for adding other extensions is similar.  Code for the Attribute
Exchange (AX) and Provider Authentication Policy Extension (PAPE) are
included with the library, and additional extensions can be implemented
subclassing `OpenID::Extension`.

### Completing the transaction

The return_to and its arguments are verified, so you need to pass in
the base URL and the arguments.  With Rails, the params method mashes
together parameters from GET, POST, and the path, so you'll need to pull
off the path "parameters" with something like

    return_to = url_for(:only_path => false,
                        :controller => 'openid',
                        :action => 'complete')
    parameters = params.reject{|k,v| request.path_parameters[k] }
    openid_response = consumer.complete(parameters, return_to)

The response still uses the status codes, but they are now namespaced
slightly differently, for example `OpenID::Consumer::SUCCESS`

In the case of failure, the error message is now found in
  `openid_response.message`

The identifier to display to the user can be found in
  `openid_response.endpoint.display_identifier`

The Simple Registration response can be read from the OpenID response
with

    sreg_response = OpenID::SReg::Response.from_success_response(openid_response)
    nickname = sreg_response['nickname']
    # etc.

## Server Upgrade

The server code is mostly the same as before, with the exception of
extensions. Also, you must pass in the endpoint URL to the server
constructor:

    @server = OpenID::Server.new(store, server_url)

I recommend looking at
`examples/rails_openid/app/controllers/server_controller.rb`
for an example of the new way of doing extensions.

--
Dag Arneson, JanRain Inc.
Please direct questions to openid@janrain.com
