# coding: utf-8
# Part of Odoo. See LICENSE file for full copyright and licensing details.

from odoo import api, fields, models


class ResPartner(models.Model):
    _inherit = 'res.partner'

    l10n_co_document_type = fields.Selection([('rut', 'NIT'),
                                              ('national_citizen_id', 'Cédula de ciudadanía'),
                                              ('civil_registration', 'Registro Civil'),
                                              ('id_card', 'Tarjeta de Identidad'),
                                              ('foreign_colombian_card', 'Tarjeta de extranjería'),
                                              ('foreign_resident_card', 'Cédula de extranjería'),
                                              ('passport', 'Pasaporte'),
                                              ('residence_document', 'Salvoconducto de Permanencia'),
                                              ('foreign_id_card', 'Cédula Extranjera'),
                                              ('external_id', 'Nit de otro país'),
                                              ('niup_id', 'NIUP'),
                                              ('diplomatic_card', 'INACTIVO - Carné Diplomatico'),
                                              ('id_document', 'INACTIVO - Cédula')], string='Document Type',
                                             help='Indicates to what document the information in here belongs to.')
    l10n_co_verification_code = fields.Char(compute='_compute_verification_code', string='VC',  # todo remove this field in master
                                            help='Redundancy check to verify the vat number has been typed in correctly.')

    @api.depends('vat')
    def _compute_verification_code(self):
        multiplication_factors = [71, 67, 59, 53, 47, 43, 41, 37, 29, 23, 19, 17, 13, 7, 3]

        for partner in self:
            if partner.vat and partner.country_id == self.env.ref('base.co') and len(partner.vat) <= len(multiplication_factors):
                number = 0
                padded_vat = partner.vat

                while len(padded_vat) < len(multiplication_factors):
                    padded_vat = '0' + padded_vat

                # if there is a single non-integer in vat the verification code should be False
                try:
                    for index, vat_number in enumerate(padded_vat):
                        number += int(vat_number) * multiplication_factors[index]

                    number %= 11

                    if number < 2:
                        partner.l10n_co_verification_code = number
                    else:
                        partner.l10n_co_verification_code = 11 - number
                except ValueError:
                    partner.l10n_co_verification_code = False
            else:
                partner.l10n_co_verification_code = False

    @api.constrains('vat', 'country_id', 'l10n_co_document_type')
    def check_vat(self):
        # check_vat is implemented by base_vat which this localization
        # doesn't directly depend on. It is however automatically
        # installed for Colombia.
        if self.sudo().env.ref('base.module_base_vat').state == 'installed':
            # don't check Colombian partners unless they have RUT (= Colombian VAT) set as document type
            self = self.filtered(lambda partner: partner.country_id != self.env.ref('base.co') or\
                                                 partner.l10n_co_document_type == 'rut')
            return super(ResPartner, self).check_vat()
        else:
            return True
