/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.tracing;

import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Supplier;

public class Tracer {
    private static long tracingStartNs;
    private static long tracingStartMs;
    private static final AtomicLong eventId;
    private static final ConcurrentLinkedDeque<Span> spans;
    private static volatile int pid;
    private static volatile long durationThreshold;
    private static volatile FileState fileState;
    private static volatile boolean running;
    private static ScheduledExecutorService executor;
    private static Thread shutdownHook;

    private Tracer() {
    }

    public static DelayedSpan start(Supplier<String> nameSupplier) {
        long eventId = Tracer.eventId.getAndIncrement();
        long threadId = Thread.currentThread().getId();
        long startNs = System.nanoTime();
        return new DelayedSpan(eventId, threadId, nameSupplier, startNs);
    }

    public static Span start(String name) {
        long eventId = Tracer.eventId.getAndIncrement();
        long threadId = Thread.currentThread().getId();
        long startNs = System.nanoTime();
        return new Span(eventId, threadId, name, startNs);
    }

    public static void runTracer(int pid, Path filePath, long threshold, Consumer<Exception> exceptionHandler) throws IOException {
        if (running) {
            throw new IllegalStateException("Tracer already started");
        }
        tracingStartMs = System.currentTimeMillis();
        tracingStartNs = System.nanoTime();
        Files.createDirectories(filePath.getParent(), new FileAttribute[0]);
        FileOutputStream fileOutputStream = new FileOutputStream(filePath.toFile());
        OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new BufferedOutputStream(fileOutputStream), StandardCharsets.UTF_8);
        fileState = new FileState(writer);
        durationThreshold = threshold;
        Tracer.pid = pid;
        executor = Tracer.createExecutor();
        FlushingTask flushingTask = new FlushingTask(fileState, false, exceptionHandler);
        executor.scheduleAtFixedRate(flushingTask, 5L, 5L, TimeUnit.SECONDS);
        shutdownHook = new Thread((Runnable)new FlushingTask(fileState, true, exceptionHandler), "Shutdown hook trace flusher");
        Runtime.getRuntime().addShutdownHook(shutdownHook);
        running = true;
    }

    public static void finishTracer(Consumer<Exception> exceptionHandler) {
        if (fileState == null) {
            return;
        }
        new FlushingTask(fileState, true, exceptionHandler).run();
        fileState = null;
        executor.shutdown();
        executor = null;
        Runtime.getRuntime().removeShutdownHook(shutdownHook);
        running = false;
    }

    public static boolean isRunning() {
        return running;
    }

    private static ScheduledExecutorService createExecutor() {
        return Executors.newScheduledThreadPool(1, r -> {
            Thread thread = new Thread(r, "Trace flusher");
            thread.setDaemon(true);
            return thread;
        });
    }

    static long getTimeUs(long timeNs) {
        return (tracingStartMs * 1000000L - tracingStartNs + timeNs) / 1000L;
    }

    static {
        eventId = new AtomicLong();
        spans = new ConcurrentLinkedDeque();
        fileState = null;
        running = false;
        executor = null;
    }

    private static class FlushingTask
    implements Runnable {
        private final FileState fileState;
        private final boolean shouldFinish;
        private final Consumer<Exception> myExceptionHandler;

        private FlushingTask(FileState fileState, boolean shouldFinish, Consumer<Exception> exceptionHandler) {
            this.fileState = fileState;
            this.shouldFinish = shouldFinish;
            this.myExceptionHandler = exceptionHandler;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            if (this.fileState == null) {
                return;
            }
            FileState fileState = this.fileState;
            synchronized (fileState) {
                if (this.fileState.finished) {
                    return;
                }
                try {
                    Span span;
                    if (!this.fileState.openBracketWritten) {
                        this.fileState.writer.write("[\n");
                        this.fileState.openBracketWritten = true;
                    }
                    while ((span = (Span)spans.pollLast()) != null) {
                        this.fileState.writer.write(FlushingTask.serialize(span, true));
                        this.fileState.writer.write(FlushingTask.serialize(span, false));
                    }
                    if (this.shouldFinish) {
                        this.fileState.writer.write("]");
                    }
                    this.fileState.writer.flush();
                }
                catch (IOException e) {
                    this.myExceptionHandler.accept(e);
                }
            }
        }

        private static String serialize(Span span, boolean isStart) {
            StringBuilder sb = new StringBuilder();
            sb.append("{\"name\": \"").append(span.name).append("\", \"cat\": \"PERF\", \"ph\": ");
            if (isStart) {
                sb.append("\"B\"");
            } else {
                sb.append("\"E\"");
            }
            sb.append(", \"pid\": ").append(pid).append(", \"tid\": ").append(span.threadId).append(", \"ts\": ");
            if (isStart) {
                sb.append(Tracer.getTimeUs(span.startTimeNs));
            } else {
                sb.append(Tracer.getTimeUs(span.finishTimeNs));
            }
            return sb.append("},\n").toString();
        }
    }

    private static class FileState {
        final Writer writer;
        boolean openBracketWritten = false;
        boolean finished = false;

        private FileState(Writer writer) {
            this.writer = writer;
        }
    }

    public static class Span {
        final long eventId;
        final long threadId;
        final String name;
        final long startTimeNs;
        long finishTimeNs;

        public Span(long eventId, long threadId, String name, long startTimeNs) {
            this.eventId = eventId;
            this.threadId = threadId;
            this.name = name;
            this.startTimeNs = startTimeNs;
        }

        public void complete() {
            if (running) {
                this.finishTimeNs = System.nanoTime();
                if (this.getDuration() > durationThreshold) {
                    spans.offerLast(this);
                }
            }
        }

        long getDuration() {
            return this.finishTimeNs - this.startTimeNs;
        }
    }

    public static class DelayedSpan {
        final long eventId;
        final long threadId;
        final Supplier<String> nameSupplier;
        final long startTimeNs;

        public DelayedSpan(long eventId, long threadId, Supplier<String> nameSupplier, long startTimeNs) {
            this.eventId = eventId;
            this.threadId = threadId;
            this.nameSupplier = nameSupplier;
            this.startTimeNs = startTimeNs;
        }

        public void complete() {
            if (running) {
                Span span = new Span(this.eventId, this.threadId, this.nameSupplier.get(), this.startTimeNs);
                span.complete();
            }
        }
    }
}

