/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.handler.admin;

import java.lang.invoke.MethodHandles;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.cloud.LeaderElector;
import org.apache.solr.cloud.OverseerTaskProcessor;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.ClusterState;
import org.apache.solr.common.cloud.DocCollection;
import org.apache.solr.common.cloud.Replica;
import org.apache.solr.common.cloud.Slice;
import org.apache.solr.common.cloud.ZkNodeProps;
import org.apache.solr.common.cloud.ZkStateReader;
import org.apache.solr.common.params.CollectionParams;
import org.apache.solr.common.util.SimpleOrderedMap;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.handler.admin.CollectionsHandler;
import org.apache.solr.request.SolrQueryRequest;
import org.apache.solr.response.SolrQueryResponse;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RebalanceLeaders {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    final SolrQueryRequest req;
    final SolrQueryResponse rsp;
    final CollectionsHandler collectionsHandler;
    final CoreContainer coreContainer;
    private final Set<String> asyncRequests = new HashSet<String>();
    static final String INACTIVE_PREFERREDS = "inactivePreferreds";
    static final String ALREADY_LEADERS = "alreadyLeaders";
    static final String SUMMARY = "Summary";
    final SimpleOrderedMap<SimpleOrderedMap<SimpleOrderedMap<String>>> results = new SimpleOrderedMap();
    final Map<String, String> pendingOps = new HashMap<String, String>();
    private String collectionName;

    RebalanceLeaders(SolrQueryRequest req, SolrQueryResponse rsp, CollectionsHandler collectionsHandler) {
        this.req = req;
        this.rsp = rsp;
        this.collectionsHandler = collectionsHandler;
        this.coreContainer = collectionsHandler.getCoreContainer();
    }

    void execute() throws KeeperException, InterruptedException {
        DocCollection dc = this.checkParams();
        int max = this.req.getParams().getInt("maxAtOnce", Integer.MAX_VALUE);
        if (max <= 0) {
            max = Integer.MAX_VALUE;
        }
        int maxWaitSecs = this.req.getParams().getInt("maxWaitSeconds", 60);
        boolean keepGoing = true;
        for (Slice slice : dc.getSlices()) {
            this.ensurePreferredIsLeader(slice);
            if (this.asyncRequests.size() != max) continue;
            log.info("Queued {} leader reassignments, waiting for some to complete.", (Object)max);
            keepGoing = this.waitAsyncRequests(maxWaitSecs, false);
            if (keepGoing) continue;
            break;
        }
        if (keepGoing) {
            keepGoing = this.waitAsyncRequests(maxWaitSecs, true);
        }
        if (keepGoing) {
            log.info("All leader reassignments completed.");
        } else {
            log.warn("Exceeded specified timeout of '{}' all leaders may not have been reassigned'", (Object)maxWaitSecs);
        }
        this.checkLeaderStatus();
        SimpleOrderedMap summary = new SimpleOrderedMap();
        if (this.pendingOps.size() == 0) {
            summary.add("Success", (Object)"All active replicas with the preferredLeader property set are leaders");
        } else {
            summary.add("Failure", (Object)"Not all active replicas with preferredLeader property are leaders");
        }
        this.rsp.getValues().add(SUMMARY, (Object)summary);
        this.rsp.getValues().addAll(this.results);
    }

    private DocCollection checkParams() throws KeeperException, InterruptedException {
        this.req.getParams().required().check(new String[]{"collection"});
        this.collectionName = this.req.getParams().get("collection");
        if (StringUtils.isBlank((CharSequence)this.collectionName)) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, String.format(Locale.ROOT, "The collection is required for the Rebalance Leaders command.", new Object[0]));
        }
        this.coreContainer.getZkController().getZkStateReader().forceUpdateCollection(this.collectionName);
        ClusterState clusterState = this.coreContainer.getZkController().getClusterState();
        DocCollection dc = clusterState.getCollection(this.collectionName);
        if (dc == null) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Collection '" + this.collectionName + "' does not exist, no action taken.");
        }
        return dc;
    }

    private void checkLeaderStatus() throws InterruptedException, KeeperException {
        for (int idx = 0; this.pendingOps.size() > 0 && idx < 600; ++idx) {
            ClusterState clusterState = this.coreContainer.getZkController().getClusterState();
            Set liveNodes = clusterState.getLiveNodes();
            DocCollection dc = clusterState.getCollection(this.collectionName);
            block1: for (Slice slice : dc.getSlices()) {
                for (Replica replica : slice.getReplicas()) {
                    if (!replica.isActive(liveNodes) || !replica.getBool("property.preferredleader", false) || !replica.getBool("leader", false) || !this.pendingOps.containsKey(slice.getName())) continue;
                    this.pendingOps.remove(slice.getName());
                    this.addToSuccesses(slice, replica);
                    continue block1;
                }
            }
            TimeUnit.MILLISECONDS.sleep(100L);
            this.coreContainer.getZkController().getZkStateReader().forciblyRefreshAllClusterStateSlow();
        }
        this.addAnyFailures();
    }

    private void ensurePreferredIsLeader(Slice slice) throws KeeperException, InterruptedException {
        for (Replica replica : slice.getReplicas()) {
            if (!replica.getBool("property.preferredleader", false)) continue;
            if (replica.getBool("leader", false)) {
                this.addAlreadyLeaderToResults(slice, replica);
                return;
            }
            ZkStateReader zkStateReader = this.coreContainer.getZkController().getZkStateReader();
            if (!replica.isActive(zkStateReader.getClusterState().getLiveNodes())) {
                this.addInactiveToResults(slice, replica);
                return;
            }
            List<String> electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath((String)this.collectionName, (String)slice.getName()));
            if (this.electionQueueInBadState(electionNodes, slice, replica)) {
                return;
            }
            String firstWatcher = electionNodes.get(1);
            if (!LeaderElector.getNodeName(firstWatcher).equals(replica.getName())) {
                this.makeReplicaFirstWatcher(slice, replica);
            }
            this.pendingOps.put(slice.getName(), replica.getName());
            String leaderElectionNode = electionNodes.get(0);
            String coreName = slice.getReplica(LeaderElector.getNodeName(leaderElectionNode)).getStr("core");
            this.rejoinElectionQueue(slice, leaderElectionNode, coreName, false);
            this.waitForNodeChange(slice, leaderElectionNode);
            return;
        }
    }

    private boolean electionQueueInBadState(List<String> electionNodes, Slice slice, Replica replica) {
        if (electionNodes.size() < 2) {
            log.warn("Rebalancing leaders and slice {} has less than two elements in the leader election queue, but replica {} doesn't think it's the leader.", (Object)slice.getName(), (Object)replica.getName());
            return true;
        }
        return false;
    }

    private void addInactiveToResults(Slice slice, Replica replica) {
        SimpleOrderedMap inactives = (SimpleOrderedMap)this.results.get(INACTIVE_PREFERREDS);
        if (inactives == null) {
            inactives = new SimpleOrderedMap();
            this.results.add(INACTIVE_PREFERREDS, (Object)inactives);
        }
        SimpleOrderedMap res = new SimpleOrderedMap();
        res.add("status", (Object)"skipped");
        res.add("msg", (Object)("Replica " + replica.getName() + " is a referredLeader for shard " + slice.getName() + ", but is inactive. No change necessary"));
        inactives.add(replica.getName(), (Object)res);
    }

    private void addAlreadyLeaderToResults(Slice slice, Replica replica) {
        SimpleOrderedMap alreadyLeaders = (SimpleOrderedMap)this.results.get(ALREADY_LEADERS);
        if (alreadyLeaders == null) {
            alreadyLeaders = new SimpleOrderedMap();
            this.results.add(ALREADY_LEADERS, (Object)alreadyLeaders);
        }
        SimpleOrderedMap res = new SimpleOrderedMap();
        res.add("status", (Object)"skipped");
        res.add("msg", (Object)("Replica " + replica.getName() + " is already the leader for shard " + slice.getName() + ". No change necessary"));
        alreadyLeaders.add(replica.getName(), (Object)res);
    }

    void makeReplicaFirstWatcher(Slice slice, Replica replica) throws KeeperException, InterruptedException {
        String coreName;
        ZkStateReader zkStateReader = this.coreContainer.getZkController().getZkStateReader();
        List<String> electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath((String)this.collectionName, (String)slice.getName()));
        int secondSeq = Integer.MAX_VALUE;
        int candidateSeq = -1;
        for (int idx = 1; idx < electionNodes.size(); ++idx) {
            String candidate = electionNodes.get(idx);
            secondSeq = Math.min(secondSeq, LeaderElector.getSeq(candidate));
            if (!LeaderElector.getNodeName(candidate).equals(replica.getName())) continue;
            candidateSeq = LeaderElector.getSeq(candidate);
        }
        int newSeq = -1;
        if (candidateSeq == secondSeq) {
            newSeq = secondSeq;
        } else {
            for (String electionNode : electionNodes) {
                if (!LeaderElector.getNodeName(electionNode).equals(replica.getName())) continue;
                coreName = slice.getReplica(LeaderElector.getNodeName(electionNode)).getStr("core");
                this.rejoinElectionQueue(slice, electionNode, coreName, true);
                newSeq = this.waitForNodeChange(slice, electionNode);
                break;
            }
        }
        if (newSeq == -1) {
            return;
        }
        electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath((String)this.collectionName, (String)slice.getName()));
        for (String thisNode : electionNodes) {
            if (LeaderElector.getSeq(thisNode) > newSeq) break;
            if (LeaderElector.getNodeName(thisNode).equals(replica.getName()) || LeaderElector.getSeq(thisNode) != newSeq) continue;
            coreName = slice.getReplica(LeaderElector.getNodeName(thisNode)).getStr("core");
            this.rejoinElectionQueue(slice, thisNode, coreName, false);
            this.waitForNodeChange(slice, thisNode);
        }
    }

    int waitForNodeChange(Slice slice, String electionNode) throws InterruptedException, KeeperException {
        String nodeName = LeaderElector.getNodeName(electionNode);
        int oldSeq = LeaderElector.getSeq(electionNode);
        for (int idx = 0; idx < 600; ++idx) {
            ZkStateReader zkStateReader = this.coreContainer.getZkController().getZkStateReader();
            List<String> electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath((String)this.collectionName, (String)slice.getName()));
            for (String testNode : electionNodes) {
                if (!LeaderElector.getNodeName(testNode).equals(nodeName) || oldSeq == LeaderElector.getSeq(testNode)) continue;
                return LeaderElector.getSeq(testNode);
            }
            TimeUnit.MILLISECONDS.sleep(100L);
            zkStateReader.forciblyRefreshAllClusterStateSlow();
        }
        return -1;
    }

    private void rejoinElectionQueue(Slice slice, String electionNode, String core, boolean rejoinAtHead) throws KeeperException, InterruptedException {
        Replica replica = slice.getReplica(LeaderElector.getNodeName(electionNode));
        CollectionParams.CollectionAction rebalanceleaders = CollectionParams.CollectionAction.REBALANCELEADERS;
        HashMap<String, Object> propMap = new HashMap<String, Object>();
        propMap.put("collection", this.collectionName);
        propMap.put("shard", slice.getName());
        propMap.put("operation", rebalanceleaders.toLower());
        propMap.put("core", core);
        propMap.put("core_node_name", replica.getName());
        propMap.put("node_name", replica.getNodeName());
        propMap.put("base_url", this.coreContainer.getZkController().getZkStateReader().getBaseUrlForNodeName(replica.getNodeName()));
        propMap.put("rejoinAtHead", Boolean.toString(rejoinAtHead));
        propMap.put("election_node", electionNode);
        String asyncId = rebalanceleaders.toLower() + "_" + core + "_" + Math.abs(System.nanoTime());
        propMap.put("async", asyncId);
        this.asyncRequests.add(asyncId);
        this.collectionsHandler.submitCollectionApiCommand(new ZkNodeProps(propMap), rebalanceleaders);
    }

    private boolean waitAsyncRequests(int maxWaitSecs, Boolean waitForAll) throws KeeperException, InterruptedException {
        if (this.asyncRequests.size() == 0) {
            return true;
        }
        for (int idx = 0; idx < maxWaitSecs * 10; ++idx) {
            Iterator<String> iter = this.asyncRequests.iterator();
            boolean foundChange = false;
            while (iter.hasNext()) {
                String asyncId = iter.next();
                if (this.coreContainer.getZkController().getOverseerFailureMap().contains(asyncId)) {
                    this.coreContainer.getZkController().getOverseerFailureMap().remove(asyncId);
                    this.coreContainer.getZkController().clearAsyncId(asyncId);
                    iter.remove();
                    foundChange = true;
                    continue;
                }
                if (!this.coreContainer.getZkController().getOverseerCompletedMap().contains(asyncId)) continue;
                this.coreContainer.getZkController().getOverseerCompletedMap().remove(asyncId);
                this.coreContainer.getZkController().clearAsyncId(asyncId);
                iter.remove();
                foundChange = true;
            }
            if (foundChange && !waitForAll.booleanValue() || this.asyncRequests.size() == 0) {
                return true;
            }
            TimeUnit.MILLISECONDS.sleep(100L);
        }
        return false;
    }

    private void addToSuccesses(Slice slice, Replica replica) {
        SimpleOrderedMap successes = (SimpleOrderedMap)this.results.get("successes");
        if (successes == null) {
            successes = new SimpleOrderedMap();
            this.results.add("successes", (Object)successes);
        }
        if (log.isInfoEnabled()) {
            log.info("Successfully changed leader of shard {} to replica {}", (Object)slice.getName(), (Object)replica.getName());
        }
        SimpleOrderedMap res = new SimpleOrderedMap();
        res.add("status", (Object)"success");
        res.add("msg", (Object)("Successfully changed leader of slice " + slice.getName() + " to " + replica.getName()));
        successes.add(slice.getName(), (Object)res);
    }

    private void addAnyFailures() {
        if (this.pendingOps.size() == 0) {
            return;
        }
        SimpleOrderedMap fails = new SimpleOrderedMap();
        this.results.add("failures", (Object)fails);
        for (Map.Entry<String, String> ent : this.pendingOps.entrySet()) {
            if (log.isInfoEnabled()) {
                log.info("Failed to change leader of shard {} to replica {}", (Object)ent.getKey(), (Object)ent.getValue());
            }
            SimpleOrderedMap res = new SimpleOrderedMap();
            res.add("status", (Object)"failed");
            res.add("msg", (Object)String.format(Locale.ROOT, "Could not change leder for slice %s to %s", ent.getKey(), ent.getValue()));
            fails.add(ent.getKey(), (Object)res);
        }
    }
}

