// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
// @ts-ignore
import { Transform as PortableTransform } from 'readable-stream';
import { needs, } from '@aws-crypto/material-management-node';
import { deserializeSignature, decodeBodyHeader, serializeMessageHeaderAuth, } from '@aws-crypto/serialize';
import { ParseHeaderStream } from './parse_header_stream';
const PortableTransformWithType = PortableTransform;
export class VerifyStream extends PortableTransformWithType {
    _headerInfo;
    _verifyState = {
        buffer: Buffer.alloc(0),
        authTagBuffer: Buffer.alloc(0),
        signatureInfo: Buffer.alloc(0),
        sequenceNumber: 0,
        finalAuthTagReceived: false,
    };
    _verify;
    _maxBodySize;
    constructor({ maxBodySize }) {
        super();
        /* Precondition: VerifyStream requires maxBodySize must be falsey or a number. */
        needs(!maxBodySize || typeof maxBodySize === 'number', 'Unsupported MaxBodySize.');
        Object.defineProperty(this, '_maxBodySize', {
            value: maxBodySize,
            enumerable: true,
        });
        this.on('pipe', (source) => {
            /* Precondition: The source must a ParseHeaderStream emit the required events. */
            needs(source instanceof ParseHeaderStream, 'Unsupported source');
            source.once('VerifyInfo', (verifyInfo) => {
                const { getDecipher, verify, headerInfo, dispose } = verifyInfo;
                const { messageId, contentType } = headerInfo.messageHeader;
                /* If I have a verify, the header needs to be flushed through.
                 * I do it here for initialize the verifier before I even
                 * add the element to the object.
                 */
                if (verify) {
                    const { rawHeader, headerAuth, messageHeader } = headerInfo;
                    const { headerIv, headerAuthTag } = headerAuth;
                    verify.update(rawHeader);
                    verify.update(serializeMessageHeaderAuth({
                        headerIv,
                        headerAuthTag,
                        messageHeader,
                    }));
                }
                Object.defineProperty(this, '_headerInfo', {
                    value: headerInfo,
                    enumerable: true,
                });
                Object.defineProperty(this, '_verify', {
                    value: verify,
                    enumerable: true,
                });
                const decipherInfo = {
                    messageId: Buffer.from(messageId.buffer || messageId, messageId.byteOffset || 0, messageId.byteLength),
                    contentType,
                    getDecipher,
                    dispose,
                };
                this.emit('DecipherInfo', decipherInfo);
            });
        });
    }
    _transform(chunk, enc, callback) {
        /* Precondition: VerifyInfo must have initialized the stream. */
        needs(this._headerInfo, 'VerifyStream not configured, VerifyInfo event not yet received.');
        // BodyHeader
        const state = this._verifyState;
        const { currentFrame } = state;
        if (!currentFrame) {
            const { buffer } = state;
            const frameBuffer = Buffer.concat([buffer, chunk]);
            const frameHeader = decodeBodyHeader(frameBuffer, this._headerInfo, 0);
            if (!frameHeader) {
                // Need more data
                state.buffer = frameBuffer;
                return callback();
            }
            /* Precondition: If maxBodySize was set I can not buffer more data than maxBodySize.
             * Before returning *any* cleartext, the stream **MUST** verify the decryption.
             * This means that I must buffer the message until the AuthTag is reached.
             */
            needs(!this._maxBodySize || this._maxBodySize >= frameHeader.contentLength, 'maxBodySize exceeded.');
            /* Keeping track of the sequence number myself. */
            state.sequenceNumber += 1;
            /* Precondition: The sequence number is required to monotonically increase, starting from 1.
             * This is to avoid a bad actor from abusing the sequence number on un-signed algorithm suites.
             * If the frame size matched the data format (say NDJSON),
             * then the data could be significantly altered just by rearranging the frames.
             * Non-framed data returns a sequenceNumber of 1.
             */
            needs(frameHeader.sequenceNumber === state.sequenceNumber, 'Encrypted body sequence out of order.');
            if (this._verify) {
                this._verify.update(frameBuffer.slice(0, frameHeader.readPos));
            }
            const tail = frameBuffer.slice(frameHeader.readPos);
            this.emit('BodyInfo', frameHeader);
            state.currentFrame = frameHeader;
            return setImmediate(() => this._transform(tail, enc, callback));
        }
        // Content
        const { contentLength } = currentFrame;
        if (chunk.length && contentLength > 0) {
            if (contentLength > chunk.length) {
                currentFrame.contentLength -= chunk.length;
                this.push(chunk);
                return callback();
            }
            else {
                const content = chunk.slice(0, contentLength);
                const tail = chunk.slice(content.length);
                this.push(content);
                currentFrame.contentLength = 0;
                return setImmediate(() => this._transform(tail, enc, callback));
            }
        }
        // AuthTag
        const { tagLength } = currentFrame;
        const tagLengthBytes = tagLength / 8;
        const { authTagBuffer } = state;
        if (chunk.length && tagLengthBytes > authTagBuffer.length) {
            const left = tagLengthBytes - authTagBuffer.length;
            if (left > chunk.length) {
                state.authTagBuffer = Buffer.concat([authTagBuffer, chunk]);
                return callback();
            }
            else {
                const finalAuthTagBuffer = Buffer.concat([authTagBuffer, chunk], tagLengthBytes);
                if (this._verify) {
                    this._verify.update(finalAuthTagBuffer);
                }
                /* Reset state.
                 * Ciphertext buffers and authTag buffers need to be cleared.
                 */
                state.buffer = Buffer.alloc(0);
                state.currentFrame = undefined;
                state.authTagBuffer = Buffer.alloc(0);
                /* After the final frame the file format is _much_ simpler.
                 * Making sure the cascading if blocks fall to the signature can be tricky and brittle.
                 * After the final frame, just moving on to concatenate the signature is much simpler.
                 */
                if (currentFrame.isFinalFrame) {
                    /* Signal that the we are at the end of the ciphertext.
                     * See decodeBodyHeader, non-framed will set isFinalFrame
                     * for the single frame.
                     */
                    this._verifyState.finalAuthTagReceived = true;
                    /* Overwriting the _transform function.
                     * Data flow control is now handled here.
                     */
                    this._transform = (chunk, _enc, callback) => {
                        if (chunk.length) {
                            state.signatureInfo = Buffer.concat([state.signatureInfo, chunk]);
                        }
                        callback();
                    };
                }
                const tail = chunk.slice(left);
                /* The decipher_stream uses the `AuthTag` event to flush the accumulated frame.
                 * This is because ciphertext should never be returned until it is verified.
                 * i.e. the auth tag checked.
                 * This can create an issue if the chucks and frame size are small.
                 * If the verify stream continues processing and sends the next auth tag,
                 * before the current auth tag has been completed.
                 * This is basically a back pressure issue.
                 * Since the frame size, and consequently the high water mark,
                 * can not be know when the stream is created,
                 * the internal stream state would need to be modified.
                 * I assert that a simple callback is a simpler way to handle this.
                 */
                const next = () => this._transform(tail, enc, callback);
                return this.emit('AuthTag', finalAuthTagBuffer, next);
            }
        }
        callback();
    }
    push(chunk, encoding) {
        // Typescript???? this._verify instanceof Verify is better....
        if (this._verify && chunk) {
            this._verify.update(chunk);
        }
        return super.push(chunk, encoding);
    }
    _flush(callback) {
        const { finalAuthTagReceived } = this._verifyState;
        /* Precondition: All ciphertext MUST have been received.
         * The verify stream has ended,
         * there will be no more data.
         * Therefore we MUST have reached the end.
         */
        if (!finalAuthTagReceived)
            return callback(new Error('Incomplete message'));
        /* Check for early return (Postcondition): If there is no verify stream do not attempt to verify. */
        if (!this._verify)
            return callback();
        try {
            const { signatureInfo } = this._verifyState;
            /* Precondition: The signature must be well formed. */
            const { buffer, byteOffset, byteLength } = deserializeSignature(signatureInfo);
            const signature = Buffer.from(buffer, byteOffset, byteLength);
            const isVerified = this._verify.awsCryptoVerify(signature);
            /* Postcondition: The signature must be valid. */
            needs(isVerified, 'Invalid Signature');
            callback();
        }
        catch (e) {
            callback(e);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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