"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class RollupService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "putRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          seqNo,
          primaryTerm
        } = request.query;
        let method = "ism.putRollup";
        let params = {
          rollupId: id,
          if_seq_no: seqNo,
          if_primary_term: primaryTerm,
          body: JSON.stringify(request.body)
        };

        if (seqNo === undefined || primaryTerm === undefined) {
          method = "ism.createRollup";
          params = {
            rollupId: id,
            body: JSON.stringify(request.body)
          };
        }

        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const putRollupResponse = await callWithRequest(method, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: putRollupResponse
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - putRollup", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "deleteRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const deleteRollupResponse = await callWithRequest("ism.deleteRollup", params);

        if (deleteRollupResponse.result !== "deleted") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: deleteRollupResponse.result
            }
          });
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: true
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - deleteRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "startRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const startResponse = await callWithRequest("ism.startRollup", params);

        const acknowledged = _lodash.default.get(startResponse, "acknowledged");

        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to start rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - startRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "stopRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const stopResponse = await callWithRequest("ism.stopRollup", params);

        const acknowledged = _lodash.default.get(stopResponse, "acknowledged");

        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to stop rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - stopRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getResponse = await callWithRequest("ism.getRollup", params);
        const metadata = await callWithRequest("ism.explainRollup", params);

        const rollup = _lodash.default.get(getResponse, "rollup", null);

        const seqNo = _lodash.default.get(getResponse, "_seq_no");

        const primaryTerm = _lodash.default.get(getResponse, "_primary_term"); //Form response


        if (rollup) {
          if (metadata) {
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  _id: id,
                  _seqNo: seqNo,
                  _primaryTerm: primaryTerm,
                  rollup: rollup,
                  metadata: metadata
                }
              }
            });
          } else return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load metadata"
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - getRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getMappings", async (context, request, response) => {
      try {
        const {
          index
        } = request.body;
        const params = {
          index: index
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const mappings = await callWithRequest("indices.getMapping", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: mappings
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - getMapping:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getRollups", async (context, request, response) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField
        } = request.query;
        const rollupSortMap = {
          _id: "rollup.rollup_id.keyword",
          "rollup.source_index": "rollup.source_index.keyword",
          "rollup.target_index": "rollup.target_index.keyword",
          "rollup.rollup.enabled": "rollup.enabled"
        };
        const params = {
          from: parseInt(from, 10),
          size: parseInt(size, 10),
          search,
          sortField: rollupSortMap[sortField] || rollupSortMap._id,
          sortDirection
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getRollupResponse = await callWithRequest("ism.getRollups", params);
        const totalRollups = getRollupResponse.total_rollups;
        const rollups = getRollupResponse.rollups.map(rollup => ({
          _seqNo: rollup._seqNo,
          _primaryTerm: rollup._primaryTerm,
          _id: rollup._id,
          rollup: rollup.rollup,
          metadata: null
        })); // Call getExplain if any rollup job exists

        if (totalRollups) {
          // Concat rollup job ids
          const ids = rollups.map(rollup => rollup._id).join(",");
          const explainResponse = await callWithRequest("ism.explainRollup", {
            rollupId: ids
          });

          if (!explainResponse.error) {
            rollups.map(rollup => {
              rollup.metadata = explainResponse[rollup._id];
            });
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  rollups: rollups,
                  totalRollups: totalRollups,
                  metadata: explainResponse
                }
              }
            });
          } else return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: explainResponse ? explainResponse.error : "An error occurred when calling getExplain API."
            }
          });
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              rollups: rollups,
              totalRollups: totalRollups,
              metadata: {}
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                rollups: [],
                totalRollups: 0,
                metadata: null
              }
            }
          });
        }

        console.error("Index Management - RollupService - getRollups", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: "Error in getRollups " + err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }
  /**
   * Calls backend Put Rollup API
   */


}

exports.default = RollupService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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