"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _constants = require("../utils/constants");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class CorrelationService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "createCorrelationRule", async (_context, request, response) => {
      try {
        const params = {
          body: request.body
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const createRulesResponse = await callWithRequest(_constants.CLIENT_CORRELATION_METHODS.CREATE_CORRELATION_RULE, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: createRulesResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - CorrelationService - createCorrelationRule:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "updateCorrelationRule", async (_context, request, response) => {
      try {
        const {
          ruleId
        } = request.params;
        const params = {
          body: request.body,
          ruleId
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const createRulesResponse = await callWithRequest(_constants.CLIENT_CORRELATION_METHODS.UPDATE_CORRELATION_RULE, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: createRulesResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - CorrelationService - updateCorrelationRule:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "getCorrelationRules", async (_context, request, response) => {
      try {
        const {
          query
        } = request.body;
        const params = {
          body: {
            from: 0,
            size: 10000,
            query
          }
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getCorrelationsResponse = await callWithRequest(_constants.CLIENT_CORRELATION_METHODS.GET_CORRELATION_RULES, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: getCorrelationsResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - CorrelationService - getCorrelationRules:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "deleteCorrelationRule", async (_context, request, response) => {
      try {
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const deleteCorrelationRuleResponse = await callWithRequest(_constants.CLIENT_CORRELATION_METHODS.DELETE_CORRELATION_RULE, request.params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: deleteCorrelationRuleResponse
          }
        });
      } catch (error) {
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "getCorrelatedFindings", async (_context, request, response) => {
      try {
        const {
          finding,
          detector_type,
          nearby_findings = 20
        } = request.query;
        const params = {
          finding,
          detector_type,
          nearby_findings
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getCorrelationFindingsResponse = await callWithRequest(_constants.CLIENT_CORRELATION_METHODS.GET_CORRELATED_FINDINGS, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: getCorrelationFindingsResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - CorrelationService - getCorrelatedFindings:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "getAllCorrelationsInTimeRange", async (_context, request, response) => {
      try {
        const {
          start_time,
          end_time
        } = request.query;
        const params = {
          start_timestamp: start_time,
          end_timestamp: end_time
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getCorrelationsResponse = await callWithRequest(_constants.CLIENT_CORRELATION_METHODS.GET_ALL_CORRELATIONS, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: getCorrelationsResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - CorrelationService - getAllCorrelationsInTimeRange:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }

}

exports.default = CorrelationService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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