// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { immutableClass, KeyringTraceFlag, needs, readOnlyProperty, } from '@aws-crypto/material-management';
import { constructArnInOtherRegion, isMultiRegionAwsKmsArn, parseAwsKmsKeyArn, } from './arn_parsing';
import { decrypt, KMS_PROVIDER_ID } from './helpers';
export function AwsKmsMrkAwareSymmetricDiscoveryKeyringClass(BaseKeyring) {
    class AwsKmsMrkAwareSymmetricDiscoveryKeyring
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.5
    //# MUST implement that AWS Encryption SDK Keyring interface (../keyring-
    //# interface.md#interface)
     extends BaseKeyring {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
        //# On initialization the caller MUST provide:
        constructor({ client, grantTokens, discoveryFilter, }) {
            super();
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# The keyring MUST know what Region the AWS KMS client is in.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# It
            //# SHOULD obtain this information directly from the client as opposed to
            //# having an additional parameter.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# However if it can not, then it MUST
            //# NOT create the client itself.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# It SHOULD have a Region parameter and
            //# SHOULD try to identify mismatched configurations.
            //
            // @ts-ignore the V3 client has set the config to protected
            const clientRegion = client.config.region;
            needs(clientRegion, 'Client must be configured to a region.');
            /* Precondition: The AwsKmsMrkAwareSymmetricDiscoveryKeyring Discovery filter *must* be able to match something.
             * I am not going to wait to tell you
             * that no CMK can match an empty account list.
             * e.g. [], [''], '' are not valid.
             */
            needs(!discoveryFilter ||
                (discoveryFilter.accountIDs &&
                    discoveryFilter.accountIDs.length &&
                    !!discoveryFilter.partition &&
                    discoveryFilter.accountIDs.every((a) => typeof a === 'string' && !!a)), 'A discovery filter must be able to match something.');
            readOnlyProperty(this, 'client', client);
            readOnlyProperty(this, 'clientRegion', clientRegion);
            readOnlyProperty(this, 'grantTokens', grantTokens);
            readOnlyProperty(this, 'discoveryFilter', discoveryFilter
                ? Object.freeze({
                    ...discoveryFilter,
                    accountIDs: Object.freeze(discoveryFilter.accountIDs),
                })
                : discoveryFilter);
        }
        async _onEncrypt() {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.7
            //# This function MUST fail.
            throw new Error('AwsKmsMrkAwareSymmetricDiscoveryKeyring cannot be used to encrypt');
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# OnDecrypt MUST take decryption materials (structures.md#decryption-
        //# materials) and a list of encrypted data keys
        //# (structures.md#encrypted-data-key) as input.
        async _onDecrypt(material, encryptedDataKeys) {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# If the decryption materials (structures.md#decryption-materials)
            //# already contained a valid plaintext data key OnDecrypt MUST
            //# immediately return the unmodified decryption materials
            //# (structures.md#decryption-materials).
            if (material.hasValidKey())
                return material;
            const { client, grantTokens, clientRegion } = this;
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# The set of encrypted data keys MUST first be filtered to match this
            //# keyring's configuration.
            const decryptableEDKs = encryptedDataKeys.filter(filterEDKs(this));
            const cmkErrors = [];
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# For each encrypted data key in the filtered set, one at a time, the
            //# OnDecrypt MUST attempt to decrypt the data key.
            for (const edk of decryptableEDKs) {
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                //# Otherwise it MUST
                //# be the provider info.
                let keyId = edk.providerInfo;
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                //# *  "KeyId": If the provider info's resource type is "key" and its
                //# resource is a multi-Region key then a new ARN MUST be created
                //# where the region part MUST equal the AWS KMS client region and
                //# every other part MUST equal the provider info.
                const keyArn = parseAwsKmsKeyArn(edk.providerInfo);
                needs(keyArn, 'Unexpected EDK ProviderInfo for AWS KMS EDK');
                if (isMultiRegionAwsKmsArn(keyArn)) {
                    keyId = constructArnInOtherRegion(keyArn, clientRegion);
                }
                let dataKey = false;
                try {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# When calling AWS KMS Decrypt
                    //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html), the keyring MUST call with a request constructed
                    //# as follows:
                    dataKey = await decrypt(
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# To attempt to decrypt a particular encrypted data key
                    //# (structures.md#encrypted-data-key), OnDecrypt MUST call AWS KMS
                    //# Decrypt (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html) with the configured AWS KMS client.
                    client, {
                        providerId: edk.providerId,
                        providerInfo: keyId,
                        encryptedDataKey: edk.encryptedDataKey,
                    }, material.encryptionContext, grantTokens);
                    /* This should be impossible given that decrypt only returns false if the client supplier does
                     * or if the providerId is not "aws-kms", which we have already filtered out
                     */
                    if (!dataKey)
                        continue;
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# *  The "KeyId" field in the response MUST equal the requested "KeyId"
                    needs(dataKey.KeyId === keyId, 'KMS Decryption key does not match the requested key id.');
                    const flags = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
                        KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
                    const trace = {
                        keyNamespace: KMS_PROVIDER_ID,
                        keyName: dataKey.KeyId,
                        flags,
                    };
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# *  The length of the response's "Plaintext" MUST equal the key
                    //# derivation input length (algorithm-suites.md#key-derivation-input-
                    //# length) specified by the algorithm suite (algorithm-suites.md)
                    //# included in the input decryption materials
                    //# (structures.md#decryption-materials).
                    //
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# Since the response does satisfies these requirements then OnDecrypt
                    //# MUST do the following with the response:
                    //
                    // setUnencryptedDataKey will throw if the plaintext does not match the algorithm suite requirements.
                    material.setUnencryptedDataKey(dataKey.Plaintext, trace);
                    return material;
                }
                catch (e) {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# If the response does not satisfies these requirements then an error
                    //# is collected and the next encrypted data key in the filtered set MUST
                    //# be attempted.
                    cmkErrors.push({ errPlus: e });
                }
            }
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# If OnDecrypt fails to successfully decrypt any encrypted data key
            //# (structures.md#encrypted-data-key), then it MUST yield an error that
            //# includes all collected errors.
            needs(material.hasValidKey(), [
                `Unable to decrypt data key${!decryptableEDKs.length ? ': No EDKs supplied' : ''}.`,
                ...cmkErrors.map((e, i) => `Error #${i + 1}  \n${e.errPlus.stack}`),
            ].join('\n'));
            return material;
        }
    }
    immutableClass(AwsKmsMrkAwareSymmetricDiscoveryKeyring);
    return AwsKmsMrkAwareSymmetricDiscoveryKeyring;
}
function filterEDKs({ discoveryFilter, clientRegion, }) {
    return function filter({ providerId, providerInfo }) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  Its provider ID MUST exactly match the value "aws-kms".
        if (providerId !== KMS_PROVIDER_ID)
            return false;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  The provider info MUST be a valid AWS KMS ARN (aws-kms-key-
        //# arn.md#a-valid-aws-kms-arn) with a resource type of "key" or
        //# OnDecrypt MUST fail.
        const edkArn = parseAwsKmsKeyArn(providerInfo);
        needs(edkArn && edkArn.ResourceType === 'key', 'Unexpected EDK ProviderInfo for AWS KMS EDK');
        const { AccountId: account, Partition: partition, Region: edkRegion, } = edkArn;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If the provider info is not identified as a multi-Region key (aws-
        //# kms-key-arn.md#identifying-an-aws-kms-multi-region-key), then the
        //# provider info's Region MUST match the AWS KMS client region.
        if (!isMultiRegionAwsKmsArn(edkArn) && clientRegion !== edkRegion) {
            return false;
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If a discovery filter is configured, its partition and the
        //# provider info partition MUST match.
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If a discovery filter is configured, its set of accounts MUST
        //# contain the provider info account.
        return (!discoveryFilter ||
            (discoveryFilter.partition === partition &&
                discoveryFilter.accountIDs.includes(account)));
    };
}
//# sourceMappingURL=data:application/json;base64,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