#!/usr/bin/env python3
"""
An example of defining reflectometry instrument
for time of flight experiment. In this example
we will use purely qz-defined beam,
without explicitly specifying
incident angle or a wavelength.
Additionally we will set pointwise resolution
to the scan.
Note that these approaches work with SLD-based
materials only.
"""
import numpy as np
import bornagain as ba
from bornagain import ba_plot as bp, std_samples


def get_sample():
    return std_samples.alternating_layers()


def get_simulation(sample):
    "Specular simulation with a qz-defined beam"
    n = 50

    qzs = np.linspace(0.01, 1, n)  # qz-values
    dq = 0.03*qzs
    distr = ba.DistributionGaussian(0., 1., 25, 2.)

    scan = ba.QzScan(qzs)
    scan.setVectorResolution(distr, dq)

    return ba.SpecularSimulation(scan, sample)


if __name__ == '__main__':
    sample = get_sample()
    simulation = get_simulation(sample)
    result = simulation.simulate()
    plotargs = bp.parse_commandline()
    bp.plot_simulation_result(result, **plotargs)
    bp.export(**plotargs)
