/*
 * SPDX-License-Identifier: GPL-2.0-only
 *
 */

// Test program for this 3-router scenario, using static routing
//
// (a.a.a.a/32)A<--x.x.x.0/30-->B<--y.y.y.0/30-->C(c.c.c.c/32)

#include "ns3/applications-module.h"
#include "ns3/core-module.h"
#include "ns3/csma-module.h"
#include "ns3/internet-module.h"
#include "ns3/ipv4-static-routing-helper.h"
#include "ns3/network-module.h"
#include "ns3/point-to-point-module.h"

#include <cassert>
#include <fstream>
#include <iostream>
#include <string>

using namespace ns3;

NS_LOG_COMPONENT_DEFINE("StaticRoutingSlash32Test");

int
main(int argc, char* argv[])
{
    // Allow the user to override any of the defaults and the above
    // DefaultValue::Bind ()s at run-time, via command-line arguments
    CommandLine cmd(__FILE__);
    cmd.Parse(argc, argv);

    Ptr<Node> nA = CreateObject<Node>();
    Ptr<Node> nB = CreateObject<Node>();
    Ptr<Node> nC = CreateObject<Node>();

    NodeContainer c = NodeContainer(nA, nB, nC);

    InternetStackHelper internet;
    internet.Install(c);

    // Point-to-point links
    NodeContainer nAnB = NodeContainer(nA, nB);
    NodeContainer nBnC = NodeContainer(nB, nC);

    // We create the channels first without any IP addressing information
    PointToPointHelper p2p;
    p2p.SetDeviceAttribute("DataRate", StringValue("5Mbps"));
    p2p.SetChannelAttribute("Delay", StringValue("2ms"));
    NetDeviceContainer dAdB = p2p.Install(nAnB);

    NetDeviceContainer dBdC = p2p.Install(nBnC);

    Ptr<CsmaNetDevice> deviceA = CreateObject<CsmaNetDevice>();
    deviceA->SetAddress(Mac48Address::Allocate());
    nA->AddDevice(deviceA);
    deviceA->SetQueue(CreateObject<DropTailQueue<Packet>>());

    Ptr<CsmaNetDevice> deviceC = CreateObject<CsmaNetDevice>();
    deviceC->SetAddress(Mac48Address::Allocate());
    nC->AddDevice(deviceC);
    deviceC->SetQueue(CreateObject<DropTailQueue<Packet>>());

    // Later, we add IP addresses.
    Ipv4AddressHelper ipv4;
    ipv4.SetBase("10.1.1.0", "255.255.255.252");
    Ipv4InterfaceContainer iAiB = ipv4.Assign(dAdB);

    ipv4.SetBase("10.1.1.4", "255.255.255.252");
    Ipv4InterfaceContainer iBiC = ipv4.Assign(dBdC);

    Ptr<Ipv4> ipv4A = nA->GetObject<Ipv4>();
    Ptr<Ipv4> ipv4B = nB->GetObject<Ipv4>();
    Ptr<Ipv4> ipv4C = nC->GetObject<Ipv4>();

    int32_t ifIndexA = ipv4A->AddInterface(deviceA);
    int32_t ifIndexC = ipv4C->AddInterface(deviceC);

    Ipv4InterfaceAddress ifInAddrA =
        Ipv4InterfaceAddress(Ipv4Address("172.16.1.1"), Ipv4Mask("/32"));
    ipv4A->AddAddress(ifIndexA, ifInAddrA);
    ipv4A->SetMetric(ifIndexA, 1);
    ipv4A->SetUp(ifIndexA);

    Ipv4InterfaceAddress ifInAddrC =
        Ipv4InterfaceAddress(Ipv4Address("192.168.1.1"), Ipv4Mask("/32"));
    ipv4C->AddAddress(ifIndexC, ifInAddrC);
    ipv4C->SetMetric(ifIndexC, 1);
    ipv4C->SetUp(ifIndexC);

    Ipv4StaticRoutingHelper ipv4RoutingHelper;
    // Create static routes from A to C
    Ptr<Ipv4StaticRouting> staticRoutingA = ipv4RoutingHelper.GetStaticRouting(ipv4A);
    // The ifIndex for this outbound route is 1; the first p2p link added
    staticRoutingA->AddHostRouteTo(Ipv4Address("192.168.1.1"), Ipv4Address("10.1.1.2"), 1);
    Ptr<Ipv4StaticRouting> staticRoutingB = ipv4RoutingHelper.GetStaticRouting(ipv4B);
    // The ifIndex we want on node B is 2; 0 corresponds to loopback, and 1 to the first point to
    // point link
    staticRoutingB->AddHostRouteTo(Ipv4Address("192.168.1.1"), Ipv4Address("10.1.1.6"), 2);
    // Create the OnOff application to send UDP datagrams of size
    // 210 bytes at a rate of 448 Kb/s
    uint16_t port = 9; // Discard port (RFC 863)
    OnOffHelper onoff("ns3::UdpSocketFactory",
                      Address(InetSocketAddress(ifInAddrC.GetLocal(), port)));
    onoff.SetConstantRate(DataRate(6000));
    ApplicationContainer apps = onoff.Install(nA);
    apps.Start(Seconds(1));
    apps.Stop(Seconds(10));

    // Create a packet sink to receive these packets
    PacketSinkHelper sink("ns3::UdpSocketFactory",
                          Address(InetSocketAddress(Ipv4Address::GetAny(), port)));
    apps = sink.Install(nC);
    apps.Start(Seconds(1));
    apps.Stop(Seconds(10));

    AsciiTraceHelper ascii;
    p2p.EnableAsciiAll(ascii.CreateFileStream("static-routing-slash32.tr"));
    p2p.EnablePcapAll("static-routing-slash32");

    Simulator::Run();
    Simulator::Destroy();

    return 0;
}
